/* phyport match
 *
 * Author:
 * Yabo Wang <wyblyf@gmail.com>
 *
 * May 2012
 */
#include <stdio.h>
#include <netdb.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <ctype.h>
#include "../include/ebtables_u.h"
#include "../include/ethernetdb.h"
#if defined(__GLIBC__) && __GLIBC__ == 2
#include <net/ethernet.h>
#else
#include <linux/if_ether.h>
#endif
#include <linux/netfilter/xt_phyport.h>

#define PORT_SOURCE		'1'
#define PORT_DESTINATION	'2'

static void phyport_help(void)
{
	printf(
"phyport match options:\n"
"[!] --port-source input number(1-6)\n"
"				Match source port number\n"
" --port-destination output number(1-6)\n"
"				Match destination port number\n");
}

static const struct option phyport_opts[] = {
	{ "port-source", required_argument, 0, PORT_SOURCE },
	{ "port-destination", required_argument, 0, PORT_DESTINATION },
	{ 0 }
};

static void
parse_phyportnum(const char *phyportnum, struct xt_phyport_info *info)
{
	long number;
	char *end;

	if(phyportnum==NULL)
		ebt_print_error("phy port number pointer is null");
	
	number=strtol(phyportnum, &end, 10);

	if((number<0)||(number>255))
		ebt_print_error("Bad phy port number `%s'", phyportnum);

	if(info->flags & PORT_SRC){
		info->srcport=(unsigned char)number;
	}
	else if (info->flags & PORT_DST){
		info->dstport=(unsigned char)number;
	}
	else{
		ebt_print_error("wrong info->flags: 0x%x", info->flags);
	}	
}
static void
phyport_init(struct ebt_entry_match *m)
{
}
static int
phyport_parse(int c, char **argv, int argc,
	        const struct ebt_u_entry *entry, unsigned int *flags,
	        struct ebt_entry_match **match)
{
	struct xt_phyport_info *phyportinfo = (struct xt_phyport_info *)((*match)->data);
        switch (c) {
        case PORT_SOURCE:
                ebt_check_option2(flags, PORT_SRC);
                phyportinfo->flags |= PORT_SRC;
				
				if (ebt_check_inverse2(optarg))
                    phyportinfo->flags |= PORT_SRC_INV;
                
                parse_phyportnum(optarg, phyportinfo);
                break;

        case PORT_DESTINATION:
				ebt_check_option2(flags, PORT_DST);
				phyportinfo->flags |= PORT_DST;
				
				if (ebt_check_inverse2(optarg))
					phyportinfo->flags |= PORT_DST_INV;
				
				parse_phyportnum(optarg, phyportinfo);
				break;

        default:
                return 0;
        }

        return 1;
}

static void print_phyportnum(struct xt_phyport_info *info)
{
	if(info->flags & PORT_SRC){
		printf("%d", info->srcport);
	}
	else if (info->flags & PORT_DST){
		printf("%d", info->dstport);
	}
	else{
		ebt_print_error("wrong info->flags: 0x%x", info->flags);
	}
		
	printf(" ");
}

static void phyport_check(const struct ebt_u_entry *entry,
	   const struct ebt_entry_match *match,
	   const char *name, unsigned int hookmask, unsigned int time)
{
	//if (!flags)
	//	exit_error(PARAMETER_PROBLEM,
	//		   "You must specify `--port-source' or `--port-destination'");
}

static void
phyport_print(const struct ebt_u_entry *entry,
	   const struct ebt_entry_match *match)
{
	struct xt_phyport_info *phyportinfo = (struct xt_phyport_info *)(match->data);

        if (phyportinfo->flags & PORT_SRC) {
                printf("source port  ");
                if (phyportinfo->flags & PORT_SRC_INV)
                        printf("! ");
                print_phyportnum(phyportinfo);
        }
        if (phyportinfo->flags & PORT_DST) {
                printf("destination port ");
                if (phyportinfo->flags & PORT_DST_INV)
                        printf("! ");
                print_phyportnum(phyportinfo);
        }
}

static void phyport_compare(const struct ebt_entry_match *m1,
   const struct ebt_entry_match *m2)
{
	struct xt_phyport_info *phyportinfo = (struct xt_phyport_info *)(m2->data);

        if (phyportinfo->flags & PORT_SRC) {
                if (phyportinfo->flags & PORT_SRC_INV)
                        printf("! ");
                printf("--port-source ");
                print_phyportnum(phyportinfo);
                if (phyportinfo->flags & PORT_DST)
                        fputc(' ', stdout);
        }
        if (phyportinfo->flags & PORT_DST) {
                if (phyportinfo->flags & PORT_DST_INV)
                        printf("! ");
                printf("--port-destination ");
                print_phyportnum(phyportinfo);
        }
}

static struct ebt_u_match phyport_match = {
 	.name		= "phyport",
	.size		= sizeof(struct xt_phyport_info),
	.help		= phyport_help,
	.init		= phyport_init,
	.parse		= phyport_parse,
	.final_check	= phyport_check,
	.print		= phyport_print,
	.compare		= phyport_compare,
	.extra_ops	= phyport_opts,
};
void _init(void)
{
	ebt_register_match(&phyport_match);
}
