
// Copyright 2005, Texas Instruments Incorporated
//
// This program has been modified from its original operation by Texas Instruments
// to do the following:
//
// 1. Prevent caching when AAAA query responses are received and don't lookup cache
//    when AAAA queries are received from clients
// 2. Allow error response with rsp code 1 & 3 to client
// 3. Cache only if a valid address or name is received in the response. There was
//    a possibility of caching incorrect values in ip and cname variables  if the
//    response is success, but there was no name or host address in the response
// 4. Use port 53 only for listening to DNS request and to send error response to
//    the clients. Use ephemeral port for forwarding request to server. Also
//    use ephemeral port for listening to responses from DNS server
// 5. NSP Policy Routing Framework
//
// THIS MODIFIED SOFTWARE AND DOCUMENTATION ARE PROVIDED
// "AS IS," AND TEXAS INSTRUMENTS MAKES NO REPRESENTATIONS
// OR WARRENTIES, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO, WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
// PARTICULAR PURPOSE OR THAT THE USE OF THE SOFTWARE OR
// DOCUMENTATION WILL NOT INFRINGE ANY THIRD PARTY PATENTS,
// COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
// See The GNU General Public License for more details.
//
// These changes are covered under version 2 of the GNU General Public License,
// dated June 1991.
//-------------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <string.h>
#include <arpa/inet.h>
#include <errno.h>
#include <stdarg.h>
#include <signal.h>
#include <syslog.h>
#include <sys/stat.h>

#include "dproxy.h"
#include "dns_decode.h"
#include "cache.h"
#include "new_conf.h"
#include "dns_list.h"
#include "dns_construct.h"
#include "dns_io.h"
//#include <linux/pr.h>
/*****************************************************************************/
/*****************************************************************************/
int dns_main_quit;
int dns_sock;
int dns_any_sock;
fd_set rfds;
dns_request_t *dns_request_list;
struct stat conf_stat;

int g_PrimaryTimeout = 0;
int g_PrimaryErrName = 0;

/*****************************************************************************/
int is_connected()
{
	return 1;
}
/*****************************************************************************/
int dns_init()
{
    struct sockaddr_in sa;
    struct in_addr ip;

    //unsigned int pr_mark = DPROXY_PR_MARK;

    /* Clear it out */
    memset((void *)&sa, 0, sizeof(sa));

    dns_sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);

    /* Error */
    if( dns_sock < 0 ){
        debug_perror("Could not create dns socket");
        exit(1);
    }

    ip.s_addr = INADDR_ANY;
    sa.sin_family = AF_INET;
    memcpy((void *)&sa.sin_addr, (void *)&ip, sizeof(struct in_addr));
    sa.sin_port = htons(PORT);

    /* bind() the socket to the interface */
    if (bind(dns_sock, (struct sockaddr *)&sa, sizeof(struct sockaddr)) < 0){
        debug_perror("dns_init: bind: Could not bind to port");
        exit(1);
    }

    /* Use ANY port to forward request to the DNS server */
    /* Clear it out */
    memset((void *)&sa, 0, sizeof(sa));

    dns_any_sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);

    /* Error */
    if( dns_any_sock < 0 ){
        debug_perror("Could not create dns any socket");
        exit(1);
    }

    ip.s_addr = INADDR_ANY;
    sa.sin_family = AF_INET;
    memcpy((void *)&sa.sin_addr, (void *)&ip, sizeof(struct in_addr));
    sa.sin_port = 0; /* Use any port */

    /* bind() the socket to the interface */
    if (bind(dns_any_sock, (struct sockaddr *)&sa, sizeof(struct sockaddr)) < 0){
        debug_perror("dns_init: bind: Could not bind to ANY port");
        exit(1);
    }

#if 0
    /* Adding setsockopt to set a mark value for a particular socket. */
    if (setsockopt(dns_any_sock, SOL_IP, POLICY_ROUTING_MARK, (void *)&pr_mark, sizeof(pr_mark)) != 0)
    {
        printf ("Failed to set PR mark for socket %d\n", dns_any_sock);
    }
#endif

    dns_main_quit = 0; 
    FD_ZERO( &rfds );
    FD_SET( dns_sock, &rfds );
    FD_SET( dns_any_sock, &rfds );

    dns_request_list = NULL;

    cache_purge(g_stConfig.nPurgeTime);	
	cache_errpurge(g_stConfig.nPurgeTime/60);

    return 1;
}
/*****************************************************************************/
void dns_handle_new_query(dns_request_t *m)
{
    struct in_addr in;
    int retval = -1;

	//m->pNodeڵΪգ˵ǰ湹첻ȫܼ
	if( m->pNode == NULL )
	{
		debug("NULL pNode\n" );
		return;
	}

	/* query timeout cache name, if hit it, return success */
	if (cache_lookup_errname( m->cname) && ( m->message.question[0].type != AAA )&&( m->message.question[0].type != SRV))
	{
		debug("hit errname, return\n");
		return;
	}
	/* end */
	
    if( m->message.question[0].type == A){
        /* standard query */
    retval = cache_lookup_name( m->cname, m->ip );
    }
	#if 0
	else if( m->message.question[0].type == AAA ){
        /* IPv6 AAAA query in IPv4 */
        /* This query is not cached */
        retval = 0 ;
    }
	#endif
	else if( m->message.question[0].type == SRV){
        /* ֧SRV͸*/
        retval = 0 ;
    }else if( m->message.question[0].type == PTR ){
        /* reverse lookup */
    retval = cache_lookup_ip( m->ip, m->cname );
    }

    debug(".......... %s ---- %s\n", m->cname, m->ip );

    switch( retval )
    {
        case 0:
            if( is_connected() ){
                debug("Adding to request list-> id: %d\n", m->message.header.id);
                dns_request_list = dns_list_add( dns_request_list, m );
                /*!!! relay the query untouched */

                //inet_aton( g_stConfig.pWanInfo->szDnsIp[0], &in );
				inet_aton(m->pNode->szDnsIp[m->ucIndex], &in );
                debug("Sent Request To %s\n",m->pNode->szDnsIp[m->ucIndex]);
                dns_write_packet( dns_any_sock, in, PORT, m, 1);
            }else{
                debug("Not connected **\n");
                dns_construct_error_reply(m);
                dns_write_packet( dns_sock, m->src_addr, m->src_port, m, 0 );
            }
            break;
        case 1:
            dns_construct_reply( m );
            dns_write_packet( dns_sock, m->src_addr, m->src_port, m, 0 );
            debug("Cache hit -  name %s -ip %s\n", m->cname, m->ip);
            break;
        default:
            debug("Unknown query type: %d\n", m->message.question[0].type );
    }

}

void dns_ExchangeMasterSlave()
{
	char pIp[IP_LEN] = {0};
	ST_LAN_INFO* pLanInfo;
	
	for (pLanInfo = g_stConfig.pLanInfo; pLanInfo; pLanInfo = pLanInfo->pNext)
	{
		strncpy(pIp, pLanInfo->szDnsIp[0], IP_LEN);
		strncpy(pLanInfo->szDnsIp[0], pLanInfo->szDnsIp[1], IP_LEN);
		strncpy(pLanInfo->szDnsIp[1], pIp, IP_LEN);

		debug("pLanInfo->szDnsIp[0] = %s, pLanInfo->szDnsIp[1] = %s\n", pLanInfo->szDnsIp[0], pLanInfo->szDnsIp[1]);
	}
}

/*****************************************************************************/
void dns_handle_request(dns_request_t *m)
{
    dns_request_t *ptr = NULL;
    int forward_resp=0;
	char *szDnsIp = NULL;
	int firstDnsCmp = 0;
    struct in_addr in;
	
    /* request may be a new query or a answer from the upstream server */
    ptr = dns_list_find_by_id( dns_request_list, m );

    if( ptr != NULL )
    {
        debug("Found query in list\n");
        /* message may be a response */
        if( m->message.header.flags.f.question == 1 )
        {
			szDnsIp = inet_ntoa(m->src_addr);			
			firstDnsCmp = strncmp(szDnsIp, ptr->pNode->szDnsIp[0], IP_LEN);
			debug("########szDnsIp=%s, ptr->pNode->szDnsIp[0]=%s, firstDnsCmp = %d##########\n", 
				  szDnsIp, ptr->pNode->szDnsIp[0], firstDnsCmp);
            switch( m->message.header.flags.f.rcode )
            {
                case DNS_NO_ERROR:
                    {
                        /* Append to the cache if not a AAA query and cache flag indicates
                           something available for cache
                         */
                        if((m->message.question[0].type != AAA ) &&( m->message.question[0].type != SRV) &&
                                (m->cache))
                        {
                            debug("Cache append: %s ----> %s\n", m->cname, m->ip );

							cache_name_append( m->cname, m->ip );
							
							/* when message from slave server, and master server timeout,
							   exchange of master-salve position */
							if (g_PrimaryTimeout)
							{
								debug("start dns_ExchangeMasterSlave..\n ");
								dns_ExchangeMasterSlave();
							}		
						
                        }
                        forward_resp=1;
                        break;
                    }
                case DNS_FMT_ERROR:
                    {
						if (g_PrimaryErrName && (m->message.question[0].type != AAA ) &&( m->message.question[0].type != SRV) && firstDnsCmp)
						{
							debug("DNS_FMT_ERROR, start add szErrCacheFile record(%s)..\n ", m->cname);
							//cache_errname_append(m->cname);								
							g_PrimaryErrName = 0;
						}
                        forward_resp=1;
                        break;
                    }
                case DNS_NAME_ERR:
                    {
						if ((g_PrimaryTimeout || g_PrimaryErrName) && (m->message.question[0].type != AAA ) &&( m->message.question[0].type != SRV) && firstDnsCmp)
						{
							debug("DNS_NAME_ERR, start add szErrCacheFile record(%s)..\n ", m->cname);
							//cache_errname_append(m->cname);					
						}

						if ((0 == g_PrimaryErrName) && (m->message.question[0].type != AAA ) &&( m->message.question[0].type != SRV))
							g_PrimaryErrName = 1;
						else if ((1 == g_PrimaryErrName) && (m->message.question[0].type != AAA ) &&( m->message.question[0].type != SRV))
							g_PrimaryErrName = 0;
						
/*յDNSصİNo such nameȨ״̬Ϊ1ʱȻDNSmodify by yangwei 20130731*/
#if 0
                        if(m->message.header.flags.f.authorative)
                        {
                            forward_resp=1;
                        }
#endif                        
                        break;

                    }
                default:
					if (g_PrimaryErrName && (m->message.question[0].type != AAA ) &&( m->message.question[0].type != SRV) && firstDnsCmp)
					{
						debug("default, start add szErrCacheFile record(%s)..\n ", m->cname);
						//cache_errname_append(m->cname);
						g_PrimaryErrName = 0;
					}
                    /* do nothing */
                    break;
            }
            if (forward_resp)
            {
                //dns_write_packet( dns_any_sock, ptr->src_addr, ptr->src_port, m );

                dns_write_packet( dns_sock, ptr->src_addr, ptr->src_port, m, 0 );
                debug("Rsp Code: %d - Replying with answer from %s\n",(int)(m->message.header.flags.f.rcode),inet_ntoa( m->src_addr ));
                dns_request_list = dns_list_remove( dns_request_list, ptr );
            }
            else
            {
                debug("Rsp Code: %d - answer from %s dropped\n",(int)(m->message.header.flags.f.rcode),inet_ntoa( m->src_addr ));
                ++ptr->ucIndex;
                if(ptr->ucIndex > 2||
                            (ptr->pNode->szDnsIp[ptr->ucIndex][0] == '\0' ))
                {
                    /* send error back */
                    debug("All Servers Tried: No Response\n");
                    dns_construct_error_reply(ptr);
                    dns_write_packet( dns_sock, ptr->src_addr, ptr->src_port, ptr, 0 );
                    dns_request_list = dns_list_remove( dns_request_list, ptr );
                    debug("Sent a DNS error message to Client \n");
                }
                else
                {
                    /* create the new server ip address and relay the query untouched */
                    inet_aton( ptr->pNode->szDnsIp[ptr->ucIndex], &in );
                    /*!!!*/
                    debug("sending a new request to %s\n", ptr->pNode->szDnsIp[ptr->ucIndex]);

                    /* send the new request to the next name server */
                    dns_write_packet( dns_any_sock, in, PORT, ptr, 1 );
                }
            }
			
			g_PrimaryTimeout = 0;
        }
        else
        {
            debug("Duplicate query\n");
        }
		
    }
    else
    {
        dns_handle_new_query( m );
    }

}
/*****************************************************************************/
int dns_main_loop()
{
    struct timeval tv;
    fd_set active_rfds;
    int retval;
    dns_request_t m;
    dns_request_t *ptr, *next;
    int purge_time = g_stConfig.nPurgeTime / 60;
	int errpurge_time = g_stConfig.nPurgeTime / 3600;
    struct stat temp_stat ;

    //int next_server_index = 0;
    struct in_addr in;

    while( !dns_main_quit ){

        /* set the one second time out */
        tv.tv_sec = 1;
        tv.tv_usec = 0;

        /* now copy the main rfds in the active one as it gets modified by select*/
        active_rfds = rfds;

        retval = select( FD_SETSIZE, &active_rfds, NULL, NULL, &tv );

        if (retval)
        {
            /* data is now available */
            if (FD_ISSET(dns_sock,  &active_rfds))
            {
                /* -2˵dproxyļҲlanڵ */
                if (-2 == dns_read_packet( dns_sock, &m, 1))
                {
                    dns_construct_error_reply(&m);
                    dns_write_packet( dns_sock, m.src_addr, m.src_port, &m, 0 );
                }
                else
                {
                    dns_handle_request( &m );
                }
            }
            else
            {
                if (FD_ISSET(dns_any_sock,  &active_rfds))
                {
                    dns_read_packet( dns_any_sock, &m, 0);
                    dns_handle_request( &m );
                }
            }
            m.time_pending = 0;
        }
        else
        {
            /* If resolv.conf has changed read it again*/
            if( !stat(g_stConfig.szConfigFile, &temp_stat))
            {
                if( conf_stat.st_ctime != temp_stat.st_ctime)
                {
                    load_config(g_stConfig.szConfigFile);
                    conf_stat = temp_stat;
                }
            }
            /* select time out */
            ptr = dns_request_list;
            while( ptr ){
                next = ptr->next;
                /* Resend query to a new nameserver if response
                 * has not been received from the current nameserver */

                ptr->time_pending++;
                if( ptr->time_pending > DNS_TIMEOUT ){
                    debug("Request timed out\n");
                    /* send error back */
                    dns_construct_error_reply(ptr);
                    dns_write_packet( dns_sock, ptr->src_addr, ptr->src_port, ptr, 0 );
                    dns_request_list = dns_list_remove( dns_request_list, ptr );
                }
                if( ( ptr && (ptr->time_pending % DNS_SERVER_TIMEOUT == 0)) &&
                        ( ptr->time_pending < DNS_TIMEOUT) )
                {
                    ++ptr->ucIndex;
                    if(ptr->ucIndex > NUM_OF_DNS||
                            (ptr->pNode->szDnsIp[ptr->ucIndex][0] == '\0' ))
                    {
                        /* send error back */
                        debug("All Servers Tried: No Response\n");
                        dns_construct_error_reply(ptr);
                        dns_write_packet( dns_sock, ptr->src_addr, ptr->src_port, ptr, 0 );
                        dns_request_list = dns_list_remove( dns_request_list, ptr );
                        debug("Sent a DNS error message to Client \n");
                    }
                    else
                    {
						/* record primary timeout state */
						//printf("get response from master server timeout \n");
						if ((1 == ptr->ucIndex) && ( ptr->message.question[0].type != AAA ) &&( ptr->message.question[0].type != SRV))
							g_PrimaryTimeout = 1;	/* master server index */
						
                        /* Create the new server IP address and relay the query untouched */
                        inet_aton( ptr->pNode->szDnsIp[ptr->ucIndex], &in );
                        /*!!!*/
                        debug("Didn't get a response from last server\n");
                        debug("Sending a new request to %s\n", ptr->pNode->szDnsIp[ptr->ucIndex]);

						/* query timeout cache name, if hit it, return success */
						if (cache_lookup_errname( ptr->cname) && ( ptr->message.question[0].type != AAA ) &&( ptr->message.question[0].type != SRV))
						{
							debug("hit errname, slave server also couldn't send packet\n");
						}
						else
						{
                        	/* Send the new request to the next name server */
                        	dns_write_packet( dns_any_sock, in, PORT, ptr, 1 );
						}
                    }
                }
                ptr = next;
            } /* while(ptr) */

            /* purge cache */
            purge_time--;
            if( !purge_time ){
                cache_purge(g_stConfig.nPurgeTime);
                purge_time = g_stConfig.nPurgeTime / 60;
            }

			/* purge cache */
            errpurge_time--;
            if( !errpurge_time ){
                cache_errpurge(g_stConfig.nPurgeTime/60);
                errpurge_time = g_stConfig.nPurgeTime / 3600;
            }

        } /* if (retval) */
    }
    return 0;
}

#if 0
/*****************************************************************************/
void debug_perror( char * msg ) {
    debug( "%s : %s\n" , msg , strerror(errno) );
}
/*****************************************************************************/
void debug(char *fmt, ...)
{
#define MAX_MESG_LEN 1024

    va_list args;
    char text[ MAX_MESG_LEN ];

    sprintf( text, "[ %d ]: ", getpid());
    va_start (args, fmt);
    vsnprintf( &text[strlen(text)], MAX_MESG_LEN - strlen(text), fmt, args);
    va_end (args);

    if( config.debug_file[0] ){
        FILE *fp;
        fp = fopen( config.debug_file, "a");
        if(!fp){
            syslog( LOG_ERR, "could not open log file %m" );
            return;
        }
        fprintf( fp, "%s", text);
        fclose(fp);
    }

    /** if not in daemon-mode stderr was not closed, use it. */
    if( ! config.daemon_mode ) {
        fprintf( stderr, "%s", text);
    }
}
#endif

/*****************************************************************************
 * print usage informations to stderr.
 *
 *****************************************************************************/
void usage(char * program , char * message ) {
    fprintf(stderr,"%s\n" , message );
    fprintf(stderr,"usage : %s [-c <config-file>] [-d] [-h] [-P]\n", program );
    fprintf(stderr,"\t-c <config-file>\tread configuration from <config-file>\n");
    fprintf(stderr,"\t-d \t\trun in debug (=non-daemon) mode.\n");
    fprintf(stderr,"\t-P \t\tprint configuration on stdout and exit.\n");
    fprintf(stderr,"\t-h \t\tthis message.\n");
}
/*****************************************************************************
 * get commandline options.
 *
 * @return 0 on success, < 0 on error.
 *****************************************************************************/
int get_options( int argc, char ** argv )
{
    int c = 0;
    int not_daemon = 0;
    int want_printout = 0;
    char * progname = argv[0];

    while( (c = getopt( argc, argv, "c:dhP")) != -1 ) {
        switch(c) {
            case 'c':
				if (0 > load_config(optarg))
				{
					//printf("load_config error\n");
					return -1;
				}
                if(g_stConfig.szConfigFile)
                    stat(g_stConfig.szConfigFile, &conf_stat);
                break;
            case 'd':
                not_daemon = 1;
                break;
            case 'h':
                usage(progname,"");
                return -1;
            case 'P':
                want_printout = 1;
                break;
            default:
                usage(progname,"");
                return -1;
        }
    }

    /** unset daemon-mode if -d was given. */
    if( not_daemon ) {
		g_stConfig.ucDeamon = 0;
    }

    if( want_printout ) {
        exit(0);
    }
    return 0;
}
/*****************************************************************************/
void sig_hup (int signo)
{
    signal(SIGHUP, sig_hup); /* set this for the next sighup */

	load_config(g_stConfig.szConfigFile);

}
/*****************************************************************************/
int main(int argc, char **argv)
{

    /* get commandline options, load config if needed. */
    if(get_options( argc, argv ) < 0 ) {
        exit(1);
    }

    signal(SIGHUP, sig_hup);

    dns_init();

    if (g_stConfig.ucDeamon) {
        /* Standard fork and background code */
        switch (fork()) {
            case -1:	/* Oh shit, something went wrong */
                debug_perror("fork");
                exit(-1);
            case 0:	/* Child: close off stdout, stdin and stderr */
                close(0);
                close(1);
                close(2);
                break;
            default:	/* Parent: Just exit */
                exit(0);
        }
    }

    dns_main_loop();

    return 0;
}


