/* vi: set sw=4 ts=4: */
/* serverpacket.c
 *
 * Construct and send DHCP server packets
 *
 * Russ Dill <Russ.Dill@asu.edu> July 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "common.h"
#include "dhcpd.h"
#include "options.h"


/* send a packet to giaddr using the kernel ip stack */
static int send_packet_to_relay(struct dhcpMessage *payload)
{
	DEBUG("Forwarding packet to relay");

	return udhcp_kernel_packet(payload, server_config.server, SERVER_PORT,
			payload->giaddr, SERVER_PORT);
}


/* send a packet to a specific arp address and ip address by creating our own ip packet */
static int send_packet_to_client(struct dhcpMessage *payload, int force_broadcast)
{
	const uint8_t *chaddr;
	uint32_t ciaddr;

	if (force_broadcast) {
		DEBUG("broadcasting packet to client (NAK)");
		ciaddr = INADDR_BROADCAST;
		chaddr = MAC_BCAST_ADDR;
	} else if (payload->ciaddr) {
		DEBUG("unicasting packet to client ciaddr");
		ciaddr = payload->ciaddr;
		chaddr = payload->chaddr;
	} else if (ntohs(payload->flags) & BROADCAST_FLAG) {
		DEBUG("broadcasting packet to client (requested)");
		ciaddr = INADDR_BROADCAST;
		chaddr = MAC_BCAST_ADDR;
	} else {
		DEBUG("unicasting packet to client yiaddr");
		ciaddr = payload->yiaddr;
		chaddr = payload->chaddr;
	}
	return udhcp_raw_packet(payload, server_config.server, SERVER_PORT,
			ciaddr, CLIENT_PORT, chaddr, server_config.ifindex);
}


/* send a dhcp packet, if force broadcast is set, the packet will be broadcast to the client */
static int send_packet(struct dhcpMessage *payload, int force_broadcast)
{
	int ret;

	if (payload->giaddr)
		ret = send_packet_to_relay(payload);
	else ret = send_packet_to_client(payload, force_broadcast);
	return ret;
}


static void init_packet(struct dhcpMessage *packet, struct dhcpMessage *oldpacket, char type)
{
	udhcp_init_header(packet, type);
	packet->xid = oldpacket->xid;
	memcpy(packet->chaddr, oldpacket->chaddr, 16);
	
	/*Added by songhaibo
	**dlinkҪserver
	*2013.10.22*/
	if(g_dhcps_brdcast == 1)
		packet->flags = 0x8000; /*ǿƹ㲥*/
	else
		packet->flags = oldpacket->flags; /*ݽյӦor㲥*/
	
	packet->giaddr = oldpacket->giaddr;
	packet->ciaddr = oldpacket->ciaddr;
	add_simple_option(packet->options, DHCP_SERVER_ID, server_config.server);
}


/* add in the bootp options */
static void add_bootp_options(struct dhcpMessage *packet)
{
	packet->siaddr = server_config.siaddr;
	if (server_config.sname)
		strncpy((char*)packet->sname, server_config.sname, sizeof(packet->sname) - 1);
	if (server_config.boot_file)
		strncpy((char*)packet->file, server_config.boot_file, sizeof(packet->file) - 1);
}

/*************************************************************************
功能: 获取DHCP Client的主机名
参数: pstPacket: 收到的DHCP包数据
      pcOutStr : 获取到的Client主机名字符串, 需要再函数外分配空间
      iStrlen  : pOutStr字符串长度
返回: 成功 -- 大于等于0, 获取的Client主机名字符串长度
      失败 -- 小于0
备注: add by laisimin for supply hostname
**************************************************************************/
int DHCPS_get_host_name( struct dhcpMessage *pstPacket, char *pcOutStr, int iStrLen )
{
    int iRet                  = 0;
    char unsigned *pcOptLen   = NULL; 
    char unsigned *pcHostName = NULL; 

    if( NULL==pstPacket || NULL==pcOutStr )
    {
        bb_error_msg("function NULL point pstPacket or pcOutStr\n");
        return -1;
    }

    pcHostName = get_option(pstPacket, DHCP_HOST_NAME);
    if( pcHostName )
    {
        pcOptLen = pcHostName -1;
        iRet = (*pcOptLen)<DHCPD_HOST_NAME_LEN? (*pcOptLen):DHCP_HOST_NAME;
        if( iRet>iStrLen )
        {
            iRet = iStrLen;
        }
        memcpy( pcOutStr, pcHostName, iRet );
        pcOutStr[iStrLen] = '\0';
        //printf("len=%d hostname=[%s]\n", *pcOptLen, pcOutStr );
    }

    return iRet;
}

/* send a DHCP OFFER to a DHCP DISCOVER */
int sendOffer(struct dhcpMessage *oldpacket)
{
	struct dhcpMessage packet;
	struct dhcpOfferedAddr *lease = NULL;
	uint32_t req_align, lease_time_align = server_config.lease;
	uint8_t *req, *lease_time;
	struct option_set *curr;
	struct in_addr addr;
	uint32_t static_lease_ip = 0;
	unsigned char flgFoundMac = 0;

	init_packet(&packet, oldpacket, DHCPOFFER);

	static_lease_ip = getIpByMac(server_config.static_leases, oldpacket->chaddr, &flgFoundMac);
	/* untrust static lease ip. MAC is exist but IP is 0.0.0.0, that's mean the MAC is untrusty. */
	if (!static_lease_ip && flgFoundMac){
		bb_info_msg("Untrusty static MAC:%02X:%02X:%02X:%02X:%02X:%02X",
			oldpacket->chaddr[0], oldpacket->chaddr[1], oldpacket->chaddr[2],
			oldpacket->chaddr[3], oldpacket->chaddr[4], oldpacket->chaddr[5]);
		return -1;
	}

	/* ADDME: if static, short circuit */
	if (!static_lease_ip) {
		/* the client is in our lease/offered table */
		lease = find_lease_by_chaddr(oldpacket->chaddr);
		if (lease) {
			if (!lease_expired(lease))
				lease_time_align = lease->expires - time(0);
			packet.yiaddr = lease->yiaddr;

		/* Or the client has a requested ip */
		} else if ((req = get_option(oldpacket, DHCP_REQUESTED_IP))
			/* Don't look here (ugly hackish thing to do) */
			&& memcpy(&req_align, req, 4)
			/* and the ip is in the lease range */
			&& ntohl(req_align) >= ntohl(server_config.start)
			&& ntohl(req_align) <= ntohl(server_config.end)
			&& !reservedIp(server_config.static_leases, req_align)/* Check whether the ip would be used as a static lease */
			/* and is not already taken/offered */
			&& (!(lease = find_lease_by_yiaddr(req_align))
				/* or its taken, but expired */ /* ADDME: or maybe in here */
				|| lease_expired(lease))
		) {
			packet.yiaddr = req_align; /* FIXME: oh my, is there a host using this IP? */
			/* otherwise, find a free IP */
		} else {
			/* Is it a static lease? (No, because find_address skips static lease) */
			packet.yiaddr = find_address(0);
			/* try for an expired lease */
			if (!packet.yiaddr) packet.yiaddr = find_address(1);
		}

		if (!packet.yiaddr) {
			bb_error_msg("no IP addresses to give - OFFER abandoned");
			return -1;
		}

		if (!add_lease(packet.chaddr, packet.yiaddr, server_config.offer_time)) {
			bb_error_msg("lease pool is full - OFFER abandoned");
			return -1;
		}
		lease_time = get_option(oldpacket, DHCP_LEASE_TIME);
		if (lease_time) {
			memcpy(&lease_time_align, lease_time, 4);
			lease_time_align = ntohl(lease_time_align);
			if (lease_time_align > server_config.lease)
				lease_time_align = server_config.lease;
		}

		/* Make sure we aren't just using the lease time from the previous offer */
		if (lease_time_align < server_config.min_lease)
			lease_time_align = server_config.lease;
		/* ADDME: end of short circuit */
	} else {
		/* It is a static lease... use it */
		packet.yiaddr = static_lease_ip;
	}

	add_simple_option(packet.options, DHCP_LEASE_TIME, htonl(lease_time_align));

	curr = server_config.options;
	while (curr) {
		if (curr->data[OPT_CODE] != DHCP_LEASE_TIME)
			add_option_string(packet.options, curr->data);
		curr = curr->next;
	}

	add_bootp_options(&packet);

	addr.s_addr = packet.yiaddr;
	bb_info_msg("Sending OFFER of %s", inet_ntoa(addr));
	return send_packet(&packet, 0);
}


int sendNAK(struct dhcpMessage *oldpacket)
{
	struct dhcpMessage packet;

	init_packet(&packet, oldpacket, DHCPNAK);

	DEBUG("Sending NAK");
	return send_packet(&packet, 1);
}


int sendACK(struct dhcpMessage *oldpacket, uint32_t yiaddr)
{
	struct dhcpMessage packet;
	struct option_set *curr;
	uint8_t *lease_time;
	uint32_t lease_time_align = server_config.lease;
	struct in_addr addr;
	uint32_t static_lease_ip = 0;
	unsigned char flgFoundMac = 0;

    /* add by laisimin for supply hostname*/
    char acHostName[DHCPD_HOST_NAME_LEN] = {0};
    DHCPS_get_host_name( oldpacket, acHostName, sizeof(acHostName)-1);

	static_lease_ip = getIpByMac(server_config.static_leases, oldpacket->chaddr, &flgFoundMac);

	init_packet(&packet, oldpacket, DHCPACK);
	packet.yiaddr = yiaddr;

	if ( (!static_lease_ip && flgFoundMac) )
	{
		//DHCPD_TRACE("Call check_valid_request fail");
		sendNAK(oldpacket);
		return -1;
	}

	if ((lease_time = get_option(oldpacket, DHCP_LEASE_TIME))) {
		memcpy(&lease_time_align, lease_time, 4);
		lease_time_align = ntohl(lease_time_align);
		if (lease_time_align > server_config.lease)
			lease_time_align = server_config.lease;
		else if (lease_time_align < server_config.min_lease)
			lease_time_align = server_config.lease;
	}

	add_simple_option(packet.options, DHCP_LEASE_TIME, htonl(lease_time_align));

	curr = server_config.options;
	while (curr) {
		if (curr->data[OPT_CODE] != DHCP_LEASE_TIME)
			add_option_string(packet.options, curr->data);
		curr = curr->next;
	}

	add_bootp_options(&packet);

	addr.s_addr = packet.yiaddr;
	//bb_info_msg("Sending ACK to %s", inet_ntoa(addr));
    /* add by laisimin: output for logic/dhcps adapter*/
    bb_info_msg("info, ADD %02x:%02x:%02x:%02x:%02x:%02x %s %u Computer %s",
            packet.chaddr[0], packet.chaddr[1], packet.chaddr[2],
            packet.chaddr[3], packet.chaddr[4], packet.chaddr[5],
            inet_ntoa(addr), lease_time_align, acHostName);

	if (send_packet(&packet, 0) < 0)
		return -1;

	add_lease(packet.chaddr, packet.yiaddr, lease_time_align);

	return 0;
}

/* output for logic/dhcps adapter*/
int outputInfo(struct dhcpMessage *oldpacket, uint32_t yiaddr)
{
	struct dhcpMessage packet;
	uint8_t *lease_time;
	uint32_t lease_time_align = server_config.lease;
	struct in_addr addr;

    /* add by laisimin for supply hostname*/
    char acHostName[DHCPD_HOST_NAME_LEN] = {0};
    DHCPS_get_host_name( oldpacket, acHostName, sizeof(acHostName)-1);


    init_packet(&packet, oldpacket, DHCPACK);

	packet.yiaddr = yiaddr;


	if ((lease_time = get_option(oldpacket, DHCP_LEASE_TIME))) {
		memcpy(&lease_time_align, lease_time, 4);
		lease_time_align = ntohl(lease_time_align);
		if (lease_time_align > server_config.lease)
			lease_time_align = server_config.lease;
		else if (lease_time_align < server_config.min_lease)
			lease_time_align = server_config.lease;
	}



	addr.s_addr = packet.yiaddr;

    bb_info_msg("info, ADD %02x:%02x:%02x:%02x:%02x:%02x %s %u Computer %s",
            packet.chaddr[0], packet.chaddr[1], packet.chaddr[2],
            packet.chaddr[3], packet.chaddr[4], packet.chaddr[5],
            inet_ntoa(addr), lease_time_align, acHostName);

	return 0;
}


int send_inform(struct dhcpMessage *oldpacket)
{
	struct dhcpMessage packet;
	struct option_set *curr;

	init_packet(&packet, oldpacket, DHCPACK);

	curr = server_config.options;
	while (curr) {
		if (curr->data[OPT_CODE] != DHCP_LEASE_TIME)
			add_option_string(packet.options, curr->data);
		curr = curr->next;
	}

	add_bootp_options(&packet);

	return send_packet(&packet, 0);
}
