/* vi: set sw=4 ts=4: */
/*
 * options.c -- DHCP server option packet tools
 * Rewrite by Russ Dill <Russ.Dill@asu.edu> July 2001
 */

#include "common.h"
#include "dhcpd.h"
#include "options.h"
#include "dhcpc.h"


/* supported options are easily added here */
const struct dhcp_option dhcp_options[] = {
	/* name[12]     flags                                   code */
	{"subnet",      OPTION_IP | OPTION_REQ,                 0x01},
	{"timezone",    OPTION_S32,                             0x02},
	{"router",      OPTION_IP | OPTION_LIST | OPTION_REQ,   0x03},
	{"timesvr",     OPTION_IP | OPTION_LIST,                0x04},
	{"namesvr",     OPTION_IP | OPTION_LIST,                0x05},
	{"dns",         OPTION_IP | OPTION_LIST | OPTION_REQ,   0x06},
	{"logsvr",      OPTION_IP | OPTION_LIST,                0x07},
	{"cookiesvr",   OPTION_IP | OPTION_LIST,                0x08},
	{"lprsvr",      OPTION_IP | OPTION_LIST,                0x09},
	{"hostname",    OPTION_STRING | OPTION_REQ,             0x0c},
	{"bootsize",    OPTION_U16,                             0x0d},
	{"domain",      OPTION_STRING | OPTION_LIST | OPTION_REQ, 0x0f},
	{"swapsvr",     OPTION_IP,                              0x10},
	{"rootpath",    OPTION_STRING,                          0x11},
	{"ipttl",       OPTION_U8,                              0x17},
	{"mtu",         OPTION_U16,                             0x1a},
	{"broadcast",   OPTION_IP | OPTION_REQ,                 0x1c},
	{"nisdomain",   OPTION_STRING | OPTION_REQ,             0x28},
	{"nissrv",      OPTION_IP | OPTION_LIST | OPTION_REQ,   0x29},
	{"ntpsrv",      OPTION_IP | OPTION_LIST | OPTION_REQ,   0x2a},
	{"wins",        OPTION_IP | OPTION_LIST,                0x2c},
	{"requestip",   OPTION_IP,                              0x32},
	{"lease",       OPTION_U32,                             0x33},
	{"dhcptype",    OPTION_U8,                              0x35},
	{"serverid",    OPTION_IP,                              0x36},
	{"message",     OPTION_STRING,                          0x38},
	{"vendorclass", OPTION_STRING,                          0x3C},
	{"clientid",    OPTION_STRING,                          0x3D},
	{"tftp",        OPTION_STRING,                          0x42},
	{"bootfile",    OPTION_STRING,                          0x43},
	{"userclass",   OPTION_STRING,                          0x4D},
#if ENABLE_FEATURE_RFC3397
	{"search",      OPTION_STR1035 | OPTION_LIST | OPTION_REQ, 0x77},
#endif
	/* MSIE's "Web Proxy Autodiscovery Protocol" support */
	{"wpad",        OPTION_STRING,                          0xfc},
	{"",            0x00,                                   0x00}
};

/* Lengths of the different option types */
const unsigned char option_lengths[] = {
	[OPTION_IP] =      4,
	[OPTION_IP_PAIR] = 8,
	[OPTION_BOOLEAN] = 1,
	[OPTION_STRING] =  1,
#if ENABLE_FEATURE_RFC3397
	[OPTION_STR1035] = 1,
#endif
	[OPTION_U8] =      1,
	[OPTION_U16] =     2,
	[OPTION_S16] =     2,
	[OPTION_U32] =     4,
	[OPTION_S32] =     4
};

u_char  dhcpc_opt60[512] = {0};

#ifdef DHCPC_OPTION_12_60
u_char  dhcpc_opt12[512] = {0}; /*BT 洢option12*/
#endif

void TW_printf(const void *buf, int len)
{
    const int LINE_WIDTH = 16;
    int i = 0, j = 0;
    int tail = 0;
    const unsigned char *pucBuf = NULL;

    if (len == 0 || NULL == buf)
    {
        return;
    }

    pucBuf = buf;

    for(; i<len; i++)
    {
        if(i>0 && i%LINE_WIDTH == 0)
        {
            j = i - LINE_WIDTH;
            printf("; ");
            for(; j<i; j++)
            {
                if(pucBuf[j] < 16)
                    printf("%c", '.');
                else
                    printf("%c", pucBuf[j]);
            }

            printf("\x0a\x0d");
        }

        printf("%02X ", pucBuf[i]);
    }

    tail = len%LINE_WIDTH == 0 ? len-LINE_WIDTH:(len/LINE_WIDTH)*LINE_WIDTH;
    if(tail != len-LINE_WIDTH)
    {
        for(i=0; i<48-(len-tail)*3; i++)
        {
            printf("%c", ' ');
        }
    }

    printf("; ");

    for(i=tail; i<len; i++)
    {
        if(pucBuf[i] < 16)
            printf("%c", '.');
        else
            printf("%c", pucBuf[i]);
    }

    printf("\x0a\x0d");
}


/* get an option with bounds checking (warning, not aligned). */
uint8_t *get_option(struct dhcpMessage *packet, int code)
{
	int i, length;
	uint8_t *optionptr;
	int over = 0, done = 0, curr = OPTION_FIELD;

	optionptr = packet->options;
	i = 0;
	length = 308;
	while (!done) {
		if (i >= length) {
			bb_error_msg("bogus packet, option fields too long");
			return NULL;
		}
		if (optionptr[i + OPT_CODE] == code) {
			if (i + 1 + optionptr[i + OPT_LEN] >= length) {
				bb_error_msg("bogus packet, option fields too long");
				return NULL;
			}
			return optionptr + i + 2;
		}
		switch (optionptr[i + OPT_CODE]) {
		case DHCP_PADDING:
			i++;
			break;
		case DHCP_OPTION_OVER:
			if (i + 1 + optionptr[i + OPT_LEN] >= length) {
				bb_error_msg("bogus packet, option fields too long");
				return NULL;
			}
			over = optionptr[i + 3];
			i += optionptr[OPT_LEN] + 2;
			break;
		case DHCP_END:
			if (curr == OPTION_FIELD && over & FILE_FIELD) {
				optionptr = packet->file;
				i = 0;
				length = 128;
				curr = FILE_FIELD;
			} else if (curr == FILE_FIELD && over & SNAME_FIELD) {
				optionptr = packet->sname;
				i = 0;
				length = 64;
				curr = SNAME_FIELD;
			} else done = 1;
			break;
		default:
			i += optionptr[OPT_LEN + i] + 2;
		}
	}
	return NULL;
}


/* return the position of the 'end' option (no bounds checking) */
int end_option(uint8_t *optionptr)
{
	int i = 0;

	while (optionptr[i] != DHCP_END) {
		if (optionptr[i] == DHCP_PADDING) i++;
		else i += optionptr[i + OPT_LEN] + 2;
	}
	return i;
}


/* add an option string to the options (an option string contains an option code,
 * length, then data) */
int add_option_string(uint8_t *optionptr, uint8_t *string)
{
	int end = end_option(optionptr);

	/* end position + string length + option code/length + end option */
	if (end + string[OPT_LEN] + 2 + 1 >= 308) {
		bb_error_msg("option 0x%02x did not fit into the packet",
				string[OPT_CODE]);
		return 0;
	}
	DEBUG("adding option 0x%02x", string[OPT_CODE]);
	memcpy(optionptr + end, string, string[OPT_LEN] + 2);
	optionptr[end + string[OPT_LEN] + 2] = DHCP_END;
	return string[OPT_LEN] + 2;
}


/* add a one to four byte option to a packet */
int add_simple_option(uint8_t *optionptr, uint8_t code, uint32_t data)
{
	const struct dhcp_option *dh;

	for (dh = dhcp_options; dh->code; dh++) {
		if (dh->code == code) {
			uint8_t option[6], len;

			option[OPT_CODE] = code;
			len = option_lengths[dh->flags & TYPE_MASK];
			option[OPT_LEN] = len;
			if (BB_BIG_ENDIAN) data <<= 8 * (4 - len);
			/* This memcpy is for broken processors which can't
			 * handle a simple unaligned 32-bit assignment */
			memcpy(&option[OPT_DATA], &data, 4);
			return add_option_string(optionptr, option);
		}
	}

	bb_error_msg("cannot add option 0x%02x", code);
	return 0;
}

char  *get_opt60_sub(char *opt, char fld)
{
	char total_len, fld_len;
	char *optionptr, *pfld;
	pfld = NULL;
	total_len = *(opt-1);
	if(total_len <= 2)
    {
		return NULL;
	}
	optionptr = opt + 2;
	total_len -= 2;
	while(total_len > 0)
    {
#if 0 /* wuyun, 2009-01-14, û */
		if (*optionptr > 0 && *optionptr < 64) /* Ҫ */
        {
#endif
			if (*optionptr == fld)
            {
				pfld = optionptr;
				break;
			}
            else
            {
				fld_len = *(optionptr+1);
				optionptr += (fld_len + 2);
				total_len -= (fld_len + 2);
			}
#if 0 /* wuyun, 2009-01-14, û */
		}
        else
        {
			return NULL;
		}
#endif
	}
	return pfld;
}

/*******************************************************************************
* Function    : verify_opt60
* Description : УDHCP Option 60
* Parameters  : opt -- option 60 ײ
* Return      : 
* Author      : wuyun / 2009-04-11
* History     :
*******************************************************************************/
int verify_opt60(u_char *opt)
{
    u_char  alllen = opt[1];
    u_char  *subopt;
    u_char  sublen;

    if (alllen < sizeof(u_short)) /* ֽҵ */
    {
        return -1;
    }

    /* TLV */
    alllen -= sizeof(u_short);
    subopt = opt + 2 + sizeof(u_short);

    while (TRUE)
    {
        if (0 == alllen)
        {
            return 0;
        }
        else if (alllen < 2)
        {
            return -1;
        }
        else
        {
            sublen = 2 + subopt[1];
            if (alllen < sublen)
            {
                return -1;
            }

            alllen -= sublen;
            subopt += sublen;
        }
    }
}

/*******************************************************************************
* Function    : verify_opt43
* Description : УDHCP Option 43
* Parameters  : opt -- option 43 ײ
* Return      : 
* Author      : wuyun / 2009-02-26
* History     :
*******************************************************************************/
int verify_opt43(u_char *opt)
{
    u_char  alllen = opt[1];
    u_char  *subopt = opt + 2;
    u_char  sublen;

    while (TRUE)
    {
        if (0 == alllen)
        {
            return 0;
        }
        else if (alllen < 2)
        {
            return -1;
        }
        else
        {
            sublen = 2 + subopt[1];
            if (alllen < sublen)
            {
                return -1;
            }

            alllen -= sublen;
            subopt += sublen;
        }
    }
}

/*******************************************************************************
* Function    : get_opt43_sub
* Description : DHCP Option 43ȡѡ
* Parameters  : opt -- option 43 ײ
*               sub -- ҵѡ
* Return      : 
* Author      : wuyun / 2009-02-26
* History     :
*******************************************************************************/
u_char *get_opt43_sub(u_char *opt, u_char sub)
{
    u_char  alllen = opt[1];
    u_char  *subopt = opt + 2;
    u_char  sublen;

    while (TRUE)
    {
        if (0 == alllen)
        {
            return NULL;
        }
        else if (alllen < 2)
        {
            return NULL;
        }
        else
        {
            sublen = 2 + subopt[1];
            if (alllen < sublen)
            {
                return NULL;
            }

            if (subopt[0] == sub)
            {
                return subopt;
            }

            alllen -= sublen;
            subopt += sublen;
        }
    }
}

#ifdef DHCPC_OPTION_12_60
/*******************************************************************************
* Function    : init_BT_opt12
* Description : ļжȡֵֶʼѡ12
* Parameters  :
* Return      : 
* Author      : songhaibo / 2013-01-21
* History     :
*******************************************************************************/
init_BT_opt12()
{
    FILE *file = NULL;
	char line[128] = {0};
	char len = 0;
	char dhcpc_hostname[128] = "BT-Mini-WiFi-Home-Hotspot";	
	char PlcMacAddr[20] = {0};
	
    /* Open the file */
	file = fopen("/var/Dhcpc_BT_Opt60", "r");
	if (NULL == file)
	{
		goto error;
	}

	/*Read the file*/
	while(NULL != fgets(line, sizeof(line) - 1, file))
	{
        if(1 == sscanf(line, "PlcMacAddr : %s\n",PlcMacAddr))/*Get Plc Mac Address*/
		{
            printf("PlcMacAddr = %s\n",PlcMacAddr);
		}
		else/*Invalid Information*/
		{            
			DEBUG("Bad line: %s\n", line);
		}
	}
	fclose(file);
	
    snprintf(dhcpc_hostname,sizeof(dhcpc_hostname)/sizeof(dhcpc_hostname[0]),"%s-%c%c%c",dhcpc_hostname,PlcMacAddr[13],PlcMacAddr[15],PlcMacAddr[16]);            

    /*Fill opt12*/
    dhcpc_opt12[0] = 12;
	dhcpc_opt12[1] = strlen(dhcpc_hostname);	
	memcpy(&dhcpc_opt12[2], dhcpc_hostname, dhcpc_opt12[1]);

	return 0;

error:/*Handle Error*/
	
	if (NULL != file)
	{
		fclose(file);/*Close File*/
	}
	return -1;
}

/*******************************************************************************
* Function    : init_BT_opt60
* Description : ļжȡֵֶʼѡ60
* Parameters  :
* Return      : 
* Author      : songhaibo / 2013-01-21
* History     :
*******************************************************************************/
init_BT_opt60()
{
	FILE *file = NULL;
	char line[128] = {0};
	char VendorInfo[256] = {0};
	char VendorName[128] = {0};
	char SoftVersion[128] = {0};
	char HardVersion[128] = {0};
	char VenIdtifyItem[] = "BT Wi-Fi Home Hotspot Mini";
	char VenIdtify[] = "Vendor Class Identify : ";
	char PlcMacAddr[20] = {0};
	
	/* Open the file */
	file = fopen("/var/Dhcpc_BT_Opt60", "r");
	if (NULL == file)
	{
		goto error;
	}

	/*Read the file*/
	while(NULL != fgets(line, sizeof(line) - 1, file))
	{
		if(0 == strncmp(line,VenIdtify,strlen(VenIdtify)))
		{
			strncpy(VendorName,line+sizeof(VenIdtify)-1,strlen(VenIdtifyItem));/*Get Vendor Name*/
			printf("VendorName = %s\n",VendorName);
		}
		else if(1 == sscanf(line,"ForCustomerHardwareVersion : %s\n",HardVersion))/*Get Hardware Version*/
		{
			printf("HardwareVersion = %s\n",HardVersion);
		}
		else if(1 == sscanf(line,"ForCustomerSoftwareVersion : %s\n",SoftVersion))/*Get Software Version*/
		{
			printf("SoftwareVersion = %s\n",SoftVersion);
		}
		else if(1 == sscanf(line, "PlcMacAddr : %s\n",PlcMacAddr))/*Get Plc Mac Address*/
		{
            printf("PlcMacAddr = %s\n",PlcMacAddr);
		}
		else/*Invalid Information*/
		{
			DEBUG("Bad line: %s\n", line);
		}
	}
	fclose(file);

	/*Combine info above*/	
	snprintf(VendorInfo,sizeof(VendorInfo)/sizeof(VendorInfo[0]),"%s %c%c%c(%s/%s)",VendorName,PlcMacAddr[13],PlcMacAddr[15],PlcMacAddr[16],HardVersion,SoftVersion);    
	printf("VendorInfo = %s",VendorInfo);
	
	/*Fill opt60*/
	dhcpc_opt60[0] = 60;
	dhcpc_opt60[1] = strlen(VendorInfo);
	memcpy(&dhcpc_opt60[2], VendorInfo, dhcpc_opt60[1]);

	return 0;

error:/*Handle Error*/
	
	if (NULL != file)
	{
		fclose(file);/*Close File*/
	}
	return -1;
}

/*******************************************************************************
* Function    : init_TalkTalk_opt12
* Description : ļжȡֵֶʼѡ12
* Parameters  :
* Return      : 
* Author      : songhaibo / 2013-01-21
* History     :
*******************************************************************************/
init_TalkTalk_opt12()
{
    FILE *file = NULL;
	char line[128] = {0};
	char len = 0;
	char dhcpc_hostname[128] = "DMG-304P";	
	char szMacAddr[20] = {0};
	
    /* Open the file */
	file = fopen("/var/TalkTalk_Dhcpc_OptFile", "r");
	if (NULL == file)
	{
		goto error;
	}

	/*Read the file*/
	while(NULL != fgets(line, sizeof(line) - 1, file))
	{
        if(1 == sscanf(line, "LanMacAddr : %s\n",szMacAddr))/*Get Mac Address*/
		{
            printf("LanMacAddr = %s\n",szMacAddr);
		}
		else/*Invalid Information*/
		{            
			DEBUG("Bad line: %s\n", line);
		}
	}
	fclose(file);
	
    snprintf(dhcpc_hostname,sizeof(dhcpc_hostname)/sizeof(dhcpc_hostname[0]),"%s-%c%c%c%c%c%c%c%c%c%c%c%c",dhcpc_hostname,
             szMacAddr[0],szMacAddr[1],szMacAddr[3],szMacAddr[4],szMacAddr[6],szMacAddr[7],
             szMacAddr[9],szMacAddr[10],szMacAddr[12],szMacAddr[13],szMacAddr[15],szMacAddr[16]);

    /*Fill opt12*/
    dhcpc_opt12[0] = 12;
	dhcpc_opt12[1] = strlen(dhcpc_hostname);	
	memcpy(&dhcpc_opt12[2], dhcpc_hostname, dhcpc_opt12[1]);

	return 0;

error:/*Handle Error*/
	
	if (NULL != file)
	{
		fclose(file);/*Close File*/
	}
	return -1;
}

/*******************************************************************************
* Function    : init_TalkTalk_opt60
* Description : ļжȡֵֶʼѡ60
* Parameters  :
* Return      : 
* Author      : songhaibo / 2013-01-21
* History     :
*******************************************************************************/
init_TalkTalk_opt60()
{
	FILE *file = NULL;
	char line[128] = {0};
	char VendorInfo[256] = {0};
	char VendorName[128] = {0};
	char SoftVersion[128] = {0};
	char HardVersion[128] = {0};
	char VenIdtifyItem[] = "BT Wi-Fi Home Hotspot Mini";
	char VenIdtify[] = "Vendor Class Identify : ";
	char PlcMacAddr[20] = {0};
	
	/* Open the file */
	file = fopen("/var/Dhcpc_BT_Opt60", "r");
	if (NULL == file)
	{
		goto error;
	}

	/*Read the file*/
	while(NULL != fgets(line, sizeof(line) - 1, file))
	{
		if(0 == strncmp(line,VenIdtify,strlen(VenIdtify)))
		{
			strncpy(VendorName,line+sizeof(VenIdtify)-1,strlen(VenIdtifyItem));/*Get Vendor Name*/
			printf("VendorName = %s\n",VendorName);
		}
		else if(1 == sscanf(line,"ForCustomerHardwareVersion : %s\n",HardVersion))/*Get Hardware Version*/
		{
			printf("HardwareVersion = %s\n",HardVersion);
		}
		else if(1 == sscanf(line,"ForCustomerSoftwareVersion : %s\n",SoftVersion))/*Get Software Version*/
		{
			printf("SoftwareVersion = %s\n",SoftVersion);
		}
		else if(1 == sscanf(line, "PlcMacAddr : %s\n",PlcMacAddr))/*Get Plc Mac Address*/
		{
            printf("PlcMacAddr = %s\n",PlcMacAddr);
		}
		else/*Invalid Information*/
		{
			DEBUG("Bad line: %s\n", line);
		}
	}
	fclose(file);

	/*Combine info above*/	
	snprintf(VendorInfo,sizeof(VendorInfo)/sizeof(VendorInfo[0]),"%s %c%c%c(%s/%s)",VendorName,PlcMacAddr[13],PlcMacAddr[15],PlcMacAddr[16],HardVersion,SoftVersion);    
	printf("VendorInfo = %s",VendorInfo);
	
	/*Fill opt60*/
	dhcpc_opt60[0] = 60;
	dhcpc_opt60[1] = strlen(VendorInfo);
	memcpy(&dhcpc_opt60[2], VendorInfo, dhcpc_opt60[1]);

	return 0;

error:/*Handle Error*/
	
	if (NULL != file)
	{
		fclose(file);/*Close File*/
	}
	return -1;
}

#endif

/*******************************************************************************
* Function    : init_opt60
* Description : ļжȡֵֶʼѡ60
* Parameters  :
* Return      : 
* Author      : wuyun / 2009-01-13
* History     :
*******************************************************************************/
int init_opt60()
{
  	FILE    *file = NULL;
	char    line[128] = {0};
	char    dhcp_wanname[20] = {0};
    u_short entrprs_nmbr;
    int     sub_opt;
    char    type[8] = {0};
    char    value[128] = {0};
	u_char  len = 0;
#if 0 /* wuyun, 2009-01-13, ĿǰйŵDHCP Option 60rfc3925ݣԺҪ޸ĳ֧rfc3925ʱѴ˱ */
    u_char  *entrprs_data_len;
#endif
    u_long  value_ulong;
    u_short value_ushort;
    u_char  *pstn = &dhcpc_opt60[2];

    /* ļ */
	file = fopen("/var/DhcpC_OptSnd_60", "r");
	if (NULL == file)
	{
		goto error;
	}

    /* ȡһжϴwanǷΪACָwan */
	if(NULL != fgets(line, sizeof(line) - 1, file))
	{
		if(1 == sscanf(line,"DHCPWanName %s\n",dhcp_wanname))
		{				
			if(strncmp(dhcp_wanname,client_config.interface,6))
			{
				DEBUG("This dhcp need not option60 info");
                fclose(file);
				return 0; 
			}
		}
		else
		{
			 fclose(file);
			 return 0; 
		}
	}
	else
	{
		fclose(file);
		dhcpc_opt60[0] = 60;
		dhcpc_opt60[1] = len;
		return 0;
		
	}
    while (NULL != fgets(line, sizeof(line) - 1, file)) /* ȡһ */
    {
        if (1 == sscanf(line, "EnterpriseNumber %hu\n", &entrprs_nmbr))
        {
            if (len + sizeof(u_short) > sizeof(dhcpc_opt60) - 2)
            {
                DEBUG("dhcpc_opt60 not long enough");
                goto error;
            }

            *(u_short *)pstn = htons(entrprs_nmbr);
            len  += sizeof(u_short);
            pstn += sizeof(u_short);

#if 0 /* wuyun, 2009-01-13, ĿǰйŵDHCP Option 60rfc3925ݣԺҪ޸ĳ֧rfc3925ʱѴ˱ */
            entrprs_data_len = pstn;
            len  += sizeof(u_char);
            pstn += sizeof(u_char);
#endif
        }
        else if (3 == sscanf(line, "SubOption %d:%[^=]=\"%[^\"]\"", &sub_opt, type, value))
        {
            if (len + sizeof(u_char) + sizeof(u_char) > sizeof(dhcpc_opt60) - 2)
            {
                //LOG(LOG_INFO, "dhcpc_opt60 not long enough");
                goto error;
            }

            pstn[0] = (u_char)sub_opt;
            if (0 == strcmp(type, "STRING"))
            {
                pstn[1] = strlen(value);
                if (len + sizeof(u_char) + sizeof(u_char) + pstn[1] > sizeof(dhcpc_opt60) - 2)
                {
                    DEBUG("dhcpc_opt60 not long enough");
                    goto error;
                }
                memcpy(&pstn[2], value, pstn[1]);
            }
            else if (0 == strcmp(type, "ULONG"))
            {
                value_ulong = htonl(atol(value));
                pstn[1] = sizeof(u_long);
                if (len + sizeof(u_char) + sizeof(u_char) + pstn[1] > sizeof(dhcpc_opt60) - 2)
                {
                    DEBUG("dhcpc_opt60 not long enough");
                    goto error;
                }
                memcpy(&pstn[2], (u_char *)&value_ulong, pstn[1]);
            }
            else if (0 == strcmp(type, "USHORT"))
            {
                value_ushort = htons(atoi(value));
                pstn[1] = sizeof(u_short);
                if (len + sizeof(u_char) + sizeof(u_char) + pstn[1] > sizeof(dhcpc_opt60) - 2)
                {
                    DEBUG("dhcpc_opt60 not long enough");
                    goto error;
                }
                memcpy(&pstn[2], (u_char *)&value_ushort, pstn[1]);
            }
            else if (0 == strcmp(type, "UCHAR"))
            {
                pstn[1] = sizeof(u_char);
                if (len + sizeof(u_char) + sizeof(u_char) + pstn[1] > sizeof(dhcpc_opt60) - 2)
                {
                    DEBUG("dhcpc_opt60 not long enough");
                    goto error;
                }
                pstn[2] = (u_char)atoi(value);
            }
            else
            {
                DEBUG("Unkown data type %s", type);
                goto error;
            }

            len  += sizeof(u_char) + sizeof(u_char) + pstn[1];
            pstn += sizeof(u_char) + sizeof(u_char) + pstn[1];
#if 0 /* wuyun, 2009-01-13, ĿǰйŵDHCP Option 60rfc3925ݣԺҪ޸ĳ֧rfc3925ʱѴ˱ */
            *entrprs_data_len += sizeof(u_char) + sizeof(u_char) + pstn[1];
#endif
        }
        else
        {
            DEBUG("Bad line: %s", line);
        }
    }

	fclose(file);
	dhcpc_opt60[0] = 60;
	dhcpc_opt60[1] = len;
	return 0;

error:
	
    if (NULL != file)
    {
        fclose(file);
    }
    return -1;
}
