/* vi: set sw=4 ts=4: */
/* dhcpc.c
 *
 * udhcp DHCP client
 *
 * Russ Dill <Russ.Dill@asu.edu> July 2001
 *
 * Licensed under the GPL v2 or later, see the file LICENSE in this tarball.
 */

#include <getopt.h>
#include <syslog.h>

#include "common.h"
#include "dhcpd.h"
#include "dhcpc.h"
#include "options.h"

/* Something is definitely wrong here. IPv4 addresses
 * in variables of type long?? BTW, we use inet_ntoa()
 * in the code. Manpage says that struct in_addr has a member of type long (!)
 * which holds IPv4 address, and the struct is passed by value (!!)
 */
static unsigned char server_mac[6];
static long init_start; 
static unsigned long timeout;
static unsigned long requested_ip; /* = 0 */
static uint32_t server_addr;
static int packet_num; /* = 0 */
static int sockfd = -1;

#define LISTEN_NONE 0
#define LISTEN_KERNEL 1
#define LISTEN_RAW 2
static smallint listen_mode;

static smallint state;

struct client_config_t client_config;


/* just a little helper */
static void change_mode(int new_mode)
{
	DEBUG("entering %s listen mode",
		new_mode ? (new_mode == 1 ? "kernel" : "raw") : "none");
	if (sockfd >= 0) {
		close(sockfd);
		sockfd = -1;
	}
	listen_mode = new_mode;
}


/* perform a renew */
static void perform_renew(void)
{
	bb_info_msg("Performing a DHCP renew");
	switch (state) {
	case BOUND:
		change_mode(LISTEN_KERNEL);
	case RENEWING:
	case REBINDING:
		state = RENEW_REQUESTED;
		break;
	case RENEW_REQUESTED: /* impatient are we? fine, square 1 */
		udhcp_run_script(NULL, "deconfig");
	case REQUESTING:
	case RELEASED:
		change_mode(LISTEN_RAW);
		state = INIT_SELECTING;
		init_start = uptime();
		break;
	case INIT_SELECTING:
		break;
	}

	/* start things over */
	packet_num = 0;

	/* Kill any timeouts because the user wants this to hurry along */
	timeout = 0;
}


/* perform a release */
static void perform_release(void)
{
	char buffer[16];
	struct in_addr temp_addr;

	/* send release packet */
	if (state == BOUND || state == RENEWING || state == REBINDING) {
		temp_addr.s_addr = server_addr;
		sprintf(buffer, "%s", inet_ntoa(temp_addr));
		temp_addr.s_addr = requested_ip;
		bb_info_msg("Unicasting a release of %s to %s",
				inet_ntoa(temp_addr), buffer);
		send_release(server_addr, requested_ip); /* unicast */
		udhcp_run_script(NULL, "deconfig");
	}
	bb_info_msg("Entering released state");

	change_mode(LISTEN_NONE);
	state = RELEASED;
	timeout = 0x7fffffff;
    /* add by wangyabo 释放地址时删除运行过程中生成的文件 */
    unlink("/var/DhcpC_OptRcv_60");
    unlink("/var/DhcpC_OptRcv_43");
	/* add end */
}


static void client_background(void)
{
#ifdef __uClinux__
	bb_error_msg("cannot background in uclinux (yet)");
/* ... mainly because udhcpc calls client_background()
 * in _the _middle _of _udhcpc _run_, not at the start!
 * If that will be properly disabled for NOMMU, client_background()
 * will work on NOMMU too */
#else
// chdir(/) is problematic. Imagine that e.g. pidfile name is RELATIVE! what will unlink do then, eh?
	bb_daemonize(DAEMON_CHDIR_ROOT);
	/* rewrite pidfile, as our pid is different now */
	if (client_config.pidfile)
		write_pidfile(client_config.pidfile);
	logmode &= ~LOGMODE_STDIO;
#endif
	client_config.foreground = 1; /* Do not fork again. */
	client_config.background_if_no_lease = 0;
}


// 电信要求企业码可修改
static  u_short get_enterprisecode(void)
{
  FILE *pFile = NULL;
  char  *pEpCode = NULL,*pCh = NULL;
  char szFileData[512]={0};
  
  if(0 != access("/data/profile/profile.dat",F_OK)) 
  {
    //DEBUG(LOG_INFO, "file /data/profile/profile.dat not exist\n");
    return 0;
  }
  pFile = fopen("/data/profile/profile.dat","r");
  if(!pFile)
  {
    //DEBUG(LOG_INFO, "open file /data/profile/profile.dat failed (%s)\n",strerror(errno));
     return 0;
  }
  
  if(0 < fread(szFileData,1,511,pFile))
  {
    fclose(pFile);
    pEpCode = strstr(szFileData,"X_TWSZ-COM_EnterpriseCode=");
    if(!pEpCode)
    {  
      //DEBUG(LOG_INFO, "file /data/profile/profile.dat has no EnterpriseCode\n");
       return 0;
    }
    
    pEpCode += strlen("X_TWSZ-COM_EnterpriseCode=");
    pCh = pEpCode;
    while(*pCh && (*pCh != 0x0a && *pCh != 0x0d)) pCh++;
    *pCh = 0x0;

    return (u_short)atoi(pEpCode);
  }
  
  fclose(pFile);
  return 0;
}

static uint8_t* alloc_dhcp_option(int code, const char *str, int extra)
{
	uint8_t *storage;
	int len = strlen(str);
	if (len > 255) len = 255;
	storage = xzalloc(len + extra + OPT_DATA);
	storage[OPT_CODE] = code;
	storage[OPT_LEN] = len + extra;
	memcpy(storage + extra + OPT_DATA, str, len);
	return storage;
}

/*************************************************************************
功能: 获取DHCP Server的DNSName
参数: pstPacket: 收到的DHCP包数据
      iOptType : 要获取的Opt类型
      ppOptStr:  获取的Opt字符串
返回: 成功 -- 大于等于0, 获取的Opt字符串长度
      失败 -- 小于0
备注: 本函数会在内部为ppOptStr使用malloc分配内存空间, 使用完后需free释放
**************************************************************************/
static int DHCPS_get_Opt_Str( struct dhcpMessage *pstPacket, int iOptType, char **ppOptStr )
{
    int iRet                  = 0;
    char unsigned *pcOptLen   = NULL; 
    char unsigned *pcOptName  = NULL; 

    /* 获取DNSName指针 */
    pcOptName = get_option( pstPacket, iOptType );
    if( pcOptName )
    {
        /* DNSName长度 */
        pcOptLen = pcOptName -1;
        iRet     = *pcOptLen;

        *ppOptStr = malloc(iRet+1);
        if( NULL == *ppOptStr )
        {
            bb_error_msg("malloc error\n" );
            return -1;
        }
        memset( *ppOptStr, 0, iRet+1 );
        memcpy( *ppOptStr, pcOptName, iRet );
        //bb_info_msg("len=%d OptStr=[%s]\n", iRet, *ppOptStr );
    }
    return iRet;
}




int udhcpc_main(int argc, char **argv);
int udhcpc_main(int argc, char **argv)
{
	uint8_t *temp, *message;
	char *str_c, *str_V, *str_h, *str_F, *str_r, *str_T, *str_t;
	unsigned long t1 = 0, t2 = 0, xid = 0;
	unsigned long start = 0, lease = 0;
	long now;
	unsigned opt;
	int max_fd;
	int sig;
	int retval, indefinite_optlen, i;
	int len;
	int no_clientid = 0;
	fd_set rfds;
	struct timeval tv;
	struct dhcpMessage packet;
	struct in_addr temp_addr, temp_inet;
	char *pcOptStr = NULL;
    char *strbuf   = NULL;
    char *bufpos   = NULL;
	u_char  *subopt = NULL;
	u_char  svrmac[6];

	/* add by wangyabo for opt60 init enterprisecode */
	u_short opt60_entrprs_nmbr = get_enterprisecode();
    
    if( 0 == opt60_entrprs_nmbr)
    {
#ifdef CONFIG_OPENSOURCE_SNMPA_ENTERPRISE_OID
      opt60_entrprs_nmbr = (u_short)atoi(CONFIG_OPENSOURCE_SNMPA_ENTERPRISE_OID);
#else
      opt60_entrprs_nmbr = 33003;
#endif
	}
	/* add end */
	enum {
		OPT_c = 1 << 0,
		OPT_C = 1 << 1,
		OPT_V = 1 << 2,
		OPT_f = 1 << 3,
		OPT_b = 1 << 4,
		OPT_H = 1 << 5,
		OPT_h = 1 << 6,
		OPT_F = 1 << 7,
		OPT_i = 1 << 8,
		OPT_n = 1 << 9,
		OPT_p = 1 << 10,
		OPT_q = 1 << 11,
		OPT_R = 1 << 12,
		OPT_r = 1 << 13,
		OPT_s = 1 << 14,
		OPT_T = 1 << 15,
		OPT_t = 1 << 16,
		OPT_v = 1 << 17,
		OPT_u = 1 << 18, /*for unicast/broadcast flag*/
	};
#if ENABLE_GETOPT_LONG
	static const struct option arg_options[] = {
		{ "clientid",   required_argument,      0, 'c' },
		{ "clientid-none", no_argument,         0, 'C' },
		{ "vendorclass", required_argument,     0, 'V' },
		{ "foreground", no_argument,            0, 'f' },
		{ "background", no_argument,            0, 'b' },
		{ "hostname",   required_argument,      0, 'H' },
		{ "hostname",   required_argument,      0, 'h' },
		{ "fqdn",       required_argument,      0, 'F' },
		{ "interface",  required_argument,      0, 'i' },
		{ "now",        no_argument,            0, 'n' },
		{ "pidfile",    required_argument,      0, 'p' },
		{ "quit",       no_argument,            0, 'q' },
		{ "release",    no_argument,            0, 'R' },
		{ "request",    required_argument,      0, 'r' },
		{ "script",     required_argument,      0, 's' },
		{ "timeout",    required_argument,      0, 'T' },
		{ "version",    no_argument,            0, 'v' },
		{ "retries",    required_argument,      0, 't' },
		{ "unicast",    no_argument,            0, 'u' },
		{ 0, 0, 0, 0 }
	};

#endif
	/* Default options. */
	client_config.interface = "eth0";
	client_config.script = DEFAULT_SCRIPT;
	client_config.retries = 3;
	client_config.timeout = 3;
	client_config.flags = 0x8000; /*ĬΪbroadcast*/

	/* Parse command line */
	opt_complementary = "?:c--C:C--c" // mutually exclusive
	                    ":hH:Hh"; // -h and -H are the same
#if ENABLE_GETOPT_LONG
	applet_long_options = arg_options;
#endif
	opt = getopt32(argc, argv, "c:CV:fbH:h:F:i:np:qRr:s:T:t:v:u",
		&str_c, &str_V, &str_h, &str_h, &str_F,
		&client_config.interface, &client_config.pidfile, &str_r,
		&client_config.script, &str_T, &str_t
		);

	if (opt & OPT_c)
		client_config.clientid = alloc_dhcp_option(DHCP_CLIENT_ID, str_c, 0);
	if (opt & OPT_C)
		no_clientid = 1;
	if (opt & OPT_V)
		client_config.vendorclass = alloc_dhcp_option(DHCP_VENDOR, str_V, 0);
	if (opt & OPT_f)
		client_config.foreground = 1;
	if (opt & OPT_b)
		client_config.background_if_no_lease = 1;
	if (opt & OPT_h)
		client_config.hostname = alloc_dhcp_option(DHCP_HOST_NAME, str_h, 0);
	if (opt & OPT_F) {
		client_config.fqdn = alloc_dhcp_option(DHCP_FQDN, str_F, 3);
		/* Flags: 0000NEOS
		S: 1 => Client requests Server to update A RR in DNS as well as PTR
		O: 1 => Server indicates to client that DNS has been updated regardless
		E: 1 => Name data is DNS format, i.e. <4>host<6>domain<4>com<0> not "host.domain.com"
		N: 1 => Client requests Server to not update DNS
		*/
		client_config.fqdn[OPT_DATA + 0] = 0x1;
		/* client_config.fqdn[OPT_DATA + 1] = 0; - redundant */
		/* client_config.fqdn[OPT_DATA + 2] = 0; - redundant */
	}
	// if (opt & OPT_i) client_config.interface = ...
	if (opt & OPT_n)
		client_config.abort_if_no_lease = 1;
	// if (opt & OPT_p) client_config.pidfile = ...
	if (opt & OPT_q)
		client_config.quit_after_lease = 1;
	if (opt & OPT_R)
		client_config.release_on_quit = 1;
	if (opt & OPT_r)
		requested_ip = inet_addr(str_r);
	// if (opt & OPT_s) client_config.script = ...
	if (opt & OPT_T)
		client_config.timeout = xatoi_u(str_T);
	if (opt & OPT_t)
		client_config.retries = xatoi_u(str_t);
	if (opt & OPT_v) {
		printf("version %s\n\n", BB_VER);
		return 0;
	}
	/*Added by songhaibo
	**According to setting of clients to appoint the broadcast or unicast of OFFER and ACK from DHCP Server
	*2013.09.11*/
	if (opt & OPT_u)
		client_config.flags = 0x0000; /*unicast*/
	else
		client_config.flags = 0x8000;/*broadcast*/


	if (ENABLE_FEATURE_UDHCP_SYSLOG) {
		openlog(applet_name, LOG_PID, LOG_LOCAL0);
		logmode |= LOGMODE_SYSLOG;
	}

	if (read_interface(client_config.interface, &client_config.ifindex,
			   NULL, client_config.arp) < 0)
		return 1;

	/* Sanitize fd's and write pidfile */
	udhcp_make_pidfile(client_config.pidfile);

	/* if not set, and not suppressed, setup the default client ID */
	if (!client_config.clientid && !no_clientid) {
		client_config.clientid = alloc_dhcp_option(DHCP_CLIENT_ID, "", 7);
		client_config.clientid[OPT_DATA] = 1;
		memcpy(client_config.clientid + OPT_DATA+1, client_config.arp, 6);
	}

	if (!client_config.vendorclass)
		client_config.vendorclass = alloc_dhcp_option(DHCP_VENDOR, "udhcp "BB_VER, 0);

	/* setup the signal pipe */
	udhcp_sp_setup();

	state = INIT_SELECTING;
	init_start = uptime();
	udhcp_run_script(NULL, "deconfig");
	change_mode(LISTEN_RAW);

	timeout = uptime();

	/* init opt60 add by wangyabo */
	unlink("/var/DhcpC_OptRcv_60");
    unlink("/var/DhcpC_OptRcv_43");

	
#ifdef DHCPC_OPTION_12_60 /*Add specialized option60 and option12*/

#ifdef COMPANY_BT
	init_BT_opt60();/*Init option60 packet*/
	init_BT_opt12();/*Init option12 packet*/
#endif

#ifdef COMPANY_TALK_TALK    
    //init_TalkTalk_opt60();/*Init option60 packet*/
	init_TalkTalk_opt12();/*Init option12 packet*/
#endif

#else
	init_opt60();
#endif

	/* add end */
	
	for (;;) {
		tv.tv_sec = timeout - uptime();
		tv.tv_usec = 0;

		if (listen_mode != LISTEN_NONE && sockfd < 0) {
			if (listen_mode == LISTEN_KERNEL)
				sockfd = listen_socket(INADDR_ANY, CLIENT_PORT, client_config.interface);
			else
				sockfd = raw_socket(client_config.ifindex);
		}
		max_fd = udhcp_sp_fd_set(&rfds, sockfd);

		if (tv.tv_sec > 0) {
			DEBUG("Waiting on select...");
			retval = select(max_fd + 1, &rfds, NULL, NULL, &tv);
		} else retval = 0; /* If we already timed out, fall through */

		now = uptime();
		if (retval == 0) {
			/* timeout dropped to zero */
			switch (state) {
			case INIT_SELECTING:
				if (packet_num < client_config.retries) {
					if (packet_num == 0)
						xid = random_xid();

					/* send discover packet */
					send_discover(xid, requested_ip); /* broadcast */

					timeout = now + client_config.timeout;
					packet_num++;
				} else {
					udhcp_run_script(NULL, "leasefail");
					if (client_config.background_if_no_lease) {
						bb_info_msg("No lease, forking to background");
						client_background();
					} else if (client_config.abort_if_no_lease) {
						bb_info_msg("No lease, failing");
						retval = 1;
						goto ret;
					}
					/* wait to try again */
					packet_num = 0;
					timeout = now + 60;
				}
				break;
			case RENEW_REQUESTED:
			case REQUESTING:
				if (packet_num < client_config.retries) {
					/* send request packet */
					if (state == RENEW_REQUESTED)
						send_renew(xid, server_addr, requested_ip); /* unicast */
					else send_selecting(xid, server_addr, requested_ip); /* broadcast */

					timeout = now + ((packet_num == 2) ? 10 : 2);
					packet_num++;
				} else {
					/* timed out, go back to init state */
					if (state == RENEW_REQUESTED)
						udhcp_run_script(NULL, "deconfig");
					state = INIT_SELECTING;
					init_start = uptime();
					timeout = now;
					packet_num = 0;
					change_mode(LISTEN_RAW);
				}
				break;
			case BOUND:
				/* Lease is starting to run out, time to enter renewing state */
				state = RENEWING;
				change_mode(LISTEN_KERNEL);
				DEBUG("Entering renew state");
				/* fall right through */
			case RENEWING:
				/* Either set a new T1, or enter REBINDING state */
				if ((t2 - t1) <= (lease / 14400 + 1)) {
					/* timed out, enter rebinding state */
					state = REBINDING;
					timeout = now + (t2 - t1);
					DEBUG("Entering rebinding state");
				} else {
					/* send a request packet */
					send_renew(xid, server_addr, requested_ip); /* unicast */

					t1 = (t2 - t1) / 2 + t1;
					timeout = t1 + start;
				}
				break;
			case REBINDING:
				/* Either set a new T2, or enter INIT state */
				if ((lease - t2) <= (lease / 14400 + 1)) {
					/* timed out, enter init state */
                    /* add by wangyabo for opt60 */
                    unlink("/var/DhcpC_OptRcv_60");
                    unlink("/var/DhcpC_OptRcv_43");
					state = INIT_SELECTING;
					init_start = uptime();
					/* add end */
					bb_info_msg("Lease lost, entering init state");
					udhcp_run_script(NULL, "deconfig");
					timeout = now;
					packet_num = 0;
					change_mode(LISTEN_RAW);
				} else {
					/* send a request packet */
					send_renew(xid, 0, requested_ip); /* broadcast */

					t2 = (lease - t2) / 2 + t2;
					timeout = t2 + start;
				}
				break;
			case RELEASED:
				/* yah, I know, *you* say it would never happen */
				timeout = 0x7fffffff;
				break;
			}
		} else if (retval > 0 && listen_mode != LISTEN_NONE && FD_ISSET(sockfd, &rfds)) {
			/* a packet is ready, read it */
GET_PACKET:
			if (listen_mode == LISTEN_KERNEL)
				len = udhcp_get_packet(&packet, sockfd);
			else len = get_raw_packet(&packet, sockfd);

			if (len == -1 && errno != EINTR) {
				DEBUG("error on read, %s, reopening socket", strerror(errno));
				change_mode(listen_mode); /* just close and reopen */
			}
			if (len < 0) continue;

			if (packet.xid != xid) {
				DEBUG("Ignoring XID %lx (our xid is %lx)",
					(unsigned long) packet.xid, xid);
				continue;
			}

			/* Ignore packets that aren't for us */
			if (memcmp(packet.chaddr, client_config.arp, 6)) {
				DEBUG("Packet does not have our chaddr - ignoring");
				continue;
			}

			message = get_option(&packet, DHCP_MESSAGE_TYPE);
			if (message == NULL) {
				bb_error_msg("cannot get option from packet - ignoring");
				continue;
			}

			switch (state) {
			case INIT_SELECTING:
				/* Must be a DHCPOFFER to one of our xid's */
                    if (*message == DHCPOFFER)
                	{
                    /* modify by wangyabo for opt60 */
                    #ifdef CONFIG_ForceCheckOption60AllTheTime
                    if (0 == access("/var/SysPara/DhcpCheckOpt60", F_OK))
                    #else
                    if ((0 == access("/var/SysPara/DhcpCheckOpt60", F_OK)) && (uptime() - init_start < 60)) /* INIT_SELECTING״̬ǰ30sѡЯƥOption 60ҵDHCP Server */
                    #endif
                    {
                        if (NULL == (temp = get_option(&packet, DHCP_VENDOR))) /* Option 60 */
                        {
                            goto GET_PACKET;
                        }

                        if (0 != verify_opt60(temp - 2)) /* Option 60У */
                        {
                            //DEBUG(LOG_INFO, "DHCP_Offer ignored due to bad DHCP_VENDOR[60] option");
                            //break; 
                            goto GET_PACKET;
                        }

                        if (opt60_entrprs_nmbr != htons(*(u_short *)temp))
                        {
                            //DEBUG(LOG_INFO, "DHCP_Offer ignored due to bad enterprise number:configed=%d, packet=%d", 
                                             //opt60_entrprs_nmbr,htons(*(u_short *)temp));
                            //break; 
                            goto GET_PACKET;
                        }
                    }
                        if ((temp = get_option(&packet, DHCP_SERVER_ID)))
                        {
                            memcpy(&server_addr, temp, 4);
                            xid = packet.xid;
                            requested_ip = packet.yiaddr;
                            memcpy(server_mac, svrmac, sizeof(server_mac)); /* wuyun, 2009-09-21, save dhcp svr mac address */

                            /* enter requesting state */
                            state = REQUESTING;
                            timeout = now;
                            packet_num = 0;
                        /* Get AC address from Option 60 */
                        if (   (NULL != (temp = get_option(&packet, DHCP_VENDOR)))
                            && (0 == verify_opt60(temp - 2))
                            && (opt60_entrprs_nmbr == htons(*(u_short *)temp))
                            && (NULL != (subopt = get_opt60_sub(temp, 100)))
                           )
                        {
                            FILE    *file = NULL;
                            char    buffer[256] = {0};

                            file = fopen("/var/DhcpC_OptRcv_60", "w+");
                            if (file != NULL)
                            {
                                memcpy(buffer, &subopt[2], subopt[1]);
                                buffer[subopt[1]] = '\0';
                                //DEBUG(LOG_INFO, "ACAddress=%s", buffer);
                                fprintf(file, "ACAddress=%s\n", buffer);
                                fclose(file);
                            }
                        }
                        else
                        {
                            unlink("/var/DhcpC_OptRcv_60");
                        }
                        /* Get AC address from Option 43 */
                        if (   (NULL != (temp = get_option(&packet, DHCP_VENDOR_SPEC_INFO))) /* Option 43 */
                            && (0 == verify_opt43(temp - 2))
                            && (NULL != (subopt = get_opt43_sub(temp - 2, 128)))
                            && ((subopt[1] != 0) && ((subopt[1] & 0x03) == 0))
                           )
                        {
                            FILE    *file = NULL;
                            u_char  *pstn;

                            file = fopen("/var/DhcpC_OptRcv_43", "w+");
                            if (NULL != file)
                            {
                                for (pstn = &subopt[2]; pstn < &subopt[2] + subopt[1]; pstn += sizeof(u_long))
                                {
                                    fprintf(file, "ACAddress=%u.%u.%u.%u\n", pstn[0], pstn[1], pstn[2], pstn[3]);
                                }
                                fclose(file);
                            }
                        }
                        else
                        {
                            unlink("/var/DhcpC_OptRcv_43"); 
                        }
                        }
						else
						{
						bb_error_msg("no server ID in message");
					}
				}
				break;
			case RENEW_REQUESTED:
			case REQUESTING:
			case RENEWING:
			case REBINDING:
				if (*message == DHCPACK) {
					temp = get_option(&packet, DHCP_LEASE_TIME);
					if (!temp) {
						bb_error_msg("no lease time with ACK, using 1 hour lease");
						lease = 60 * 60;
					} else {
						/* can be misaligned, thus memcpy */
						memcpy(&lease, temp, 4);
						lease = ntohl(lease);
					}

					/* enter bound state */
					t1 = lease / 2;

					/* little fixed point for n * .875 */
					t2 = (lease * 0x7) >> 3;
					temp_addr.s_addr = packet.yiaddr;
					bb_info_msg("Lease of %s obtained, lease time %ld",
						inet_ntoa(temp_addr), lease);

                    /* add by laisimin: 输出subnet和router用于pc匹配 */
                    if (!(temp = get_option(&packet, DHCP_SUBNET)))
                    {
                        memset(&temp_inet.s_addr, 0x00, 4); 
                    }
                    else
                    {
                        memcpy(&temp_inet.s_addr, temp, 4);                                                              
                    }
                    bb_info_msg("Subnet mask %s obtained", inet_ntoa(temp_inet));

                    if(temp = get_option(&packet,DHCP_ROUTER))
                    {   
                        memcpy(&temp_inet.s_addr,temp,4);
                        bb_info_msg("Router %s obtained", inet_ntoa(temp_inet));
                    }  

                    /* 输出DNSServer用于适配模块匹配 */
                    pcOptStr = NULL;
                    indefinite_optlen = DHCPS_get_Opt_Str( &packet, DHCP_DNS_SERVER, &pcOptStr );
                    if( pcOptStr )
                    {
                        strbuf = (char *)malloc(indefinite_optlen*4 + 1);
                        if (strbuf) 
                        {
                            memset(strbuf, 0, indefinite_optlen*4 + 1);
                            i = 0; 
                            bufpos = strbuf;
                            while (i < indefinite_optlen)
                            {
                                sprintf(bufpos, "%d.%d.%d.%d%s", pcOptStr[i],
                                        pcOptStr[i+1],pcOptStr[i+2],pcOptStr[i+3],
                                         ((i+4)<indefinite_optlen)?",":"");
                                //bb_info_msg("len=%d i=%d", indefinite_optlen, i );
                                i += 4;
                                bufpos += strlen(bufpos);
                            }

                            *bufpos = '\0';
                            bb_info_msg("Dns server %s obtained", strbuf);
                            free(strbuf);
                        }    
                        free(pcOptStr);
                    }

                    /* 用于确定匹配完成*/
                    bb_info_msg("Lease obtained, entering bound state");
                    /* add by laisimin: add end */

					start = now;
					timeout = t1 + start;
					requested_ip = packet.yiaddr;
					udhcp_run_script(&packet,
						   ((state == RENEWING || state == REBINDING) ? "renew" : "bound"));

					state = BOUND;
					change_mode(LISTEN_NONE);
					if (client_config.quit_after_lease) {
						if (client_config.release_on_quit)
							perform_release();
						goto ret0;
					}
                    /* add by laisimin: 防止udhcpc进程在获取到IP租约后自动退出 */
                    /*
					if (!client_config.foreground)
						client_background();
                        */

				} else if (*message == DHCPNAK) {
					/* return to init state */
					bb_info_msg("Received DHCP NAK");
					udhcp_run_script(&packet, "nak");
					if (state != REQUESTING)
						udhcp_run_script(NULL, "deconfig");
                    /* add by wangyabo for opt60 */
                    unlink("/var/DhcpC_OptRcv_60");
                    unlink("/var/DhcpC_OptRcv_43");
					state = INIT_SELECTING;
					init_start = uptime();
					/* add end */
					timeout = now;
					requested_ip = 0;
					packet_num = 0;
					change_mode(LISTEN_RAW);
					sleep(3); /* avoid excessive network traffic */
				}
				break;
			/* case BOUND, RELEASED: - ignore all packets */
			}
		} else if (retval > 0 && (sig = udhcp_sp_read(&rfds))) {
			switch (sig) {
			case SIGUSR1:
				perform_renew();
				break;
			case SIGUSR2:
				perform_release();
				break;
			case SIGTERM:
				bb_info_msg("Received SIGTERM");
				if (client_config.release_on_quit)
					perform_release();
				goto ret0;
			}
		} else if (retval == -1 && errno == EINTR) {
			/* a signal was caught */
		} else {
			/* An error occured */
			bb_perror_msg("select");
		}
	} /* for (;;) */
 ret0:
	retval = 0;
 ret:
	if (client_config.pidfile)
		remove_pidfile(client_config.pidfile);
	return retval;
}
