/*
 * Cryptographic API.
 *
 * SHA1 Secure Hash Algorithm.
 *
 * Derived from cryptoapi implementation, adapted for in-place
 * scatterlist interface.
 *
 * Copyright (c) Alan Smithee.
 * Copyright (c) Andrew McDonald <andrew@mcdonald.org.uk>
 * Copyright (c) Jean-Francois Dive <jef@linuxbe.org>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 */
#include <crypto/internal/hash.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <linux/cryptohash.h>
#include <linux/types.h>
#include <crypto/sha.h>
#include <asm/byteorder.h>

#ifdef CONFIG_CRYPTO_DEV_REALTEK
#include "rtl_crypto_helper.h"
#endif // CONFIG_CRYPTO_DEV_REALTEK

struct sha1_ctx {
        u64 count;
        u32 state[5];
        u8 buffer[64];
#ifdef CONFIG_CRYPTO_DEV_REALTEK
		struct rtl_hash_ctx rtl_ctx;
#endif
};

static int sha1_init(struct shash_desc *desc)
{
#ifdef CONFIG_CRYPTO_DEV_REALTEK
	struct sha1_ctx *sctx = shash_desc_ctx(desc);

	rtl_hash_init_ctx(crypto_shash_tfm(desc->tfm), &sctx->rtl_ctx);
	return 0;
#else
	struct sha1_ctx *sctx = shash_desc_ctx(desc);

	static const struct sha1_ctx initstate = {
	  0,
	  { SHA1_H0, SHA1_H1, SHA1_H2, SHA1_H3, SHA1_H4 },
	  { 0, }
	};

	*sctx = initstate;

	return 0;
#endif
}

static int sha1_update(struct shash_desc *desc, const u8 *data,
			unsigned int len)
{
#ifdef CONFIG_CRYPTO_DEV_REALTEK
	struct sha1_ctx *sctx = shash_desc_ctx(desc);

	rtl_hash_update(&sctx->rtl_ctx, (u8 *) data, len);
	return 0;
#else
	struct sha1_ctx *sctx = shash_desc_ctx(desc);
	unsigned int partial, done;
	const u8 *src;

	partial = sctx->count & 0x3f;
	sctx->count += len;
	done = 0;
	src = data;

	if ((partial + len) > 63) {
		u32 temp[SHA_WORKSPACE_WORDS];

		if (partial) {
			done = -partial;
			memcpy(sctx->buffer + partial, data, done + 64);
			src = sctx->buffer;
		}

		do {
			sha_transform(sctx->state, src, temp);
			done += 64;
			src = data + done;
		} while (done + 63 < len);

		memset(temp, 0, sizeof(temp));
		partial = 0;
	}
	memcpy(sctx->buffer + partial, src, len - done);

	return 0;
#endif
}

#ifdef CONFIG_CRYPTO_DEV_REALTEK
static int sha1_digest(struct shash_desc *desc, const u8 *data,
	unsigned int len, u8 *out)
{
	struct sha1_ctx *sctx = shash_desc_ctx(desc);

	return rtl_hash_digest(&sctx->rtl_ctx, (u8 *) data, len, out);
}
#endif

/* Add padding and return the message digest. */
static int sha1_final(struct shash_desc *desc, u8 *out)
{
#ifdef CONFIG_CRYPTO_DEV_REALTEK
	struct sha1_ctx *sctx = shash_desc_ctx(desc);

	return rtl_hash_final(&sctx->rtl_ctx, out);
#else
	struct sha1_ctx *sctx = shash_desc_ctx(desc);
	__be32 *dst = (__be32 *)out;
	u32 i, index, padlen;
	__be64 bits;
	static const u8 padding[64] = { 0x80, };

	bits = cpu_to_be64(sctx->count << 3);

	/* Pad out to 56 mod 64 */
	index = sctx->count & 0x3f;
	padlen = (index < 56) ? (56 - index) : ((64+56) - index);
	sha1_update(desc, padding, padlen);

	/* Append length */
	sha1_update(desc, (const u8 *)&bits, sizeof(bits));

	/* Store state in digest */
	for (i = 0; i < 5; i++)
		dst[i] = cpu_to_be32(sctx->state[i]);

	/* Wipe context */
	memset(sctx, 0, sizeof *sctx);

	return 0;
#endif
}

static struct shash_alg alg = {
	.digestsize	=	SHA1_DIGEST_SIZE,
	.init		=	sha1_init,
	.update		=	sha1_update,
	.final		=	sha1_final,
#ifdef CONFIG_CRYPTO_DEV_REALTEK
	.digest		=	sha1_digest,
#endif
	.descsize	=	sizeof(struct sha1_ctx),
	.base		=	{
		.cra_name	=	"sha1",
		.cra_driver_name=	"sha1-generic",
		.cra_flags	=	CRYPTO_ALG_TYPE_SHASH,
		.cra_blocksize	=	SHA1_BLOCK_SIZE,
		.cra_module	=	THIS_MODULE,
	}
};

static int __init sha1_generic_mod_init(void)
{
	return crypto_register_shash(&alg);
}

static void __exit sha1_generic_mod_fini(void)
{
	crypto_unregister_shash(&alg);
}

module_init(sha1_generic_mod_init);
module_exit(sha1_generic_mod_fini);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("SHA1 Secure Hash Algorithm");

MODULE_ALIAS("sha1");
