/** Copyright (c) 2015 Cvisionhk */

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>

#include <event.h>
#include <frame.h>
#include <stream.h>
#include <pmsg.h>
#include <rtp.h>
#include <conf_parse.h>
#include "mylogger.h"
#include "hw_platform_feature.h"

static int tcp_base64[] =
    {
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, -1, 63,
    52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1,  0, -1, -1,
    -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
    15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1,
    -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
    41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1
    };

void rtsp_write_log(struct sockaddr *addr, int i32Id, char *pCharReq,
                    int i32Len, char *pCharRef, char *pCharUser );

struct tcp_listener
    {
    int fd;
    };

static struct rtp_conn *psRtpConnList = NULL;

static void tcp_do_read( struct event_info *psEvInfo, void *d );

static void tcp_drop_conn( struct rtp_conn *psRtpConn )
    {
    if(psRtpConn == NULL)
        return;

    if( psRtpConn->proto_state )
        switch( psRtpConn->proto )
            {
            case RTP_CONN_PROTO_HTTP:
                /*http_conn_disconnect( psRtpConn );*/
                break;
            case RTP_CONN_PROTO_RTSP:
                rtsp_conn_disconnect( psRtpConn );
                break;
            default:
                //nothing
                break;
            }
    event_remove( psRtpConn->read_event, NULL);
    if( psRtpConn->second_read_event )
        {
        event_remove( psRtpConn->second_read_event, NULL);
        }
    event_remove( psRtpConn->write_event, NULL);
    if( psRtpConn->fd >= 0 ) close( psRtpConn->fd );
    psRtpConn->fd = -1;
    if( psRtpConn->second_fd >= 0 ) close( psRtpConn->second_fd );
    psRtpConn->second_fd = -1;
    if( psRtpConn->next ) psRtpConn->next->prev = psRtpConn->prev;
    if( psRtpConn->prev ) psRtpConn->prev->next = psRtpConn->next;
    else psRtpConnList = psRtpConn->next;
    free( psRtpConn );
    }

static void tcp_conn_write( struct event_info *psEvInfo, void *d )
    {
    struct rtp_conn *psRtpConn = (struct rtp_conn *)d;
    int i32Ret, len;

    while( psRtpConn->send_buf_r != psRtpConn->send_buf_w )
        {
        if( psRtpConn->send_buf_w < psRtpConn->send_buf_r )
            len = sizeof( psRtpConn->send_buf ) - psRtpConn->send_buf_r;
        else
            len = psRtpConn->send_buf_w - psRtpConn->send_buf_r;

        i32Ret = write( psRtpConn->fd, psRtpConn->send_buf + psRtpConn->send_buf_r, len );
        if( i32Ret <= 0 )
            {
            perror("write");
            if( i32Ret < 0 && errno == EAGAIN )
                {
                return;
                }
            tcp_drop_conn( psRtpConn );
            return;
            }
        psRtpConn->send_buf_r += i32Ret;
        if( psRtpConn->send_buf_r == sizeof( psRtpConn->send_buf ) )
            psRtpConn->send_buf_r = 0;
        }
    if( psRtpConn->drop_after ) tcp_drop_conn( psRtpConn );
    else event_set_event_enabled( psRtpConn->write_event, 0, NULL);
    }

int tcp_avail_send_buf( struct rtp_conn *psRtpConn )
    {
    if( psRtpConn->send_buf_r > psRtpConn->send_buf_w )
        return psRtpConn->send_buf_r - psRtpConn->send_buf_w - 1;
    else return sizeof( psRtpConn->send_buf ) - psRtpConn->send_buf_w + psRtpConn->send_buf_r - 1;
    }

int tcp_send_data( struct rtp_conn *psRtpConn, unsigned char *pucData, int i32Len )
    {
    if( tcp_avail_send_buf( psRtpConn ) < i32Len ) return 1;
    while( --i32Len >= 0 )
        {
        psRtpConn->send_buf[psRtpConn->send_buf_w++] = *(pucData++);
        if( psRtpConn->send_buf_w == sizeof( psRtpConn->send_buf ) )
            psRtpConn->send_buf_w = 0;
        }
    event_set_event_enabled( psRtpConn->write_event, 1, NULL);
    return 0;
    }

int tcp_send_pmsg( struct rtp_conn *psRtpConn, struct pmsg *psMsg, int i32Len )
    {
    unsigned char pucBuf[2048];
    int i, f, totlen;

    if( psMsg->type != PMSG_RESP ) return -1;
    i = sprintf( (char *)pucBuf, "%s %d %s\r\n", psMsg->proto_id, psMsg->sl.stat.code,
                 psMsg->sl.stat.reason );
    for( f = 0; f < psMsg->header_count; ++f )
        i += sprintf( (char *)pucBuf + i, "%s: %s\r\n", psMsg->fields[f].name,
                      psMsg->fields[f].value );
    if( i32Len >= 0 )
        {
        i += sprintf( (char *)pucBuf + i, "Content-Length: %d\r\n\r\n", i32Len );
        totlen = i + i32Len;
        }
    else
        {
        i += sprintf( (char *)pucBuf + i, "\r\n" );
        totlen = i;
        }
    if( tcp_avail_send_buf( psRtpConn ) < totlen ) return -1;
    tcp_send_data( psRtpConn, pucBuf, i );
    return 0;
    }

static void log_request( struct req *psReq, int i32Code, int i32Len )
    {
    char *pcRef, *pcUsrAgent;
    pcRef = pmsg_get_header( psReq->req, "referer" );
    pcUsrAgent = pmsg_get_header( psReq->req, "user-agent" );
    rtsp_write_log((struct sockaddr *)&psReq->conn->client_addr,
                   i32Code, (char *)psReq->conn->req_buf, i32Len,
                   pcRef ? pcRef : "-", pcUsrAgent ? pcUsrAgent : "-" );
    }

static int handle_unknown( struct req *psReq )
    {
    log_request( psReq, 501, 0 );
    psReq->conn->drop_after = 1;
    psReq->resp = pmsg_new_pmsg( 256 );
    psReq->resp->type = PMSG_RESP;
    psReq->resp->proto_id = pmsg_add_pmsg_string( psReq->resp, psReq->req->proto_id );
    psReq->resp->sl.stat.code = 501;
    psReq->resp->sl.stat.reason =
        pmsg_add_pmsg_string( psReq->resp, "Not Implemented" );
    pmsg_copy_headers( psReq->resp, psReq->req, "CSeq" );
    tcp_send_pmsg( psReq->conn, psReq->resp, -1 );
    return 0;
    }

static int handle_request( struct rtp_conn *psRtpConn )
    {
    int ret = -1;
    struct req *psReq;

    if( ( psReq = malloc( sizeof( struct req ) ) ) == NULL )
    {
        return -1;
    }

    psReq->conn = psRtpConn;
    psReq->resp = NULL;
    psReq->req = pmsg_new_pmsg( psRtpConn->req_len );
    psReq->req->msg_len = psRtpConn->req_len;
    memset(psReq->req->msg, 0x00,psRtpConn->req_len);
    memcpy( psReq->req->msg, psRtpConn->req_buf, psReq->req->msg_len );
    if( pmsg_parse_pmsg( psReq->req ) < 0 )
    {
        if(psReq)
        {

            pmsg_free_pmsg( psReq->req );
            free( psReq );
        }
        return -1;
    }

    mylog_debug("client request: '%s' '%s' '%s'\n",
                psReq->req->sl.req.method, psReq->req->sl.req.uri,
                psReq->req->proto_id );

    switch( psRtpConn->proto )
        {
        case RTP_CONN_PROTO_HTTP:
            break;
        case RTP_CONN_PROTO_RTSP:
            ret = rtsp_handle_msg( psReq );
            break;
        default:
            ret = handle_unknown( psReq );
            break;
        }
    if( ret <= 0 )
    {
        if(psReq)
        {
            pmsg_free_pmsg( psReq->req );
            if( psReq->resp ) pmsg_free_pmsg( psReq->resp );
            free( psReq );
        }
    }

    return ret < 0 ? -1 : 0;
    }

static int parse_client_data( struct rtp_conn *psRtpConn )
    {
    char *a, *b;
    if(psRtpConn == NULL)
    {
        return -1;
    }
    switch( psRtpConn->proto )
        {
        case RTP_CONN_PROTO_START:
            if( ( a = strchr( (char *)psRtpConn->req_buf, '\n' ) ) )
            {
                *a = 0;
                if( ! ( b = strrchr( (char *)psRtpConn->req_buf, ' ' ) ) )
                {
                    return -1;
                }

                if( ! strncmp( b + 1, "HTTP/", 5 ) )
                {
                    psRtpConn->proto = RTP_CONN_PROTO_HTTP;
                }

                else if( ! strncmp( b + 1, "RTSP/", 5 ) )
                {
                    psRtpConn->proto = RTP_CONN_PROTO_RTSP;
                }
                else return -1;
                *a = '\n';
                return 1;
            }
            break;
        case RTP_CONN_PROTO_RTSP:
            if( psRtpConn->req_len < 4 )
            {
                return 0;
            }
            if( psRtpConn->req_buf[0] == '$' )
            {
                if( psRtpConn->req_len < RTP_GET_16( psRtpConn->req_buf + 2 ) + 4 )
                {
                    return 0;
                }
                rtsp_interleave_recv( psRtpConn, psRtpConn->req_buf[1], psRtpConn->req_buf + 4,
                                      RTP_GET_16( psRtpConn->req_buf + 2 ) );
                /* should really just delete the packet from the
                 * buffer, not the entire buffer */
                psRtpConn->req_len = 0;
            }
            else
            {
                if( ! strstr( (char *)psRtpConn->req_buf, "\r\n\r\n" ) )
                {
                    return 0;
                }
                if( handle_request( psRtpConn ) < 0 )
                {
                    return -1;
                }
                /* should really just delete the request from the
                 * buffer, not the entire buffer */
                psRtpConn->req_len = 0;
            }
            break;
        case RTP_CONN_PROTO_HTTP:
            if( ! strstr( (char *)psRtpConn->req_buf, "\r\n\r\n" ) ) return 0;
            if( handle_request( psRtpConn ) < 0 ) return -1;
            /* should really just delete the request from the
             * buffer, not the entire buffer */
            psRtpConn->req_len = 0;
            break;
        }
    return 0;
    }

static int unbase64( unsigned char *d, int len, int *remain )
    {
    int i32Src = 0, i32Dest = 0, i;
    unsigned char enc[4];

    for(;;)
        {
        for( i = 0; i < 4; ++i32Src )
            {
            if( i32Src >= len )
                {
                if( i > 0 ) memcpy( d + i32Dest, enc, i );
                *remain = i;
                return i32Dest + i;
                }
            if( tcp_base64[d[i32Src]] < 0 ) continue;
            enc[i++] = d[i32Src];
            }
        d[i32Dest] = ( ( tcp_base64[enc[0]] << 2 ) & 0xFC ) |
                     ( tcp_base64[enc[1]] >> 4 );
        d[i32Dest + 1] = ( ( tcp_base64[enc[1]] << 4 ) & 0xF0 ) |
                         ( tcp_base64[enc[2]] >> 2 );
        d[i32Dest + 2] = ( ( tcp_base64[enc[2]] << 6 ) & 0xC0 ) |
                         tcp_base64[enc[3]];
        if( enc[3] == '=' )
            {
            if( enc[2] == '=' ) i32Dest += 1;
            else i32Dest += 2;
            }
        else i32Dest += 3;
        }
    return 0;
    }

static void tcp_do_read( struct event_info *psEvInfo, void *d )
    {
    struct rtp_conn *psRtpConn = (struct rtp_conn *)d;
    int ret, second;

    if(psRtpConn == NULL)
    {
        return;
    }
    second = psEvInfo->e == psRtpConn->second_read_event;
    for(;;)
    {
        ret = read( second ? psRtpConn->second_fd : psRtpConn->fd,
                    psRtpConn->req_buf + psRtpConn->req_len,
                    sizeof( psRtpConn->req_buf ) - psRtpConn->req_len - 1 );
        /* VINH:TODO */
        if( ret <= 0 )
        {
            if( ret < 0 )
            {
                if( errno == EAGAIN )
                {
                    return;
                }
                mylog_info("closing TCP connection to client due to read error: %s",
                           strerror( errno ) );
            }
            else
            {
                mylog_info("client closed TCP connection" );
            }

            if( second )
            {
                event_remove( psRtpConn->second_read_event, NULL );
                psRtpConn->second_read_event = NULL;
                close( psRtpConn->second_fd );
                psRtpConn->second_fd = -1;
            }
            else
            {
                tcp_drop_conn( psRtpConn );
            }
            return;
        }

        if( psRtpConn->base64_count >= 0 )
        {
            psRtpConn->req_len -= psRtpConn->base64_count;
            psRtpConn->req_len +=
                unbase64( psRtpConn->req_buf + psRtpConn->req_len,
                          ret + psRtpConn->base64_count,
                          &psRtpConn->base64_count );
        }
        else
        {
            psRtpConn->req_len += ret;
        }

        if( psRtpConn->req_len == sizeof( psRtpConn->req_buf ) - 1 )
        {
            mylog_info("malformed request from client; exceeded maximum size" );
            tcp_drop_conn( psRtpConn );
            return;
        }

        if( psRtpConn->base64_count > 0 )
        {
            continue;
        }

        psRtpConn->req_buf[psRtpConn->req_len] = 0;

        while( ( ret = parse_client_data( psRtpConn ) ) > 0 );
        if( ret < 0 )
        {
            tcp_drop_conn( psRtpConn);
            return;
        }
    }
}

static void do_accept( struct event_info *psEvInfo, void *d )
    {
    struct tcp_listener *psTcpListener = (struct tcp_listener *)d;
    int i32fd, i;
    struct sockaddr_in addr;
    struct rtp_conn *psRtpConn;
    int sockBufSize = 400*1024;

    i = sizeof( addr );
    if( ( i32fd = accept( psTcpListener->fd, (struct sockaddr *)&addr, (socklen_t *)&i ) ) < 0 )
        {
        mylog_info("error accepting TCP connection: %s",
                   strerror( errno ) );
        return;
        }
    if( fcntl( i32fd, F_SETFL, O_NONBLOCK ) < 0 )
        mylog_info("error setting O_NONBLOCK on socket: %s",
                   strerror( errno ) );
    i = 1;
    if( setsockopt( i32fd, IPPROTO_TCP, TCP_NODELAY, &i, sizeof( i ) ) < 0 )
        mylog_info("error setting TCP_NODELAY on socket: %s",
                   strerror( errno ) );
    mylog_debug("do_accept() ALLOCATE %d byte", sizeof( struct rtp_conn ));
    psRtpConn = (struct rtp_conn *)malloc( sizeof( struct rtp_conn ) );
    psRtpConn->next = psRtpConnList;
    if( psRtpConn->next ) psRtpConn->next->prev = psRtpConn;
    psRtpConn->prev = NULL;
    psRtpConn->fd = i32fd;
    psRtpConn->second_fd = -1;
    psRtpConn->client_addr = addr;
    psRtpConn->proto = RTP_CONN_PROTO_START;
    psRtpConn->http_tunnel_cookie[0] = 0;
    psRtpConn->base64_count = -1;
    psRtpConn->req_len = 0;
    psRtpConn->req_list = NULL;
    psRtpConn->read_event = event_add_fd( i32fd, 0, 0, tcp_do_read, psRtpConn, NULL);
    psRtpConn->second_read_event = NULL;
    psRtpConn->write_event = event_add_fd( i32fd, 1, 0, tcp_conn_write, psRtpConn, NULL);
    setsockopt(i32fd, SOL_SOCKET, SO_SNDBUF, &sockBufSize, sizeof(sockBufSize));
    event_set_event_enabled( psRtpConn->write_event, 0, NULL);
    psRtpConn->send_buf_r = psRtpConn->send_buf_w = 0;
    psRtpConn->drop_after = 0;
    psRtpConn->proto_state = NULL;
    psRtpConnList = psRtpConn;
    }

static int tcp_listen( int i32Port )
    {
    struct sockaddr_in sSockAddrIn;
    struct tcp_listener *psTcpListener;
    int opt, i32fd;

    sSockAddrIn.sin_family = AF_INET;
#ifdef PUBLIC_PORT_STREAMING_RTSP
    /*Check lock shell will close port 6667*/
    FILE * fp = NULL;
    fp = fopen (FACTORY_SHELL_LOGIN_FLAG, "r");
    if(fp != NULL)
    {
        sSockAddrIn.sin_addr.s_addr = inet_addr("127.0.0.1");
        fclose (fp);
    }
    else
    {
        sSockAddrIn.sin_addr.s_addr = 0;
    }
#else
    sSockAddrIn.sin_addr.s_addr = 0;
#endif
    sSockAddrIn.sin_port = htons( i32Port );

    if( ( i32fd = socket( PF_INET, SOCK_STREAM, 0 ) ) < 0 )
        {
        mylog_error("error creating listen socket: %s",
                    strerror( errno ) );
        return -1;
        }
    opt = 1;
    if( setsockopt( i32fd, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof( opt ) ) < 0 )
        mylog_info("ignoring error on setsockopt: %s",
                   strerror( errno ) );
    if( bind( i32fd, (struct sockaddr *)&sSockAddrIn, sizeof( sSockAddrIn ) ) < 0 )
        {
        mylog_error("unable to bind to tcp socket: %s",
                    strerror( errno ) );
        close( i32fd );
        return -1;
        }
    if( listen( i32fd, 5 ) < 0 )
        {
        mylog_error("error when attempting to listen on tcp socket: %s",
                    strerror( errno ) );
        close( i32fd );
        return -1;
        }

    psTcpListener = (struct tcp_listener *)malloc( sizeof( struct tcp_listener ) );
    psTcpListener->fd = i32fd;

    event_add_fd( i32fd, 0, 0, do_accept, psTcpListener, NULL);

    mylog_info("Socket %d listening on tcp port %d", i32fd, i32Port );

    return 0;
    }

/********************* GLOBAL CONFIGURATION DIRECTIVES ********************/
int tcp_config_port( int num_tokens, struct token *psToken, void *d )
    {
    int i32Port;
    i32Port = psToken[1].v.num;

    if( i32Port <= 0 || i32Port > 65535 )
        {
        mylog_error("invalid listen i32Port %d", i32Port );
        return -1;
        }

    return tcp_listen( i32Port );
    }
