/** Copyright (c) 2015 Cvisionhk */

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#include <event.h>
#include <frame.h>
#include <stream.h>
#include "mylogger.h"

/* define macro for YUV */
#define STREAM_YUV_TO_RGB(i32y, i32u, i32v, i32r, i32g, i32b)\
  i32r = i32y + ( ((i32v << 10) + (i32v << 8) + (i32v << 7) + (i32v << 4) + (i32v << 3) + (i32v << 1)) >> 11);\
  i32g = i32y - ( ((i32u << 8) + (i32u << 7) + (i32u << 4) + (i32u << 2) + (i32u << 1)) >> 11 ) - \
          ( ((i32v << 9) + (i32v << 6) + (i32v << 4) + (i32v << 1) + i32v) >> 11 );\
  i32b = i32y + ( ((i32u << 11) + (i32u << 5) - (i32u << 1)) >> 11);\
  i32r = i32r < 0 ? 0 : i32r;\
  i32g = i32g < 0 ? 0 : i32g;\
  i32b = i32b < 0 ? 0 : i32b;\
  i32r = i32r > 255 ? 255 : i32r;\
  i32g = i32g > 255 ? 255 : i32g;\
  i32b = i32b > 255 ? 255 : i32b

#if 0
#define STREAM_RGB_TO_YUV(r, g, b, y, u, v)\
  y = (9798*r + 19235*g + 3736*b)  >> 15;\
  u = ((-4784*r - 9437*g + 14221*b) >> 15)  + 128;\
  v = ((20218*r - 16941*g - 3277*b) >> 15) + 128;\
  y = y < 0 ? 0 : y;\
  u = u < 0 ? 0 : u;\
  v = v < 0 ? 0 : v;\
  y = y > 255 ? 255 : y;\
  u = u > 255 ? 255 : u;\
  v = v > 255 ? 255 : v

#endif

void uyvy2rgb (char *pcYUV, char *pcRGB, int NumPixels)
    {
    register int y0, y1, u, v;
    register int i32R, i32G, i32B;
    int i, j;

    for (i = 0, j = 0; i < (NumPixels << 1); i += 4, j += 6)
        {
        u = (unsigned char) pcYUV[i + 0] - 128;
        y0 = (unsigned char) pcYUV[i + 1];
        v = (unsigned char) pcYUV[i + 2] - 128;
        y1 = (unsigned char) pcYUV[i + 3];
        STREAM_YUV_TO_RGB (y0, u, v, i32R, i32G, i32B);
        pcRGB[j + 0] = i32R;
        pcRGB[j + 1] = i32G;
        pcRGB[j + 2] = i32B;
        STREAM_YUV_TO_RGB (y1, u, v, i32R, i32G, i32B);
        pcRGB[j + 3] = i32R;
        pcRGB[j + 4] = i32G;
        pcRGB[j + 5] = i32B;
        }
    }

/* end of define */
struct converter
    {
    struct stream_destination *input;
    struct stream *output;
    };

static struct stream *stream_list = NULL;

static void convert_uyvy_to_rgb24( struct frame *uyvy, void *d )
    {
    struct stream *s = (struct stream *)d;
    struct frame *rgb;

    rgb = frame_new('c');
    if(rgb == NULL)
        return;

    rgb->format = RTSP_FORMAT_RAW_RGB24;
    rgb->width = uyvy->width;
    rgb->height = uyvy->height;
    rgb->length = uyvy->height * uyvy->width * 3;
    rgb->key = uyvy->key;
    uyvy2rgb((char *)uyvy->d, (char *)rgb->d, uyvy->length / 2 );
    frame_unref_frame( uyvy, NULL);
    stream_deliver_frame_to_stream( rgb, s );
    }

static void get_framerate( struct stream *s, int *fincr, int *fbase )
    {
    struct stream_destination *dest =
        (struct stream_destination *)s->private;

    dest->stream->get_framerate( dest->stream, fincr, fbase );
    }

static void set_running( struct stream *s, int running )
    {
    stream_set_waiting( (struct stream_destination *)s->private, running );
    }

void stream_deliver_frame_to_stream( struct frame *psFrame, void *d )
    {
    struct stream *psStream = (struct stream *)d;
    struct stream_destination *dest;
    for( dest = psStream->dest_list; dest; dest = dest->next )
        {
        if( ! dest->waiting )
            {
            continue;
            }
        frame_ref_frame( psFrame, NULL );
        dest->process_frame( psFrame, dest->d );
        }
    frame_unref_frame( psFrame, NULL);
    }

int format_match( int format, int *formats, int format_count )
    {
    int i;

    if( format_count == 0 ) return 1;

    for( i = 0; i < format_count; ++i )
        {

        if( format == formats[i] ) return 1;
        }
    return 0;
    }

static struct stream_destination *new_dest( struct stream *s,
        frame_deliver_func process_frame, void *d )
    {
    struct stream_destination *dest;

    dest = (struct stream_destination *)
           malloc( sizeof( struct stream_destination ) );

    dest->next = s->dest_list;
    dest->prev = NULL;
    if( dest->next ) dest->next->prev = dest;
    s->dest_list = dest;
    dest->stream = s;
    dest->waiting = 0;
    dest->process_frame = process_frame;
    dest->d = d;

    return dest;
    }

struct stream *new_convert_stream( struct stream *psStreamBase, int format,
                                   frame_deliver_func convert )
    {
    struct stream *psStream;

    psStream = (struct stream *)malloc( sizeof( struct stream ) );
    psStream->next = psStreamBase->next;
    psStream->prev = psStreamBase;
    psStreamBase->next = psStream;
    if( psStream->next ) psStream->next->prev = psStream;
    strcpy( psStream->name, psStreamBase->name );
    psStream->format = format;
    psStream->dest_list = NULL;
    psStream->get_framerate = get_framerate;
    psStream->set_running = set_running;
    psStream->private = new_dest( psStreamBase, convert, psStream );
    return psStream;
    }

struct stream_destination *connect_to_stream( char *name,
        frame_deliver_func process_frame, void *d,
        int *formats, int format_count )
    {
    struct stream *s;
    int found_one = 0;

    for( s = stream_list; s; s = s->next )
        {
        if( ! strcmp( s->name, name ) && format_match( s->format, formats, format_count ) )
            {
            return new_dest( s, process_frame, d );
            }
        }

    for( s = stream_list; s; s = s->next )
        {
        if( ! strcmp( s->name, name ) )
            {
            found_one = 1;
            switch( s->format )
                {
                case RTSP_FORMAT_RAW_UYVY:
                    if( format_match( RTSP_FORMAT_RAW_RGB24,
                                      formats, format_count ) )
                        {
                        s = new_convert_stream( s,
                                                RTSP_FORMAT_RAW_RGB24,
                                                convert_uyvy_to_rgb24 );
                        return new_dest( s, process_frame, d );
                        }
                    break;
                }
            }
        }

    if( found_one )
        mylog_error("unable to convert stream %s", name );
    return NULL;
    }

void stream_del_stream( struct stream *psStream )
    {
    if(psStream)
        {
        if( psStream->next ) psStream->next->prev = psStream->prev;
        if( psStream->prev ) psStream->prev->next = psStream->next;
        else stream_list = psStream->next;
        free( psStream );
        }
    }

struct stream *stream_new_stream( char *pcName, int i32Format, void *d )
    {
    struct stream *psStream;

    psStream = (struct stream *)malloc( sizeof( struct stream ) );
    psStream->next = stream_list;
    psStream->prev = NULL;
    if( psStream->next ) psStream->next->prev = psStream;
    stream_list = psStream;
    strcpy( psStream->name, pcName );
    psStream->format = i32Format;
    psStream->dest_list = NULL;
    psStream->get_framerate = NULL;
    psStream->set_running = NULL;
    psStream->private = d;
    return psStream;
    }

void stream_set_waiting( struct stream_destination *dest, int waiting )
    {
    struct stream *s = dest->stream;

    /* We call set_running every time a destination starts listening,
     * or when the last destination stops listening.  It is good to know
     * when new listeners come so maybe the source can send a keyframe. */

    mylog_info( "%s Track %s dest->waiting:%d waiting:%d", __FUNCTION__, s->name, dest->waiting, waiting) ;
    if( dest->waiting )
        {
        if( waiting )
            {
            return; /* no change in status */
            }
        dest->waiting = 0;

        /* see if anybody else is listening */
        for( dest = s->dest_list; dest; dest = dest->next )
            waiting |= dest->waiting;

        if( waiting )
            {
            return; /* others are still listening */
            }
        }
    else
        {
        if( ! waiting )
            return; /* no change in status */

        dest->waiting = 1;
        }

    s->set_running( s, waiting );
    }
