/*
 * Copyright (C) 2004 Nathan Lutchansky <lutchann@litech.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <pthread.h>
#include <errno.h>

#include <event.h>
#include <frame.h>
#include <outputs.h>
#include <conf_parse.h>
#include "../../inc/rtsp/service_rtsp.h"
#include "mylogger.h"

//int read_config_file( char *config_file );
int rtsp_log_init( int min );
//int alaw_init(void);
int tcp_config_port( int num_tokens, struct token *psToken, void *d );
void *find_context(char *szType, char *szName);
void *run_context_start_block(void *pvCtx);
int run_context_end_block(void *pvCtx, void *pvCtxData);
int run_context_statement(void *pvCtx, void *pvCtxData, struct token *t, int num_tokens);

static int init_random(void)
    {
    int fd;
    unsigned int seed;

    if( ( fd = open( "/dev/urandom", O_RDONLY ) ) < 0 )
        {
        mylog_error("unable to open /dev/urandom: %s",
                    strerror( errno ) );
        return -1;
        }
    if( read( fd, &seed, sizeof( seed ) ) < 0 )
        {
        mylog_error("unable to read from /dev/urandom: %s",
                    strerror( errno ) );
        return -1;
        }
    close( fd );
    srandom( seed );
    return 0;
    }

void rtsp_origin_random_bytes( unsigned char *dest, int len )
    {
    int i;

    for( i = 0; i < len; ++i )
        dest[i] = random() & 0xff;
    }

void rtsp_origin_random_id( unsigned char *dest, int len )
    {
    int i;

    for( i = 0; i < len / 2; ++i )
        sprintf((char*)dest + i * 2, "%02X",
                (unsigned int)( random() & 0xff ));
    dest[len] = 0;
    }

int rtsp_origin_init(void)
    {
    if(init_random() < 0)
        return -1;
    h264_init();
    alaw_init();
    live_init();
    audio_only_init();
    return 0;
    }

/********************* GLOBAL CONFIGURATION DIRECTIVES ********************/

int rtsp_origin_config_frameheap( int num_tokens, struct token *tokens, void *d )
    {
    int size, count;

    //mylog_info("%s enter", __FUNCTION__ );

    signal( SIGPIPE, SIG_IGN );

    count = tokens[1].v.num;
    if(num_tokens == 3)
        size = tokens[2].v.num;
    else
        size = 352 * 288 * 3;

    //mylog_info("%s framesize is :%d", __FUNCTION__, size );
    mylog_info("frame size is %d, count %d. Total %uKBs", size, count, size * count / 1024);

    if( count < 10 )
        {
        mylog_error("frame heap of %d frames is too small, use at least 10", count );
        return -1;
        }

    frame_init_frame_heap( size, count, 'c');

    return 0;
    }

/*************************************************************************/
static int h264_config(
    S_RTSP_CONFIG *psConfig
)
    {
    char *szCtxType = "encoder";
    char *szCtxName = "h264";
    struct token tokens[2];

    void *pvH264Ctx = NULL;
    void *pvCtxData = NULL;
    int i32Ret = 0;

    pvH264Ctx = find_context(szCtxType, szCtxName);
    if(pvH264Ctx == NULL)
        return -1;

    pvCtxData = run_context_start_block(pvH264Ctx);
    if(pvCtxData == NULL)
        return -1;

    tokens[0].type = TOKEN_STR;
    strcpy(tokens[0].v.str, "output");
    tokens[1].type = TOKEN_STR;
    sprintf(tokens[1].v.str, "\"compressed-h264%d\"", 0);
    run_context_statement(pvH264Ctx, pvCtxData, tokens, 2);

    i32Ret = run_context_end_block(pvH264Ctx, pvCtxData);
    if(i32Ret < 0)
        return i32Ret;

    mylog_info("%s return:%d\n ", __FUNCTION__, i32Ret);
    return i32Ret;
    }

static int
audio_config(S_RTSP_CONFIG *psConfig, char* type)
    {
    char *szCtxType = "encoder";
    char *szCtxName = type;
    struct token tokens[2];

    void *pvAACCtx = NULL;
    void *pvCtxData = NULL;
    int i32Ret = 0;

    pvAACCtx = find_context(szCtxType, szCtxName);
    if(pvAACCtx == NULL)
        return -1;

    pvCtxData = run_context_start_block(pvAACCtx);
    if(pvCtxData == NULL)
        return -1;

    tokens[0].type = TOKEN_STR;
    strcpy(tokens[0].v.str, "output");
    tokens[1].type = TOKEN_STR;
    sprintf(tokens[1].v.str, "\"%s\"", type);

    run_context_statement(pvAACCtx, pvCtxData, tokens, 2);

    i32Ret = run_context_end_block(pvAACCtx, pvCtxData);

    return i32Ret;
    }

static int h264_live_config(S_RTSP_CONFIG *psConfig)
    {
    char *szCtxType = "rtsp-handler";
    char *szCtxName = "live";
    struct token tokens[5];

    void *pvLiveCtx = NULL;
    void *pvCtxData = NULL;

    pvLiveCtx = find_context(szCtxType, szCtxName);
    if(pvLiveCtx == NULL)
        return -1;

    pvCtxData = run_context_start_block(pvLiveCtx);
    if(pvCtxData == NULL)
        return -1;

    // TOKEN[0]: PATH -- calls --> (live.c) set_path
    tokens[0].type = TOKEN_STR;
    strcpy(tokens[0].v.str, "Path");
    tokens[1].type = TOKEN_STR;

    if(psConfig->m_szH264CamPath)
        strcpy(tokens[1].v.str, psConfig->m_szH264CamPath);
    else
        strcpy(tokens[1].v.str, "/webcam-h264");

    if(psConfig->m_uiAuthen)
        {
        /* VINH : set username and password */
        tokens[2].type = TOKEN_STR;
        strcpy(tokens[2].v.str, "realm");
        tokens[3].type = TOKEN_STR;
        strcpy(tokens[3].v.str, psConfig->m_szUser);
        tokens[4].type = TOKEN_STR;
        strcpy(tokens[4].v.str, psConfig->m_szPass);
        run_context_statement(pvLiveCtx, pvCtxData, tokens, 5);
        }
    else
        run_context_statement(pvLiveCtx, pvCtxData, tokens, 2);

    mylog_info( "%s,  setup normal RTP tracks", __FUNCTION__);
    ///////// SETUP NORMAL RTP Tracks ///////////////
    // TOKEN[0]: Track -- calls --> (live.c) set_track (h264)
    tokens[0].type = TOKEN_STR;
    strcpy(tokens[0].v.str, "Track");
    tokens[1].type = TOKEN_STR;
    sprintf(tokens[1].v.str, "\"compressed-h264%d\"", 0);
    run_context_statement(pvLiveCtx, pvCtxData, tokens, 2);

    // TOKEN[0]: Track -- calls --> (live.c) set_track (Alaw)
    tokens[0].type = TOKEN_STR;
    strcpy(tokens[0].v.str, "Track");
    tokens[1].type = TOKEN_STR;
    strcpy(tokens[1].v.str, "\"alaw\"");
    run_context_statement(pvLiveCtx, pvCtxData, tokens, 2);
    run_context_end_block(pvLiveCtx, pvCtxData);
    return 0;
    }

static int audio_only_config(S_RTSP_CONFIG *psConfig)
    {
    char *szCtxType = "rtp";
    char *szCtxName = "audio_only";
    struct token tokens[5];

    void *pvLiveCtx = NULL;
    void *pvCtxData = NULL;

    pvLiveCtx = find_context(szCtxType, szCtxName);
    if(pvLiveCtx == NULL)
        return -1;

    pvCtxData = run_context_start_block(pvLiveCtx);
    if(pvCtxData == NULL)
        return -1;

    // TOKEN[0]: PATH -- calls --> (live.c) set_path
    tokens[0].type = TOKEN_STR;
    strcpy(tokens[0].v.str, "Path");
    tokens[1].type = TOKEN_STR;
    strcpy(tokens[1].v.str, psConfig->m_szAudioOnlyPath);
    if(psConfig->m_uiAuthen)
        {
        /* VINH : set username and password */
        tokens[2].type = TOKEN_STR;
        strcpy(tokens[2].v.str, "realm");
        tokens[3].type = TOKEN_STR;
        strcpy(tokens[3].v.str, psConfig->m_szUser);
        tokens[4].type = TOKEN_STR;
        strcpy(tokens[4].v.str, psConfig->m_szPass);
        run_context_statement(pvLiveCtx, pvCtxData, tokens, 5);
        }
    else
        run_context_statement(pvLiveCtx, pvCtxData, tokens, 2);

    mylog_info( "%s,  setup normal RTP tracks", __FUNCTION__);
    ///////// SETUP NORMAL RTP Tracks ///////////////
    // TOKEN[0]: Track -- calls --> (live.c) set_track (Alaw)
    tokens[0].type = TOKEN_STR;
    strcpy(tokens[0].v.str, "Track");
    tokens[1].type = TOKEN_STR;
    strcpy(tokens[1].v.str, "\"alaw\"");
    run_context_statement(pvLiveCtx, pvCtxData, tokens, 2);
    run_context_end_block(pvLiveCtx, pvCtxData);
    return 0;
    }

int rtsp_origin_config(
    S_RTSP_CONFIG *psConfig
)
    {
    struct token tokens[3];
    int i32H264Ret;
    int i32AudioRet;

    if(psConfig == NULL)
        {
        return -1;
        }
    else if(psConfig->m_uiIsInitialized)
        {
        mylog_info("%s -1 before h264_live_config\n", __FUNCTION__);
        if(h264_live_config(psConfig) < 0)
            mylog_info("%s Failed to configure h264 live cam", __FUNCTION__);
        return 0;
        }

    tokens[1].v.num = psConfig->m_uiFrameCnt;
    tokens[2].v.num = psConfig->m_uiFrameSize;

    if(rtsp_origin_config_frameheap(3, tokens, NULL) < 0)
        {
        mylog_info("%s Failed to configure frame heap\n", __FUNCTION__);
        return -1;
        }

    if(psConfig->m_uiPort)
        tokens[1].v.num = psConfig->m_uiPort;
    else
        tokens[1].v.num = 6667;
    if(tcp_config_port(2, tokens, NULL) < 0)
        {
        mylog_info("%s Failed to configure RTSP port", __FUNCTION__);
        return -1;
        }

    i32H264Ret = h264_config(psConfig);
    if(i32H264Ret < 0)
        {
        mylog_info("%s Failed to configure h264", __FUNCTION__);
        }

    i32AudioRet = audio_config(psConfig, "alaw");
    if(i32AudioRet < 0)
        {
        mylog_info("%s: Failed to configure alaw", __FUNCTION__);
        return -1;
        }

    if( (i32H264Ret == 0) && (i32AudioRet == 0))
        {
        if(h264_live_config(psConfig) < 0)
            mylog_info("%s Failed to configure h264 live cam", __FUNCTION__);
        if(audio_only_config(psConfig) < 0)
            mylog_info("%s Failed to configure audio only stream", __FUNCTION__);
        }
    else
        {
        mylog_info("%s Unable to enable h264 live cam", __FUNCTION__);
        }
    psConfig->m_uiIsInitialized = 1;
    return 0;
    }

void rtsp_origin_run(void)
    {
    event_loop(1, NULL);
    }

void rtsp_origin_stop(void)
    {
    rtsp_set_closing(1);
    event_exit_event_loop();
    }

