/*
 * Copyright (C) 2015 CVision.
 *
 * This unpublished material is proprietary to CVision.
 * All rights reserved. The methods and
 * techniques described herein are considered trade secrets
 * and/or confidential. Reproduction or distribution, in whole
 * or in part, is forbidden except by express written permission
 * of CVision.
 */

/* FILE   : encoder-h264.c
 * AUTHOR : leon nguyen ☻
 * DATE   : Feb 17, 2016
 * DESC   : This file contains all API related to get AAC frame
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdint.h>
#include <sys/ioctl.h>

#include <event.h>
#include <frame.h>
#include <stream.h>
#include <conf_parse.h>

#include "service_rtsp.h"
#include "extra.h"
#include "mylogger.h"

typedef struct tRTSPVidEncoder
    {
    struct stream           *output;
    struct frame_exchanger  *ex;
    int                     format;
    int                     running;
    pthread_t               thread;
    }tRTSPVidEncoder;

static tRTSPVidEncoder  *gRTSPVidEncPtr = NULL;

static void set_running(struct stream *s, int running)
    {
    struct tRTSPVidEncoder *en = (struct tRTSPVidEncoder *) s->private;
    en->running = running;
    if(running)
        mylog_info("Enable h264 encode pipe resource\n");
    else
        mylog_info("Disable h264 encode pipe resource\n");
    }

/************************ CONFIGURATION DIRECTIVES ************************/

static void *start_block(void)
    {
    struct tRTSPVidEncoder *en;
    en = (struct tRTSPVidEncoder *) malloc(sizeof(struct tRTSPVidEncoder));
    if(en == NULL)
        mylog_info("%s, malloc Failed", __FUNCTION__);
    else
        {
        memset(en, 0x00, sizeof(struct tRTSPVidEncoder));
        en->output = NULL;
        }
    return en;
    }

static int end_block(void *d)
    {
    struct tRTSPVidEncoder *en = (struct tRTSPVidEncoder *) d;
    if(!en->output)
        {
        mylog_error("h264: missing output stream name");
        return -1;
        }
    gRTSPVidEncPtr = en;
    en->ex = frame_new_exchanger(8, stream_deliver_frame_to_stream, en->output, NULL);
    return 0;
    }

static void get_framerate(struct stream *s, int *fincr, int *fbase)
    {
    *fincr = 1;
    *fbase = 15; //Frame rate
    if(*fbase == 0)
        *fbase = 15;
    }

static int set_output(int num_tokens, struct token *tokens, void *d)
    {
    struct tRTSPVidEncoder *en = (struct tRTSPVidEncoder *) d;
    en->output = stream_new_stream(tokens[1].v.str, RTSP_FORMAT_H264, en);
    if(!en->output)
        {
        mylog_error("h264: unable to create stream \"%s\"", tokens[1].v.str);
        return -1;
        }
    en->output->get_framerate = get_framerate;
    en->output->set_running = set_running;
    return 0;
    }

static struct statement config_statements[] =
    {
    /* directive name, process function, min args, max args, arg types */
        { "output", set_output, 1, 1, { TOKEN_STR } },
    /* empty terminator -- do not remove */
        { NULL, NULL, 0, 0, {} }
    };

static int isH264KeyFrame(char* data, int size)
    {
    const int syncWordLength = 4;
    int fragment_type = data[0] & 0x1F;
    int nal_type = data[1] & 0x1F;
    int start_bit = data[1] & 0x80;

    if(size < syncWordLength + 2)
        return 0;

    if(((fragment_type == 28 || fragment_type == 29) && nal_type == 5 && start_bit == 128) ||
         fragment_type == 5)
        {
        return 1;
        }
    return 0;
   }


int h264_init(void)
    {
    register_config_context("encoder", "h264", start_block, end_block, config_statements);
    return 0;
    }

void rtsp_deliver_video_frame(void* data, int size, int isKeyFrame, int pts)
    {
    struct frame    *psRTSPFrame = NULL;

    if(gRTSPVidEncPtr == NULL)
        return;
    if(!gRTSPVidEncPtr->running)
        return;

    if(size <= 0)
        return;

    if(frame_get_max_frame_size('c') < size)
        {
        mylog_error("Frame too big %u. Drop", size);
        return;
        }

    psRTSPFrame = frame_new('c');
    if(psRTSPFrame == NULL)
        {
        mylog_error("%s: ERROR: can't allocate frame", __FUNCTION__);
        return;
        }
    psRTSPFrame->format = RTSP_FORMAT_H264;
    psRTSPFrame->length = size;
    psRTSPFrame->size = size;
    memcpy(psRTSPFrame->d, data, size);
    if(isKeyFrame <= 0)
        psRTSPFrame->key = isH264KeyFrame(data, size);
    else
        psRTSPFrame->key = isKeyFrame;
    psRTSPFrame->pts = 0;
    frame_deliver_frame(gRTSPVidEncPtr->ex, psRTSPFrame, NULL);
    }
