/*
 * Copyright (C) 2015 CVision.
 *
 * This unpublished material is proprietary to CVision.
 * All rights reserved. The methods and
 * techniques described herein are considered trade secrets
 * and/or confidential. Reproduction or distribution, in whole
 * or in part, is forbidden except by express written permission
 * of CVision.
 */

/* FILE   : encoder-alaw.c
 * AUTHOR : leon nguyen ☻
 * DATE   : Feb 19, 2016
 * DESC   : This file is implement for change RTSP stream format
 */

#include <stdint.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <pthread.h>
#include <math.h>

#include <event.h>
#include <frame.h>
#include <stream.h>
#include <conf_parse.h>
#include "service_rtsp.h"

#include "extra.h"
#include "mylogger.h"
#include "audio_conv.h"

typedef struct tRTSPAudConfig
    {
    uint32_t m_uiSampleRate;
    uint16_t m_ui16Channel;
    uint16_t m_ui16SmplPerFrame;
    uint16_t m_ui16BytePerFrame;
    }tRTSPAudConfig;

typedef struct tRTSPAudEncoder
    {
    struct stream           *output;
    struct frame_exchanger  *ex;
    int                     format;
    int                     running;
    pthread_t               thread;
    tRTSPAudConfig          cfg;
    }tRTSPAudEncoder;

/*****************************************************************************
 *                                                                           *
 *                              GLOBAL  VARIABLES                            *
 *                                                                           *
 ****************************************************************************/
static tRTSPAudEncoder *gRTSPAudEncPtr = NULL;

void
rtsp_audio_cfg_set(int sampleRate, int channel, int sampleSize, int frameSize)
    {
    if(gRTSPAudEncPtr)
        {
        gRTSPAudEncPtr->cfg.m_uiSampleRate = sampleRate;
        gRTSPAudEncPtr->cfg.m_ui16Channel = channel;
        gRTSPAudEncPtr->cfg.m_ui16SmplPerFrame = sampleSize;
        gRTSPAudEncPtr->cfg.m_ui16BytePerFrame = frameSize;
        }
    }

void rtsp_audio_cfg_get(int* sampleRate, int* channel, int* sampleSize, int* frameSize)
    {
    if(sampleRate && gRTSPAudEncPtr)
        *sampleRate = gRTSPAudEncPtr->cfg.m_uiSampleRate;
    if(channel && gRTSPAudEncPtr)
        *channel = gRTSPAudEncPtr->cfg.m_ui16Channel;
    if(sampleSize)
        *sampleSize = 2;//gAudioConfig->m_ui16BytePerFrame / gAudioConfig->m_ui16SmplPerFrame;
    if(frameSize)
        *frameSize = 2048; // temporary hard code
    }


/*******************************************************************
 *                     Platform function register
 *******************************************************************/
static void get_framerate( struct stream *s, int *fincr, int *fbase )
    {
    struct tRTSPAudEncoder *en = (struct tRTSPAudEncoder *)s->private;
    *fincr = 1;
    *fbase = en->cfg.m_uiSampleRate;
    }

static void set_running( struct stream *s, int running )
    {
    struct tRTSPAudEncoder *en = (struct tRTSPAudEncoder *)s->private;
    mylog_info("%s enter running is :%d \n", __FUNCTION__, running);
    en->running = running;
    }

/************************ CONFIGURATION DIRECTIVES ************************/

static void *start_block(void)
    {
    struct tRTSPAudEncoder *en;

    en = (struct tRTSPAudEncoder *)malloc( sizeof( struct tRTSPAudEncoder ) );
    memset(en, 0x00, sizeof(struct tRTSPAudEncoder));
    en->output = NULL;
    return en;
    }

static int end_block( void *d )
    {
    struct tRTSPAudEncoder *en = (struct tRTSPAudEncoder *)d;
    if( ! en->output )
        {
        mylog_error("alaw: missing output stream name" );
        return -1;
        }

    gRTSPAudEncPtr = en;
    rtsp_audio_cfg_set(8000, 2, 2, 2048);

    //create import alaw thread
    en->ex = frame_new_exchanger( 4, stream_deliver_frame_to_stream, en->output, NULL);
    return 0;
    }


static int set_output( int num_tokens, struct token *psToken, void *arg)
    {
    struct tRTSPAudEncoder *en = (struct tRTSPAudEncoder *)arg;

    en->output = stream_new_stream( psToken[1].v.str, RTSP_FORMAT_ALAW, en );
    if( ! en->output )
        {
        mylog_error("alaw: unable to create stream \"%s\"",
                    psToken[1].v.str );
        return -1;
        }
    en->output->get_framerate = get_framerate;
    en->output->set_running = set_running;
    return 0;
    }

static struct statement alaw_config_statements[] =
    {
    /* directive name, process function, min args, max args, arg types */
        { "output", set_output, 1, 1, { TOKEN_STR } },
    /* empty terminator -- do not remove */
        { NULL, NULL, 0, 0, {} }
    };


int alaw_init(void)
    {
    register_config_context( "encoder", "alaw", start_block, end_block,
                             alaw_config_statements );
    return 0;
    }

void rtsp_deliver_audio_frame(void* data, int size, int isKeyFrame, int pts)
    {
    struct frame    *psRTSPFrame = NULL;

    if(gRTSPAudEncPtr == NULL)
        return;
    if(!gRTSPAudEncPtr->running)
        return;

    if(size <= 0)
        return;

    if(frame_get_max_frame_size('c') < size)
        {
        mylog_error("Frame too big %u. Drop", size);
        return;
        }

    if(size > gRTSPAudEncPtr->cfg.m_ui16BytePerFrame)
        size = gRTSPAudEncPtr->cfg.m_ui16BytePerFrame;

    psRTSPFrame = frame_new('c');
    if(psRTSPFrame == NULL)
        {
        mylog_error("%s: ERROR: can't allocate frame", __FUNCTION__);
        return;
        }
    psRTSPFrame->format = RTSP_FORMAT_ALAW;
    psRTSPFrame->width = 0;
    psRTSPFrame->height = 0;
    psRTSPFrame->key = isKeyFrame;
    psRTSPFrame->pts = 0;
    psRTSPFrame->length = size / 2;
    psRTSPFrame->size = size / 2;

    /* Encode S16LE to Alaw data */
    sound_alaw_le_enc((char *) data, (char *) psRTSPFrame->d, size);
    frame_deliver_frame(gRTSPAudEncPtr->ex, psRTSPFrame, NULL);
    }
