/** Copyright (c) 2015 Cvisionhk */

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <event.h>
#include <frame.h>
#include <stream.h>
#include <pmsg.h>
#include <rtp.h>
#include <conf_parse.h>
#include "mylogger.h"
#include "cli_event.h"

void rtsp_write_log(struct sockaddr *addr, int i32Id, char *pCharReq,
                    int i32Len, char *pCharRef, char *pCharUser );

struct rtsp_session
    {
    struct rtsp_session *next;
    struct rtsp_session *prev;
    char id[32];
    struct session *sess;
    };

struct rtsp_location
    {
        struct rtp_loc_node node;
        char realm[128];
        char username[128];
        char password[128];
        open_func open;
        void *private;
    };

struct rtsp_conn
    {
    struct
        {
        struct rtp_endpoint *ep;
        int rtcp;
        } ichan[RTP_MAX_INTERLEAVE_CHANNELS];
    };

static struct rtsp_location *rtsp_loc_list = NULL;
static struct rtsp_session *sess_list = NULL;

static int nb_ext_client = 0;
static int nb_client = 0;
static int force_closing = 0;

int rtsp_increase_nb_ext_clients();
int rtsp_decrease_nb_ext_clients();
int rtsp_set_closing(int value);

static char *rtsp_get_local_path( char *pcPath )
    {
    char *pc;
    static char *pcRootPath = "/";
    if( strncasecmp( pcPath, "rtsp://", 7 ) ) return NULL;
    for( pc = pcPath + 7; *pc != '/'; ++pc ) if( *pc == 0 ) return pcRootPath;
    return pc;
    }

static struct rtp_loc_node *rtsp_node_find_location( struct rtp_loc_node *psLocNodeList,
        char *pcPath, int i32Len )
    {
    struct rtp_loc_node *psLocNode;

    for( psLocNode = psLocNodeList; psLocNode; psLocNode = psLocNode->next )
        if( ! strncmp( pcPath, psLocNode->path, strlen( psLocNode->path ) )
                && ( psLocNode->path[i32Len] == '/'
                     || psLocNode->path[i32Len] == 0 ) )
            return psLocNode;
    return NULL;
    }

static struct rtsp_location *rtsp_find_location( char *pcUri, char *pcBase, int *track )
    {
    char *pcPath, *c, *end;
    int len;

    if( ! ( pcPath = rtsp_get_local_path( pcUri ) ) ) return NULL;

    len = strlen( pcPath );

    if( track )
        {
        *track = -1;
        if( ( c = strrchr( pcPath, '/' ) ) &&
                ! strncmp( c, "/track", 6 ) )
            {
            *track = strtol( c + 6, &end, 10 );
            if( ! *end ) len -= strlen( c );
            }
        }

    if( len > 1 && pcPath[len - 1] == '/' ) --len;

    if( pcBase )
        {
        strncpy( pcBase, pcPath, len );
        pcBase[len] = 0;
        }

    return (struct rtsp_location *)rtsp_node_find_location(
               (struct rtp_loc_node *)rtsp_loc_list, pcPath, len );
    }

static void rtsp_init_location( struct rtp_loc_node *psLocNode, char *pcPath,
                                struct rtp_loc_node **ppsLocNodeList )
    {
    psLocNode->next = *ppsLocNodeList;
    psLocNode->prev = NULL;
    if( psLocNode->next ) psLocNode->next->prev = psLocNode;
    *ppsLocNodeList = psLocNode;
    strcpy( psLocNode->path, pcPath );
    }

void rtsp_new_location( char *pcPath, char *pcRealm, char *pcUsername, char *pcPassword,
                        open_func open, void *private )
    {
    struct rtsp_location *psRtspLoc;
    psRtspLoc = (struct rtsp_location *)malloc( sizeof( struct rtsp_location ) );
    rtsp_init_location( (struct rtp_loc_node *)psRtspLoc, pcPath,
                        (struct rtp_loc_node **)&rtsp_loc_list );
    if( pcRealm ) strcpy( psRtspLoc->realm, pcRealm );
    else psRtspLoc->realm[0] = 0;
    if( pcUsername ) strcpy( psRtspLoc->username, pcUsername );
    else psRtspLoc->username[0] = 0;
    if( pcPassword ) strcpy( psRtspLoc->password, pcPassword );
    else psRtspLoc->password[0] = 0;
    psRtspLoc->open = open;
    psRtspLoc->private = private;
    }

static void rtsp_session_close( struct session *sess )
    {
    struct rtsp_session *rs = (struct rtsp_session *)sess->control_private;


    //Use NEO, judge every clients as real streaming client
    //if ( strcmp(sess->addr,"IP4 127.0.0.1") !=0)
        {
        //Non-local client
        rtsp_decrease_nb_ext_clients();
        }

    nb_client--;
    if (nb_client < 0)
        {
        nb_client = 0;
        }
    mylog_debug("freeing session %s @ %s ext_client:%d", "xxxx" , "xxxx", 0xFF);

    if(rs)
        {
        if( rs->next ) rs->next->prev = rs->prev;
        if( rs->prev ) rs->prev->next = rs->next;
        else sess_list = rs->next;
        free( rs );
        }
    }

static struct rtsp_session *rtsp_new_session( struct session *sess, int isLocal)
    {
    struct rtsp_session *psRtspSess;

    mylog_info("rtsp_new_session() ALLOCATE sesstion %d bytes", sizeof( struct rtsp_session ));
    psRtspSess = (struct rtsp_session *)malloc( sizeof( struct rtsp_session ) );
    psRtspSess->next = sess_list;
    psRtspSess->prev = NULL;
    if( psRtspSess->next ) psRtspSess->next->prev = psRtspSess;
    sess_list = psRtspSess;
    rtsp_origin_random_id( (unsigned char *)psRtspSess->id, 30 );
    psRtspSess->sess = sess;
    sess->control_private = psRtspSess;
    sess->control_close = rtsp_session_close;
    sess->isLocal = isLocal;
    return psRtspSess;
    }

static struct rtsp_session *rtsp_get_session( char *id )
    {
    struct rtsp_session *psRtspSess;

    if( ! id ) return NULL;
    for( psRtspSess = sess_list; psRtspSess; psRtspSess = psRtspSess->next )
        if( ! strcmp( psRtspSess->id, id ) ) break;
    return psRtspSess;
    }

void rtsp_conn_disconnect( struct rtp_conn *psRtpConn )
    {
    struct rtsp_conn *psRc = (struct rtsp_conn *)psRtpConn->proto_state;
    int i;

    for( i = 0; i < RTP_MAX_INTERLEAVE_CHANNELS; ++i )
        {
        if( psRc->ichan[i].ep && ! psRc->ichan[i].rtcp )
            {
            psRc->ichan[i].ep->session->teardown(
                psRc->ichan[i].ep->session,
                psRc->ichan[i].ep );
            }
        }
    if(psRc)
        {
        mylog_debug("rtsp_conn_disconnect() free rtsp_conn %p", psRc);
        free( psRc );
        }
    }

void rtsp_interleave_disconnect( struct rtp_conn *psRtpConn, int i32Chan )
    {
    struct rtsp_conn *rc = (struct rtsp_conn *)psRtpConn->proto_state;
    rc->ichan[i32Chan].ep = NULL;
    }

int rtsp_interleave_recv( struct rtp_conn *psRtpConn, int i32Chan, unsigned char *d, int i32Len )
    {
    struct rtsp_conn *rc = (struct rtsp_conn *)psRtpConn->proto_state;

    if( i32Chan >= RTP_MAX_INTERLEAVE_CHANNELS || ! rc->ichan[i32Chan].ep ) return -1;
    if( rc->ichan[i32Chan].rtcp )
        rtp_interleave_recv_rtcp( rc->ichan[i32Chan].ep, d, i32Len );
    return 0;
    }

int rtsp_interleave_send( struct rtp_conn *psRtpConn, int i32Chan, struct iovec *v, int i32Cnt, int isAudio)
    {
    unsigned char head[4];
    int len = 0, i;

    if(force_closing)
        return -1;

    for(i = 0; i < i32Cnt; ++i) len += v[i].iov_len;
    if(isAudio == 1)
        {
        if(tcp_avail_send_buf(psRtpConn) < (64 * 1024 - 16 * 1024))
            {
            mylog_debug(" return 1 here, chan: %davail:%d", i32Chan, tcp_avail_send_buf(psRtpConn));
            return 1;
            }
        }
    else
        {
        if(tcp_avail_send_buf(psRtpConn) < len + 4)
            return 1;
        }
    head[0] = '$';
    head[1] = i32Chan;
    RTP_PUT_16( head + 2, len );
    tcp_send_data( psRtpConn, head, 4 );
    for( i = 0; i < i32Cnt; ++i )
        tcp_send_data( psRtpConn, v[i].iov_base, v[i].iov_len );

    return 0;
    }

static void rtsp_log_request( struct req *psReq, int i32Code, int length )
    {
    char *ref, *ua;
    ref = pmsg_get_header( psReq->req, "referer" );
    ua = pmsg_get_header( psReq->req, "user-agent" );
    rtsp_write_log((struct sockaddr *)&psReq->conn->client_addr,
                   i32Code, (char *)psReq->conn->req_buf, length,
                   ref ? ref : "-", ua ? ua : "-" );
    }

static int rtsp_create_reply( struct req *psReq, int i32Code, char *pcReason )
    {
    if( ! ( psReq->resp = pmsg_new_pmsg( 512 ) ) ) return -1;
    psReq->resp->type = PMSG_RESP;
    psReq->resp->proto_id = pmsg_add_pmsg_string( psReq->resp, "RTSP/1.0" );
    psReq->resp->sl.stat.code = i32Code;
    psReq->resp->sl.stat.reason = pmsg_add_pmsg_string( psReq->resp, pcReason );
    pmsg_copy_headers( psReq->resp, psReq->req, "CSeq" );
    return 0;
    }

static void rtsp_send_error( struct req *psReq, int i32Code, char *pcReason )
    {
    rtsp_log_request( psReq, i32Code, 0 );
    rtsp_create_reply( psReq, i32Code, pcReason );
    tcp_send_pmsg( psReq->conn, psReq->resp, -1 );
    }

static int rtsp_verify_auth( struct req *psReq, char *pcRealm,
                             char *pcUsername, char *pcPassword )
    {
    int ret = httpauth_check_digest_response( psReq->req, pcRealm, pcUsername, pcPassword );

    if( ret > 0 ) return 0;

    rtsp_log_request( psReq, 401, 0 );
    rtsp_create_reply( psReq, 401, "Unauthorized" );
    add_digest_challenge( psReq->resp, pcRealm, ret == 0 ? 1 : 0 );
    tcp_send_pmsg( psReq->conn, psReq->resp, -1 );
    return -1;
    }

static int handle_OPTIONS( struct req *psReq )
    {
    rtsp_create_reply( psReq, 200, "OK" );
    pmsg_add_header( psReq->resp, "Public",
                     "DESCRIBE, SETUP, TEARDOWN, PLAY, PAUSE" );
    rtsp_log_request( psReq, 200, 0 );
    tcp_send_pmsg( psReq->conn, psReq->resp, -1 );
    return 0;
    }

static int handle_DESCRIBE( struct req *psReq )
    {
    char sdp[8192], path[256], hdr[512];
    int sdp_len;
    struct rtsp_location *psRtspLoc;
    struct session *psSess;

    psRtspLoc = rtsp_find_location( psReq->req->sl.req.uri, path, NULL );
    mylog_info("%s enter uri:%s->loc: %p\n",__FUNCTION__ , psReq->req->sl.req.uri, psRtspLoc);

    if(psRtspLoc != NULL)
        {
        psSess = psRtspLoc->open( path, psRtspLoc->private );
        if(psSess == NULL)
            {
            mylog_info("Can not open session");
            rtsp_send_error( psReq, 405, "Failed to open RTSP session" );
            return 0;
            }
        mylog_info( "session found :%p\n", psSess);
        }
    else
        {
        mylog_info("Can not find rtsp location");
        rtsp_send_error( psReq, 404, "RTSP location not found" );
        return 0;
        }

    if( psRtspLoc->realm[0] && rtsp_verify_auth( psReq, psRtspLoc->realm,psRtspLoc->username, psRtspLoc->password ) < 0 )
        {
        // Fix rtspserver MEM LEAK
        if (psSess)
            {
            psSess->teardown( psSess, NULL );
            }
        mylog_debug("Exiting handle_DESCRIBE()1");
        return 0;
        }

    sdp_len = sizeof( sdp ) - 2;

    if( psSess->get_sdp( psSess, sdp, &sdp_len, psReq->req->sl.req.uri ) > 0 )
        {
        rtsp_create_reply( psReq, 200, "OK" );
        sprintf( hdr, "%s/", psReq->req->sl.req.uri );
        pmsg_add_header( psReq->resp, "Content-Base", hdr );
        pmsg_add_header( psReq->resp, "Content-Type", "application/sdp" );
        rtsp_log_request( psReq, 200, sdp_len );

        if( tcp_send_pmsg( psReq->conn, psReq->resp, sdp_len ) >= 0 )
            {
            tcp_send_data( psReq->conn, (unsigned char *)sdp, sdp_len );
            }
        }
    else
        {
        rtsp_send_error( psReq, 406, "Failed to get SDP" );
        }

    psSess->teardown( psSess, NULL );

    mylog_debug("Exiting handle_DESCRIBE()");
    return 0;
    }

static int rtsp_udp_setup( struct session *psSess, int track,
                           struct req *psReq, char *t )
    {
    char *p, *end, trans[128];
    int i32ClientPort = 0, i32ServerPort = 0;

    if( ! ( p = strstr( t, "client_port" ) ) || *(p + 11) != '=' )
        return -1;
    i32ClientPort = strtol( p + 12, &end, 10 );
    if( end == p + 12 ) return -1;
    mylog_info("client requested UDP port %d", 0xFF );

    if( rtp_connect_udp_endpoint( psSess->ep[track], psReq->conn->client_addr.sin_addr,
                                  i32ClientPort, &i32ServerPort ) < 0 )
        return -1;

    mylog_info("our port is %d, client port is %d", 0xFF, 0xFF );

    sprintf( trans, "RTP/AVP;unicast;client_port=%d-%d;server_port=%d-%d",
             i32ClientPort, i32ClientPort + 1, i32ServerPort, i32ServerPort + 1 );
    pmsg_add_header( psReq->resp, "Transport", trans );
    return 0;
    }

static int rtsp_interleave_setup( struct session *psSess, int track,
                                  struct req *psReq, char *t )
    {
    struct rtsp_conn *psRc = (struct rtsp_conn *)psReq->conn->proto_state;
    char *pc, *end, trans[128];
    int i32RtpChan = -1, i32RtcpChan = -1, i;

    if( ( pc = strstr( t, "interleaved" ) ) )
        {
        if( *(pc + 11) != '=' ) return -1;
        i32RtpChan = strtol( pc + 12, &end, 10 );
        i32RtcpChan = i32RtpChan + 1; // XXX make better parser
        if( end == pc + 12 ) return -1;
        if( i32RtpChan < 0 || i32RtcpChan < 0 ||
                i32RtpChan >= RTP_MAX_INTERLEAVE_CHANNELS ||
                i32RtcpChan >= RTP_MAX_INTERLEAVE_CHANNELS )
            return -1;
        if( psRc && ( psRc->ichan[i32RtpChan].ep ||
                      psRc->ichan[i32RtcpChan].ep ) )
            return -1;
        }
    else
        {
        mylog_info("requested any interleave channel" );
        if( psRc )
            {
            for( i = 0; i < RTP_MAX_INTERLEAVE_CHANNELS; i += 2 )
                if( ! psRc->ichan[i].ep && ! psRc->ichan[i + 1].ep )
                    break;
            if( i >= RTP_MAX_INTERLEAVE_CHANNELS ) return -1;
            i32RtpChan = i;
            i32RtcpChan = i + 1;
            }
        else
            {
            i32RtpChan = 0;
            i32RtcpChan = 1;
            }
        }

    if( !psRc )
        {
        mylog_debug("ALLOCATE rtp_conn %d bytes", sizeof( struct rtsp_conn ));
        psRc = (struct rtsp_conn *)malloc( sizeof( struct rtsp_conn ) );
        if( ! psRc )
            {
            mylog_error("out of memory on malloc rtsp_conn" );
            return -1;
            }
        for( i = 0; i < RTP_MAX_INTERLEAVE_CHANNELS; ++i )
            {
            psRc->ichan[i].ep = NULL;
            }
        psReq->conn->proto_state = psRc;
        }

    psRc->ichan[i32RtpChan].ep = psSess->ep[track];
    psRc->ichan[i32RtpChan].rtcp = 0;
    psRc->ichan[i32RtcpChan].ep = psSess->ep[track];
    psRc->ichan[i32RtcpChan].rtcp = 1;

    rtp_connect_interleaved_endpoint( psSess->ep[track], psReq->conn,
                                      i32RtpChan, i32RtcpChan );

    sprintf( trans, "RTP/AVP/TCP;unicast;interleaved=%d-%d",
             i32RtpChan, i32RtcpChan );
    pmsg_add_header( psReq->resp, "Transport", trans );
    return 0;
    }

static int handle_SETUP( struct req *psReq )
    {
    char *t, *sh, path[256];
    int track, i32Ret;
    struct session *psSess;
    struct rtsp_session *psRtspSess = NULL;
    struct rtsp_location *psRtspLoc;

    if( ! ( psRtspLoc = rtsp_find_location( psReq->req->sl.req.uri,
                                            path, &track ) ) ||
            track < 0 || track >= RTP_MAX_TRACKS )
        {
        rtsp_send_error( psReq, 404, "Not Found" );
        return 0;
        }

    if( psRtspLoc->realm[0] && rtsp_verify_auth( psReq, psRtspLoc->realm,
            psRtspLoc->username, psRtspLoc->password ) < 0 )
        return 0;

    if( ! ( t = pmsg_get_header( psReq->req, "Transport" ) ) )
        {
        // XXX better error reply
        rtsp_send_error( psReq, 461, "Unspecified Transport" );
        return 0;
        }

    if( ! ( sh = pmsg_get_header( psReq->req, "Session" ) ) )
        {
        if( ! ( psSess = psRtspLoc->open( path, psRtspLoc->private ) ) )
            {
            rtsp_send_error( psReq, 404, "Not Found" );
            return 0;
            }
        sprintf( psSess->addr, "IP4 %s",
                 inet_ntoa( psReq->conn->client_addr.sin_addr ) );
        }
    else if( ! ( psRtspSess = rtsp_get_session( sh ) ) )
        {
        rtsp_send_error( psReq, 454, "Session Not Found" );
        return 0;
        }
    else psSess = psRtspSess->sess;

    if( psSess->ep[track] )
        {
        // XXX better error reply
        rtsp_send_error( psReq, 461, "Unsupported Transport" );
        return 0;
        }

    if( psSess->setup( psSess, track ) < 0 )
        {
        rtsp_send_error( psReq, 404, "Not Found" );
        if( ! psRtspSess ) psSess->teardown( psSess, NULL );
        return 0;
        }


    rtsp_create_reply( psReq, 200, "OK" );

    if( ! strncasecmp( t, "RTP/AVP/TCP", 11 ) )
        i32Ret = rtsp_interleave_setup( psSess, track, psReq, t );
    else if( ( ! strncasecmp( t, "RTP/AVP", 7 ) && t[7] != '/' ) ||
             ! strncasecmp( t, "RTP/AVP/UDP", 11 ) )
        i32Ret = rtsp_udp_setup( psSess, track, psReq, t );
    else i32Ret = -1;

    if( i32Ret < 0 )
        {
        pmsg_free_pmsg( psReq->resp );
        rtsp_send_error( psReq, 461, "Unsupported Transport" );
        psSess->teardown( psSess, psSess->ep[track] );
        }
    else
        {
        if( ! psRtspSess ) psRtspSess = rtsp_new_session( psSess, 1 );
        pmsg_add_header( psReq->resp, "Session", psRtspSess->id );
        rtsp_log_request( psReq, 200, 0 );
        tcp_send_pmsg( psReq->conn, psReq->resp, -1 );
        }

    return 0;
    }

static int handle_PLAY( struct req *psReq )
    {
    struct rtsp_session *psRtspSess;
    double start;
    int i32HaveStart = 0, i, i32p, ret;
    char *psRange, hdr[512];

    if( ! ( psRtspSess = rtsp_get_session( pmsg_get_header( psReq->req, "Session" ) ) ) )
        {
        rtsp_send_error( psReq, 454, "Session Not Found" );
        return 0;
        }

    psRange = pmsg_get_header( psReq->req, "Range" );
    if( psRange && sscanf( psRange, "npt=%lf-", &start ) == 1 )
        {
        i32HaveStart = 1;
        }
    else
        {
        // Nothing
        }

    psRtspSess->sess->play( psRtspSess->sess, i32HaveStart ? &start : NULL );

    rtsp_create_reply( psReq, 200, "OK" );
    pmsg_add_header( psReq->resp, "Session", psRtspSess->id );
    if( i32HaveStart && start >= 0 )
        {
        sprintf( hdr, "npt=%.4f-", start );
        pmsg_add_header( psReq->resp, "Range", hdr );
        i32p = 0;
        for( i = 0; i < RTP_MAX_TRACKS; ++i )
            {
            if( ! psRtspSess->sess->ep[i] ) continue;
            ret = snprintf( hdr + i32p, sizeof( hdr ) - i32p,
                            "url=%s/track%d;seq=%d;rtptime=%u,",
                            psReq->req->sl.req.uri, i,
                            psRtspSess->sess->ep[i]->seqnum,
                            psRtspSess->sess->ep[i]->last_timestamp );
            if( ret >= sizeof( hdr ) - i32p )
                {
                i32p = -1;
                break;
                }
            i32p += ret;
            }
        if( i32p > 0 )
            {
            hdr[i32p - 1] = 0; /* Kill last comma */
            pmsg_add_header( psReq->resp, "RTP-Info", hdr );
            }
        }
    rtsp_log_request( psReq, 200, 0 );
    nb_client ++;
    //Use NEO, increase number for every client
    //if (psReq->conn->client_addr.sin_addr.s_addr != 0x0100007F) //If the client is NOT Local client
        {
        rtsp_increase_nb_ext_clients();
        }
    tcp_send_pmsg( psReq->conn, psReq->resp, -1 );
    return 0;
    }

static int handle_PAUSE( struct req *psReq )
    {
    struct rtsp_session *psRtspSess;
    if( ! ( psRtspSess = rtsp_get_session( pmsg_get_header( psReq->req, "Session" ) ) ) )
        {
        rtsp_send_error( psReq, 454, "Session Not Found" );
        return 0;
        }

    if( ! psRtspSess->sess->pause )
        {
        rtsp_send_error( psReq, 501, "Not Implemented" );
        return 0;
        }

    psRtspSess->sess->pause( psRtspSess->sess );

    rtsp_create_reply( psReq, 200, "OK" );
    pmsg_add_header( psReq->resp, "Session", psRtspSess->id );
    rtsp_log_request( psReq, 200, 0 );
    tcp_send_pmsg( psReq->conn, psReq->resp, -1 );
    return 0;
    }

static int handle_TEARDOWN( struct req *psReq )
    {
    struct rtsp_location *psRtspLoc;
    struct rtsp_session *psRtspSess;
    int track;
    if( ! ( psRtspLoc = rtsp_find_location( psReq->req->sl.req.uri,
                                            NULL, &track ) ) ||
            track >= RTP_MAX_TRACKS )
        {
        rtsp_send_error( psReq, 404, "Not Found" );
        return 0;
        }
    mylog_info("%s enter found loc: %s", __FUNCTION__, psReq->req->sl.req.uri);

    if( ! ( psRtspSess = rtsp_get_session( pmsg_get_header( psReq->req, "Session" ) ) ) )
        {
        rtsp_send_error( psReq, 454, "Session Not Found" );
        return 0;
        }
    mylog_info("%s got  psRtspSess:%p psRtspSess->sess:%p", __FUNCTION__, psRtspSess, psRtspSess->sess);


    rtsp_create_reply( psReq, 200, "OK" );
    pmsg_add_header( psReq->resp, "Session", psRtspSess->id );

    /* This might destroy the session, so do it after creating the reply */
    psRtspSess->sess->teardown( psRtspSess->sess, track < 0 ? NULL : psRtspSess->sess->ep[track] );
    mylog_info("%s exit", __FUNCTION__);

    rtsp_log_request( psReq, 200, 0 );
    tcp_send_pmsg( psReq->conn, psReq->resp, -1 );
    return 0;
    }

static int handle_unknown( struct req *psReq )
    {
    rtsp_send_error( psReq, 501, "Not Implemented" );
    return 0;
    }

int rtsp_handle_msg( struct req *psReq )
    {
    int ret;
    if(psReq == NULL)
    {
        return -1;
    }

    if( ! strcasecmp( psReq->req->sl.req.method, "OPTIONS" ) )
        ret = handle_OPTIONS( psReq );
    else if( ! strcasecmp( psReq->req->sl.req.method, "DESCRIBE" ) )
        ret = handle_DESCRIBE( psReq );
    else if( ! strcasecmp( psReq->req->sl.req.method, "SETUP" ) )
        ret = handle_SETUP( psReq );
    else if( ! strcasecmp( psReq->req->sl.req.method, "PLAY" ) )
        ret = handle_PLAY( psReq );
    else if( ! strcasecmp( psReq->req->sl.req.method, "PAUSE" ) )
        ret = handle_PAUSE( psReq );
    else if( ! strcasecmp( psReq->req->sl.req.method, "TEARDOWN" ) )
        ret = handle_TEARDOWN( psReq );
    else ret = handle_unknown( psReq );

    return ret;
    }

int rtsp_nb_ext_clients()
    {
    return nb_ext_client;
    }

int rtsp_increase_nb_ext_clients()
    {
    int n = ++nb_ext_client;
    mylog_info("rtsp_increase_nb_ext_clients  %d", n);
    thirdparty_rtsp_stream_client_update(n);
    return n;
    }

int rtsp_decrease_nb_ext_clients()
    {
    --nb_ext_client;
    if (nb_ext_client <= 0)
        nb_ext_client = 0;
    mylog_info("rtsp_decrease_nb_ext_clients  %d", nb_ext_client);
    thirdparty_rtsp_stream_client_update(nb_ext_client);
    return nb_ext_client;
    }

int rtsp_set_closing(int value)
    {
    force_closing = value;
    return 0;
    }

