/** Copyright (c) 2015 Cvisionhk */

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <time.h>

#include <event.h>
#include <frame.h>
#include <rtp.h>
#include <conf_parse.h>
#include "mylogger.h"

static int g_i32RtpPortStart = 50000, g_i32RtpPortEnd = 60000;

static int rtcp_send( struct rtp_endpoint *psEndpoint );

static void rtcp_fire( struct event_info *psEvInfo, void *d )
    {
    struct rtp_endpoint *psEndpoint = (struct rtp_endpoint *)d;
    rtcp_send( psEndpoint );
    }

struct rtp_endpoint *rtp_new_endpoint( int i32Payload, int isLocal)
    {
    struct rtp_endpoint *psEndpoint;

    if( ! ( psEndpoint = (struct rtp_endpoint *)
                         malloc( sizeof( struct rtp_endpoint ) ) ) )
        return NULL;

    psEndpoint->isLocal = isLocal;
    psEndpoint->payload = i32Payload;
    psEndpoint->max_data_size = 8192; /* default maximum */
    psEndpoint->ssrc = 0;
    rtsp_origin_random_bytes( (unsigned char *)&psEndpoint->ssrc, 4 );
    psEndpoint->start_timestamp = 0;
    //Phung : don't randomize the start timestamp
    //rtsp_origin_random_bytes( (unsigned char *)&psEndpoint->start_timestamp, 4 );
    psEndpoint->last_timestamp = psEndpoint->start_timestamp;
    psEndpoint->seqnum = 0;
    rtsp_origin_random_bytes( (unsigned char *)&psEndpoint->seqnum, 2 );
    psEndpoint->packet_count = 0;
    psEndpoint->octet_count = 0;
    psEndpoint->rtcp_send_event = event_add_timer( 5000, 0, rtcp_fire, psEndpoint, NULL);
    event_set_event_enabled( psEndpoint->rtcp_send_event, 0, NULL);
    psEndpoint->force_rtcp = 1;
    clock_gettime(CLOCK_MONOTONIC,  &psEndpoint->last_rtcp_recv);
    psEndpoint->trans_type = 0;
    psEndpoint->rtpPacketBuffer = NULL;

    return psEndpoint;
    }

void rtp_del_rtp_endpoint( struct rtp_endpoint *psEndpoint )
    {
    event_remove( psEndpoint->rtcp_send_event, NULL);

    switch( psEndpoint->trans_type )
        {
        case RTP_TRANSACTION_UDP:
            event_remove( psEndpoint->trans.udp.rtp_event, NULL);
            close( psEndpoint->trans.udp.rtp_fd );
            event_remove( psEndpoint->trans.udp.rtcp_event, NULL);
            close( psEndpoint->trans.udp.rtcp_fd );
            break;
        case RTP_TRANSACTION_INTER:
            rtsp_interleave_disconnect( psEndpoint->trans.inter.conn,
                                        psEndpoint->trans.inter.rtp_chan );
            rtsp_interleave_disconnect( psEndpoint->trans.inter.conn,
                                        psEndpoint->trans.inter.rtcp_chan );
            break;
        }

    if(psEndpoint)
        {
        free( psEndpoint );
        }
    }

void rtp_update_rtp_timestamp( struct rtp_endpoint *psEndpoint, int time_increment )
    {
    psEndpoint->last_timestamp += time_increment;
    psEndpoint->last_timestamp &= 0xFFFFFFFF;
    }

static void udp_rtp_read( struct event_info *psEvInfo, void *d )
    {
    struct rtp_endpoint *psEndpoint = (struct rtp_endpoint *)d;
    //Origin size 16384 bytes
    unsigned char ucBuf[128];
    int ret;

    ret = read( psEndpoint->trans.udp.rtp_fd, ucBuf, sizeof( ucBuf ) );
    if( ret > 0 )
        {
        /* some SIP phones don't send RTCP */
        clock_gettime(CLOCK_MONOTONIC, &psEndpoint->last_rtcp_recv);
        return;
        }
    psEndpoint->session->teardown( psEndpoint->session, psEndpoint );
    }

static void udp_rtcp_read( struct event_info *psEvInfo, void *d )
    {
    struct rtp_endpoint *psEndpoint = (struct rtp_endpoint *)d;
    //Origin size 16384 bytes
    unsigned char ucBuf[1024];
    int ret;

    ret = read( psEndpoint->trans.udp.rtcp_fd, ucBuf, sizeof( ucBuf ) );
    if( ret > 0 )
        {
        clock_gettime(CLOCK_MONOTONIC, &psEndpoint->last_rtcp_recv);
        return;
        }
    psEndpoint->session->teardown( psEndpoint->session, psEndpoint );
    }

void rtp_interleave_recv_rtcp( struct rtp_endpoint *psEndpoint, unsigned char *d, int len )
    {
    clock_gettime(CLOCK_MONOTONIC, &psEndpoint->last_rtcp_recv);
    }

static int rtcp_send( struct rtp_endpoint *psEndpoint )
    {
    struct timespec now;
    // TODO: reduce to 128 bytes
    unsigned char ucBuf[64];
    unsigned int ntp_sec, ntp_usec;
    struct iovec v[1];

    clock_gettime(CLOCK_MONOTONIC, &now);
    psEndpoint->force_rtcp = 0;

    /* Grrr...  QuickTime apparently doesn't send RTCP over TCP */
    if( psEndpoint->trans_type == RTP_TRANSACTION_UDP)
        {
        //Phung: skip timeout for ALL loop back connection
        if ( strcmp(psEndpoint->trans.udp.sdp_addr,"IP4 127.0.0.1") ==0)
            {

            }
        else if( now.tv_sec - psEndpoint->last_rtcp_recv.tv_sec > 20 )
            {
            mylog_info("client with payload %d timeout (no RTCP received in 20 seconds), ssrc %u",
                       psEndpoint->payload, psEndpoint->ssrc+1 );
            psEndpoint->session->teardown( psEndpoint->session, psEndpoint );
            return -1;
            }
        }

    ntp_sec = now.tv_sec + 0x83AA7E80;
    ntp_usec = (double)( (double)now.tv_nsec/1000 * (double)0x4000000 ) / 15625.0;

    ucBuf[0] = 2 << 6; // version
    ucBuf[1] = 200; // packet type is Sender Report
    RTP_PUT_16( ucBuf + 2, 6 ); // length in words minus one
    RTP_PUT_32( ucBuf + 4, psEndpoint->ssrc );
    RTP_PUT_32( ucBuf + 8, ntp_sec );
    RTP_PUT_32( ucBuf + 12, ntp_usec );
    RTP_PUT_32( ucBuf + 16, psEndpoint->last_timestamp );
    RTP_PUT_32( ucBuf + 20, psEndpoint->packet_count );
    RTP_PUT_32( ucBuf + 24, psEndpoint->octet_count );
    ucBuf[28] = ( 2 << 6 ) | 1; // version; source count = 1
    ucBuf[29] = 202; // packet type is Source Description
    RTP_PUT_16( ucBuf + 30, 5 ); // length in words minus one
    RTP_PUT_32( ucBuf + 32, psEndpoint->ssrc );
    ucBuf[36] = 0x01; // field type is CNAME
    ucBuf[37] = 14; // text length
    memcpy( ucBuf + 38, "Unnamed stream", 14 );

    switch( psEndpoint->trans_type )
        {
        case RTP_TRANSACTION_UDP:
            if(send( psEndpoint->trans.udp.rtcp_fd, ucBuf, 52, 0 ) >= 0)
                return 0;
            break;
        case RTP_TRANSACTION_INTER:
            v[0].iov_base = ucBuf;
            v[0].iov_len = 52;
            if( rtsp_interleave_send( psEndpoint->trans.inter.conn,
                                      psEndpoint->trans.inter.rtcp_chan, v, 1, 0 ) >= 0 )
                return 0;
            break;
        }
    psEndpoint->session->teardown( psEndpoint->session, psEndpoint );
    return -1;
    }

int rtp_connect_udp_endpoint( struct rtp_endpoint *psEndpoint,
                              struct in_addr dest_ip, int dest_port, int *piOurPort )
    {
    struct sockaddr_in sockRtpAddr, sockRtcpAddr;
    int i32Port, success = 0, i, max_tries, i32Rtpfd = -1, i32Rtcpfd = -1;

    sockRtpAddr.sin_family = sockRtcpAddr.sin_family = AF_INET;
    sockRtpAddr.sin_addr.s_addr = sockRtcpAddr.sin_addr.s_addr = 0;

    i32Port = g_i32RtpPortStart + random() % ( g_i32RtpPortEnd - g_i32RtpPortStart );
    if( i32Port & 0x1 ) ++i32Port;
    max_tries = ( g_i32RtpPortEnd - g_i32RtpPortStart + 1 ) / 2;

    for( i = 0; i < max_tries; ++i )
        {
        if( i32Port + 1 > g_i32RtpPortEnd ) i32Port = g_i32RtpPortStart;
        sockRtpAddr.sin_port = htons( i32Port );
        sockRtcpAddr.sin_port = htons( i32Port + 1 );
        if( i32Rtpfd < 0 &&
                ( i32Rtpfd = socket( PF_INET, SOCK_DGRAM, 0 ) ) < 0 )
            {
            mylog_info("unable to create UDP RTP socket: %s",
                       strerror( errno ) );
            return -1;
            }
        if( i32Rtcpfd < 0 &&
                ( i32Rtcpfd = socket( PF_INET, SOCK_DGRAM, 0 ) ) < 0 )
            {
            mylog_info("unable to create UDP RTCP socket: %s",
                       strerror( errno ) );
            close( i32Rtpfd );
            return -1;
            }
        if( bind( i32Rtpfd, (struct sockaddr *)&sockRtpAddr,
                  sizeof( sockRtpAddr ) ) < 0 )
            {
            if( errno == EADDRINUSE )
                {
                i32Port += 2;
                continue;
                }
            mylog_info("strange error when binding RTP socket: %s",
                       strerror( errno ) );
            close( i32Rtpfd );
            close( i32Rtcpfd );
            return -1;
            }
        if( bind( i32Rtcpfd, (struct sockaddr *)&sockRtcpAddr,
                  sizeof( sockRtcpAddr ) ) < 0 )
            {
            if( errno == EADDRINUSE )
                {
                close( i32Rtpfd );
                i32Rtpfd = -1;
                i32Port += 2;
                continue;
                }
            mylog_info("strange error when binding RTCP socket: %s",
                       strerror( errno ) );
            close( i32Rtpfd );
            close( i32Rtcpfd );
            return -1;
            }
        success = 1;
        break;
        }
    if( ! success )
        {
        mylog_info("ran out of UDP RTP ports!" );
        return -1;
        }
    sockRtpAddr.sin_family = sockRtcpAddr.sin_family = AF_INET;
    sockRtpAddr.sin_addr = sockRtcpAddr.sin_addr = dest_ip;
    sockRtpAddr.sin_port = htons( dest_port );
    sockRtcpAddr.sin_port = htons( dest_port + 1 );
    if( connect( i32Rtpfd, (struct sockaddr *)&sockRtpAddr,
                 sizeof( sockRtpAddr ) ) < 0 )
        {
        mylog_info("strange error when connecting RTP socket: %s",
                   strerror( errno ) );
        close( i32Rtpfd );
        close( i32Rtcpfd );
        return -1;
        }
    if( connect( i32Rtcpfd, (struct sockaddr *)&sockRtcpAddr,
                 sizeof( sockRtcpAddr ) ) < 0 )
        {
        mylog_info("strange error when connecting RTCP socket: %s",
                   strerror( errno ) );
        close( i32Rtpfd );
        close( i32Rtcpfd );
        return -1;
        }
    i = sizeof( sockRtpAddr );
    if( getsockname( i32Rtpfd, (struct sockaddr *)&sockRtpAddr, (socklen_t *)&i ) < 0 )
        {
        mylog_info("strange error from getsockname: %s",
                   strerror( errno ) );
        close( i32Rtpfd );
        close( i32Rtcpfd );
        return -1;
        }
    psEndpoint->trans_type = RTP_TRANSACTION_UDP;
    sprintf( psEndpoint->trans.udp.sdp_addr, "IP4 %s",
             inet_ntoa( sockRtpAddr.sin_addr ) );
    psEndpoint->trans.udp.sdp_port = ntohs( sockRtpAddr.sin_port );
    psEndpoint->trans.udp.rtp_fd = i32Rtpfd;
    psEndpoint->trans.udp.rtcp_fd = i32Rtcpfd;
    psEndpoint->trans.udp.rtp_event = event_add_fd( i32Rtpfd, 0, 0,
                                      udp_rtp_read, psEndpoint, NULL);
    psEndpoint->trans.udp.rtcp_event =
        event_add_fd( i32Rtcpfd, 0, 0, udp_rtcp_read, psEndpoint, NULL);

    if(psEndpoint->isLocal) //Local client
        {
        int ttl = 64;
        int tos = 0x1C;
        int sockBufSize = 512*1024;

        setsockopt(i32Rtpfd, IPPROTO_IP, IP_TTL, &(ttl), sizeof(ttl));
        setsockopt(i32Rtcpfd, IPPROTO_IP, IP_TTL, &(ttl), sizeof(ttl));
        setsockopt(i32Rtpfd, SOL_SOCKET, SO_SNDBUF, &sockBufSize, sizeof(sockBufSize));
        setsockopt(i32Rtpfd, IPPROTO_IP, IP_TOS, &(tos), sizeof(tos));
        setsockopt(i32Rtcpfd, IPPROTO_IP, IP_TOS, &(tos), sizeof(tos));
        psEndpoint->force_rtcp = 1;
        //psEndpoint->max_data_size = 1500;
        }
    else //Remote client
        {
        psEndpoint->force_rtcp = 0;
        psEndpoint->max_data_size = 912;
        }

    *piOurPort = i32Port;
    return 0;
    }

void rtp_connect_interleaved_endpoint( struct rtp_endpoint *psEndpoint,
                                       struct rtp_conn *conn, int rtp_chan, int rtcp_chan )
    {
    psEndpoint->trans_type = RTP_TRANSACTION_INTER;
    psEndpoint->trans.inter.conn = conn;
    psEndpoint->trans.inter.rtp_chan = rtp_chan;
    psEndpoint->trans.inter.rtcp_chan = rtcp_chan;
    }

int rtp_send_rtp_packet( struct rtp_endpoint *psEndpoint, struct iovec *v, int i32Cnt,
                         unsigned int timestamp, int marker )
    {
    unsigned char rtphdr[12];
    struct msghdr mh;
    int i;

    psEndpoint->last_timestamp = ( psEndpoint->start_timestamp + timestamp ) & 0xFFFFFFFF;

    rtphdr[0] = 2 << 6; /* version */
    rtphdr[1] = psEndpoint->payload;
    if( marker ) rtphdr[1] |= 0x80;
    RTP_PUT_16( rtphdr + 2, psEndpoint->seqnum );
    RTP_PUT_32( rtphdr + 4, psEndpoint->last_timestamp );
    RTP_PUT_32( rtphdr + 8, psEndpoint->ssrc );

    v[0].iov_base = rtphdr;
    v[0].iov_len = 12;


    switch( psEndpoint->trans_type )
        {
        case RTP_TRANSACTION_UDP:
            if(psEndpoint->isLocal)
                {
                memset( &mh, 0, sizeof( mh ) );
                mh.msg_iov = v;
                mh.msg_iovlen = i32Cnt;
                if( sendmsg( psEndpoint->trans.udp.rtp_fd, &mh, 0 ) < 0 )
                    {
                    psEndpoint->session->teardown( psEndpoint->session, psEndpoint );
                    return -1;
                    }
                }
            else
                {
                }
            break;
        case RTP_TRANSACTION_INTER:
            if( rtsp_interleave_send( psEndpoint->trans.inter.conn,
                                      psEndpoint->trans.inter.rtp_chan, v, i32Cnt, (psEndpoint->payload != 96)) < 0 )
                {
                psEndpoint->session->teardown( psEndpoint->session, psEndpoint );
                return -1;
                }
            break;
        }

    for( i = 0; i < i32Cnt; ++i )
        psEndpoint->octet_count += v[i].iov_len;
    ++psEndpoint->packet_count;

    if( psEndpoint->force_rtcp )
        {
        if( rtcp_send( psEndpoint ) < 0 ) return -1;
        event_set_event_enabled( psEndpoint->rtcp_send_event, 1, NULL);
        }
    psEndpoint->seqnum = ( psEndpoint->seqnum + 1 ) & 0xFFFF;
    return 0;
    }

/********************* GLOBAL CONFIGURATION DIRECTIVES ********************/

int rtp_config_rtprange( int num_tokens, struct token *psToken, void *d )
    {
    g_i32RtpPortStart = psToken[1].v.num;
    g_i32RtpPortEnd = psToken[2].v.num;

    if( g_i32RtpPortStart & 0x1 ) ++g_i32RtpPortStart;
    if( ! ( g_i32RtpPortEnd & 0x1 ) ) --g_i32RtpPortEnd;

    if( g_i32RtpPortEnd - g_i32RtpPortStart + 1 < 8 )
        {
        mylog_error("at least 8 ports are needed for RTP" );
        exit( 1 );
        }

    return 0;
    }
