// ----------------------------------------------------------------
// Copyright (c) 2013 Nuvoton Technology Corp. All rights reserved.
// ----------------------------------------------------------------

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
// MA 02110-1301, USA.

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <errno.h>

#include <event.h>
#include <frame.h>
#include <stream.h>
#include <outputs.h>
#include <rtp.h>
#include <rtp_media.h>
#include <conf_parse.h>

#include "extra.h"
#include "mylogger.h"

struct rtp_h264
    {
    unsigned char *d;
    int len;
    int key;	// key(I) frame
    int init_done;
    int ts_incr;
    unsigned int timestamp;
    uint64_t last_frame;
    };

typedef struct tH264ParamSet
    {
    uint8_t *data;
    int     lenght;
    } tH264ParamSet;

static tH264ParamSet gSPS = {0, 0};
static tH264ParamSet gPPS = {0, 0};

/*****************************************************************************
 * b64_encode: Stolen from VLC's http.c.
 * Simplified by Michael.
 * Fixed edge cases and made it work from data (vs. strings) by Ryan.
 *****************************************************************************/

char *base64_encode(char *out, int out_size, const uint8_t *in, int in_size)
    {
    static const char b64[] =
        "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
    char *ret, *dst;
    unsigned i_bits = 0;
    int i_shift = 0;
    int bytes_remaining = in_size;

    if (out_size < (in_size+2) / 3 * 4 + 1)
        return NULL;
    ret = dst = out;
    while (bytes_remaining)
        {
        i_bits = (i_bits << 8) + *in++;
        bytes_remaining--;
        i_shift += 8;

        do
            {
            *dst++ = b64[(i_bits << 6 >> i_shift) & 0x3f];
            i_shift -= 6;
            }
        while (i_shift > 6 || (bytes_remaining == 0 && i_shift > 0));
        }
    while ((dst - ret) & 3)
        *dst++ = '=';
    *dst = '\0';

    return ret;
    }

#define  VIDEO_PROFILE_H264_BASELINE 66
#define  VIDEO_PROFILE_H264_MAIN     77
#define  VIDEO_PROFILE_H264_HIGH 	 100

void
EncoderH264SetSPS(char* sps, int length)
    {
    if(sps != NULL && length > 0)
        {
        if(length > gSPS.lenght)
            gSPS.data = realloc(gSPS.data, length);
        if(gSPS.data != NULL)
            {
            memcpy(gSPS.data, sps, length);
            gSPS.lenght = length;
            }
        }
    }

void
EncoderH264SetPPS(char* pps, int length)
    {
    if(pps != NULL && length > 0)
        {
        if(length > gPPS.lenght)
            gPPS.data = realloc(gPPS.data, length);
        if(gPPS.data != NULL)
            {
            memcpy(gPPS.data, pps, length);
            gPPS.lenght = length;
            }
        }
    }

uint64_t utils_get_ms_time(void)
    {
    struct timespec ts;
    uint64_t time;

    clock_gettime(CLOCK_MONOTONIC, &ts);

    time = ts.tv_sec * 1000 + ts.tv_nsec/1000000;
    return time;
    }
static int h264_process_frame( struct frame *f, void *d )
    {
    struct rtp_h264 *out = (struct rtp_h264 *)d;
    uint64_t frames;

    /* Discard the start code 0x00000001*/
    out->d = f->d + 4;
    out->len = f->length - 4;
    out->key = f->key;

    if(f->pts == 0)
        {
        if(out->ts_incr == 0)
            out->ts_incr = 1;
        frames = (utils_get_ms_time() * 90) / out->ts_incr;
        out->timestamp = (unsigned int)(frames * out->ts_incr);
        out->last_frame = frames;
        }
    else
        {
        if(out->timestamp == f->pts)
            out->timestamp += 1;
        else
            out->timestamp = f->pts;
        out->last_frame++;
        }
    return out->init_done;
    }

static int h264_get_sdp( char *dest, int len, int payload, int port, void *d )
    {
    char szProfileStr[30];
    char szSPSBase64[100];
    char szPPSBase64[100];
    char szPrameterSet[200];
    int ret = 0 ;
    int profile = VIDEO_PROFILE_H264_MAIN;
    int level = 41;

    sprintf(szProfileStr, "profile-level-id=%02x%02x%02x;", profile, 0, level);
    base64_encode(szSPSBase64, 100, gSPS.data, gSPS.lenght);
    base64_encode(szPPSBase64, 100, gPPS.data, gPPS.lenght);
    sprintf(szPrameterSet, "sprop-parameter-sets=%s,%s", szSPSBase64, szPPSBase64);
    ret =  snprintf( dest, len, "m=video %d RTP/AVP %d\r\n"
                     "a=rtpmap:%d H264/90001\r\n"
                     "a=fmtp:%d packetization-mode=1; %s; %s\r\n",
                     port,
                     payload,
                     payload,
                     payload,
                     szProfileStr,
                     szPrameterSet );
    return ret;
    }

static int h264_send( struct rtp_endpoint *ep, void *d )
    {
    struct rtp_h264 *out = (struct rtp_h264 *)d;
    uint32_t i, plen;
    struct iovec v[3];
    unsigned char vhdr[2]; //FU indicator and FU header
    uint8_t u8NALHeader;

    if((ep->packet_count == 0) && (out->key == 0)) // Skip P frame to make sure the endpoint first frame is key frame.
        return 0;

    u8NALHeader = out->d[0];
    if(out->len <= ep->max_data_size)
        {
        v[1].iov_base = out->d;
        v[1].iov_len = out->len;

        if(rtp_send_rtp_packet( ep, v, 2, out->timestamp, 1) < 0)
            return -1;
        }
    else
        {
        uint8_t u8NALType;
        uint8_t u8NRI;
        uint32_t u32FULen = 2;
        unsigned char *pFrameData;
        uint32_t u32FrameLen;

        pFrameData = out->d + 1;
        u32FrameLen = out->len -1;
        v[1].iov_base = vhdr;
        v[1].iov_len = 2;
        u8NALType = u8NALHeader & 0x1F;
        u8NRI = u8NALHeader & 0x60;
        vhdr[0] = 28; //FU Indicator; Type = 28 ---> FU-A
        vhdr[0] |= u8NRI;
        vhdr[1] = u8NALType;
        vhdr[1] |= 1 << 7; //Set start bit

        for(i = 0; i < u32FrameLen; i += plen)
            {

            plen = u32FrameLen - i;
            if(plen > ep->max_data_size - u32FULen)
                plen = ep->max_data_size - u32FULen;

            if((plen + i) == u32FrameLen)
                {
                vhdr[1] |= 1 << 6; //Set end bit
                }

            v[2].iov_base = pFrameData + i;
            v[2].iov_len = plen;

            if(rtp_send_rtp_packet( ep, v, 3, out->timestamp, plen + i == u32FrameLen ) < 0)
                return -1;

            vhdr[1] &= ~(1 << 7); //Clear start bit
            }
        }
    return 0;
    }

struct rtp_media *new_rtp_media_h264_stream( struct stream *stream )
    {
    struct rtp_h264 *out;
    int fincr, fbase;

    stream->get_framerate( stream, &fincr, &fbase );
    out = (struct rtp_h264 *)malloc( sizeof( struct rtp_h264 ) );
    out->init_done = 1;
    out->timestamp = 0;
    if(fbase == 0)
        fbase == 1;
    out->ts_incr = 90000 * fincr / fbase;
    out->last_frame = 0;
    return session_new_rtp_media( h264_get_sdp, NULL,
                                  h264_process_frame, h264_send, out );
    }
