/** Copyright (c) 2015 Cvisionhk */

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <pthread.h>
#include <errno.h>
#include <pmsg.h>

#include "mylogger.h"

char *pmsg_add_pmsg_string( struct pmsg *psMsg, char *pcs )
    {
    int len = strlen( pcs ) + 1;
    char *pcRet;

    if( psMsg->msg_len + len > psMsg->max_len ) return NULL;
    pcRet = (char *)psMsg->msg + psMsg->msg_len;
    psMsg->msg_len += len;
    strcpy( pcRet, pcs );
    return pcRet;
    }

static int pmsg_delim( char *pcd, int i32Len, char cTerm1, char cTerm2 )
    {
    int i = 0;
    for( i = 0; i < i32Len; ++i )
        if( pcd[i] == cTerm1 || pcd[i] == cTerm2 ||
                pcd[i] == '\r' || pcd[i] == '\n' )
            break;
    return i;
    }

#define MAX_SUPPORT_COMMAND 6
#define MAX_COMMAND_LEN     10
const char rtsp_support_command[MAX_SUPPORT_COMMAND][MAX_COMMAND_LEN] =
{
        "OPTIONS",
        "DESCRIBE",
        "SETUP",
        "PLAY",
        "PAUSE",
        "TEARDOWN"
};

// return 0 is OK otherwise -1
int check_valid_request(char *pData, int len)
{
    int ret = -1;
    int i = 0;
    for (i = 0; i < MAX_SUPPORT_COMMAND; i++)
    {
        if (strstr(pData, rtsp_support_command[i]) != 0)
        {
            ret = 0;
            break;
        }
    }
    return ret;
}

int pmsg_parse_pmsg( struct pmsg *psMsg )
    {
    int i = 0;
    int i32Len = 0;
    unsigned char *pcData;

    if (psMsg == NULL)
    {
        return -1;
    }

    pcData = psMsg->msg;
    i32Len = psMsg->msg_len;

    psMsg->header_count = 0;

    if (check_valid_request(pcData, i32Len) < 0)
    {
        mylog_error("Invalid request\n");
        return -1;
    }

    /* Was it all whitespace? */
    if( i32Len - i < 4 )
    {
        return -1;
    }
    /* Check for a slash in the first word */
    for( i = 0; i < i32Len && pcData[i] != ' ' && pcData[i] != '/'; ++i );
    if( i == i32Len )
    {
        return -1;
    }

    /* Responses begin with "PROTO/" */
    if( pcData[i] == '/' )
    {
        psMsg->type = PMSG_RESP;
        /* The first word is the protocol name and version */
        psMsg->proto_id = (char *)pcData;
        i = pmsg_delim( (char *)pcData, i32Len, ' ', 0 );
        /* Check for a space following the version number */
        if( pcData[i] != ' ' )
        {
            return -1;
        }
        pcData[i++] = 0;
        while( i < i32Len && pcData[i] == ' ' ) ++i;
        /* The next word is the 3-digit response code, then a space */
        if( i32Len - i < 4 || ! isdigit( pcData[i] ) || ! isdigit( pcData[i+1] )
                || ! isdigit( pcData[i+2] ) || pcData[i+3] != ' ' )
        {
            return -1;
        }
        psMsg->sl.stat.code = atoi( (char *)pcData + i );
        i += 4;
        while( i < i32Len && pcData[i] == ' ' ) ++i;
        /* The rest of the line is the textual response */
        psMsg->sl.stat.reason = (char *)pcData + i;
        i += pmsg_delim( (char *)pcData + i, i32Len - i, 0, 0 );
        pcData[i++] = 0;
    }
    else
    {
        psMsg->type = PMSG_REQ;
        /* The first word is the method */
        psMsg->sl.req.method = (char *)pcData;
        i = pmsg_delim( (char *)pcData, i32Len, ' ', 0 );
        /* Then a space */
        if( i >= i32Len || pcData[i] != ' ' )
        {
            return -1;
        }
        pcData[i++] = 0;
        while( i < i32Len && pcData[i] == ' ' ) ++i;
        /* The second word is the URI */
        psMsg->sl.req.uri = (char *)pcData + i;
        i += pmsg_delim( (char *)pcData + i, i32Len - i, ' ', 0 );
        /* Then a space */
        if( i >= i32Len || pcData[i] != ' ' )
        {
            return -1;
        }
        pcData[i++] = 0;
        while( i < i32Len && pcData[i] == ' ' ) ++i;
        /* The last word is the protocol name and version */
        psMsg->proto_id = (char *)pcData + i;
        i += pmsg_delim( (char *)pcData + i, i32Len - i, 0, 0 );
        pcData[i++] = 0;
    }

    /* Skip any trailing space */
    while( i < i32Len && pcData[i] == ' ' ) ++i;
    /* Skip the \r if we didn't kill it already */
    if( i < i32Len && pcData[i] == '\r' ) ++i;
    /* We should be at the end of the line now */
    if( i >= i32Len || pcData[i++] != '\n' )
    {
        return -1;
    }

    /* Now, parse all the header lines */
    for(;;)
    {
        /* There may be a \r here if we're at the end of the headers */
        if( i < i32Len && pcData[i] == '\r' )
        {
            ++i;
        }
        /* If there's no more data, we're done */
        if( i == i32Len )
        {
            return i32Len;
        }
        /* If there's a newline, we're at the end of the headers */
        if( pcData[i] == '\n' )
        {
            return i + 1;
        }

        /* XXX headers beginning with whitespace are continuations */
        if( pcData[i] == '\t' || pcData[i] == ' ' )
        {
            return -1;
        }
        /* The first thing on the line is the header name */
        psMsg->fields[psMsg->header_count].name = (char *)pcData + i;
        /* The name ends with optional spaces then a colon */
        i += pmsg_delim( (char *)pcData + i, i32Len - i, ' ', ':' );
        if( i >= i32Len )
        {
            return -1;
        }
        /* If the optional spaces are present, skip them */
        if( pcData[i] == ' ' )
        {
            pcData[i++] = 0;
            while( i < i32Len && pcData[i] == ' ' ) ++i;
        }
        /* Make sure the colon is present */
        if( i >= i32Len || pcData[i] != ':' )
        {
            return -1;
        }
        pcData[i++] = 0;
        /* Skip any whitespace after the colon */
        while( i < i32Len && ( pcData[i] == ' ' || pcData[i] == '\t' ) ) ++i;
        /* Everything else on the line is the header data */
        psMsg->fields[psMsg->header_count].value = (char *)pcData + i;
        i += pmsg_delim( (char *)pcData + i, i32Len - i, 0, 0 );
        pcData[i++] = 0;
        /* We should be at the end of the line now */
        if( i >= i32Len || pcData[i++] != '\n' )
        {
            return -1;
        }
        ++psMsg->header_count;
    }
    return 0;
    }

char *pmsg_get_header( struct pmsg *psMsg, char *pcName )
    {
    int i;

    for( i = 0; i < psMsg->header_count; ++i )
        if( ! strcasecmp( psMsg->fields[i].name, pcName ) )
            return psMsg->fields[i].value;
    return NULL;
    }

int pmsg_add_header( struct pmsg *psMsg, char *pcName, char *pcValue )
    {
    if( psMsg->header_count == PMSG_MAX_FIELDS ) return -1;
    /* Put the name and value back-to-back at the end of the message */
    psMsg->fields[psMsg->header_count].name = pmsg_add_pmsg_string( psMsg, pcName );
    if( ! psMsg->fields[psMsg->header_count].name ) return -1;
    psMsg->fields[psMsg->header_count].value = pmsg_add_pmsg_string( psMsg, pcValue );
    if( ! psMsg->fields[psMsg->header_count].value ) return -1;
    ++psMsg->header_count;
    return 0;
    }

int pmsg_add_header_printf( struct pmsg *psMsg, char *pcName, char *pcFormat, ... )
    {
    va_list sAp;
    int len;

    /* do the vsnprintf first to clean up the stack */
    va_start( sAp, pcFormat );
    len = vsnprintf( (char *)psMsg->msg + psMsg->msg_len, psMsg->max_len - psMsg->msg_len,
                     pcFormat, sAp );
    va_end( sAp );

    /* check for errors */
    if( psMsg->header_count == PMSG_MAX_FIELDS ) return -1;
    /* vsnprintf will return the length of the formatted string
     * regardless of length, although the actual output may be truncated */
    if( psMsg->msg_len + len >= psMsg->max_len ) return -1;
    psMsg->fields[psMsg->header_count].value = (char *)psMsg->msg + psMsg->msg_len;
    psMsg->msg_len += len + 1;
    /* add the pcName */
    psMsg->fields[psMsg->header_count].name = pmsg_add_pmsg_string( psMsg, pcName );
    if( ! psMsg->fields[psMsg->header_count].name ) return -1;
    ++psMsg->header_count;
    return 0;
    }

int pmsg_replace_header( struct pmsg *psMsg, char *pcName, char *pcValue )
    {
    int i = 0;
    for( i = 0; i < psMsg->header_count; ++i )
        if( ! strcasecmp( psMsg->fields[i].name, pcName ) )
            {
            /* If we can't reuse the space from the original pcValue,
             * we have to allocate space for the new pcValue at the
             * end of the message */
            if( strlen( psMsg->fields[i].value ) < strlen( pcValue ) )
                {
                char *p = pmsg_add_pmsg_string( psMsg, pcValue );
                if( ! p ) return -1;
                psMsg->fields[i].value = p;
                }
            else strcpy( psMsg->fields[i].value, pcValue );
            return 0;
            }
    /* It doesn't exist, so we'll insert it as new */
    return pmsg_add_header( psMsg, pcName, pcValue );
    }

int pmsg_copy_headers( struct pmsg *psMsgDest, struct pmsg *psMsgSrc, char *pcName )
    {
    int i = 0, i32Cnt = 0;

    for( i = 0; i < psMsgSrc->header_count; ++i )
        if( ! strcasecmp( psMsgSrc->fields[i].name, pcName ) )
            {
            pmsg_add_header( psMsgDest, psMsgSrc->fields[i].name,
                             psMsgSrc->fields[i].value );
            ++i32Cnt;
            }
    return i32Cnt;
    }

/* pmsg_get_param() returns 1 for found, 0 for not found, -1 for error */
int pmsg_get_param( char *pcValue, char *pcTag, char *pcDest, int i32Size )
    {
    int i32TagLen = 0, i = 0;
    char *pc;

    if( ! pcValue ) return -1;

    i32TagLen = strlen( pcTag );

    for( pc = pcValue; pc; pc = strchr( pc, ';' ) )
        {
        ++pc;
        if( ! strncasecmp( pc, pcTag, i32TagLen ) )
            {
            pc += i32TagLen;
            if( *pc == 0 || *pc == ';' ) /* Value-less pcTag */
                {
                if( pcDest && i32Size > 0 ) *pcDest = 0;
                return 1;
                }
            else if( *pc == '=' ) /* Tag has an associated pcValue */
                {
                if( ! pcDest || i32Size <= 0 ) return 1;
                ++pc;
                for( i = 0; *pc && *pc != ';' && i < i32Size;
                        ++i, ++pc )
                    pcDest[i] = *pc;
                if( i == i32Size ) return -1;
                pcDest[i] = 0;
                return 1;
                } /* Otherwise, it's a false hit */
            }
        }
    return 0;
    }

struct pmsg *pmsg_new_pmsg( int i32Size )
    {
    struct pmsg *psMsg;
    void *v;

    if( ! ( v = malloc( sizeof( struct pmsg ) + i32Size ) ) )
        {
        mylog_error("unable to allocate memory for message" );
        return NULL;
        }
    psMsg = (struct pmsg *)v;
    psMsg->msg = v + sizeof( struct pmsg );
    psMsg->max_len = i32Size;
    psMsg->msg_len = 0;
    psMsg->header_count = 0;
    psMsg->proto_id = NULL;
    return psMsg;
    }

void pmsg_free_pmsg( struct pmsg *psMsg )
    {
    if(psMsg)
        {
        free( psMsg );
        }
    }

