/** Copyright (c) 2015 Cvisionhk */
#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <errno.h>
#include <pthread.h>
#include <string.h>

#include <event.h>
#include <frame.h>
#include <stream.h>
#include <outputs.h>
#include <rtp.h>
#include <rtp_media.h>
#include <conf_parse.h>
#include <arpa/inet.h>	//fix warning for inet_addr

#include "stream.h"
#include "mylogger.h"

#define GET_SDP_RETRY_MAX 20 // ~60 seconds

struct live_source;

struct live_session
    {
    struct live_session *next;
    struct live_session *prev;
    struct session *sess;
    struct live_source * source;
    int playing;
    };

struct live_track
    {
    int index;
    struct live_source *source;
    struct stream_destination *stream;
    int ready;
    struct rtp_media * rtp;
    };

struct live_source
    {
    struct live_session *sess_list;
    struct live_track track[RTP_MAX_TRACKS];
    };

/* Global variable - Improve p2p socket creation */
/* Start define */
#define LIVE_MAX_NB_EXT_CLIENTS				3//MAX_NB_EXT_CLIENTS
#define LIVE_SOCKET_INIT_FLG_NO_INIT		0		// No Init
#define LIVE_SOCKET_INIT_FLG_AVAIL			1		// Initialized and available
#define LIVE_SOCKET_INIT_FLG_IN_USE			2		// Live p2p is using this socket

#define LIVE_TIME_SEND_KEEP_ALIVE	60	// seconds
#define LIVE_TIME_RESET_SOCKET		30	// LIVE_TIME_RESET_SOCKET* LIVE_TIME_SEND_KEEP_ALIVE	
#define LIVE_SOCKET_NO_INIT			-1
#define LIVE_P2P_NO_PORT			0

#define LIVE_TIME_INTERVAL_MAX		4000	// ms
#define LIVE_TIME_INTERVAL_MIN		2000	// ms

static int live_get_sdp( struct session *s, char *dest, int *len,
                         char *path )
    {
    struct live_session *psLiveSess = (struct live_session *)s->private;
    int i = 0, t;
    char *addr = "IP4 0.0.0.0";
    static int get_video_sdp_retry = GET_SDP_RETRY_MAX;
    static int get_audio_sdp_retry = GET_SDP_RETRY_MAX;

    // Check video sdp
    if( ! psLiveSess->source->track[0].rtp || ! psLiveSess->source->track[0].ready )
        {
        mylog_error("Can not get video SDP, retry %d %p %d", get_video_sdp_retry, psLiveSess->source->track[0].rtp, psLiveSess->source->track[0].ready);
        if (get_video_sdp_retry < 0)
            mylog_error("Can not get any frame now. Please reboot system");
        else
            get_video_sdp_retry--;
        return 0;
        }
    else
        {
        // Reset SDP getting retry number
        get_video_sdp_retry = GET_SDP_RETRY_MAX;
        }

    // Check audio sdp
    if( ! psLiveSess->source->track[RTP_MAX_TRACKS/2].rtp && ! psLiveSess->source->track[RTP_MAX_TRACKS/2].ready )
        {
        mylog_error("Can not get audio SDP, retry %d", get_audio_sdp_retry);
        if (get_audio_sdp_retry < 0)
            mylog_error("Can not get any frame now. Please reboot system");
        else
            get_audio_sdp_retry--;
        return 0;
        }
    else
        {
        // Reset SDP getting retry number
        get_audio_sdp_retry = GET_SDP_RETRY_MAX;
        }


    if( s->ep[0] && s->ep[0]->trans_type == RTP_TRANSACTION_UDP )
        addr = s->ep[0]->trans.udp.sdp_addr;

    i = snprintf( dest, *len,
                  "v=0\r\no=- 1 1 IN IP4 127.0.0.1\r\ns=Test\r\na=type:broadcast\r\nt=0 0\r\nc=IN %s\r\n", addr );

    for( t = 0; t < RTP_MAX_TRACKS && psLiveSess->source->track[t].rtp; ++t )
        {
        int i32Port;

        if( ! psLiveSess->source->track[t].ready ) return 0;

        if( s->ep[t] && s->ep[t]->trans_type == RTP_TRANSACTION_UDP )
            i32Port = s->ep[t]->trans.udp.sdp_port;
        else
            i32Port = 0;

        i += psLiveSess->source->track[t].rtp->get_sdp( dest + i, *len - i,
                96 + t, i32Port,
                psLiveSess->source->track[t].rtp->private );
        if( i32Port == 0 ) // XXX What's a better way to do this?
            i += sprintf( dest + i, "a=control:track%d\r\n", t );
        }
    *len = i;
    return t;
    }

static int live_setup( struct session *psSess, int t )
    {
    struct live_session *psLiveSess = (struct live_session *)psSess->private;
    int payload = 96 + t;

    if( ! psLiveSess->source->track[t].rtp ) return -1;

    if( psLiveSess->source->track[t].rtp->get_payload )
        payload = psLiveSess->source->track[t].rtp->get_payload( payload,
                  psLiveSess->source->track[t].rtp->private );
    psSess->ep[t] = rtp_new_endpoint( payload, psSess->isLocal);
    psSess->ep[t]->session = psSess;

    return 0;
    }

static void live_play( struct session *psSess, double *dStart )
    {
    struct live_session *psLiveSess = (struct live_session *)psSess->private;
    int t;

    if( dStart ) *dStart = -1;
    psLiveSess->playing = 1;
    for( t = 0; t < RTP_MAX_TRACKS && psLiveSess->source->track[t].rtp; ++t )
        if( psSess->ep[t] )
            {
            stream_set_waiting( psLiveSess->source->track[t].stream, 1 );
            }
    }

static void track_check_running( struct live_source *psLiveSrc, int t )
    {
    struct live_session *psLiveSess;

    for( psLiveSess = psLiveSrc->sess_list; psLiveSess; psLiveSess = psLiveSess->next )
        if( psLiveSess->playing && psLiveSess->sess->ep[t] ) return;
    stream_set_waiting( psLiveSrc->track[t].stream, 0 );
    }

static void live_teardown( struct session *psSess, struct rtp_endpoint *psRtpEndpoint )
    {
    struct live_session *psLiveSess = (struct live_session *)psSess->private;
    int i, remaining = 0;

    for( i = 0; i < RTP_MAX_TRACKS && psLiveSess->source->track[i].rtp; ++i )
        {
        if (psLiveSess->source->track[i].rtp == NULL)
            continue;
        if( ! psSess->ep[i] )
            continue;
        if( ! psRtpEndpoint || psSess->ep[i] == psRtpEndpoint )
            {
            rtp_del_rtp_endpoint( psSess->ep[i] );
            psSess->ep[i] = NULL;
            track_check_running( psLiveSess->source, i );
            }
        else ++remaining;
        }

    mylog_info("%s remaining:%d ", __FUNCTION__, remaining);

    if( remaining == 0 )
        {
        if(psLiveSess)
            {
            if( psLiveSess->next ) psLiveSess->next->prev = psLiveSess->prev;
            if( psLiveSess->prev ) psLiveSess->prev->next = psLiveSess->next;
            else psLiveSess->source->sess_list = psLiveSess->next;
            mylog_debug("free struct live_session");
            free( psLiveSess );
            }

        session_del_session( psSess );
        }
    }

static struct session *live_open( char *pcPath, void *d )
    {
    struct live_source *psLiveSrc = (struct live_source *)d;
    struct live_session *psLiveSess;

    psLiveSess = (struct live_session *)malloc( sizeof( struct live_session ) );
    psLiveSess->next = psLiveSrc->sess_list;
    if( psLiveSess->next ) psLiveSess->next->prev = psLiveSess;
    psLiveSrc->sess_list = psLiveSess;
    psLiveSess->prev = NULL;
    psLiveSess->sess = session_new_session(1);
    psLiveSess->source = psLiveSrc;
    psLiveSess->playing = 0;
    psLiveSess->sess->get_sdp = live_get_sdp;
    psLiveSess->sess->setup = live_setup;
    psLiveSess->sess->play = live_play;
    psLiveSess->sess->teardown = live_teardown;
    psLiveSess->sess->private = psLiveSess;

    return psLiveSess->sess;
    }

static void next_live_frame( struct frame * mFrame, void *d )
    {
    struct live_track *psLiveTrack = (struct live_track *)d;
    struct live_session *psLiveSess, *next;

    if( ! psLiveTrack->rtp->frame( mFrame, psLiveTrack->rtp->private ) )
        {
        frame_unref_frame( mFrame, NULL);
        return;
        }

    if( ! psLiveTrack->ready )
        {
        mylog_info("Track %d Got frame type %d", psLiveTrack->index, mFrame->format);
        stream_set_waiting( psLiveTrack->stream, 0 );
        psLiveTrack->ready = 1;
        }

    for( psLiveSess = psLiveTrack->source->sess_list; psLiveSess; psLiveSess = next )
        {
        next = psLiveSess->next;

        if( psLiveSess->playing && psLiveSess->sess->ep[psLiveTrack->index] )
            {
            psLiveTrack->rtp->send( psLiveSess->sess->ep[psLiveTrack->index],
                                    psLiveTrack->rtp->private );
            }
        }
    frame_unref_frame( mFrame, NULL);
    }

/************************ CONFIGURATION DIRECTIVES ************************/

static void *start_block(void)
    {
    struct live_source *source;
    int i;

    source = (struct live_source *)malloc( sizeof( struct live_source ) );
    source->sess_list = NULL;
    for( i = 0; i < RTP_MAX_TRACKS; ++i )
        {
        source->track[i].index = i;
        source->track[i].source = source;
        source->track[i].stream = NULL;
        source->track[i].ready = 0;
        source->track[i].rtp = NULL;
        }

    return source;
    }

static int end_block( void *d )
    {
    struct live_source *source = (struct live_source *)d;

    if( ! source->track[0].rtp )
        {
        mylog_error("live: no media sources specified!" );
        return -1;
        }
    return 0;
    }

static int set_track( int num_tokens, struct token *tokens, void *d )
    {
    struct live_source *source = (struct live_source *)d;
    int t, formats[] = { RTSP_FORMAT_MPEG4, RTSP_FORMAT_MPV, RTSP_FORMAT_H263, RTSP_FORMAT_JPEG,
                         RTSP_FORMAT_PCM, RTSP_FORMAT_ALAW, RTSP_FORMAT_MPA, RTSP_FORMAT_H264, RTSP_FORMAT_ADPCM, RTSP_FORMAT_ULAW, RTSP_FORMAT_AAC,
                         RTSP_FORMAT_G722
                       }; //TODO: Added new format

    for( t = 0; t < RTP_MAX_TRACKS && (source->track[t].rtp); ++t );
    if( t == RTP_MAX_TRACKS )
        {
        mylog_error("live: exceeded maximum number of tracks" );
        return -1;
        }

    if( ! ( source->track[t].stream = connect_to_stream( tokens[1].v.str,
                                      next_live_frame, &source->track[t], formats, sizeof(formats) / sizeof(formats[0]) ) ) ) //TODO: Add new format, 10: count of formats[]
        {
        mylog_error("live: unable to connect to stream \"%s\"", tokens[1].v.str );
        return -1;
        }

    switch( source->track[t].stream->stream->format )
        {
        case RTSP_FORMAT_MPEG4:
        case RTSP_FORMAT_MPV:
        case RTSP_FORMAT_H263:
        case RTSP_FORMAT_JPEG:
        case RTSP_FORMAT_MPA:
        case RTSP_FORMAT_ADPCM:
        case RTSP_FORMAT_AAC:
        case RTSP_FORMAT_G722:
        default:
            mylog_error("UN SUPPORTED MEDIA TYPE\n");
            break;
        case RTSP_FORMAT_PCM:
        case RTSP_FORMAT_ALAW:
        case RTSP_FORMAT_ULAW:
            source->track[t].rtp = new_rtp_media_rawaudio_stream(
                                       source->track[t].stream->stream );
            break;
        case RTSP_FORMAT_H264:
            source->track[t].rtp = new_rtp_media_h264_stream(
                                       source->track[t].stream->stream );
            break;
        }

    if( ! source->track[t].rtp ) return -1;

    stream_set_waiting( source->track[t].stream, 1 );
    return 0;
    }

static int set_path( int num_tokens, struct token *psToken, void *d )
    {
    if( num_tokens == 2 )
        {
        rtsp_new_location( psToken[1].v.str, NULL, NULL, NULL,
                           live_open, d );
        return 0;
        }
    if( num_tokens == 5 )
        {
        rtsp_new_location( psToken[1].v.str, psToken[2].v.str,
                           psToken[3].v.str, psToken[4].v.str,
                           live_open, d );
        return 0;
        }
    mylog_error("rtsp-handler: syntax: Path <path> [<realm> <username> <password>]" );
    return -1;
    }


static struct statement live_config_statements[] =
    {
    /* directive name, process function, min args, max args, arg types */
        { "track", set_track, 1, 1, { TOKEN_STR } },
        { "path", set_path, 1, 4, { TOKEN_STR, TOKEN_STR, TOKEN_STR, TOKEN_STR } },
    /* empty terminator -- do not remove */
        { NULL, NULL, 0, 0, {} }
    };

int live_init(void)
    {
    register_config_context( "rtsp-handler", "live", start_block, end_block,
                             live_config_statements );
    return 0;
    }
