/** Copyright (c) 2015 Cvisionhk */

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <event.h>
#include <pmsg.h>
#include <rtp.h>

#include "mylogger.h"
#include "md5.h"

struct digest_auth_info
    {
    char realm[128];
    char uri[512];
    char nonce[128];
    char opaque[128];
    char username[128];
    char response[33];
    };

static char g_cDigestSecret[17];
static int g_i32SecretCreated = 0;

static void httpauth_create_digest_secret(void)
    {
    rtsp_origin_random_id( (unsigned char *)g_cDigestSecret, 16 );
    g_i32SecretCreated = 1;
    }

static void httpauth_md5_hash( char **ppc, int i32Count, char *pcHash )
    {
    struct MD5Context sMD5;
    int i;
    unsigned char ucBin[16];

    MD5Init( &sMD5 );
    for( i = 0; i < i32Count; ++i )
        {
        if( i > 0 ) MD5Update( &sMD5, (unsigned char const *)":", 1 );
        MD5Update( &sMD5, (unsigned char const *)ppc[i], strlen( ppc[i] ) );
        }
    MD5Final( ucBin, &sMD5 );
    for( i = 0; i < 16; ++i ) sprintf( pcHash + (i<<1), "%02x", ucBin[i] );
    pcHash[32] = 0;
    }

/*
 * Nonce format:
 *
 * Bytes  0- 3: date of nonce creation in seconds since epoch
 * Bytes  4-15: random bytes
 * Bytes 16-31: MD5( bytes[0..15] + ":" + digest secret )
 */
static void httpauth_create_nonce( struct digest_auth_info *psDigestAuth )
    {
    unsigned char ucToken[16];
    char *v[2] = { psDigestAuth->nonce, g_cDigestSecret };
    struct timeval sTimeNow;
    int i = 0;

    if( ! g_i32SecretCreated ) httpauth_create_digest_secret();

    gettimeofday( &sTimeNow, NULL );
    RTP_PUT_32( ucToken, sTimeNow.tv_sec );
    rtsp_origin_random_bytes( ucToken + 4, sizeof( ucToken ) - 4 );
    for( i = 0; i < 16; ++i )
        sprintf( psDigestAuth->nonce + (i<<1), "%02x", ucToken[i] );
    psDigestAuth->nonce[32] = 0;
    httpauth_md5_hash( v, 2, psDigestAuth->nonce + 32 );
    }

static int httpauth_parse_auth_header( char *pcHeader, struct digest_auth_info *psDigestAuth )
    {
    char *pcn, *pcv;
    int i32Len;

    if( strncasecmp( pcHeader, "digest ", 7 ) ) return -1;
    pcn = pcHeader + 7;
    while( *pcn )
        {
        /* Look for the '=' after the directive name */
        for( pcv = pcn; *pcv != '='; ++pcv ) if( ! *pcv ) return -1;
        /* Step past the '=' */
        if( ! *(++pcv) ) return -1;
        /* See if the value is quoted */
        if( *pcv == '"' )
            {
            ++pcv;
            /* Count characters until we find another quote */
            for( i32Len = 0; pcv[i32Len] != '"'; ++i32Len )
                if( ! pcv[i32Len] ) return -1;
            }
        else
            {
            /* Count characters until we find a comma, space, NUL */
            for( i32Len = 0; pcv[i32Len] != ',' && pcv[i32Len] != ' ' &&
                    pcv[i32Len] != 0; ++i32Len );
            }
        /* Do the appropriate thing for each directive that we handle */
        if( ! strncasecmp( pcn, "realm=", 6 ) )
            {
            if( i32Len >= sizeof( psDigestAuth->realm ) ) return -1;
            strncpy( psDigestAuth->realm, pcv, i32Len );
            psDigestAuth->realm[i32Len] = 0;
            }
        else if( ! strncasecmp( pcn, "uri=", 4 ) )
            {
            if( i32Len >= sizeof( psDigestAuth->uri ) ) return -1;
            strncpy( psDigestAuth->uri, pcv, i32Len );
            psDigestAuth->uri[i32Len] = 0;
            }
        else if( ! strncasecmp( pcn, "nonce=", 6 ) )
            {
            if( i32Len >= sizeof( psDigestAuth->nonce ) ) return -1;
            strncpy( psDigestAuth->nonce, pcv, i32Len );
            psDigestAuth->nonce[i32Len] = 0;
            }
        else if( ! strncasecmp( pcn, "opaque=", 7 ) )
            {
            if( i32Len >= sizeof( psDigestAuth->opaque ) ) return -1;
            strncpy( psDigestAuth->opaque, pcv, i32Len );
            psDigestAuth->opaque[i32Len] = 0;
            }
        else if( ! strncasecmp( pcn, "username=", 9 ) )
            {
            if( i32Len >= sizeof( psDigestAuth->username ) ) return -1;
            strncpy( psDigestAuth->username, pcv, i32Len );
            psDigestAuth->username[i32Len] = 0;
            }
        else if( ! strncasecmp( pcn, "response=", 9 ) )
            {
            if( i32Len >= sizeof( psDigestAuth->response ) ) return -1;
            strncpy( psDigestAuth->response, pcv, i32Len );
            psDigestAuth->response[i32Len] = 0;
            }
        else if( ! strncasecmp( pcn, "algorithm=", 10 ) )
            {
            /* If this is included, just make sure it is "MD5" */
            if( strncasecmp( pcv, "md5", 3 ) ) return -1;
            }
        /* Advance past the value */
        pcn = pcv + i32Len;
        /* Advance past any trailing quotes, commas or spaces */
        while( *pcn == '"' || *pcn == ',' || *pcn == ' ' ) ++pcn;
        }
    return 0;
    }

static void httpauth_create_response( char *pcResponse, struct digest_auth_info *psDigestAuth,
                                      char *pcMethod, char *pcPassword )
    {
    char *elem[3], cHA1[33], cHA2[33];

    elem[0] = psDigestAuth->username;
    elem[1] = psDigestAuth->realm;
    elem[2] = pcPassword;
    httpauth_md5_hash( elem, 3, cHA1 );
    elem[0] = pcMethod;
    elem[1] = psDigestAuth->uri;
    httpauth_md5_hash( elem, 2, cHA2 );
    elem[0] = cHA1;
    elem[1] = psDigestAuth->nonce;
    elem[2] = cHA2;
    httpauth_md5_hash( elem, 3, pcResponse );
    }

static unsigned int httpauth_get_hex_u32( char *pcHex )
    {
    int i = 0;
    unsigned int val = 0;

    for( i = 0; i < 8; ++i )
        {
        if( pcHex[i] >= '0' && pcHex[i] <= '9' )
            val = ( val << 4 ) | ( pcHex[i] - 0 );
        else if( pcHex[i] >= 'A' && pcHex[i] <= 'F' )
            val = ( val << 4 ) | ( pcHex[i] - 'A' + 10 );
        else if( pcHex[i] >= 'a' && pcHex[i] <= 'f' )
            val = ( val << 4 ) | ( pcHex[i] - 'a' + 10 );
        else break;
        }
    return val;
    }

int httpauth_check_digest_response( struct pmsg *psMsg, char *pcRealm,
                                    char *pcUsername, char *pcPassword )
    {
    char *pcHdr, pcExpected[33], pcToken[33];
    struct digest_auth_info sDigestAuth;
    char *v[2] = { pcToken, g_cDigestSecret };
    struct timeval now;

    memset( &sDigestAuth, 0, sizeof( sDigestAuth ) );

    if( ! ( pcHdr = pmsg_get_header( psMsg, "authorization" ) ) ) return -1;
    if( httpauth_parse_auth_header( pcHdr, &sDigestAuth ) < 0 )
        {
        mylog_debug("digest-auth: unable to parse www-authenticate header" );
        return -1;
        }

    /* Case-sensitive pcRealm (should just be parroted back by the client) */
    if( strcmp( sDigestAuth.realm, pcRealm ) )
        {
        mylog_debug("digest-auth: pcRealm \"%s\" is not correct",
                    sDigestAuth.realm );
        return -1;
        }
    /* Case-insensitive usernames */
    if( strcasecmp( sDigestAuth.username, pcUsername ) )
        {
        mylog_debug("digest-auth: pcUsername \"%s\" is not correct",
                    sDigestAuth.username );
        return -1;
        }

    /* The client may or may not have included the digest-uri directive */
    if( ! sDigestAuth.uri[0] )
        {
        if( strlen( psMsg->sl.req.uri ) >= sizeof( sDigestAuth.uri ) )
            {
            mylog_info("URI is too long for digest-auth!" );
            return -1;
            }
        strcpy( sDigestAuth.uri, psMsg->sl.req.uri );
        }

    /* Figure out what the response should be */
    httpauth_create_response( pcExpected, &sDigestAuth, psMsg->sl.req.method, pcPassword );
    if( strcasecmp( sDigestAuth.response, pcExpected ) )
        {
        mylog_debug("digest-auth: incorrect password" );
        return -1;
        }

    /* From this point on, it appears that the client knows the correct
     * username and password, it's just a matter of whether the nonce
     * was generated by us and has not yet expired.  If the nonce is
     * invalid, we can use the "stale=true" directive in the 401 so
     * the client can retry authentication with the same username and
     * password, if it still has it. */

    if( ! g_i32SecretCreated || strlen( sDigestAuth.nonce ) != 64 )
        {
        mylog_debug("digest-auth: this is not our nonce!" );
        return 0;
        }

    /* Check that the nonce validates against our secret */
    strncpy( pcToken, sDigestAuth.nonce, 32 );
    pcToken[32] = 0;
    httpauth_md5_hash( v, 2, pcExpected );
    if( strcmp( sDigestAuth.nonce + 32, pcExpected ) )
        {
        mylog_debug("digest-auth: this is not our nonce!" );
        return 0;
        }

    /* Check that the nonce is not more than 15 seconds old */
    gettimeofday( &now, NULL );
    if( now.tv_sec > httpauth_get_hex_u32( pcToken ) + 15 )
        {
        mylog_debug("digest-auth: nonce is more than 15 seconds old" );
        return 0;
        }

    mylog_debug("digest-auth authentication succeeded" );
    return 1;
    }

int add_digest_challenge( struct pmsg *msg, char *realm, int stale )
    {
    struct digest_auth_info auth;

    strcpy( auth.realm, realm );
    httpauth_create_nonce( &auth );

    /* LIVE.COM expects the challenge in exactly this format */
    return pmsg_add_header_printf( msg, "WWW-Authenticate",
                                   "Digest realm=\"%s\", nonce=\"%s\"%s",
                                   auth.realm, auth.nonce,
                                   stale ? ", stale=true" : "" );
    }
