/** Copyright (c) 2015 Cvisionhk */
#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <errno.h>
#include <pthread.h>
#include <string.h>

#include <event.h>
#include <frame.h>
#include <stream.h>
#include <outputs.h>
#include <rtp.h>
#include <rtp_media.h>
#include <conf_parse.h>
#include <arpa/inet.h>	//fix warning for inet_addr

#include "stream.h"
#include "mylogger.h"

#define GET_SDP_RETRY_MAX 20 // ~60 seconds

struct audio_only_source;

struct audio_only_session
    {
    struct audio_only_session *next;
    struct audio_only_session *prev;
    struct session *sess;
    struct audio_only_source * source;
    int playing;
    };

struct audio_only_track
    {
    int index;
    struct audio_only_source *source;
    struct stream_destination *stream;
    int ready;
    struct rtp_media * rtp;
    };

struct audio_only_source
    {
    struct audio_only_session *sess_list;
    struct audio_only_track track[1];
    };

static int audio_only_get_sdp( struct session *s, char *dest, int *len,
                         char *path )
    {
    struct audio_only_session *psLiveSess = (struct audio_only_session *)s->private;
    int i = 0, t = 0;
    char *addr = "IP4 0.0.0.0";

    if( s->ep[0] && s->ep[0]->trans_type == RTP_TRANSACTION_UDP )
        addr = s->ep[0]->trans.udp.sdp_addr;

    i = snprintf( dest, *len,
                  "v=0\r\no=- 1 1 IN IP4 127.0.0.1\r\ns=Test\r\na=type:broadcast\r\nt=0 0\r\nc=IN %s\r\n", addr );

    int i32Port;

    mylog_info("Get SDP of stream %s", path);
    if( ! psLiveSess->source->track[t].ready )
        {
        mylog_error("Track%d is not ready", t);
        return 0;
        }

    if( s->ep[t] && s->ep[t]->trans_type == RTP_TRANSACTION_UDP )
        i32Port = s->ep[t]->trans.udp.sdp_port;
    else
        i32Port = 0;

    i += psLiveSess->source->track[t].rtp->get_sdp( dest + i, *len - i,
            97, i32Port,
            psLiveSess->source->track[t].rtp->private );
    if( i32Port == 0 ) // XXX What's a better way to do this?
        i += sprintf( dest + i, "a=control:track%d\r\n", t );
    *len = i;
    return i;
    }

static int audio_only_setup( struct session *psSess, int t )
    {
    struct audio_only_session *psLiveSess = (struct audio_only_session *)psSess->private;
    int payload = 97;

    if( ! psLiveSess->source->track[t].rtp ) return -1;

    if( psLiveSess->source->track[t].rtp->get_payload )
        payload = psLiveSess->source->track[t].rtp->get_payload( payload,
                  psLiveSess->source->track[t].rtp->private );
    psSess->ep[t] = rtp_new_endpoint( payload, psSess->isLocal);
    psSess->ep[t]->session = psSess;

    return 0;
    }

static void audio_only_play( struct session *psSess, double *dStart )
    {
    struct audio_only_session *psLiveSess = (struct audio_only_session *)psSess->private;
    int t = 0;;

    if( dStart ) *dStart = -1;
    psLiveSess->playing = 1;
    if( psSess->ep[t] )
        stream_set_waiting( psLiveSess->source->track[t].stream, 1 );
    }

static void track_check_running( struct audio_only_source *psLiveSrc, int t )
    {
    struct audio_only_session *psLiveSess;

    for( psLiveSess = psLiveSrc->sess_list; psLiveSess; psLiveSess = psLiveSess->next )
        if( psLiveSess->playing && psLiveSess->sess->ep[t] ) return;
    stream_set_waiting( psLiveSrc->track[t].stream, 0 );
    }

static void audio_only_teardown( struct session *psSess, struct rtp_endpoint *psRtpEndpoint )
    {
    struct audio_only_session *psLiveSess = (struct audio_only_session *)psSess->private;
    int i = 0, remaining = 0;

    if (psLiveSess->source->track[i].rtp == NULL)
        return;
    if( ! psSess->ep[i] )
        return;
    if( ! psRtpEndpoint || psSess->ep[i] == psRtpEndpoint )
        {
        rtp_del_rtp_endpoint( psSess->ep[i] );
        psSess->ep[i] = NULL;
        track_check_running( psLiveSess->source, i );
        }
    else ++remaining;

    if( remaining == 0 )
        {
        if(psLiveSess)
            {
            if( psLiveSess->next ) psLiveSess->next->prev = psLiveSess->prev;
            if( psLiveSess->prev ) psLiveSess->prev->next = psLiveSess->next;
            else psLiveSess->source->sess_list = psLiveSess->next;
            mylog_debug("free struct audio_only_session");
            free( psLiveSess );
            }
        session_del_session( psSess );
        }
    }

static struct session *audio_only_open( char *pcPath, void *d )
    {
    struct audio_only_source *psLiveSrc = (struct audio_only_source *)d;
    struct audio_only_session *psLiveSess;

    psLiveSess = (struct audio_only_session *)malloc( sizeof( struct audio_only_session ) );
    psLiveSess->next = psLiveSrc->sess_list;
    if( psLiveSess->next ) psLiveSess->next->prev = psLiveSess;
    psLiveSrc->sess_list = psLiveSess;
    psLiveSess->prev = NULL;
    psLiveSess->sess = session_new_session(1);
    psLiveSess->source = psLiveSrc;
    psLiveSess->playing = 0;
    psLiveSess->sess->get_sdp = audio_only_get_sdp;
    psLiveSess->sess->setup = audio_only_setup;
    psLiveSess->sess->play = audio_only_play;
    psLiveSess->sess->teardown = audio_only_teardown;
    psLiveSess->sess->private = psLiveSess;

    return psLiveSess->sess;
    }

static void next_audio_only_frame( struct frame * mFrame, void *d )
    {
    struct audio_only_track *psLiveTrack = (struct audio_only_track *)d;
    struct audio_only_session *psLiveSess, *next;

    if( ! psLiveTrack->rtp->frame( mFrame, psLiveTrack->rtp->private ) )
        {
        frame_unref_frame( mFrame, NULL);
        return;
        }

    if( ! psLiveTrack->ready )
        {
        mylog_info("Track %d Got frame type %d", psLiveTrack->index, mFrame->format);
        stream_set_waiting( psLiveTrack->stream, 0 );
        psLiveTrack->ready = 1;
        }

    for( psLiveSess = psLiveTrack->source->sess_list; psLiveSess; psLiveSess = next )
        {
        next = psLiveSess->next;

        if( psLiveSess->playing && psLiveSess->sess->ep[psLiveTrack->index] )
            {
            psLiveTrack->rtp->send( psLiveSess->sess->ep[psLiveTrack->index],
                                    psLiveTrack->rtp->private );
            }
        }
    frame_unref_frame( mFrame, NULL);
    }

/************************ CONFIGURATION DIRECTIVES ************************/

static void *start_block(void)
    {
    struct audio_only_source *source;
    int i = 0;

    source = (struct audio_only_source *)malloc( sizeof( struct audio_only_source ) );
    source->sess_list = NULL;
    source->track[i].index = i;
    source->track[i].source = source;
    source->track[i].stream = NULL;
    source->track[i].ready = 0;
    source->track[i].rtp = NULL;
    return source;
    }

static int end_block( void *d )
    {
    struct audio_only_source *source = (struct audio_only_source *)d;

    if( ! source->track[0].rtp )
        {
        mylog_error("live: no media sources specified!" );
        return -1;
        }
    return 0;
    }

static int set_track( int num_tokens, struct token *tokens, void *d )
    {
    struct audio_only_source *source = (struct audio_only_source *)d;
    int t = 0, formats[] = { RTSP_FORMAT_MPEG4, RTSP_FORMAT_MPV, RTSP_FORMAT_H263, RTSP_FORMAT_JPEG,
                         RTSP_FORMAT_PCM, RTSP_FORMAT_ALAW, RTSP_FORMAT_MPA, RTSP_FORMAT_H264, RTSP_FORMAT_ADPCM, RTSP_FORMAT_ULAW, RTSP_FORMAT_AAC,
                         RTSP_FORMAT_G722
                       }; //TODO: Added new format

    if( ! ( source->track[t].stream = connect_to_stream( tokens[1].v.str,
                                      next_audio_only_frame, &source->track[t], formats, sizeof(formats) / sizeof(formats[0]) ) ) ) //TODO: Add new format, 10: count of formats[]
        {
        mylog_error("live: unable to connect to stream \"%s\"", tokens[1].v.str );
        return -1;
        }

    switch( source->track[t].stream->stream->format )
        {
        case RTSP_FORMAT_MPEG4:
        case RTSP_FORMAT_MPV:
        case RTSP_FORMAT_H263:
        case RTSP_FORMAT_JPEG:
        case RTSP_FORMAT_MPA:
        case RTSP_FORMAT_ADPCM:
        case RTSP_FORMAT_AAC:
        case RTSP_FORMAT_G722:
        default:
            mylog_error("UN SUPPORTED MEDIA TYPE\n");
            break;
        case RTSP_FORMAT_PCM:
        case RTSP_FORMAT_ALAW:
        case RTSP_FORMAT_ULAW:
            source->track[t].rtp = new_rtp_media_rawaudio_stream(
                                       source->track[t].stream->stream );
            break;
        }

    if( ! source->track[t].rtp ) return -1;

    stream_set_waiting( source->track[t].stream, 1 );
    return 0;
    }

static int set_path( int num_tokens, struct token *psToken, void *d )
    {
    if( num_tokens == 2 )
        {
        rtsp_new_location( psToken[1].v.str, NULL, NULL, NULL,
                           audio_only_open, d );
        return 0;
        }
    if( num_tokens == 5 )
        {
        rtsp_new_location( psToken[1].v.str, psToken[2].v.str,
                           psToken[3].v.str, psToken[4].v.str,
                           audio_only_open, d );
        return 0;
        }
    return -1;
    }


static struct statement audio_only_config_statements[] =
    {
    /* directive name, process function, min args, max args, arg types */
        { "track", set_track, 1, 1, { TOKEN_STR } },
        { "path", set_path, 1, 4, { TOKEN_STR, TOKEN_STR, TOKEN_STR, TOKEN_STR } },
    /* empty terminator -- do not remove */
        { NULL, NULL, 0, 0, {} }
    };

int audio_only_init(void)
    {
    register_config_context( "rtp", "audio_only", start_block, end_block,
                             audio_only_config_statements );
    return 0;
    }
