/*******************************************************************************
 * Copyright (c) 2012, 2013 IBM Corp.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v1.0 which accompany this distribution.
 *
 * The Eclipse Public License is available at
 *   http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 *   http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *    Ian Craggs - initial contribution
 *    Ian Craggs - change delimiter option from char to string
 *    Al Stockdill-Mander - Version using the embedded C client
 *******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <sys/ioctl.h>
#include <sys/msg.h>

#include <fcntl.h>

#include "MQTTClient.h"
#include "os_port.h"
#include <pthread.h>
#include "pb_encode.h"
#include "pb_decode.h"
#include "iot_packet.pb.h"
#include "curl/curl.h"
#include "time.h"
#include <sys/types.h>
#include <assert.h>
#include "hubble_config.h"

#define TRUE            1

#define TRANSACTION_ID_LENGTH 36
#define MQTT_THREAD_SLEEPTIME 100 //ms
#define KEEPALIVE_TIMER 10*(1000/MQTT_THREAD_SLEEPTIME) //30s
#define CAMERA_PORT 80
#define RTMP_STREAM_LENGTH          12
#define HEX_RTMP_STREAM_LENGTH      RTMP_STREAM_LENGTH * 2
#define SKIP_DOWNLOAD_CERT -3
#define COMMAND_TIME_OUT             10000000
#define COMMAND_IN_PROGRESS_TIME     2000000
#define PACKET_ARRAY_LENGTH 6
#define MULTI_CMD_SUPPORTED 5
#define DEFAULT_INDEX_SINGLE_CMD 0
#define MAX_WAIT_MSECS 10*1000 /* Wait max. 10 seconds */

struct MemoryStruct
{
    char* buffer;
    size_t size;
};
struct MemoryStruct chunk[6];
//Variable for mqtt handle thread
typedef enum eHandleCmdSrvStat
{
    eHandleCommandStarted = 0, // Set by messageArrived if no command is in progress. messageArrived will send command to messageHandle in packet
    eHandleCommandRunning, // Set by messageHandle. messageHandle reads packet sent by messageArrived and pass to mqtts_handle_thread to handle command
    eHandleCommandStopped,
    eHandleCommandFinished, // Set by mqtts_handle_thread once command handler finished
    eHandleCommandTimeOut,
    eHandleCommandNone,
}eHandleCmdSrvStat;

typedef struct
{
    Packet packetSend;
    char cmd[128]; // Store command string
    char response[128]; // Store response for command.
    eHandleCmdSrvStat handleStatus; // Store handle_status
    char recv_topic[MQTT_TOPIC_LENGTH];
    struct timeval start_handle_time;
}messagePacket;
messagePacket packet_array[6]; // Index 0 for Set commands, Index [1-6] for handling get cmd paralelly.


enum
{
    MQTTS_STATE_RUNNING = 0, // start to listen the subscribe topic
    MQTTS_STATE_STOP, // end the mqtts_main, need to re-init all to continue the mqtts
    MQTTS_STATE_RESTART, // re-init the mqtts, re-subscribe all topic in storage list
    MQTTS_STATE_PAUSE // temporaly stop the MQTT, actually need this state ?????
};
static volatile int mqtts_state = MQTTS_STATE_STOP;
extern MQTT_LOGGER mqtt_logger;
static volatile int mqtt_enable = 1;
static volatile int mqtt_continue = 1;
static Client client;
static char client_id[MQTT_CLIENTID_LENGTH] = {0};
static unsigned char buf[MQTT_BUF_SIZE];
static unsigned char readbuf[MQTT_READBUF_SIZE];
static int count_sub_topic = 0;
//static int subscribe_recover = 0;
static char topic_array[MAX_SUB_TOPIC_SUPPORT][MQTT_TOPIC_LENGTH];
static struct Network network;
static char gHost[64] = {0};
static int gPort;
static char gUsername[32] = {0};
static char gPassword[64] = {0};
static int mqttsRestartFailedCount = 0;
static struct timeval startHandleMessage;
static char transID[37] = {0};
char recvTopicStr[MQTT_TOPIC_LENGTH] = {0};
static pthread_mutex_t mqtts_data_access = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t mqtts_multi_cmd_data_access = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t mqtts_data_cond = PTHREAD_COND_INITIALIZER;
static pthread_cond_t mqtts_multi_cmd_data_cond = PTHREAD_COND_INITIALIZER;
static char multi_curl_error_buffer[6][128];
static int msg_received = 0;
static int cmd_in_progress = 0;
static int curl_multi_ready = 0;

static char command_list[][64] =
{
    "action=command&command=camera_setting2",
    "action=command&command=get_blink_led",
    "action=command&command=camera_parameter_setting",
    "action=command&command=get_recording_parameter",
    "action=command&command=value_temperature",
    "action=command&command=get_wifi_strength",
    "action=command&command=get_soc_version",
    "action=command&command=get_resolution",
    "action=command&command=get_time_zone",
    "action=command&command=get_version"
};
static int command_list_size = 0;

CALLBACKHANDLER commandHanderFunc = NULL;
void messageArrived(MessageData* md);

static size_t WriteMemoryCallback(void *contents, size_t size, size_t nmemb, void *userp)
{
    size_t realsize = size * nmemb;
    struct MemoryStruct *mem = (struct MemoryStruct *)userp;
    if(realsize < 8000)
    {
        assert(mem);
        if(mem->buffer == NULL)
        {
            assert(mem->buffer);
            hlog_error("Memory NULL");
            return 0;
        }
        memcpy(mem->buffer, contents, realsize);
        mem->size = realsize;
        mem->buffer[mem->size] = 0;
    }
    else
    {
        hlog_warn("REPLY SIZE %d CAN NOT BE BIGGER THAN 8 KBs", realsize);
        return -1;
    }
    return realsize;
}

int transfer_with_multi_curl(CURLM *curlMulti, int command_count)
{
    int retries = 3;
    int transferLeft = command_count;
    int transfer_status = -1;
    int http_status_code;
    CURLMsg *msg=NULL;
    CURL *easyCurlHandle=NULL;
    int msgs_left=0;
    int stillRuning;
    CURLcode return_code=0;
    const char *szUrl;
    while(transferLeft > 0 && transfer_status!= -2)
    {
        stillRuning = 0;
        // Downloading all Certs at once
        do {
            int numfds=0;
            curl_multi_perform(curlMulti, &stillRuning);
            // Wait for Certs processing
            int res = curl_multi_wait(curlMulti, NULL, 0, MAX_WAIT_MSECS, &numfds);
            if(res != CURLM_OK)
            {
                hlog_error("error: curl_multi_wait() returned %d . Removing all curl_easy", res);
                transfer_status = -2;
                break;
            }
        } while(stillRuning);
        hlog_info("CURL_MULTI: All transfer is done , Checking for success status");
        // Finish transfering with curl_multi, check for transfer status
        while ((msg = curl_multi_info_read(curlMulti, &msgs_left)))
        {
            easyCurlHandle = msg->easy_handle;
            if (transfer_status == -2)
            {
                // If an operation failed to proceed with CURL, remove all others curl_easy_handle
                curl_multi_remove_handle(curlMulti, easyCurlHandle);
                curl_easy_cleanup(easyCurlHandle);
                continue;
            }
            // Check easy_curl_handle if the transfer was done or not
            if (msg->msg == CURLMSG_DONE)
            {
                return_code = msg->data.result;
                // Check if CURL transfer successfully of not
                if(return_code!=CURLE_OK)
                {
                    hlog_error("CURL error code: %d\n", msg->data.result);
                    hlog_error("Operation number %d failed", command_count - transferLeft + 1);
                    hlog_error("CURL error msg: '%s'", multi_curl_error_buffer[command_count - transferLeft]);
                    transfer_status = -2;
                    curl_multi_remove_handle(curlMulti, easyCurlHandle);
                    curl_easy_cleanup(easyCurlHandle);
                    continue;
                }
                else
                {
                    transferLeft--;
                    transfer_status = 0;
                }

                // Get HTTP status code
                http_status_code=0;
                szUrl = NULL;
                curl_easy_getinfo(easyCurlHandle, CURLINFO_RESPONSE_CODE, &http_status_code);
                curl_easy_getinfo(easyCurlHandle, CURLINFO_PRIVATE, &szUrl);
                if(http_status_code==200)
                {
                    hlog_debug("Curl transfer return HTTP Code 200 %s\n", szUrl);
                }
                else
                {
                    hlog_error("Curl transfer with http return code = %d \n", http_status_code);
                }
                curl_multi_remove_handle(curlMulti, easyCurlHandle);
                curl_easy_cleanup(easyCurlHandle);
            }
            else
            {
                // Retry failed CURL_easy
                transfer_status = -1;
                retries --;
                hlog_error("Curl_Multi Download Failed with CURLMsg=%d\n", msg->msg);
                if (retries == 0)
                {   // Remove Curl_easy handle if failed more than 3 times and restart Downloading Certs state
                    curl_multi_remove_handle(curlMulti, easyCurlHandle);
                    curl_easy_cleanup(easyCurlHandle);
                }
                hlog_error("Checking next curl_easy, Failed file will be re-transfer with retries = %d",retries);
            }
        }
        usleep(10000);
    }
    return transfer_status;
}

int add_multi_curl(CURLM *curlMulti,char *cmd,char *response,int packet_idx)
{
    // Add Cert's URL and path to array
    CURL *curl = curl_easy_init();
    char command[256] = {0};
    int port = CAMERA_PORT;
    if (curl == NULL)
    {
        hlog_error("Cant Init CURL\n");
        return -1;
    }
#if(STREAMMGR_FULL_PROTOCOL)
    /* Forward get_session_key commadn to port 9090 */
    pTmpStr = strstr(cmddata, "get_session_key");
    if(pTmpStr)
    {
        /* Right now we do not handle RTMP stream and record */
        if(strstr(pTmpStr, "rtmp") == NULL)
            port = 9090;
    }
#endif
    sprintf(command, "http://%s:%d/?%s", CAMERA_IP, port, cmd);
    hlog_debug("Add [Index]:%d | cmd: '%s' to multi curl",packet_idx,command);
    curl_easy_setopt(curl, CURLOPT_URL, command);
    // NOTICE WE DONT VERIFY PEER FOR NOW
    curl_easy_setopt(curl, CURLOPT_TIMEOUT, 10);
    curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 1);
    curl_easy_setopt(curl, CURLOPT_FAILONERROR, 1);
    curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);

    // use for debugging curl
    curl_easy_setopt(curl, CURLOPT_VERBOSE, 0L);
    curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, multi_curl_error_buffer[packet_idx]);
    /* write the page body to this file handle */
    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0L);

    chunk[packet_idx].buffer = response;
    // Set Callback to download cert
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void * )&chunk[packet_idx]);

    curl_multi_add_handle(curlMulti, curl);
    return 0;
}

void *mqtts_handle_multi_cmd_thread ()
{
    int retries  = 0;
    int packet_idx; // Index packet_array[1-5] are reserved for multiple commands.
    int i32ret = 0;
    int cmd_count;
    int index_mark_array[6] = {0}; // This array is used to mark which index is used to transfer command - 0 : not used | 1 : used
    CURLM *curl_multi_handler = NULL;
    curl_multi_handler = curl_multi_init();
    while(1)
    {
        cmd_count = 0;
        curl_multi_ready = 1;
        pthread_mutex_lock(&mqtts_multi_cmd_data_access);
        pthread_cond_wait(&mqtts_multi_cmd_data_cond, &mqtts_multi_cmd_data_access);
        curl_multi_ready = 0;
        for(packet_idx = 1; packet_idx < PACKET_ARRAY_LENGTH ; packet_idx++)
        {
            if(packet_array[packet_idx].handleStatus == eHandleCommandRunning)
            {
                index_mark_array[packet_idx] = 1;
                add_multi_curl(curl_multi_handler, packet_array[packet_idx].cmd, packet_array[packet_idx].response, packet_idx);
                cmd_count ++;
            }
            else
                hlog_debug("MQTT: Got signal for command handling. Current state: %d", packet_array[packet_idx].handleStatus);
        }
        pthread_mutex_unlock(&mqtts_multi_cmd_data_access);
        i32ret = transfer_with_multi_curl(curl_multi_handler,cmd_count);
        if(i32ret < 0)
        {
            hlog_error("Handle multiple commands with CURL failed!!!");
        }
        for(packet_idx = 1; packet_idx < PACKET_ARRAY_LENGTH ; packet_idx++)
        {
            // If all commands are served already, no need to loop anymore
            if(cmd_count == 0)
                break;
            if(index_mark_array[packet_idx] == 0) // If index wasn't used to transfer command
                continue;
            retries = 3;
            while(retries > 0) // Fix in case camera in stopped mode
            {
                pthread_mutex_lock(&mqtts_multi_cmd_data_access);
                if(packet_array[packet_idx].handleStatus == eHandleCommandRunning)
                {
                    cmd_count --;
                    packet_array[packet_idx].handleStatus = eHandleCommandFinished;
                    pthread_mutex_unlock(&mqtts_multi_cmd_data_access);
                    break;
                }
                else
                    hlog_info("......Current status %d...........", packet_array[packet_idx].handleStatus); // In case falling to this, something wrong
                pthread_mutex_unlock(&mqtts_multi_cmd_data_access);
                retries--;
                usleep(100000);
            }
            //Reset mark
            index_mark_array[packet_idx] = 0;
        }
    }
    curl_multi_cleanup(curl_multi_handler);
}
/*@brief        Handle single SET command for MQTT 
 * @details     1) Loop forever and wait for commands from messageHandle 
 *              2) On condition signaled, to local HTTP
 *              3) Send back reponse to messageHandle
 *              This function is for single command only, which is reserved at Index 0 of packet_array
 * @param[in]   
 * @param[out] 
 */
void *mqtts_handle_thread ()
{
    //    hlog_info("%s - %d\n", __func__,__LINE__);
    //    pthread_mutex_lock(&mqtts_data_access);
    int retries  = 0;
    char urlhandle[128] = {0};
    char resulthandle[128] = {0};
    int i32ret = 0;
    CURL* curl_handle_command;
    curl_handle_command = curl_easy_init();
    while(1)
    {
        pthread_mutex_lock(&mqtts_data_access);
        pthread_cond_wait(&mqtts_data_cond,&mqtts_data_access);
        if(packet_array[DEFAULT_INDEX_SINGLE_CMD].handleStatus == eHandleCommandRunning)
        {
            sprintf(urlhandle,"%s",packet_array[DEFAULT_INDEX_SINGLE_CMD].cmd);
            memset(resulthandle,0,sizeof(resulthandle));
            pthread_mutex_unlock(&mqtts_data_access);
            //            hlog_info("%s - %d\n", __func__,__LINE__);

            /* After receive reset_factory cmd, mqtt wait for success returning from plugin_http,
             * But after plugin_http do reboot and return to mqtt, mqtt don't have enough time to make the response to server.
             * Workarround: Force this result to success
             */
            if(strncmp(urlhandle, "action=command&command=reset_factory", sizeof("action=command&command=reset_factory")) == 0)
            {
                sprintf(packet_array[DEFAULT_INDEX_SINGLE_CMD].response, "reset_factory: 0");
                packet_array[DEFAULT_INDEX_SINGLE_CMD].handleStatus = eHandleCommandFinished;
            }

            retries = 2;
            while(retries > 0)
            {
                i32ret = CvCmdMgrBridge(curl_handle_command, urlhandle,resulthandle);
                if(i32ret == -1)
                {
                    curl_easy_cleanup(curl_handle_command);
                    curl_handle_command = curl_easy_init();
                    hlog_debug("Can not receive HTTP %s, reinit curl handle", urlhandle);
                }
                else if(i32ret == 1)
                {
                    curl_handle_command = curl_easy_init(); // Reinit curl easy handle
                    hlog_debug("Reinit curl_easy_handler");
                }
                else
                    break;
                retries--;
            }
            if(retries <= 0)
            {
                sprintf(resulthandle,"NA");
            }
            sprintf(packet_array[DEFAULT_INDEX_SINGLE_CMD].response,"%s",resulthandle);
            hlog_info("Result: %s \n",packet_array[DEFAULT_INDEX_SINGLE_CMD].response);
            retries = 3;
            while(retries > 0) // Fix in case camera in stopped mode
            {
                pthread_mutex_lock(&mqtts_data_access);
                if(packet_array[DEFAULT_INDEX_SINGLE_CMD].handleStatus == eHandleCommandRunning)
                {
                    packet_array[DEFAULT_INDEX_SINGLE_CMD].handleStatus = eHandleCommandFinished;
                    pthread_mutex_unlock(&mqtts_data_access);
                    break;
                }
                else
                    hlog_info("......Current status %d...........", packet_array[DEFAULT_INDEX_SINGLE_CMD].handleStatus); // In case falling to this, something wrong
                pthread_mutex_unlock(&mqtts_data_access);
                retries--;
                usleep(100000);
            }
        }
        else
        {
            hlog_debug("MQTT: Got signal for command handling. Current state: %d", packet_array[DEFAULT_INDEX_SINGLE_CMD].handleStatus);
            pthread_mutex_unlock(&mqtts_data_access);
        }
    }
    curl_easy_cleanup(curl_handle_command);
}

//Nuvoton RTMP compatible
//Change RTMP streamname to HEX
#ifdef NUVOTON_PLATFORM_SUPPORT
void CheckForStream(char*newCmdPadding)
{
    if(strstr(newCmdPadding,"relay_rtmp") != NULL)
    {
        // Convert stream name to hex string
        // Avoid stream name generated by server include camera parser's key (e.g "type", "grid", zone", ... in httpd.c)
        int i =0;
        char* pStreamname;
        char cHexStreamName[HEX_RTMP_STREAM_LENGTH + 1] = {0};

        pStreamname = strstr(newCmdPadding,"streamname");
        if(pStreamname != NULL)
        {
            pStreamname += strlen("streamname="); // Move pointer to value of stream name
            for(i = 0; i < RTMP_STREAM_LENGTH; i++)
            {
                sprintf(cHexStreamName + i * 2, "%02X", pStreamname[i]);
            }
            hlog_info("Hex Stream name %s", cHexStreamName);
            memcpy(pStreamname, cHexStreamName , sizeof(cHexStreamName));
        }
    }
}
#endif

void mqtts_control(int id)
{
    switch(id)
    {
        case MQTTS_CMD_STOP: //off
            hlog_info("mqtt_control: MQTTS_CMD_STOP\n");
            mqtts_state = MQTTS_STATE_STOP;
            break;
        case MQTTS_CMD_START: //on
            hlog_info("mqtt_control: MQTTS_CMD_START\n");
            mqtts_state = MQTTS_STATE_RUNNING;
            break;
        case MQTTS_CMD_RESTART: //restart
            hlog_info("mqtt_control: MQTTS_CMD_RESTART\n");
            mqtts_state = MQTTS_STATE_RESTART;
            break;
    }
}


int checking_input()
{
    if((strlen(client_id) == 0) || commandHanderFunc == NULL)
        return -1;
    return 0;
}

int mqtts_publish(char *topic, char *msg)
{
    int rc;
    int i = 0;
    MQTTMessage message;
    if(mqtt_enable && mqtt_continue)
    {
        memset(&message, 0, sizeof(message));
        message.qos = QOS0;
        message.id = 0;
        message.payloadlen = strlen(msg);
        message.payload = (void *)msg;
        do
        {
            rc = MQTTPublish(&client, topic, &message);
            hlog_info("Publish topic: %s, msg: %s, %d\n", topic, msg, rc);
            if(i++ > 3)
                break;
        }
        while(rc != 0);
        return rc;
    }
    else
    {
        hlog_info("The mqtts is not running\n");
        return -1;
    }
}

//LongLP
int mqtts_publish_1(char *topic, uint8_t *msg,int len)
{
    if(mqtt_enable && mqtt_continue)
    {
        int i = 0;
        int rc;
        MQTTMessage message;
        memset(&message, 0, sizeof(message));
        message.qos = QOS1;//0
        message.id = 0;
        message.payloadlen = len;
        //        mqtt_logger("%d\n", message.payloadlen);
        message.payload = (void *)msg;
        do
        {
            rc = MQTTPublish(&client, topic, &message);
            //hlog_info("Publish topic: %s, msg: %s, %d\n", topic, msg, rc);
            if(i++ > 3)
                break;
        }
        while(rc != 0);
        return rc;
    }else
    {
        mqtt_logger("The mqtts is not running\n");
        return -1;
    }
}

int mqtts_unsubscribe(char *topic)
{
    if(topic && mqtt_enable && mqtt_continue && (count_sub_topic > 0) && (strlen(topic) > 0) && (strlen(topic) < MQTT_TOPIC_LENGTH))
    {
        int rc = MQTTUnsubscribe(&client, topic);
        hlog_info("Unsubcribe topic: %s, %d\n", topic, rc);
        if(rc == 0) // store to use in case need to re-subscribe
        {
            int i = 0;
            for(i = 0; i < MAX_SUB_TOPIC_SUPPORT; i++)
            {
                if(strcmp(topic_array[i], topic) == 0)
                {
                    memset(topic_array[i], 0, MQTT_TOPIC_LENGTH);
                    count_sub_topic--;
                    break;
                }
            }
        }
        return rc;
    }
    else
    {
        hlog_info("The mqtts is not running, or no topic to unsubcribe, or topic len is out of [1-%d]\n", MQTT_TOPIC_LENGTH);
        return -1;
    }
}

int mqtts_subscribe(char *topic)
{
    hlog_info("topic: '%s'\n", topic);
    hlog_info("mqtt_enable :'%d'\n", mqtt_enable);
    hlog_info("mqtt_continue: '%d'\n", mqtt_continue);
    hlog_info("count_sub_topic: '%d'\n", count_sub_topic);

    if(topic && mqtt_enable && mqtt_continue && count_sub_topic < MAX_SUB_TOPIC_SUPPORT && strlen(topic) > 0 && strlen(topic) < MQTT_TOPIC_LENGTH)
    {
        int rc;
        int i;
        // check whether the current topic is already subscribed
        if(count_sub_topic > 0)
        {
            for(i = 0; i < MAX_SUB_TOPIC_SUPPORT; i++)
            {
                if(strcmp(topic_array[i], topic) == 0)
                {
                    hlog_info("This topic has already subscribed\n");
                    return 0;
                }
            }
        }
        // do subscribe and store to the subscribed list
        i = 0;
        do
        {
            rc = MQTTSubscribe(&client, topic, QOS1, messageArrived);
            i++;
        }
        while(rc != 0 && i < 3);
        hlog_info("Subcribe to topic: %s, %d\n", topic, rc);
        if(rc == 0) // store to use in case need to re-subscribe
        {
            for(i = 0; i < MAX_SUB_TOPIC_SUPPORT; i++)
            {
                if(strlen(topic_array[i]) == 0)
                {
                    sprintf(topic_array[i], "%s", topic);
                    count_sub_topic++;
                    break;
                }
            }
        }
        return rc;
    }
    else
    {
        hlog_info("The mqtts is not running, or reach max topic support %d, or topic len is out of [1-%d]\n", MAX_SUB_TOPIC_SUPPORT, MQTT_TOPIC_LENGTH);
        return -1;
    }
}

int mqtts_set_clientid(char *clientid)
{
    //Borrow this place to init the topic array
    memset(topic_array, 0, sizeof(topic_array));
    if(clientid && strlen(clientid) < sizeof(client_id))
    {
        sprintf(client_id, "%s", clientid);
        return 0;
    }
    else
        return -1;
}

int mqtts_set_callbackhandler(CALLBACKHANDLER handler_func)
{
    commandHanderFunc = handler_func;
    return 0;
}

void messageArrived(MessageData* md)
{
    bool status;
    size_t message_length;
    uint8_t buffer[2048] = {0};
    MQTTMessage* message = md->message;
    MQTTString *mqString = md->topicName;
    int recvTopicLen = 0;
    char *msg = (char *)message->payload;
    char *topic = (char *)(mqString->lenstring.data);
    char retained = message->retained;
    char messageBody [1028] = {0};
    int packet_idx = 0;
    int iter = 0;
    int mqtt_is_busy = 0;
    char *temp_pointer = NULL;
    gettimeofday(&startHandleMessage,NULL);
    recvTopicLen = MQTTstrlen(*mqString);
    if(recvTopicLen <= 0 || recvTopicLen > MQTT_TOPIC_LENGTH)
    {
        hlog_info("messageArrived: topic len is not in range [1, %d]\n", MQTT_TOPIC_LENGTH);
        return;
    }
    memcpy(recvTopicStr, topic, recvTopicLen);

    if(!msg || message->payloadlen <= 0)
    {
        hlog_info("messageArrived: message data is not correct \n");
        return;
    }
    msg[message->payloadlen] = '\0';
    hlog_debug("messageArrived, retained %x: %s", retained, recvTopicStr);
    /* Allocate space for the decoded message. */

    Packet packet = Packet_init_zero;
    /* Create a stream that reads from the buffer. */
    pb_istream_t stream = pb_istream_from_buffer(msg, message->payloadlen);

    /* Now we are ready to decode the message. */
    status = pb_decode(&stream, Packet_fields, &packet);
    /* Check for errors... */
    if (!status)
    {
        hlog_error("Decoding failed: %s\n", PB_GET_ERROR(&stream));
        return ;
    }
    Parse_RestAPI(&packet,messageBody);
    hlog_debug("Transaction id: %s, payload length: %d",packet.header.transaction_id,message->payloadlen);
    hlog_info("Message: %s",messageBody);
    if ((message->dup == 1) && (message->qos == 1))
    {

        return;
    }
    if(strncmp(transID,packet.header.transaction_id,TRANSACTION_ID_LENGTH)== 0)
    {
        //Return if received duplicated message to previous one
        hlog_info("Transaction ID duplicated with previous command %s!!! Discard command",transID);
        return;
    }
    else
    {
        for(iter = 0 ;iter < command_list_size; iter ++ )
        {
            temp_pointer=strstr(messageBody, command_list[iter]);
            if(temp_pointer != NULL)
            {
                hlog_debug("command supported multi process found!!!!");
                packet_idx = 1; // Packet array reserve index [1-5] for multi commands process
                break;
            }
        }
        if(packet_idx == 0) // Packet index 0 is reserved for single command.
            pthread_mutex_lock(&mqtts_data_access);
        else
            pthread_mutex_lock(&mqtts_multi_cmd_data_access);
        for(; packet_idx < PACKET_ARRAY_LENGTH; packet_idx++)
        {
            if(packet_idx == 0)
            {
                // Handle single command.
                if(packet_array[packet_idx].handleStatus == eHandleCommandRunning || packet_array[packet_idx].handleStatus == eHandleCommandStopped || 
                        packet_array[packet_idx].handleStatus == eHandleCommandFinished)
                    mqtt_is_busy = 1;
                break;
            }
            else
            {
                // Handle multi command.
                if(packet_array[packet_idx].handleStatus == eHandleCommandRunning || packet_array[packet_idx].handleStatus == eHandleCommandStopped || 
                        packet_array[packet_idx].handleStatus == eHandleCommandFinished || packet_array[packet_idx].handleStatus == eHandleCommandStarted)
                {
                    if(packet_idx == PACKET_ARRAY_LENGTH - 1) // If all 5 reserved index are busy.
                        mqtt_is_busy = 1;
                    continue;
                }
                else
                    break; // Found an available index to serve command.
            }
        }
        if(packet_idx == 0)
            pthread_mutex_unlock(&mqtts_data_access);
        else
            pthread_mutex_unlock(&mqtts_multi_cmd_data_access);

        if(mqtt_is_busy == 1)
        {
            packet.header.request_type = RequestType_SUCCESS;
            packet.header.has_code = 1;
            packet.header.code = StatusCodes_COMMAND_BUSY;
            packet.header.has_retry_after = 1;
            sprintf(packet.header.retry_after,"2");
            /* Create a stream that will write to our buffer. */
            pb_ostream_t stream = pb_ostream_from_buffer( buffer, sizeof(buffer));
            /* Now we are ready to encode the message! */
            status = pb_encode(&stream, Packet_fields, &packet);
            message_length = stream.bytes_written;
            mqtt_logger("message_length : %d",message_length);
            if (!status)
            {
                hlog_error("Encoding failed: %s\n", PB_GET_ERROR(&stream));
                return;
            }
            if(commandHanderFunc && msg)
            {
                commandHanderFunc(recvTopicStr, buffer, message_length);
                hlog_debug("Camera busy, return code %d", StatusCodes_COMMAND_BUSY);
            }
            return;
        }
        else
        {
            msg_received ++;
        }
    }
    if(packet_idx == 0)
        pthread_mutex_lock(&mqtts_data_access);
    else
        pthread_mutex_lock(&mqtts_multi_cmd_data_access);
    packet_array[packet_idx].handleStatus = eHandleCommandStarted;
    if(packet_idx == 0)
        pthread_mutex_unlock(&mqtts_data_access);
    else
        pthread_mutex_unlock(&mqtts_multi_cmd_data_access);

    strncpy(transID,packet.header.transaction_id, TRANSACTION_ID_LENGTH);
    transID[TRANSACTION_ID_LENGTH] = '\0';
    // Store packet information into struct
    memcpy(&packet_array[packet_idx].packetSend, &packet, sizeof(Packet));
    sprintf(packet_array[packet_idx].cmd,"%s", messageBody);
    memcpy(packet_array[packet_idx].recv_topic, topic, recvTopicLen);
    hlog_debug("[Cmd(s) in queue: %d] Pushed cmd: %s -> Index: %d", msg_received, packet_array[packet_idx].cmd, packet_idx);
}
#define NUMBER_CYCLE_1_SECOND    4
#define TIME_SEND_PROCESS        2
#define TIME_OUT_HANDLE          100
void messageHandle()
{ // Handle command and Camera's HTTP server status
    bool status;
    size_t message_length;
    uint8_t buffer[2048] = {0};
    int command_index = 0;
    char *temp_pointer;
    int packet_idx = 0;
    long process_time;
    long time_till_respond_in_progress;
    struct timeval curr_time;

    for(; packet_idx < PACKET_ARRAY_LENGTH ; packet_idx ++)
    {
        temp_pointer = NULL;
        if(msg_received == 0)
            return;
        if(packet_array[packet_idx].handleStatus == eHandleCommandNone)
        {
            continue;
        }
        cmd_in_progress = 0; 
        if(packet_array[packet_idx].handleStatus == eHandleCommandStarted)
        {
            //For RTMP backward nuvoton
            if(packet_idx == 0)
            {
                //For RTMP backward nuvoton
#ifdef NUVOTON_PLATFORM_SUPPORT
                CheckForStream(packet_array[packet_idx].cmd);
#endif
                pthread_mutex_lock(&mqtts_data_access);
                packet_array[packet_idx].handleStatus = eHandleCommandRunning;
                pthread_cond_signal(&mqtts_data_cond);
                pthread_mutex_unlock(&mqtts_data_access);
                gettimeofday(&packet_array[packet_idx].start_handle_time,NULL);
            }
            else
            {
                if(curl_multi_ready == 0) // If thread handling multi cmd is busy handling another cmd, temporary skip this
                {
                    hlog_debug("CURL_MULTI is not ready, skip for now!!!");
                    continue;
                }
                pthread_mutex_lock(&mqtts_multi_cmd_data_access);
                packet_array[packet_idx].handleStatus = eHandleCommandRunning;
                pthread_cond_signal(&mqtts_multi_cmd_data_cond);
                pthread_mutex_unlock(&mqtts_multi_cmd_data_access);
                gettimeofday(&packet_array[packet_idx].start_handle_time,NULL);
            }
        }

        if(packet_array[packet_idx].handleStatus == eHandleCommandRunning)
        {
            gettimeofday(&curr_time,NULL);
            process_time = (curr_time.tv_sec - packet_array[packet_idx].start_handle_time.tv_sec)*1000000L + curr_time.tv_usec - packet_array[packet_idx].start_handle_time.tv_usec;
            time_till_respond_in_progress = process_time % COMMAND_IN_PROGRESS_TIME; 
            if(packet_idx == 0)
            {
                if(process_time >= COMMAND_TIME_OUT)
                {
                    hlog_info("[TIME-OUT | Index: %d]Total process time: %ld ", packet_idx,process_time);
                    pthread_mutex_lock(&mqtts_data_access);
                    packet_array[packet_idx].handleStatus = eHandleCommandTimeOut;
                    pthread_mutex_unlock(&mqtts_data_access);
                }
                else if(time_till_respond_in_progress > COMMAND_IN_PROGRESS_TIME)
                {
                    hlog_info("[IN-PROGRESS | Index: %d] Total process time: %ld , Time count from last check: %ld", packet_idx, process_time, time_till_respond_in_progress);
                    pthread_mutex_lock(&mqtts_data_access);
                    packet_array[packet_idx].handleStatus = eHandleCommandStopped;
                    pthread_mutex_unlock(&mqtts_data_access);
                    gettimeofday(&packet_array[packet_idx].start_handle_time,NULL);
                }
            }
            else
            {
                if(process_time >= COMMAND_TIME_OUT)
                {
                    hlog_info("[TIME-OUT | Index: %d]Total process time: %ld ", packet_idx, process_time);
                    pthread_mutex_lock(&mqtts_multi_cmd_data_access);
                    packet_array[packet_idx].handleStatus = eHandleCommandTimeOut;
                    pthread_mutex_unlock(&mqtts_multi_cmd_data_access);
                }
                else if(time_till_respond_in_progress > COMMAND_IN_PROGRESS_TIME)
                {
                    hlog_info("[IN-PROGRESS | Index: %d] Total process time: %ld , Time count from last check: %ld", packet_idx, process_time, time_till_respond_in_progress);
                    pthread_mutex_lock(&mqtts_multi_cmd_data_access);
                    packet_array[packet_idx].handleStatus = eHandleCommandStopped;
                    pthread_mutex_unlock(&mqtts_multi_cmd_data_access);
                    gettimeofday(&packet_array[packet_idx].start_handle_time,NULL);
                }
            }
        }
        if(packet_idx == 0)
            pthread_mutex_lock(&mqtts_data_access);
        else
            pthread_mutex_lock(&mqtts_multi_cmd_data_access);

        if(packet_array[packet_idx].handleStatus == eHandleCommandFinished || packet_array[packet_idx].handleStatus == eHandleCommandTimeOut || 
                packet_array[packet_idx].handleStatus == eHandleCommandStopped)
        {
            Parse_Protobuf( packet_array[packet_idx].response, &packet_array[packet_idx].packetSend, packet_idx);
            if(packet_idx == 0)
                pthread_mutex_unlock(&mqtts_data_access);
            else
                pthread_mutex_unlock(&mqtts_multi_cmd_data_access);
            if(packet_array[packet_idx].handleStatus == eHandleCommandFinished)
                mqtt_logger("Packet  response.body after decoding : %s!", packet_array[packet_idx].packetSend.response.body);
            /* Create a stream that will write to our buffer. */
            pb_ostream_t stream = pb_ostream_from_buffer( buffer, sizeof(buffer));
            /* Now we are ready to encode the message! */
            status = pb_encode(&stream, Packet_fields, &packet_array[packet_idx].packetSend);
            message_length = stream.bytes_written;
            if (!status)
            {
                hlog_error("Encoding failed: %s\n", PB_GET_ERROR(&stream));
                return;
            }
            if(commandHanderFunc)
            {
                hlog_info("[Index: %d]Reponse: %s -> Command: %s | Status: %d", packet_idx, packet_array[packet_idx].response, packet_array[packet_idx].cmd, packet_array[packet_idx].handleStatus);
                commandHanderFunc(packet_array[packet_idx].recv_topic, buffer, message_length);
                if(cmd_in_progress != 1)
                {
                    // Publish back reponse to server
                    temp_pointer = NULL;
                    for(command_index = 0 ; command_index < command_list_size; command_index ++ )
                    {
                        temp_pointer=strstr(packet_array[packet_idx].cmd, command_list[command_index]);
                        if(temp_pointer != NULL)
                        {
                            // Finish handling command, release it so that camera would accept this command again in future 
                            hlog_debug("Release mark for cmd %s", packet_array[packet_idx].cmd);
                            break;
                        }
                    }
                    msg_received --;
                    hlog_debug(" Remove Index: %d  -> Received cmd(s): %d", packet_idx, msg_received);
                }
                else
                {
                    // Command is in progress. Do nothing 
                    hlog_debug("[ Index: %d] Could not remove msg_received, status = %d", packet_idx, packet_array[packet_idx].handleStatus);
                }
            }
        }
        else
        {
            if(packet_idx == 0)
                pthread_mutex_unlock(&mqtts_data_access);
            else
                pthread_mutex_unlock(&mqtts_multi_cmd_data_access);
        }
    }
}

#if 1
int mqtts_init(char *host, int port, char *username, char *password)
{
    int rc = 0 ;
    int retries = 3;
    int nwk_retries = 0;
    struct timeval start,stop;
    if(!host)
        return -1;
    if(checking_input() != 0)
    {
        hlog_info("Fail to start the mqtt, due to missing some input from user\n");
        return -1;
    }
    hlog_info("client id: %s\n", client_id);
    hlog_debug("user: [%s] - password: [%s]\n", username, password);
    hlog_info("MQTT start init, connecting to %s:%d\n",host,port);
    // Create a new connection to MQTT server
    NewNetwork(&network);
    gettimeofday(&start,NULL);
    while((ConnectNetwork(&network, host, port)) < 0)
    {
        hlog_error("Fail to connect to tcp/%s/%d , retries = %d\n\r", host, port,retries);
#if (SUPPORT_MQTT_IDLE == 1)
        if(wait_network_connection(__FILE__,__FUNCTION__,__LINE__) == 0 && nwk_retries != 0)
        {
            // Don't log error on 1st retry
        }
        else
            nwk_retries++;
#endif
        porting_msleep(MQTT_THREAD_SLEEPTIME);
        NewNetwork(&network);
        retries--;
        if(retries == 0)
        {
            porting_msleep(3000);
            return -1;
        }
    }
    gettimeofday(&stop,NULL);
    hlog_debug("_________________________MQTTS Init: ConnectNetwork took %lu__________________________ ",
            (stop.tv_sec - start.tv_sec)*1000000L+stop.tv_usec - start.tv_usec);
    retries = 3;
    MQTTClient(&client, &network, 2000, buf, sizeof(buf), readbuf, sizeof(readbuf));

    MQTTPacket_connectData data = MQTTPacket_connectData_initializer;
    data.willFlag = 0;

    data.clientID.cstring = client_id;
    memcpy(gUsername, username, strlen(username));
    memcpy(gPassword, password, strlen(password));
    data.username.cstring = gUsername;
    data.password.cstring = gPassword;

    data.keepAliveInterval = 40;
    data.cleansession = 1;
    data.MQTTVersion = 3;
    rc = -1;
    nwk_retries = 0;
    while(rc != 0)
    {
#ifdef DEBUG
        hlog_info("\nConnecting to %s %d\n", host, port);
#endif
        gettimeofday(&start,NULL);
        rc = MQTTConnect(&client, &data);
        gettimeofday(&stop,NULL);
        hlog_debug("_________________________MQTTS Init: MQTTConnect took %lu__________________________ ",
                (stop.tv_sec - start.tv_sec)*1000000L+stop.tv_usec - start.tv_usec);
        porting_msleep(MQTT_THREAD_SLEEPTIME);
        if (rc != 0)
            retries--;
        if(retries == 0)
        {
            hlog_error("\nMQTTConnect Failed with rc = %d\n",rc);
#if (SUPPORT_MQTT_IDLE == 1)
            if(wait_network_connection(__FILE__,__FUNCTION__,__LINE__) == 0 && nwk_retries != 0)
            {
                // Don't log error on 1st retry
            }
            else
                nwk_retries++;
#endif
            // In case server return error code -0x7880, skip re-download certs.
            if (get_skip_download_cert_flag() == 1)
            {
                set_skip_download_cert_flag(0);
                // Disconnect previous connection
                mqtts_stop();
                return -3;
            }
            porting_msleep(3000);
            return -2;
        }
    }
#ifdef DEBUG
    hlog_info("Connected %d\n", rc);
#endif
    hlog_info("mqtt Init done\n");
    mqtt_continue = 1;
    mqtt_set_non_block();
    sprintf(gHost, "%s", host);
    gPort = port;
    return rc;
}
#else
//LongLP

int mqtts_init_new(char *host, int port)
{
    int retries = 3;
    int rc = 0;
    hlog_info("Start the mqtt thread %s %d\n", host, port);
    if(!host)
        return -1;

    if(checking_input()!=0)
    {
        mqtt_logger("Fail to start the mqtt, due to missing some input from user\n");
        return -1;
    }
    hlog_info("client id: %s\n", client_id);
    mqtt_logger("MQTT START\n");
    NewNetwork(&n);
    while(ConnectNetwork(&n, host, port)<0)
    {
        mqtt_logger("Fail to connect to tcp/%s/%d\n\r", host, port);
        porting_msleep(MQTT_THREAD_SLEEPTIME);
        NewNetwork(&n);
        retries--;
        if(retries == 0)
        {
            porting_msleep(3000);
            return -1;
        }
    }

    retries = 3;
    MQTTClient(&c, &n, 1000, buf, sizeof(buf), readbuf, sizeof(readbuf));

    MQTTPacket_connectData data = MQTTPacket_connectData_initializer;
    data.willFlag = 0;

    data.clientID.cstring = client_id;
    data.username.cstring = NULL;
    data.password.cstring = NULL;

    data.keepAliveInterval = 60;
    data.cleansession = 1;
    data.MQTTVersion = 3;
    rc = -1;
    while(rc!=0)
    {
#ifdef DEBUG
        mqtt_logger("\nConnecting to %s %d\n", host, port);
#endif
        rc = MQTTConnect(&c, &data);
        porting_msleep(MQTT_THREAD_SLEEPTIME);
        if (rc != 0)
            retries--;
        if(retries == 0)
        {
            porting_msleep(3000);
            return -2;
        }
    }
#ifdef DEBUG
    mqtt_logger("Connected %d\n", rc);
#endif
    mqtt_logger("Init done\n");
    mqtt_continue = 1;
    mqtt_set_non_block();
    sprintf(gHost, "%s", host);
    gPort = port;
    return 0;
}
#endif
int mqtts_stop()
{
    hlog_info("MQTT STOP\n");

    MQTTDisconnect(&client);
    network.disconnect(&network);
    return 0;
}

int mqtts_stop_state()
{
    mqtt_enable = 0;
    return 0;
}

int mqtts_start_state()
{
    mqtt_enable = 1;
    return 0;
}

int mqtts_resubscribe()
{
    int i;
    int rc;
    int tmp = -1;
    for(i = 0; i < MAX_SUB_TOPIC_SUPPORT; i++)
    {
        if(strlen(topic_array[i]) > 0)
        {
            rc = MQTTSubscribe(&client, topic_array[i], QOS1, messageArrived); //LongLP
            if (rc == 0)
            {
                tmp = rc; // VinhPQ: If resubcrie failed all, return -1, if even one success, return 0
            }
            hlog_info("Subcribe topic: %s, %d\n", topic_array[i], rc);
        }
    }
    return tmp;
}

int mqtts_reset_topic_array()
{
    if(count_sub_topic > 0) // reset the topic array in case there is no re-subscribe topic request
    {
        int i;
        for(i = 0; i < MAX_SUB_TOPIC_SUPPORT; i++)
        {
            memset(topic_array[i], 0, MQTT_TOPIC_LENGTH);
        }
        count_sub_topic = 0;
    }
    return 0;
}

int init_cmd_handle_state()
{
    int index = 0;
    for(; index < PACKET_ARRAY_LENGTH ; index++)
        packet_array[index].handleStatus = eHandleCommandNone;
    command_list_size = sizeof(command_list) / sizeof(command_list[0]);
    hlog_info("There are %d commands in multiple commands support list",command_list_size);
}
void mqtts_main()
{
    int status = -1;
    msg_received = 0;
    init_cmd_handle_state();
    while(mqtt_enable)
    {
        switch(mqtts_state)
        {
            case MQTTS_STATE_RUNNING:
                MQTTYield(&client, 2000);
                break;
            case MQTTS_STATE_STOP:
                mqtts_stop();
                mqtts_reset_topic_array();
                mqtt_enable = 0;
                break;
            case MQTTS_STATE_RESTART:
                msg_received = 0;
                mqtts_stop();
#if 0
                if(mqtts_init_new(gHost, gPort) == 0)
#endif
                    if(mqttsRestartFailedCount >10)
                    {
                        // Switch to STOP state if Restart failed 10 times
                        mqtts_state = MQTTS_STATE_STOP;
                        mqttsRestartFailedCount = 0;
                        break;
                    }
                if(mqtts_init(gHost, gPort,gUsername,gPassword) == 0)
                {
                    status = mqtts_resubscribe();
                    if(status == 0)
                    {
                        mqtts_state = MQTTS_STATE_RUNNING; // Success resubcribe once
                        mqttsRestartFailedCount = 0;
                    }
                    else
                        mqttsRestartFailedCount++;
                }
                else
                    mqttsRestartFailedCount++;
                break;
            case MQTTS_STATE_PAUSE:
                break;
            default:
                break;
        }
        porting_msleep(MQTT_THREAD_SLEEPTIME);
    }// end while(mqtt_enable)
    hlog_info("mqtts_main: end\n");
}



// sret = string return. The mem needs to be avail at least 256 bytes
    int
CvCmdMgrBridge(CURL* curl_handle_command, char* cmddata, char* sret)
{
    CURLcode res;
    //    CURL* curl_handle_command;

    char cmd[256];
    int  exit_status = 0;
    int  port = CAMERA_PORT;
    //    char *pTmpStr = NULL;

    //    curl_handle_command = curl_easy_init();
    if(curl_handle_command == NULL)
    {
        hlog_error("Error Creating Curl_handleCommand");
        return 1;
    }

#if(STREAMMGR_FULL_PROTOCOL)
    /* Forward get_session_key commadn to port 9090 */
    pTmpStr = strstr(cmddata, "get_session_key");
    if(pTmpStr)
    {
        /* Right now we do not handle RTMP stream and record */
        if(strstr(pTmpStr, "rtmp") == NULL)
            port = 9090;
    }
#endif

    sprintf(cmd, "http://%s:%d/?%s", CAMERA_IP, port, cmddata);
    hlog_info("Send command to Camera's HTTP Server: '%s'",cmd);
    /* specify URL to get */
    curl_easy_setopt(curl_handle_command, CURLOPT_URL, cmd);

    /* send all data to this function  */
    curl_easy_setopt(curl_handle_command, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);

    /* we pass our 'chunk' struct to the callback function */
    chunk[DEFAULT_INDEX_SINGLE_CMD].buffer = sret;
    curl_easy_setopt(curl_handle_command, CURLOPT_WRITEDATA, (void * )&chunk[DEFAULT_INDEX_SINGLE_CMD]);
    curl_easy_setopt(curl_handle_command, CURLOPT_CONNECTTIMEOUT, 1);
    curl_easy_setopt(curl_handle_command, CURLOPT_TIMEOUT, 10);
    curl_easy_setopt(curl_handle_command, CURLOPT_NOSIGNAL, 1);
    curl_easy_setopt(curl_handle_command, CURLOPT_FAILONERROR, TRUE);

    /* some servers don't like requests that are made without a user-agent
       field, so we provide one */
    curl_easy_setopt(curl_handle_command, CURLOPT_USERAGENT, "libcurl-agent/1.0");
    /* get it! */
    res = curl_easy_perform(curl_handle_command);
    if(res != CURLE_OK)
    {
        hlog_error("curl_easy_perform return error %d %s", res, curl_easy_strerror(res));
        exit_status = -1;
    }

    return exit_status;
}

#define DEF_MAX_CMD_URL_LIST        (sizeof(CommandURLList)/sizeof(CommandURL))

typedef struct _CommandURL
{
    char name[64];
    Commands command;
} CommandURL;

CommandURL CommandURLList[] =
{
    {"get_session_key"                  ,Commands_GET_SESSION_KEY                       }, //1
    {"close_relay_rtmp"                 ,Commands_CLOSE_RELAY_RTMP                      }, //2
    {"set_city_timezone"                ,Commands_SET_CITY_TIMEZONE                     }, //3
    {"get_city_timezone"                ,Commands_GET_CITY_TIMEZONE                     }, //4
    {"set_time_zone"                    ,Commands_SET_TIME_ZONE                         }, //5
    {"get_time_zone"                    ,Commands_GET_TIME_ZONE                         }, //6
    {"set_motion_sensitivity"           ,Commands_SET_MOTION_SENSITIVITY                }, //7
    {"value_motion_sensitivity"         ,Commands_VALUE_MOTION_SENSITIVITY              }, //8
    {"set_motion_area"                  ,Commands_SET_MOTION_AREA                       }, //9
    {"get_motion_area"                  ,Commands_GET_MOTION_AREA                       }, //10
    {"set_subscription"                 ,Commands_SET_SUBSCRIPTION                      }, //11
    {"value_subscription"                ,Commands_VALUE_SUBSCRIPTION                    }, //12
    {"set_record_fps"                   ,Commands_SET_RECORD_FPS                        }, //13
    {"value_record_fps"                 ,Commands_VALUE_RECORD_FPS                      }, //14
    {"triggered_recording"              ,Commands_TRIGGERED_RECORDING                   }, //15
    {"value_triggered_recording"        ,Commands_VALUE_TRIGGERED_RECORDING             }, //16
    {"recording_active_duration"        ,Commands_RECORDING_ACTIVE_DURATION             }, //17
    {"value_recording_active_duration"  ,Commands_VALUE_RECORDING_ACTIVE_DURATION       }, //18
    {"recording_cooloff_duration"       ,Commands_RECORDING_COOLOFF_DURATION            }, //19
    {"value_recording_cooloff_duration" ,Commands_VALUE_RECORDING_COOLOFF_DURATION      }, //20
    {"time_travel"                      ,Commands_TIME_TRAVEL                           }, //21
    {"value_time_travel"                ,Commands_VALUE_TIME_TRAVEL                     }, //22
    {"value_contrast"                   ,Commands_VALUE_CONTRAST                        }, //23
    {"value_brightness"                 ,Commands_VALUE_BRIGHTNESS                      }, //24
    {"set_contrast"                     ,Commands_SET_CONTRAST                          }, //25
    {"set_brightness"                   ,Commands_SET_BRIGHTNESS                        }, //26
    {"setup_wireless_save"              ,Commands_SETUP_WIRELESS_SAVE                   }, //27
    {"get_wifi_strength"                ,Commands_GET_WIFI_STRENGTH                     }, //28
    {"move_forward"                     ,Commands_MOVE_FORWARD                          }, //29
    {"move_backward"                    ,Commands_MOVE_BACKWARD                         }, //30
    {"move_left"                        ,Commands_MOVE_LEFT                             }, //31
    {"move_right"                       ,Commands_MOVE_RIGHT                            }, //32
    {"move_forward_duration"            ,Commands_MOVE_FORWARD_DURATION                 }, //33
    {"move_backward_duration"           ,Commands_MOVE_BACKWARD_DURATION                }, //34
    {"move_left_duration"               ,Commands_MOVE_LEFT_DURATION                    }, //35
    {"move_right_duration"              ,Commands_MOVE_RIGHT_DURATION                   }, //36
    {"fb_stop"                          ,Commands_FB_STOP                               }, //37
    {"lr_stop"                          ,Commands_LR_STOP                               }, //38
    {"restart_system"                   ,Commands_RESTART_SYSTEM                        }, //39
    {"melody1"                          ,Commands_MELODY1                               }, //40
    {"melody2"                          ,Commands_MELODY2                               }, //41
    {"melody3"                          ,Commands_MELODY3                               }, //42
    {"melody4"                          ,Commands_MELODY4                               }, //43
    {"melody5"                          ,Commands_MELODY5                               }, //44
    {"melody_ud"                        ,Commands_MELODY_UD                             }, //45
    {"melodystop"                       ,Commands_MELODYSTOP                            }, //46
    {"value_melody"                     ,Commands_VALUE_MELODY                          }, //47
    {"value_temperature"                ,Commands_VALUE_TEMPERATURE                     }, //48
    {"reset_factory"                    ,Commands_RESET_FACTORY                         }, //49
    {"vox_get_threshold"                ,Commands_VOX_GET_THRESHOLD                     }, //50
    {"vox_set_threshold"                ,Commands_VOX_SET_THRESHOLD                     }, //51
    {"vox_enable"                       ,Commands_VOX_ENABLE                            }, //52
    {"vox_disable"                      ,Commands_VOX_DISABLE                           }, //53
    {"vox_get_status"                   ,Commands_VOX_GET_STATUS                        }, //54
    {"get_version"                      ,Commands_GET_VERSION                           }, //55
    {"set_flipup"                       ,Commands_SET_FLIPUP                            }, //56
    {"value_flipup"                     ,Commands_VALUE_FLIPUP                          }, //57
    {"set_spk_volume"                   ,Commands_SET_SPK_VOLUME                        }, //58
    {"get_spk_volume"                   ,Commands_GET_SPK_VOLUME                        }, //59
    {"set_master_key"                   ,Commands_SET_MASTER_KEY                        }, //60
    {"get_udid"                         ,Commands_GET_UDID                              }, //61
    {"get_mac_address"                  ,Commands_GET_MAC_ADDRESS                       }, //62
    {"enable_telnet"                    ,Commands_ENABLE_TELNET                         }, //63
    {"request_log"                      ,Commands_REQUEST_LOG                           }, //64
    {"mini_device_status"               ,Commands_MINI_DEVICE_STATUS                    }, //65
    {"set_temp_hi_threshold"            ,Commands_SET_TEMP_HI_THRESHOLD                 }, //66
    {"set_temp_lo_threshold"            ,Commands_SET_TEMP_LO_THRESHOLD                 }, //67
    {"get_temp_hi_threshold"            ,Commands_GET_TEMP_HI_THRESHOLD                 }, //68
    {"get_temp_lo_threshold"            ,Commands_GET_TEMP_LO_THRESHOLD                 }, //69
    {"set_flicker"                      ,Commands_SET_FLICKER                           }, //70
    {"get_flicker"                      ,Commands_GET_FLICKER                           }, //71
    {"device_setting"                   ,Commands_DEVICE_SETTING                        }, //72
    {"camera_parameter_setting"         ,Commands_CAMERA_PARAMETER_SETTING              }, //73
    {"set_temp_hi_enable"               ,Commands_SET_TEMP_HI_ENABLE                    }, //74
    {"get_temp_hi_stat"                 ,Commands_GET_TEMP_HI_STAT                      }, //75
    {"set_temp_lo_enable"               ,Commands_SET_TEMP_LO_ENABLE                    }, //76
    {"get_temp_lo_stat"                 ,Commands_GET_TEMP_LO_STAT                      }, //77
    {"set_smart_motion"                 ,Commands_SET_SMART_MOTION                      }, //78
    {"get_smart_motion"                 ,Commands_GET_SMART_MOTION                      }, //79
    {"get_image_snapshot"               ,Commands_GET_IMAGE_SNAPSHOT                    }, //80
    {"check_fw_upgrade"                 ,Commands_CHECK_FW_UPGRADE                      }, //81
    {"request_fw_upgrade"               ,Commands_REQUEST_FW_UPGRADE                    }, //82
    {"set_recording_parameter"          ,Commands_SET_RECORDING_PARAMETER               }, //83
    {"get_recording_parameter"          ,Commands_GET_RECORDING_PARAMETER               }, //84
    {"set_video_bitrate"                ,Commands_SET_VIDEO_BITRATE                     }, //85
    {"get_video_bitrate"                ,Commands_GET_VIDEO_BITRATE                     }, //86
    {"vd_debug_on"                      ,Commands_VD_DEBUG_ON                           }, //87
    {"vd_debug_off"                     ,Commands_VD_DEBUG_OFF                          }, //88
    {"stream_enc_on"                    ,Commands_STREAM_ENC_ON                         }, //89
    {"stream_enc_off"                   ,Commands_STREAM_ENC_OFF                        }, //90
    {"tune_vd"                          ,Commands_TUNE_VD                               }, //91
    {"set_log_profile"                  ,Commands_SET_LOG_PROFILE                       }, //92
    {"camera_setting2"                  ,Commands_CAMERA_SETTING2                       }, //93
    {"logdownload_cgi"                  ,Commands_LOGDOWNLOAD_CGI                       }, //94
    {"disable_reboot_offline_mode"      ,Commands_DISABLE_REBOOT_OFFLINE_MODE           }, //95
    {"request_to_update_mvr_schedule"   ,Commands_REQUEST_TO_UPDATE_MVR_SCHEDULE        }, //96
    {"get_recording_plan"               ,Commands_GET_RECORDING_PLAN                    }, //97
    {"set_recording_plan"               ,Commands_SET_RECORDING_PLAN                    }, //98
    {"set_recording_destination"        ,Commands_SET_RECORDING_DESTINATION             }, //99
    {"create_sdses"                     ,Commands_CREATE_SDSES                          }, //100
    {"close_sdcard_relay_rtmp"          ,Commands_CLOSE_SDCARD_RELAY_RTMP               }, //101
    {"sdcard_delete_md_clip"            ,Commands_SDCARD_DELETE_MD_CLIP                 }, //102
    {"sdcard_delete_md_clip_all"        ,Commands_SDCARD_DELETE_MD_CLIP_ALL             }, //103
    {"sd_localfileplay"                 ,Commands_SD_LOCALFILEPLAY                      }, //104
    {"set_sensor_register"              ,Commands_SET_SENSOR_REGISTER                   }, //105
    {"start_sd_recording"               ,Commands_START_SD_RECORDING                    }, //106
    {"stop_sd_recording"                ,Commands_STOP_SD_RECORDING                     }, //107
    {"get_sd_recording_list"            ,Commands_GET_SD_RECORING_LIST                  }, //108
    {"delete_sd_recording_file"         ,Commands_DELETE_SD_RECORDING_FILE              }, //109
    {"delete_sd_recording_all"          ,Commands_DELETE_SD_RECORDING_ALL               }, //110
    {"download_sd_recording"            ,Commands_DOWNLOAD_SD_RECORDING                 }, //111
    {"sd_safe_move"                     ,Commands_SD_SAFE_MOVE                          }, //112
    {"sd_card_format"                   ,Commands_SD_CARD_FORMAT                        }, //113
    {"sd_card_free"                     ,Commands_SD_CARD_FREE                          }, //114
    {"sd_card_cap"                      ,Commands_SD_CARD_CAP                           }, //115
    {"get_lan_mac_address"              ,Commands_GET_LAN_MAC_ADDRESS                   }, //116
    {"get_serial_number"                ,Commands_GET_SERIAL_NUMBER                     }, //117
    {"set_night_vision"                 ,Commands_SET_NIGHT_VISION                      }, //118
    {"get_night_vision"                 ,Commands_GET_NIGHT_VISION                      }, //119
    {"set_ir_pwn"                       ,Commands_SET_IR_PWN                            }, //120
    {"get_ir_pwn"                       ,Commands_GET_IR_PWN                            }, //121
    {"set_lan_static"                   ,Commands_SET_LAN_STATIC                        }, //122
    {"get_lan_static"                   ,Commands_GET_LAN_STATIC                        }, //123
    {"set_lan_dhcp"                     ,Commands_SET_LAN_DHCP                          }, //124
    {"set_facssid"                      ,Commands_SET_FACSSID                           }, //125
    {"get_facssid"                      ,Commands_GET_FACSSID                           }, //126
    {"set_wifi_static"                  ,Commands_SET_WIFI_STATIC                       }, //127
    {"get_wifi_static"                  ,Commands_GET_WIFI_STATIC                       }, //128
    {"set_wifi_dhcp"                    ,Commands_SET_WIFI_DHCP                         }, //129
    {"set_factory_default"              ,Commands_SET_FACTORY_DEFAULT                   }, //130
    {"set_adaptive_bitrate"             ,Commands_SET_ADAPTIVE_BITRATE                  }, //133
    {"get_adaptive_bitrate"             ,Commands_GET_ADAPTIVE_BITRATE                  }, //134
    {"set_resolution"                   ,Commands_SET_RESOLUTION                        }, //135
    {"get_resolution"                   ,Commands_GET_RESOLUTION                        }, //136
    {"set_server_auth"                  ,Commands_SET_SERVER_AUTH                       }, //137
    {"move_left_step"                   ,Commands_MOVE_LEFT_STEP                        }, //138
    {"move_right_step"                  ,Commands_MOVE_RIGHT_STEP                       }, //139
    {"move_backward_step"               ,Commands_MOVE_BACKWARD_STEP                    }, //140
    {"move_forward_step"                ,Commands_MOVE_FORWARD_STEP                     }, //141
    {"set_smr_recording"                ,Commands_SET_SMR_RECORDING                     }, //142
    {"get_smr_recording"                ,Commands_GET_SMR_RECORDING                     }, //143
    {"get_soc_version"                  ,Commands_GET_SOC_VERSION                       }, //144
    {"set_led_func"                     ,Commands_SET_LED_FUNC                          }, //145
    {"get_led_func"                     ,Commands_GET_LED_FUNC                          }, //146
    {"cds_value"                        ,Commands_CDS_VALUE                             }, //147
    {"beeper_en"                        ,Commands_BEEPER_EN                             }, //148
    {"beeper_dis"                       ,Commands_BEEPER_DIS                            }, //149
    {"set_sdcard_log"                   ,Commands_SET_SDCARD_LOG                        }, //150
    {"get_sdcard_log"                   ,Commands_GET_SDCARD_LOG                        }, //151
    {"set_blegateway"                   ,Commands_SET_BLEGATEWAY                        }, //152
    {"add_ble_tag"                      ,Commands_ADD_BLE_TAG                           }, //153
    {"remove_ble_tag"                   ,Commands_REMOVE_BLE_TAG                        }, //154
    {"set_video_saturation"             ,Commands_SET_VIDEO_SATURATION                  }, //155
    {"update_md_verylow_sen_param"      ,Commands_UPDATE_MD_VERYLOW_SEN_PARAM           }, //156
    {"update_md_low_sen_param"          ,Commands_UPDATE_MD_LOW_SEN_PARAM               }, //157
    {"update_md_medium_sen_param"       ,Commands_UPDATE_MD_MEDIUM_SEN_PARAM            }, //158
    {"update_md_high_sen_param"         ,Commands_UPDATE_MD_HIGH_SEN_PARAM              }, //159
    {"update_md_veryhight_sen_param"    ,Commands_UPDATE_MD_VERYHIGH_SEN_PARAM          }, //160
    {"mini_device_info"                 ,Commands_MINI_DEVICE_INFO                      }, //161
    {"read_sensor"                      ,Commands_READ_SENSOR                           }, //162
    {"write_sensor"                     ,Commands_WRITE_SENSOR                          }, //163
    {"overlay_date_set"                 ,Commands_OVERLAY_DATE_SET                      }, //164
    {"set_ir_mode"                      ,Commands_SET_IR_MODE                           }, //165
    {"get_ir_mode"                      ,Commands_GET_IR_MODE                           }, //166
    {"get_humidity"                     ,Commands_GET_HUMIDITY                          }, //167
    {"get_humidity_raw"                 ,Commands_GET_HUMIDITY_RAW                      }, //168
    {"get_battery_value"                ,Commands_GET_BATTERY_VALUE                     }, //169
    {"get_battery_charging"             ,Commands_GET_BATTERY_CHARGING                  }, //170
    {"get_battery_percent"              ,Commands_GET_BATTERY_PERCENT                   }, //171
    {"switch_video_driver_view_mode"    ,Commands_SWITCH_VIDEO_DRIVER_VIEW_MODE         }, //172
    {"get_video_driver_view_mode"       ,Commands_GET_VIDEO_DRIVER_VIEW_MODE            }, //173
    {"start_vda"                        ,Commands_START_VDA                             }, //174
    {"start_ait_md"                     ,Commands_START_AIT_MD                          }, //175
    {"get_md_type"                      ,Commands_GET_MD_TYPE                           }, //176
    {"set_vda_bsc_threshold_movement"   ,Commands_SET_VDA_BSC_THRESHOLD_MOVEMENT        }, //177
    {"set_bsc_bed_time"                 ,Commands_SET_BSC_BED_TIME                      }, //178
    {"check_vda_license"                ,Commands_CHECK_VDA_LICENSE                     }, //179
    {"soft_reset"                       ,Commands_SOFT_RESET                            }, //180
    {"get_tcl_setting"                  ,Commands_GET_TCL_SETTING                       }, //181
    {"ext_trigger_rec"                  ,Commands_EXT_TRIGGER_REC                       }, //182
    {"ext_trigger_dur"                  ,Commands_EXT_TRIGGER_DUR                       }, //183
    {"ext_trigger_snap"                 ,Commands_EXT_TRIGGER_SNAP                      }, //184
    {"ext_trigger_dogbarks"             ,Commands_EXT_TRIGGER_DOGBARKS                  }, //185
    {"ext_trigger_alarm"                ,Commands_EXT_TRIGGER_ALARM                     }, //186
    {"set_video_quality"                ,Commands_SET_VIDEO_QUALITY                     }, //187
    {"get_sd_percent_free"              ,Commands_GET_SD_PERCENT_FREE                   }, //188
    {"get_sd_recording_status"          ,Commands_GET_SD_RECORDING_STATUS               }, //189
    {"get_sd_mount_status"              ,Commands_GET_SD_MOUNT_STATUS                   }, //190
    {"start_talk_back"                  ,Commands_START_TALK_BACK                       }, //191
    {"audio_stop"                       ,Commands_AUDIO_STOP                            }, //192
    {"get_brightness"                   ,Commands_GET_BRIGHTNESS                        }, //193
    {"get_model_name"                   ,Commands_GET_MODEL_NAME                        }, //194
    {"get_model"                        ,Commands_GET_MODEL                             }, //195
    {"close_p2p_rtsp_stun"              ,Commands_CLOSE_P2P_RTSP_STUN                   }, //196
    {"ext_trigger"                      ,Commands_EXT_TRIGGER                           }, //197
    {"get_session_key_t"                ,Commands_GET_SESSION_KEY_T                     }, //198
    {"get_video_quality"                ,Commands_GET_VIDEO_QUALITY                     }, //199
    {"sd_card_format_status"            ,Commands_SD_CARD_FORMAT_STATUS                 }, //200
    {"request_user_fw_upgrade"          ,Commands_REQUEST_USER_FW_UPGRADE               }, //201
    {"get_fw_mini_ver"                  ,Commands_GET_FW_MINI_VER                       }, //202
    {"set_stream_adaptive_mode"         ,Commands_SET_STREAM_ADAPTIVE_MODE              }, //203
    {"get_stream_adaptive_mode"         ,Commands_GET_STREAM_ADAPTIVE_MODE              }, //204
    {"update_bandwidth"                 ,Commands_UPDATE_BANDWIDTH                      }, //205
    {"set_siren_status"                 ,Commands_SET_SIREN_STATUS                      }, //206
    {"get_siren_status"                 ,Commands_GET_SIREN_STATUS                      }, //207
    {"set_siren_melody"                 ,Commands_SET_SIREN_MELODY                      }, //208
    {"get_siren_melody"                 ,Commands_GET_SIREN_MELODY                      }, //209
    {"set_voiceprompt_language"         ,Commands_SET_VOICEPROMPT_LANGUAGE              }, //210
    {"get_voiceprompt_language"         ,Commands_GET_VOICEPROMPT_LANGUAGE              }, //211
    {"p2p_ses_mode_set"                 ,Commands_P2P_SES_MODE_SET                      }, //212
    {"set_multi_mode"                   ,Commands_SET_MULTI_MODE                        }, //213
    {"get_multi_mode"                   ,Commands_GET_MULTI_MODE                        }, //214
    {"get_registration_status"          ,Commands_GET_REGISTRATION_STATUS               }, //215
    {"get_mic_status"                   ,Commands_GET_MIC_STATUS                        }, //216
    {"overlay_date_get"                 ,Commands_OVERLAY_DATE_GET                      }, //217
    {"get_video_gop"                    ,Commands_GET_VIDEO_GOP                         }, //218
    {"set_video_gop"                    ,Commands_SET_VIDEO_GOP                         }, //219
    {"set_humi_hi_threshold"            ,Commands_SET_HUMI_HI_THRESHOLD                 }, //220
    {"set_humi_lo_threshold"            ,Commands_SET_HUMI_LO_THRESHOLD                 }, //221
    {"get_humi_hi_threshold"            ,Commands_GET_HUMI_HI_THRESHOLD                 }, //222
    {"get_humi_lo_threshold"            ,Commands_GET_HUMI_LO_THRESHOLD                 }, //223
    {"get_humi_hi_stat"                 ,Commands_GET_HUMI_HI_STAT                      }, //224
    {"get_humi_lo_stat"                 ,Commands_GET_HUMI_LO_STAT                      }, //225
    {"set_humi_lo_enable"               ,Commands_SET_HUMI_LO_ENABLE                    }, //226
    {"set_humi_hi_enable"               ,Commands_SET_HUMI_HI_ENABLE                    }, //227
    {"get_hum_setting"                  ,Commands_GET_HUM_SETTING                       }, //228
    {"stand_by_mode"                    ,Commands_STAND_BY_MODE                         }, //229
    {"free_storage_space"               ,Commands_FREE_STORAGE_SPACE                    }, //230
    {"get_device_mode"                  ,Commands_GET_DEVICE_MODE                       }, //231
    {"remaining_time"                   ,Commands_REMAINING_TIME                        }, //232
    {"get_mic_gain"                     ,Commands_GET_MIC_GAIN                          }, //233
    {"set_mic_gain"                     ,Commands_SET_MIC_GAIN                          }, //234
    {"set_video_sharpness"              ,Commands_SET_VIDEO_SHARPNESS                   }, //235
    {"value_pir_sensitivity"            ,Commands_VALUE_PIR_SENSITIVITY                 }, //236
    {"set_pir_sensitivity"              ,Commands_SET_PIR_SENSITIVITY                   }, //237
    {"get_mcu_version"                  ,Commands_GET_MCU_VERSION                       }, //238
    {"get_bpi_version"                  ,Commands_GET_BPI_VERSION                       }, //239
    {"get_enable_bframe"                ,Commands_GET_ENABLE_BFRAME                     }, //240
    {"set_enable_bframe"                ,Commands_SET_ENABLE_BFRAME                     }, //241
    {"sdcard_delete_md_clip_last"       ,Commands_SDCARD_DELETE_MD_CLIP_LAST            }, //242
    {"get_ldc_status"                   ,Commands_GET_LDC_STATUS                        }, //243
    {"set_ldc_status"                   ,Commands_SET_LDC_STATUS                        }, //244
    {"disable_telnet"                   ,Commands_DISABLE_TELNET                        }, //245
    {"set_power_save_mode"              ,Commands_SET_POWER_SAVE_MODE                   }, //246
    {"get_power_save_mode"              ,Commands_GET_POWER_SAVE_MODE                   }, //247
    {"set_night_light_status"           ,Commands_SET_NIGHT_LIGHT_STATUS                }, //248
    {"get_night_light_status"           ,Commands_GET_NIGHT_LIGHT_STATUS                }, //249
    {"start_upload_measure"             ,Commands_START_UPLOAD_MEASURE                  }, //250
    {"check_upload_measure"             ,Commands_CHECK_UPLOAD_MEASURE                  }, //251
    {"set_motion_source"                ,Commands_SET_MOTION_SOURCE                     }, //252
    {"cmd_get_multi_zone"               ,Commands_CMD_GET_MULTI_ZONE                    }, //253
    {"set_multi_zone"                   ,Commands_SET_MULTI_ZONE                        }, //254
    {"set_soc_projector"                ,Commands_SET_SOC_PROJECTOR                     }, //255
    {"set_soc_night_light"              ,Commands_SET_SOC_NIGHT_LIGHT                   }, //256
    {"get_soc_projector"                ,Commands_GET_SOC_PROJECTOR                     }, //257
    {"get_soc_night_light"              ,Commands_GET_SOC_NIGHT_LIGHT                   }, //258
    {"set_soc_night_light_on"           ,Commands_SET_SOC_NIGHT_LIGHT_ON                }, //259
    {"set_soc_projector_on"             ,Commands_SET_SOC_PROJECTOR_ON                  }, //260
    {"get_soc_devices_status"           ,Commands_GET_SOC_DEVICES_STATUS                }, //261
    {"set_v_cruise"                     ,Commands_SET_V_CRUISE                          }, //262
    {"get_v_cruise"                     ,Commands_GET_V_CRUISE                          }, //263
    {"set_h_cruise"                     ,Commands_SET_H_CRUISE                          }, //264
    {"get_h_cruise"                     ,Commands_GET_H_CRUISE                          }, //265
    {"set_blink_led"                    ,Commands_SET_BLINK_LED                         }, //266
    {"get_blink_led"                    ,Commands_GET_BLINK_LED                         }, //267
    {"get_time_bsc"                     ,Commands_GET_TIME_BSC                          }, //268
    {"get_image_snapshot_register"      ,Commands_GET_IMAGE_SNAPSHOT_REGISTER           }, //269
    {"camera_parameter_setting_3"       ,Commands_CAMERA_PARAMETER_SETTING_3            }, //270
    {"get_motion_source"                ,Commands_GET_MOTION_SOURCE                     }, //271
    {"get_media_list_version"           ,Commands_GET_MEDIA_LIST_VERSION                }, //272
    {"device_dowload_audio"             ,Commands_DEVICE_DOWLOAD_AUDIO                  }, //273
    {"get_playing_media"                ,Commands_GET_PLAYING_MEDIA                     }, //274
    {"del_media_file"                   ,Commands_DEL_MEDIA_FILE                        }, //275
    {"get_media_list"                   ,Commands_GET_MEDIA_LIST                        }, //276
    {"media_previous"                   ,Commands_MEDIA_PREVIOUS                        }, //277
    {"media_next"                       ,Commands_MEDIA_NEXT                            }, //278
    {"media_stop"                       ,Commands_MEDIA_STOP                            }, //279
    {"media_play"                       ,Commands_MEDIA_PLAY                            }, //280
    {"get_sensor_register"              ,Commands_GET_SENSOR_REGISTER                   }, //281
    {"get_ir_pwm"                       ,Commands_GET_IR_PWM                            }, //282
    {"get_park_timer"                   ,Commands_GET_PARK_TIMER                        }, //283
    {"get_cam_park"                     ,Commands_GET_CAM_PARK                          }, //284
    {"get_bta_running_status"           ,Commands_GET_BTA_RUNNING_STATUS                }, //285
    {"get_bsc_remain_duration"          ,Commands_GET_BSC_REMAIN_DURATION               }, //286
    {"get_framerate"                    ,Commands_GET_FRAMERATE                         }, //287
    {"set_framerate"                    ,Commands_SET_FRAMERATE                         }, //288
    {"sd_card_delete_md_clip_last"      ,Commands_SD_CARD_DELETE_MD_CLIP_LAST           }, //289
    {"reboot"                           ,Commands_REBOOT                                }, //290
    {"neorestart"                       ,Commands_NEORESTART                            }, //291
    {"get_video_qos"                    ,Commands_GET_VIDEO_QOS                         }, //292
    {"general_setting"                  ,Commands_GENERAL_SETTING                       }, //293
    {"request_fw_upgrade2"              ,Commands_REQUEST_FW_UPGRADE2                   }, //294
    {"shresto"                          ,Commands_SHRESTO                               }, //295
    {"dis_shresto"                      ,Commands_DIS_SHRESTO                           }  //296
};


int Parse_RestAPI(Packet *packet,char *url)
{
    int i;
    //char action[30] = "action=command";
    char command[50] ={0};
    char value[120] ={0};
    char setup[120] ={0};
    char mode[30] ={0};
    char nat_ip[30] ={0};
    char nat_port[30] ={0};
    char stream_name[50] ={0};
    char rtmp_ip[30] ={0};
    char rtmp_port[30] ={0};
    char modestr[40] = {0};
    char year[30] = {0};
    char time_stamp[40] = {0};
    char duty_cylce[30] ={0};
    char grid[30] ={0};
    char zone[30] ={0};
    char tag_id[40]={0};
    char tag_mac[30] ={0};
    char tag_type[30] ={0};
    char ip[30] = {0};
    char stream_id[30] = {0};
    char clip[60] ={0};
    char sum[50] ={0};
    char stream_name_[50] = {0};
    char manual_snap[20] = {0};
    char manual_rec[20] = {0};
    char dur[20] = {0};
    char alarm[25] ={0};
    char dogbarks[25] = {0};
    char snap[25] = {0};
    char rec[25] = {0};
    char size[16] = {0};
    //  if(packet->header.request_type == RequestType_REQUEST)
    //  {
    //      action[30] = "action=command";
    //        //snprintf(action,sizeof("action=")+sizeof("command"),"action=%s","command");
    //  }
    packet->header.has_code = 1;
    packet->header.code = StatusCodes_BAD_REQUEST;
    for(i = 0; i < DEF_MAX_CMD_URL_LIST; i++)
    {
        if(packet->header.command == CommandURLList[i].command)
        {
            packet->header.code = StatusCodes_OK;
            //printf("%s\n",CommandURLList[i].name);
            snprintf(command,sizeof("&command=")+sizeof(CommandURLList[i].name),"&command=%s",CommandURLList[i].name);
            break;
        }
    }

    if(packet->header.has_attribute == 1)
    {
        if(packet->header.attribute.has_value == 1)
        {
            snprintf(value,sizeof("&value=")+sizeof(packet->header.attribute.value),"&value=%s",packet->header.attribute.value);
        }
        else
        {
            memset(value, '\0', sizeof(value));
        }
        //      hlog_info("value: %s \n",value);
        if(packet->header.attribute.has_setup == 1)
        {
            snprintf(setup,sizeof("&setup=")+sizeof(packet->header.attribute.setup),"&setup=%s",packet->header.attribute.setup);
        }
        else
        {
            memset(setup, '\0', sizeof(setup));
        }
        //      hlog_info("setup: %s \n",setup);
        if(packet->header.attribute.has_mode == 1)
        {
            snprintf(mode,sizeof("&mode=")+sizeof(packet->header.attribute.mode),"&mode=%s",packet->header.attribute.mode);
        }
        else
        {
            memset(mode, '\0', sizeof(mode));
        }
        //      hlog_info("mode: %s \n",mode);
        if(packet->header.attribute.has_stream_attribute == 1)
        {
            if(packet->header.attribute.stream_attribute.has_nat_ip == 1)
            {
                if(strstr(packet->header.attribute.stream_attribute.mode,"remote") != NULL || strstr(packet->header.attribute.stream_attribute.mode,"local") != NULL )
                    snprintf(nat_ip ,sizeof("&ip=")+sizeof(packet->header.attribute.stream_attribute.nat_ip),"&ip=%s",packet->header.attribute.stream_attribute.nat_ip);
                else
                    snprintf(nat_ip ,sizeof("&natip=")+sizeof(packet->header.attribute.stream_attribute.nat_ip),"&natip=%s",packet->header.attribute.stream_attribute.nat_ip);
            }
            else
            {
                memset(nat_ip, '\0', sizeof(nat_ip));
            }
            //          hlog_info("packet->header.attribute.stream_attribute.nat_ip: %s \n",packet->header.attribute.stream_attribute.nat_ip);
            if(packet->header.attribute.stream_attribute.has_nat_port == 1)
            {
                if(strstr(packet->header.attribute.stream_attribute.mode,"remote") != NULL || strstr(packet->header.attribute.stream_attribute.mode,"local") != NULL )
                    snprintf(nat_port ,sizeof("&port1=")+sizeof(packet->header.attribute.stream_attribute.nat_port),"&port1=%s",packet->header.attribute.stream_attribute.nat_port);
                else
                    snprintf(nat_port ,sizeof("&natport=")+sizeof(packet->header.attribute.stream_attribute.nat_port),"&natport=%s",packet->header.attribute.stream_attribute.nat_port);
            }
            else
            {
                memset(nat_port, '\0', sizeof(nat_port));
            }
            //          hlog_info("packet->header.attribute.stream_attribute.nat_port: %s \n",packet->header.attribute.stream_attribute.nat_port);
            if(packet->header.attribute.stream_attribute.has_stream_id == 1)
            {
                snprintf(stream_name ,sizeof("&streamname=")+sizeof(packet->header.attribute.stream_attribute.stream_id),"&streamname=%s",packet->header.attribute.stream_attribute.stream_id);
            }
            else
            {
                memset(stream_name, '\0', sizeof(stream_name));
            }
            //          hlog_info("packet->header.attribute.stream_attribute.stream_id: %s \n",packet->header.attribute.stream_attribute.stream_id);
            //          hlog_info("stream_name: %s \n",stream_name);
            if(packet->header.attribute.stream_attribute.has_rtmp_ip == 1)
            {
                snprintf(rtmp_ip ,sizeof("&rtmp_ip=")+sizeof(packet->header.attribute.stream_attribute.rtmp_ip),"&rtmp_ip=%s",packet->header.attribute.stream_attribute.rtmp_ip);
            }
            else
            {
                memset(rtmp_ip, '\0', sizeof(rtmp_ip));
            }
            //          hlog_info("rtmp_ip: %s \n",rtmp_ip);
            if(packet->header.attribute.stream_attribute.has_rtmp_port == 1)
            {
                snprintf(rtmp_port,sizeof("&rtmp_port=")+sizeof(packet->header.attribute.stream_attribute.rtmp_port),"&rtmp_port=%s", packet->header.attribute.stream_attribute.rtmp_port);
            }
            else
            {
                memset(rtmp_port, '\0', sizeof(rtmp_port));
            }
            //          hlog_info("rtmp_port: %s \n",rtmp_port);
            if(packet->header.attribute.stream_attribute.has_mode == 1)
            {
                snprintf(modestr,sizeof("&mode=")+sizeof(packet->header.attribute.stream_attribute.mode),"&mode=%s", packet->header.attribute.stream_attribute.mode);
            }
            else
            {
                memset(modestr, '\0', sizeof(modestr));
            }
            //          hlog_info("packet->header.attribute.stream_attribute.mode: %s \n",packet->header.attribute.stream_attribute.mode);
            if(packet->header.attribute.stream_attribute.has_year == 1)
            {
                snprintf(year,sizeof("&yr=")+sizeof(packet->header.attribute.stream_attribute.year),"&yr=%s", packet->header.attribute.stream_attribute.year);
            }
            else
            {
                memset(year, '\0', sizeof(year));
            }
            if(packet->header.attribute.stream_attribute.has_time_stamp == 1)
            {
                snprintf(time_stamp,sizeof("&ts=")+sizeof(packet->header.attribute.stream_attribute.time_stamp),"&ts=%s", packet->header.attribute.stream_attribute.time_stamp);
            }
            else
            {
                memset(time_stamp, '\0', sizeof(time_stamp));
            }
            //          hlog_info("packet->header.attribute.stream_attribute.time_stamp: %s \n",packet->header.attribute.stream_attribute.time_stamp);
        }
        else
        {
            memset(nat_ip, '\0', sizeof(nat_ip));
            memset(nat_port, '\0', sizeof(nat_port));
            memset(stream_name, '\0', sizeof(stream_name));
            memset(rtmp_ip, '\0', sizeof(rtmp_ip));
            memset(rtmp_port, '\0', sizeof(rtmp_port));
            memset(year, '\0', sizeof(year));
            memset(time_stamp, '\0', sizeof(time_stamp));
        }
        //      hlog_info("nat_port: %s \n",nat_port);
        if(packet->header.attribute.has_duty_cylce == 1)
        {
            snprintf(duty_cylce,sizeof("<>")+sizeof(packet->header.attribute.duty_cylce),"<%s>",packet->header.attribute.duty_cylce);
        }
        else
        {
            memset(duty_cylce, '\0', sizeof(duty_cylce));
        }
        //      hlog_info("duty_cylce: %s \n",duty_cylce);
        if(packet->header.attribute.has_mvr_toggle_attribute == 1)
        {
            if(packet->header.attribute.mvr_toggle_attribute.has_grid == 1)
            {
                snprintf(grid,sizeof("&grid=")+sizeof(packet->header.attribute.mvr_toggle_attribute.grid),"&grid=%s",packet->header.attribute.mvr_toggle_attribute.grid);
            }
            else
            {
                memset(grid, '\0', sizeof(grid));
            }
            //          hlog_info("grid: %s \n",grid);
            if(packet->header.attribute.mvr_toggle_attribute.has_zone == 1)
            {
                snprintf(zone,sizeof("&zone=")+sizeof(packet->header.attribute.mvr_toggle_attribute.zone),"&zone=%s",packet->header.attribute.mvr_toggle_attribute.zone);
            }
            else
            {
                memset(zone, '\0', sizeof(zone));
            }
            //          hlog_info("zone: %s \n",zone);
        }
        else
        {
            memset(grid, '\0', sizeof(grid));
            memset(zone, '\0', sizeof(zone));
        }
        if(packet->header.attribute.has_tag_attribute == 1)
        {
            if(packet->header.attribute.tag_attribute.has_tag_id == 1)
            {
                snprintf(tag_id,sizeof("&tag_id=")+sizeof(packet->header.attribute.tag_attribute.tag_id),"&tag_id=%s",packet->header.attribute.tag_attribute.tag_id);
            }
            else
            {
                memset(tag_id, '\0', sizeof(tag_id));
            }
            //          hlog_info("tag_id: %s \n",tag_id);
            if(packet->header.attribute.tag_attribute.has_tag_mac == 1)
            {
                snprintf(tag_mac,sizeof("&tag_mac=")+sizeof(packet->header.attribute.tag_attribute.tag_mac),"&tag_mac=%s",packet->header.attribute.tag_attribute.tag_mac);
            }
            else
            {
                memset(tag_mac, '\0', sizeof(tag_mac));
            }
            //          hlog_info("tag_mac: %s \n",tag_mac);
            if(packet->header.attribute.tag_attribute.has_tag_type == 1)
            {
                snprintf(tag_type,sizeof("&tag_type=")+sizeof(packet->header.attribute.tag_attribute.tag_type),"&tag_type=%s",packet->header.attribute.tag_attribute.tag_type);
            }
            else
            {
                memset(tag_type, '\0', sizeof(tag_type));
            }
            //          hlog_info("tag_type: %s \n",tag_type);
        }
        else
        {
            memset(tag_id, '\0', sizeof(tag_id));
            memset(tag_mac, '\0', sizeof(tag_mac));
            memset(tag_type, '\0', sizeof(tag_type));
        }
        if(packet->header.attribute.has_sd_card_attribute == 1)
        {
            if(packet->header.attribute.sd_card_attribute.has_ip == 1)
            {
                snprintf(ip,sizeof("&ip=")+sizeof(packet->header.attribute.sd_card_attribute.ip),"&ip=%s",packet->header.attribute.sd_card_attribute.ip);
            }
            else
            {
                memset(ip, '\0', sizeof(ip));
            }
            //          hlog_info("ip: %s \n",ip);
            if(packet->header.attribute.sd_card_attribute.has_stream_id == 1)
            {
                snprintf(stream_id,sizeof("&id=")+sizeof(packet->header.attribute.sd_card_attribute.stream_id),"&id=%s",packet->header.attribute.sd_card_attribute.stream_id);
            }
            else
            {
                memset(stream_id, '\0', sizeof(stream_id));
            }
            //          hlog_info("stream_id: %s \n",stream_id);
            if(packet->header.attribute.sd_card_attribute.has_clip == 1)
            {
                snprintf(clip,sizeof("&clip=")+sizeof(packet->header.attribute.sd_card_attribute.clip),"&clip=%s",packet->header.attribute.sd_card_attribute.clip);
            }
            else
            {
                memset(clip, '\0', sizeof(clip));
            }
            //          hlog_info("clip: %s \n",clip);
            if(packet->header.attribute.sd_card_attribute.has_sum == 1)
            {
                snprintf(sum,sizeof("&sum=")+sizeof(packet->header.attribute.sd_card_attribute.sum),"&sum=%s",packet->header.attribute.sd_card_attribute.sum);
            }
            else
            {
                memset(sum, '\0', sizeof(sum));
            }
            //          hlog_info("sum: %s \n",sum);
        }
        if(packet->header.attribute.has_stream_name == 1)
        {
            snprintf(stream_name_,sizeof("&streamname=")+sizeof(packet->header.attribute.stream_name),"&streamname=%s",packet->header.attribute.stream_name);
        }
        else
        {
            memset(stream_name_, '\0', sizeof(stream_name_));
        }
        if(packet->header.attribute.has_manual_snap == 1)
        {
            snprintf(manual_snap,sizeof("&manual_snap=")+sizeof(packet->header.attribute.manual_snap),"&manual_snap=%s",packet->header.attribute.manual_snap);
        }
        else
        {
            memset(manual_snap, '\0', sizeof(manual_snap));
        }
        if(packet->header.attribute.has_manual_rec == 1)
        {
            snprintf(manual_rec,sizeof("&manual_rec=")+sizeof(packet->header.attribute.manual_rec),"&manual_rec=%s",packet->header.attribute.manual_rec);
        }
        else
        {
            memset(manual_rec, '\0', sizeof(manual_rec));
        }
        if(packet->header.attribute.has_dur == 1)
        {
            snprintf(dur,sizeof("&dur=")+sizeof(packet->header.attribute.dur),"&dur=%s",packet->header.attribute.dur);
        }
        else
        {
            memset(dur, '\0', sizeof(dur));
        }
        if(packet->header.attribute.has_alarm == 1)
        {
            snprintf(alarm,sizeof("&alarm=")+sizeof(packet->header.attribute.alarm),"&alarm=%s",packet->header.attribute.alarm);
        }
        else
        {
            memset(alarm, '\0', sizeof(alarm));
        }
        if(packet->header.attribute.has_dogbarks == 1)
        {
            snprintf(dogbarks,sizeof("&dogbarks=")+sizeof(packet->header.attribute.dogbarks),"&dogbarks=%s",packet->header.attribute.dogbarks);
        }
        else
        {
            memset(dogbarks, '\0', sizeof(dogbarks));
        }
        if(packet->header.attribute.has_snap == 1)
        {
            snprintf(snap,sizeof("&snap=")+sizeof(packet->header.attribute.snap),"&snap=%s",packet->header.attribute.snap);
        }
        else
        {
            memset(snap, '\0', sizeof(snap));
        }
        if(packet->header.attribute.has_rec == 1)
        {
            snprintf(rec,sizeof("&rec=")+sizeof(packet->header.attribute.rec),"&rec=%s",packet->header.attribute.rec);
        }
        else
        {
            memset(rec, '\0', sizeof(rec));
        }
        if(packet->header.attribute.has_sum == 1)
        {
            snprintf(sum,sizeof("&sum=")+sizeof(packet->header.attribute.sum),"&sum=%s",packet->header.attribute.sum);
        }
        else
        {
            memset(sum, '\0', sizeof(sum));
        }
        if(packet->header.attribute.has_size == 1)
        {
            snprintf(size,sizeof("&size=")+sizeof(packet->header.attribute.size),"&size=%s",packet->header.attribute.size);
        }
        else
        {
            memset(size, '\0', sizeof(size));
        }
    }
    else
    {
        memset(duty_cylce, '\0', sizeof(duty_cylce));
        memset(value, '\0', sizeof(value));
        memset(setup, '\0', sizeof(setup));
        memset(mode, '\0', sizeof(mode));
        memset(nat_ip, '\0', sizeof(nat_ip));
        memset(nat_port, '\0', sizeof(nat_port));
        memset(rtmp_ip, '\0', sizeof(rtmp_ip));
        memset(rtmp_port, '\0', sizeof(rtmp_port));
        memset(modestr, '\0', sizeof(modestr));
        memset(year, '\0', sizeof(year));
        memset(time_stamp, '\0', sizeof(time_stamp));
        memset(stream_name, '\0', sizeof(stream_name));
        memset(grid, '\0', sizeof(grid));
        memset(zone, '\0', sizeof(zone));
        memset(tag_id, '\0', sizeof(tag_id));
        memset(tag_mac, '\0', sizeof(tag_mac));
        memset(tag_type, '\0', sizeof(tag_type));
        memset(ip, '\0', sizeof(ip));
        memset(stream_id, '\0', sizeof(stream_id));
        memset(clip, '\0', sizeof(clip));
        memset(sum, '\0', sizeof(sum));
        memset(stream_name_, '\0', sizeof(stream_name_));
        memset(manual_snap, '\0', sizeof(manual_snap));
        memset(manual_rec, '\0', sizeof(manual_rec));
        memset(dur, '\0', sizeof(dur));
        memset(alarm, '\0', sizeof(alarm));
        memset(dogbarks, '\0', sizeof(dogbarks));
        memset(snap, '\0', sizeof(snap));
        memset(rec, '\0', sizeof(rec));
        memset(size, '\0', sizeof(size));
    }
    //  hlog_info("Debug");
    //  hlog_info("command: %s \n",command);
    //  hlog_info("duty_cylce: %s \n",duty_cylce);
    //  hlog_info("value: %s \n",value);
    //  hlog_info("setup: %s \n",setup);
    //  hlog_info("mode: %s \n",mode);
    //  hlog_info("nat_ip: %s \n",nat_ip);
    //  hlog_info("nat_port: %s \n",nat_port);
    //  hlog_info("rtmp_ip: %s \n",rtmp_ip);
    //  hlog_info("modestr: %s \n",modestr);
    //  hlog_info("stream_name: %s \n",stream_name);
    //  hlog_info("grid: %s \n",grid);
    //  hlog_info("zone: %s \n",zone);
    //  hlog_info("tag_id: %s \n",tag_id);
    //  hlog_info("tag_mac: %s \n",tag_mac);
    //  hlog_info("tag_type: %s \n",tag_type);
    //  hlog_info("ip: %s \n",ip);
    //  hlog_info("stream_id: %s \n",stream_id);
    //  hlog_info("clip: %s \n",clip);
    //  hlog_info("sum: %s \n",sum);
    if(strstr(packet->header.attribute.stream_attribute.mode,"remote") != NULL || strstr(packet->header.attribute.stream_attribute.mode,"local") != NULL )
        sprintf(url,"action=command%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s"/*,action*/,command,duty_cylce,value,setup,mode,modestr,nat_port,nat_ip,rtmp_ip,stream_name,year,time_stamp,grid,zone,tag_id,tag_mac,tag_type,ip,stream_id,clip,sum,stream_name_,manual_snap,manual_rec,dur,alarm,dogbarks,snap,rec,size);
    else
        sprintf(url,"action=command%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s"/*,action*/,command,duty_cylce,value,setup,mode,modestr,nat_ip,nat_port,rtmp_ip,stream_name,year,time_stamp,grid,zone,tag_id,tag_mac,tag_type,ip,stream_id,clip,sum,stream_name_,manual_snap,manual_rec,dur,alarm,dogbarks,snap,rec,size);
    return 0;
}
static int countRetry = 0;
int Parse_Protobuf(char*result, Packet *packet, int packet_idx)
{
    //  pthread_mutex_lock(&mqtts_data_access);
    if(packet_array[packet_idx].handleStatus == eHandleCommandTimeOut)
    {
        packet->header.request_type = RequestType_SUCCESS;
        packet->header.has_code = 1;
        packet->header.code = StatusCodes_TIMEOUT;
        packet_array[packet_idx].handleStatus = eHandleCommandNone;
    }
    else if(packet_array[packet_idx].handleStatus == eHandleCommandStopped)
    {
        packet->header.request_type = RequestType_SUCCESS;
        packet->header.has_code = 1;
        packet->header.code = StatusCodes_COMMAND_INPROGRESS;
        countRetry = countRetry + 1;
        packet->header.has_retry_after = 1;
        sprintf(packet->header.retry_after,"2");
        packet_array[packet_idx].handleStatus = eHandleCommandRunning;
        cmd_in_progress = 1;
        hlog_info("[IN-PROGRESS]Camera processing command, Index: %d", packet_idx);
    }
    else if(packet_array[packet_idx].handleStatus == eHandleCommandFinished)
    {
        packet->header.has_retry_after = 0;
        packet->header.has_code = 1;
        //packet->has_response = 1;
        if(result != NULL)
        {
            packet->has_response = 1;
            packet->response.has_body = 1;
            sprintf(packet->response.body,"%s",result);
            if(strncmp(result,"Unsupported command",sizeof("Unsupported command")) == 0)
            {
                packet->header.request_type = RequestType_SUCCESS;
                packet->header.code = StatusCodes_BAD_REQUEST;

            }
            else
            {

                packet->header.request_type = RequestType_SUCCESS;
                packet->header.code = StatusCodes_OK;
            }
        }
        else
        {
            packet->header.request_type = RequestType_ERROR;
            packet->header.code = StatusCodes_TIMEOUT;
            packet->response.has_body = 0;
            packet->has_response = 0;
        }
        packet_array[packet_idx].handleStatus = eHandleCommandNone;
    }
    //  pthread_mutex_unlock(&mqtts_data_access);
    //strcpy(packet->header.firmware_version,"03.00.00");
    return 0;
}

