#include <stdio.h>
#include <string.h>
#include "hubble_log.h"
#include "jsmn.h"


/*
 * A small example of jsmn parsing when JSON structure is known and number of
 * tokens is predictable.
 */

static int jsoneq(const char *json, jsmntok_t *tok, const char *s)
{
    if(tok->type == JSMN_STRING && (int) strlen(s) == tok->end - tok->start && strncmp(json + tok->start, s, tok->end - tok->start) == 0)
    {
        return 0;
    }
    return -1;
}

int get_json_string(char *JSON_STRING, char *key, char *result)
{
    int i;
    int r;
    jsmn_parser p;
    jsmntok_t t[128]; /* We expect no more than 128 tokens */

    jsmn_init(&p);
    r = jsmn_parse(&p, JSON_STRING, strlen(JSON_STRING), t, sizeof(t) / sizeof(t[0]));
    if(r < 0)
    {
        hlog_error("Failed to parse JSON: %d\n", r);
        return -1;
    }

    /* Assume the top-level element is an object */
    if(r < 1 || t[0].type != JSMN_OBJECT)
    {
        hlog_error("Object expected\n");
        return -1;
    }

    /* Loop over all keys of the root object */
    for(i = 1; i < r; i++)
    {
        if(jsoneq(JSON_STRING, &t[i], key) == 0)
        {
            memcpy(result, JSON_STRING + t[i + 1].start, t[i + 1].end - t[i + 1].start);
            return 0;
        }
    }
    return -1;
}

int parser_json_response(char *raw_data, int data_len, jsmntok_t *token, int max_token)
{
    int number_token = 0;
    jsmn_parser parser;

    memset(token, 0x00, sizeof(jsmntok_t));
    jsmn_init(&parser);
    number_token = jsmn_parse(&parser, raw_data, strlen(raw_data), token, max_token);

    if(number_token < 0)
    {
        hlog_error("Failed to parse JSON: %d\n", number_token);
        return number_token;
    }

    if(number_token < 1 || token[0].type != JSMN_OBJECT)
    {
        hlog_error("Object expected\n");
        return -1;
    }

    return number_token;
}

/**
 * @name          get_token_value
 * @brief         get the value of given token
 * @param[in]     raw_data: data buffer
 * @param[in]     data_len: length of data buffer
 * @param[in]     token: parsed token
 * @param[in]     num_token: number of token
 * @param[in]     retrive_token: wanted token
 * @param[out]    value: value of given token
 * @param[in]     buffer_size: size of output buffer
 * @return < 0    ERROR
 *         = 0    Cannot found token
 *         > 0    len of value
 */
int get_token_value(char *raw_data, int data_len, jsmntok_t *token, int num_token, char *retrive_token, char *value, int buffer_size)
{
    int i;
    int str_len;
    for(i = 1; i < num_token; i++)
    {
        if(jsoneq(raw_data, &token[i], retrive_token) == 0)
        {
            str_len = token[i + 1].end - token[i + 1].start; // end str - start str
            if((str_len + 1) > buffer_size)
            { // 1 for the terminate string char
                return -1;
            }

            hlog_debug("raw_data: %p, start: %d, len: %d\n", raw_data, token[i + 1].start, str_len);
            memcpy(value, raw_data + token[i + 1].start, str_len);
            value[str_len] = '\0';
            return str_len;
        }
    }
    return -2;
}

