/*
 *  
 *  Copyright (c) by Shuu Yamaguchi <shuu@wondernetworkresources.com>
 *
 *  $Id: parse_device.c,v 1.1 2004/08/23 14:53:40 tgu Exp $
 *
 *  Can be freely distributed and used under the terms of the GNU GPL.
 *
 */
#include	<stdio.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<sys/poll.h>
#include	<fcntl.h>
#include	<ctype.h>
#include	<limits.h>
#include	<string.h>
#include	<stdlib.h>
#include	<unistd.h>
#include	<sys/wait.h>
#include	<a.out.h>

#include	"usbmgr.h"

extern int poll_fd;	/* main.c */

#ifndef	DEVFILE
#define	DEVFILE		USB_PROCDIR "/devices"
#endif
#ifndef	USBMGR_BLOCK_SIZE
#define	USBMGR_BLOCK_SIZE	PAGE_SIZE		/* unit of allocation */
#endif
#ifndef USBMGR_BLOCK_COUNT
#define	USBMGR_BLOCK_COUNT		3		/* count of units */
#endif
#ifndef	FLUSH_TIME
#define	FLUSH_TIME	(500 * 1000)
#endif

#define	DEVICE_LINE_SIZE	128	/* enough size for a line on devices */
/* 
 * refreshing memory cycle
 * It should be greater then 2.
 */
#define REFRESH_COUNT	8

#define	ST_NONE		0x00
#define	ST_NEW			0x01
#define	ST_PRODUCT		0x02
#define	ST_INTERFACE	0x03
#define	ST_ENDPOINT	0x04
#define	ST_TOPOLOGY	0x05
#define	ST_DEVICE	       0x06
#define	ST_SYSTEM	       0x07
#define	ST_CONFIG	       0x08
#define	ST_END			-1

#define CHECK_NULL( pointer )  \
  if (NULL == pointer ) break ;


int check_class_path(__u8 class);

/* 
 * using I(interface descriptor) instead of D(Device descriptor)
 */

void strcut( char *string )
{
   int i = 0;

   if( NULL == string )
   {
      return;
   }

   for( i = strlen( string ) -1; i >=0; i-- )
   {
      if( isspace(string[i]) || string[i] == '\0' )
      {
	  string[i] = '\0';
      }
      else
      {
         break;
      }
   }
   return;
}

static void
parse_line(char *linep,int status,struct usbmgr_device *dev, int configuration_num, int interface_num, int endp_num )
{
	char *tmpstr = NULL;
	int class = -1,i;
	int l_interface_num = 1;
	static int addEPs = 0;
	static useConfig = 0;
	
	if (debug)
	  syslog(LOG_INFO,"**parsing string: %s, dev=0x%x**", linep, dev );
	
	switch(status) {
	case ST_PRODUCT:
		/* P:  Vendor=xxxx ProdID=xxxx Rev=xx.xx */
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
		DEVICE_VENDOR(dev) = ( unsigned short )strtoul(linep,NULL,16);
              USB_DEV_VENDOR(dev) = strtoul(linep,NULL,16);
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
		DEVICE_PRODUCT(dev) = strtoul(linep,NULL,16);
              USB_DEV_PRODID(dev) = strtoul(linep,NULL,16);
		if (debug)
			syslog(LOG_INFO,"parsing vender=0x%x product=0x%x",DEVICE_VENDOR(dev),DEVICE_PRODUCT(dev));
		break;
	case ST_INTERFACE:	
		/* I:  If#=dd Alt=dd #EPs=dd Cls=xx(sssss) Sub=xx Prot=xx Driver=ssss
	 	 */
		if (!useConfig) 
		{
			return;
		}
	  /*If there is already a known interface defined, we skip this one*/
	  /*This fixes some devices that report themselves as a HID and a MSC device.*/
	  /*It is not a general fix for all devices that have more than one class, but is solves the problem with devices that have two classes where one of them is unknown */
	  /*The unknown class will be ignored in that case.*/
	  /*To fix the problem in a general way, code needs to be added to add multiple entries for devices with multiple classes. That requires a lot more work and we should take a look at alternatives in that case, e.g. hotplug */ 
	  /* Koen Muylkens */
	  //if  ((interface_num > 1) && (check_class_path(DEVICE_CLASS(dev))>=0)) 
	  //  break;
				
		// Printer has the highest priority. If a printer has different interfaces (also a MS), the printer is always the mean device
		// MS has the second priority
		addEPs = 0; 
		for (i=0; i<(interface_num-1) ;i++)
		{
			if (USB_INTF_CLASS(dev, 0, i) == 7)
			{
				//printer device already added
				return;
			}
			// search the class
			class = GetClass(linep);
			if ((class != 8) && (class != 7))
			{
				//accept only printers and MS, refuse the other ones
				return;
			}
		}
		USB_CONF_NUMINTF(dev, configuration_num-1 ) = 1;
		addEPs = 1; //only add the endpoints, if the interface is allowed

		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
		USB_INTF_INTFNO(dev,configuration_num-1, l_interface_num-1 ) = strtoul(linep,NULL,10);
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
		USB_INTF_ALT(dev,configuration_num-1, l_interface_num-1 ) = strtoul(linep,NULL,10);
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
		USB_INTF_NUMENDP(dev,configuration_num-1, l_interface_num-1 ) = strtoul(linep,NULL,10);
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
       	
	  DEVICE_CLASS(dev) =  strtoul(linep,NULL,16);       
		USB_INTF_CLASS(dev,configuration_num-1,l_interface_num-1 ) = strtoul(linep,NULL,16);
		linep++;
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		  linep++;
		DEVICE_SUBCLASS(dev) = strtoul(linep,NULL,16);
		USB_INTF_SUBCLASS(dev,configuration_num-1,l_interface_num-1 ) = strtoul(linep,NULL,16);
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		  linep++;
		DEVICE_PROTOCOL(dev) = strtoul(linep,NULL,16);
		USB_INTF_PROTOCOL(dev,configuration_num-1,l_interface_num-1 ) = strtoul(linep,NULL,16);
		tmpstr = strstr(linep,"Driver=");
		CHECK_NULL( tmpstr );
		linep = tmpstr;
		linep = linep+7;
		memset( USB_INTF_DRIVER(dev,configuration_num-1,l_interface_num-1 ), 0, 30 );
		strncpy( USB_INTF_DRIVER(dev,configuration_num-1,l_interface_num-1 ), linep, 29 );
		strcut( USB_INTF_DRIVER(dev,configuration_num-1,l_interface_num-1 ) );
    
		if (debug)
			syslog(LOG_INFO,"parsing class=0x%x subclass=0x%x protocol=0x%x",DEVICE_CLASS(dev),DEVICE_SUBCLASS(dev),DEVICE_PROTOCOL(dev));
		break;
	case ST_TOPOLOGY:
		/* T:  Bus=dd Lev=dd Prnt=dd Port=dd Cnt=dd Dev#=ddd Spd=ddd MxCh=dd
		 */
		 linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
		DEVICE_BUS(dev) = strtoul(linep,NULL,10);
		if (debug)
			syslog(LOG_INFO,"parseing bus=%03d\n", DEVICE_BUS(dev));
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
              USB_DEV_PORT(dev) = strtoul(linep,NULL,16);
			  
		linep = strchr(linep,'#');
		CHECK_NULL( linep )
		linep += 2;
		DEVICE_DEVNO(dev) = strtoul(linep,NULL,10);
		USB_DEV_DEVNO(dev) = strtoul(linep,NULL,10);
		if (debug)
			syslog(LOG_INFO,"parsing dev#=%03d\n", DEVICE_DEVNO(dev));
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
              USB_DEV_SPEED(dev) = strtoul(linep,NULL,10);
		break;

	case ST_DEVICE:
		/* D:  Ver= 2.00 Cls=ff(vend.) Sub=00 Prot=00 MxPS=64 #Cfgs=  1
		 */
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
	       USB_DEV_VERION(dev) = strtod(linep,NULL);
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
              USB_DEV_CLASS(dev) = strtoul(linep,NULL,16);
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
              USB_DEV_SUBCLASS(dev) = strtoul(linep,NULL,16);
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
              USB_DEV_PROTOCOL(dev) = strtoul(linep,NULL,16);
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
              USB_DEV_MAXPKSIZE(dev) = strtoul(linep,NULL,10);
		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
              USB_DEV_NUMCONF(dev) = strtoul(linep,NULL,10);

              break;

	case ST_CONFIG:
		/*C:* #Ifs= 1 Cfg#= 1 Atr=80 MxPwr=300mA*/
		// Printer has the highest priority. If a printer has different interfaces (also a MS), the printer is always the mean device
		// MS has the second priority
		useConfig = 0;
		for (i=0; i<(configuration_num-1) ;i++)
		{
			if ((DEVICE_CLASS(dev) == 7) || (DEVICE_CLASS(dev) == 8))
			{
				//printer device or MS already added
				return;
			}
		}
		useConfig = 1;

		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
    USB_CONF_NUMINTF(dev, configuration_num-1 ) = strtoul(linep,NULL,10);

		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
		USB_CONF_CONFNO(dev, configuration_num-1  ) = strtoul(linep,NULL,10);

		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
		USB_CONF_ATTR(dev, configuration_num-1  ) = strtoul(linep,NULL,16);

		linep = strchr(linep,'=');
		CHECK_NULL( linep )
		linep++;
		USB_CONF_MAXPOW(dev, configuration_num-1  ) = strtoul(linep,NULL,10);

    break;

	case ST_ENDPOINT:
    /*E:  Ad=03(O) Atr=02(Bulk) MxPS= 512 Ivl=0ms*/
		if (addEPs && useConfig)
		{
			linep = strchr(linep,'=');
			CHECK_NULL( linep )
			linep++;
			USB_ENDP_ADDR(dev, configuration_num-1, l_interface_num-1,endp_num-1  ) = strtoul(linep,NULL,16);
					
			linep = strchr(linep,'=');
			CHECK_NULL( linep )
			linep++;
			USB_ENDP_ATTR(dev, configuration_num-1, l_interface_num-1,endp_num-1  ) = strtoul(linep,NULL,16);
	
			linep = strchr(linep,'=');
			CHECK_NULL( linep )
			linep++;
			USB_ENDP_MAXPKSIZE(dev, configuration_num-1, l_interface_num-1,endp_num-1  ) = strtoul(linep,NULL,10);
	
			linep = strchr(linep,'=');
			CHECK_NULL( linep )
			linep++;
      USB_ENDP_INTERVAL(dev, configuration_num-1, l_interface_num-1,endp_num-1  ) = strtoul(linep,NULL,10);
		}  
		break;
			  
	case ST_SYSTEM:
		tmpstr = strstr(linep,"Manufacturer=");
		if( NULL != tmpstr )
		{
       	   linep = tmpstr;
       	   linep = linep + 13;
       	   memset( USB_DEV_MANUFACTURER( dev ), 0, 64 );
       	   strncpy( USB_DEV_MANUFACTURER( dev ), linep, 63 );
       	   strcut( USB_DEV_MANUFACTURER( dev ) );
       	   break;
		}
		else if( NULL != ( tmpstr = strstr(linep,"Product=")) )
		{
 		  linep = tmpstr;
 		  linep = linep + 8;
 		  memset( USB_DEV_PRODUCT( dev ), 0, 64 );
 		  strncpy( USB_DEV_PRODUCT( dev ), linep, 63 );
 		  strcut( USB_DEV_PRODUCT( dev ) );
 		  break;
		}
		else if( NULL != ( tmpstr = strstr(linep,"SerialNumber=")) )
		{
      	         linep = tmpstr;
      	         linep = linep + 13;
      	         memset( USB_DEV_SERIALNO( dev ), 0, 64 );
      	         strncpy( USB_DEV_SERIALNO( dev ), linep, 63 );
      	         strcut( USB_DEV_SERIALNO( dev ) );
		}
		else
		{
		  syslog( LOG_ERR, "parsing one wrong string" );
		}
		break;
         default:
              break;
		
	}
}

static int
parse_devices(char **curp,char *endp,struct usbmgr_device *dev) {
	char line[DEVICE_LINE_SIZE];
	int status = ST_NEW;
	int i = 0;	/* verbose */
	char *ptr;
	int first = 1;
	int flag = 0;
	int configuration_num = 0;
	int interface_num = 0;
	int endpoint_num = 0;

	if (*curp == endp)
		return ST_END;
	ptr = *curp;

	for(;ptr < endp;ptr++) {
		if ((*ptr >= 'A') && (*ptr <= 'Z')) {
			syslog(LOG_INFO,"buffer line: remove %c", *ptr);
			break;
		}
	}
	
	if (*ptr != 'T') {
		syslog(LOG_INFO,"invalid buffer line");
		return ST_END;
	}
	for(;ptr < endp;ptr++) {
		if (status == ST_NEW) {
			i=0;
			switch(*ptr) {
			case 'P':
				flag = 1;
				status = ST_PRODUCT;
				break;
			case 'I':
				flag = 1;
				interface_num++;
				endpoint_num = 0;
				status = ST_INTERFACE;
				break;
			case 'T':
				flag = 1;
				status = ST_TOPOLOGY;
				if (!first)
					goto loop_end;
				first = 0;
				break;
			case 'D':
				flag = 1;
				status = ST_DEVICE;
				break;
			case 'C':
				flag = 1;
				interface_num = 0;
				configuration_num++;
				status = ST_CONFIG;
				break;
			case 'S':
				flag = 1;
				status = ST_SYSTEM;
				break;
			case 'E':
				flag = 1;
				endpoint_num++;
				status = ST_ENDPOINT;
				break;
				
			default:
				status = ST_NONE;
				break;
			}
		} 
		//else if (status & (ST_PRODUCT|ST_INTERFACE|ST_TOPOLOGY  ) ) {
		else if ( 1 == flag ) {
			line[i++] = *ptr;
		}
		
		/* parse the line */
		if ((*ptr == '\n') || ( ptr == endp-1   )) {
			line[i] = '\0';
			if( 1 == flag )
			{
			  parse_line( line,status,dev,configuration_num,interface_num, endpoint_num );
			}
			flag = 0;
			status = ST_NEW;
		}
	}
loop_end:
	if (debug)
		syslog(LOG_DEBUG,"left %d",endp - ptr);
	*curp = ptr;

	return 0;
}

void
observ_devices(char *bufp,ssize_t size)
{
	struct node *np;
	struct usbmgr_device dev_tmp;
	char *curp;
	char path[PATH_MAX+1];
		
	curp = bufp;
	memset(&dev_tmp, 0, sizeof(struct usbmgr_device));

	 while(parse_devices(&curp,bufp+size-1,&dev_tmp) != ST_END) 
	 {
			/* check if dev is loaded on memory */
			if (find_device(&dev_tmp) == NULL) 
			{/* find New device */
				if (debug)
					syslog(LOG_DEBUG,mesg[MSG_NEWDEVICE]);
				
				//create the unique id number and string
				DEVICE_UID(&dev_tmp) = USB_DEV_DEVNO(&dev_tmp) | (DEVICE_BUS(&dev_tmp) << 8);
									
				np = create_device(&dev_tmp);
				validate_desc(np);
				print_device(np);
				NODE_STATUS(np) |= USBMGR_ST_ACTIVE;
				
				//printf("observ_devices: class %d UID 0x%x bus %d\n", DEVICE_CLASS(&dev_tmp), DEVICE_UID(NODE2DEV(np)), dev_tmp.bus);
				if (USB_DEV_DEVNO(NODE2DEV(np)) != 1) // do not rapport the root hubs to mbus
				{		
					usb_device_sndevent_add_del("USB.Device", (unsigned long) DEVICE_UID(NODE2DEV(np)), 0 );
				}
				
				/* search module file */
				if (check_vendor_file(np,path,BUILD_MODULE) == -1) 
				{
					if (search_class_file(np,path,BUILD_MODULE) == INVALID) 
					{
						 beep(INVALID);
						 syslog(LOG_INFO,mesg[MSG_NOT_MATCH]);
						 memset(&dev_tmp, 0, sizeof(struct usbmgr_device));
						 continue;
					 }
				}
				beep(GOOD);
				syslog(LOG_INFO,mesg[MSG_MATCH]);
				load_from_file(np,path,MODULE_LOAD);
				
				/* search script file */
				if (check_vendor_file(np,path,BUILD_SCRIPT) == -1) 
				{
					if (search_class_file(np,path,BUILD_SCRIPT) == INVALID) 
					{
						memset(&dev_tmp, 0, sizeof(struct usbmgr_device));
						continue;
					}
				}
				
			 	NODE_TYPE(np) |= USBMGR_TYPE_SCRIPT;
			 	if (load_from_file( np,path,SCRIPT_START ) == 0)
			 		syslog(LOG_INFO,mesg[MSG_START],path);
     } 
		 else
     {  
		 		/* devices are not changed */
        if (debug)
          syslog(LOG_DEBUG,mesg[MSG_NOCHANGE]);
     }
		 memset(&dev_tmp, 0, sizeof(struct usbmgr_device));
  }
  delete_device(USBMGR_DO_AUTO);	/* not force */
}

void
observe(void)
{
	struct pollfd pd;
	char *devfile = DEVFILE;
	char *buf,*bufp;
	int i,len;
	int bnum = USBMGR_BLOCK_COUNT;	/* count of block */
	int sum,refresh;

	buf = malloc(bnum * USBMGR_BLOCK_SIZE);
	if (buf == NULL)
		return;
	memset(buf,0,bnum * USBMGR_BLOCK_SIZE);
		
	if ((pd.fd = open(devfile,O_RDONLY)) == -1) {
		load_from_file(NULL,USBMGR_HOST_FILE,MODULE_LOAD);
		do_mount(USB_MOUNT);
		if ((pd.fd = open(devfile,O_RDONLY)) == -1)
              exit(1);
	}
	pd.events = POLLIN;
	poll_fd = pd.fd;	/* global */
	for(refresh = 0;refresh < REFRESH_COUNT;refresh++) {
		poll(&pd,1,sleep_time); 	/* Hey poll! :-) */
#ifdef	WAIT_FLUSH_PROC
		usleep(FLUSH_TIME);	/* wait for flushing proc data */
#endif
		lseek(poll_fd,0,SEEK_SET);
		bufp = buf;
		for(i = 0,sum = 0;(len = read(poll_fd,bufp,USBMGR_BLOCK_SIZE)) != 0;i++) {
			if (i + 1 >= bnum) {	/* filled */
				bnum *= 2;
				buf = realloc(buf,bnum * USBMGR_BLOCK_SIZE);
				if (buf == NULL) {
					syslog(LOG_ERR,mesg[MSG_ALLOC_ERR]);
					goto return_close;	/* error */
				}
				memset(buf,0,bnum * USBMGR_BLOCK_SIZE);
			}
			sum += len;
			if (len < USBMGR_BLOCK_SIZE)	/* maybe end of file */
				break;
			bufp = buf + sum;
		}
		observ_devices(buf,sum);
		if (debug)
			syslog(LOG_DEBUG,"------- size %d refresh %d/%d",sum,refresh,REFRESH_COUNT);
	} 

return_close:
	close(poll_fd);
	free(buf);

	return;
}

/*
 * mount -t usbdevfs /proc/bus/usb /proc/bus/usb
 *                    and 
 * umount /proc/bus/usb
 *
 * Don't use mount(2) and umount(2),because /etc/mtab is unnecessary.
 */
void
do_mount(int action)
{
	char *mount_arg[] = {"mount","-t","usbdevfs",USB_PROCDIR,USB_PROCDIR,NULL};
	char *unmount_arg[] = {"umount",USB_PROCDIR,NULL};
	char **mnt_arg;
	static int mnt_flag = 0;	/* verbose */

	DPRINTF(LOG_INFO,"do_mount action %d",action);
	/* usbmgr didn't mount /proc/bus/usb */
	if (action == USB_UNMOUNT && mnt_flag == 0) {
		DPRINTF(LOG_INFO,"do_mount need not to unmount");
		return;
	}

	switch(fork()){
	case 0:		/* child */
		if (action == USB_MOUNT)
			mnt_arg = mount_arg;
		else
			mnt_arg = unmount_arg;
		syslog(LOG_INFO,mesg[MSG_2STRING],mnt_arg[0],USB_PROCDIR);
		execvp(mnt_arg[0],mnt_arg);
		syslog(LOG_ERR,mesg[MSG_EXEC_ERR],mnt_arg[0]);
		exit(1);
		break;
	case -1:	/* error */
		DPRINTF(LOG_INFO,"do_mount fork error");
		break;
	default:	/* parent */
		DPRINTF(LOG_INFO,"do_mount mnt_flag %d",mnt_flag);
		if (action == USB_MOUNT)
			mnt_flag = 1;
		else
			mnt_flag = 0;
		DPRINTF(LOG_INFO,"do_mount mnt_flag -> %d",mnt_flag);
		wait(NULL);
		break;
	}
}

/*
 * search class file
 */
int
search_class_file(struct node *np,char *path,int flag)
{
	char *file;

	if (!(NODE_TYPE(np) & USBMGR_TYPE_CLASS))
		return INVALID;

	if (flag & BUILD_MODULE)
		file = USBMGR_MODULE_FILE;
	else if (flag & BUILD_SCRIPT)
		file = USBMGR_SCRIPT_FILE;
	else
		return INVALID;

	/* class/subclass/protocol/XXX */
	sprintf(path,"%s/%s/%02x/%02x/%02x/%s",
		conf_dir,class_dir,
		NODE_CLASS(np),NODE_SUBCLASS(np),NODE_PROTOCOL(np),file);
	DPRINTF(LOG_DEBUG,"try %s",path);
	if (access(path,R_OK) == 0)
		return GOOD;

	/* class/subclass/XXX */
	sprintf(path,"%s/%s/%02x/%02x/%s",
		conf_dir,class_dir,
		NODE_CLASS(np),NODE_SUBCLASS(np),file);
	DPRINTF(LOG_DEBUG,"try %s",path);
	if (access(path,R_OK) == 0)
		return GOOD;

	/* class/XXX */
	sprintf(path,"%s/%s/%02x/%s",
		conf_dir,class_dir,NODE_CLASS(np), file);
	DPRINTF(LOG_DEBUG,"try %s",path);
	if (access(path,R_OK) == 0)
		return GOOD;

	/* XXX */
	sprintf(path,"%s/%s/%s",
		conf_dir,class_dir, file);
	DPRINTF(LOG_DEBUG,"try %s",path);
	if (access(path,R_OK) == 0)
		return GOOD;
	return -1;
}

/* get the class from the I line*/
int GetClass(char *str)
{
	char *ptr = str;
	int i;
	int charFound = 0;
	
	while (charFound < 4)
	{
		ptr = strchr(ptr,'=');
		if (ptr == NULL) return -1;
		ptr++;
		charFound++;
	}
	return strtoul(ptr,NULL,16);
}
/* simple function to check whether or not a certain class has a script defined */
int check_class_path(__u8 class)
{	
  char path[PATH_MAX+1];
  sprintf(path,"%s/%s/%02x",
	  conf_dir,class_dir,class);
  if (access(path,R_OK) == 0)
      return 0;
  return -1;
}



