/*
 * Broadcom mapping driver for thomson speedtouch boards
 * based on drivers of phillipe deswert / ronald vanschoren
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#include <asm/io.h>


static map_word physmap_read16(struct map_info *map, unsigned long ofs)
{
        map_word val;

        val.x[0] = __raw_readw(map->phys + ofs);

        return val;
}

static void physmap_write16(struct map_info *map, map_word data, unsigned long ofs)
{
	__raw_writew(data.x[0], (map->phys + ofs));
}


static void physmap_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
        memcpy_fromio(to, map->phys + from, len);
}


/* 
 * MAP DRIVER STUFF
 */


struct map_info speedtouch_map={
	.name = "flash",
	.size = CONFIG_SPEEDTOUCH_PHYS_SIZE,
	.phys = CONFIG_SPEEDTOUCH_FLASH_START,
	.bankwidth = 2,
	.read = physmap_read16,
	.write = physmap_write16,
	.copy_from = physmap_copy_from,
};

#ifdef CONFIG_MTD_SPEEDTOUCH_DYNAMIC
static const char *part_probes[] __initdata = {"BTHub", NULL};
#else
/*
 * MTD 'PARTITIONING' STUFF 
*/
static struct mtd_partition speedtouch_partitions[3] = {
	{	
		.name = CONFIG_SPEEDTOUCH_PART1_NAME,
		.size = CONFIG_SPEEDTOUCH_PART1_SIZE,
		.offset = CONFIG_SPEEDTOUCH_PART1_OFFSET
	},
	{
		.name = CONFIG_SPEEDTOUCH_PART2_NAME,
		.size = CONFIG_SPEEDTOUCH_PART2_SIZE,
		.offset = CONFIG_SPEEDTOUCH_PART2_OFFSET
	},
	{
		.name = CONFIG_SPEEDTOUCH_PART3_NAME,
		.size =  CONFIG_SPEEDTOUCH_PART3_SIZE,
		.offset = CONFIG_SPEEDTOUCH_PART3_OFFSET
	}
};
#endif /* CONFIG_MTD_SPEEDTOUCH_DYNAMIC */

/* 
 * This is the master MTD device for which all the others are just
 * auto-relocating aliases.
 */
static struct mtd_info *mymtd;

int bcm_flash_read(loff_t from, size_t len, size_t *retlen, unsigned char *buf)
{
	if (!mymtd) {
		printk(KERN_ERR "bcm_flash_read: MTD layer is not yet up!\n");
		return -1;
	}

	return MTD_READ(mymtd, from, len, retlen, buf);
}
EXPORT_SYMBOL(bcm_flash_read);

int __init init_bcm(void)
{
	int ret = -ENXIO;

	printk(KERN_NOTICE "Thomson Speedtouch flash mapping\n");
	 mymtd = do_map_probe("cfi_probe", &speedtouch_map);
         printk(KERN_NOTICE "flash mapping initialized\n");
         if (mymtd) {
	                 mymtd->owner = THIS_MODULE;
#ifdef CONFIG_MTD_SPEEDTOUCH_DYNAMIC
			{
				struct mtd_partition *mtd_parts = NULL;
				int mtd_parts_nb;

				mtd_parts_nb = parse_mtd_partitions(mymtd, part_probes, &mtd_parts, 0);
				ret = add_mtd_partitions(mymtd, mtd_parts, mtd_parts_nb);
				if (mtd_parts)
					kfree(mtd_parts);
			}
#else
    	                 ret = add_mtd_partitions(mymtd, speedtouch_partitions, 3);
#endif
         }
    
         return ret;
}

static void __exit cleanup_bcm(void)
{
	if (mymtd) {
		del_mtd_partitions(mymtd);
		map_destroy(mymtd);
	}
}

module_init(init_bcm);
module_exit(cleanup_bcm);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Joris Lijssens <Joris.Lijssens@thomson.net");
MODULE_DESCRIPTION("MTD map driver for speedtouch boards");
