/*
 * Common Flash Interface support:
 *   SST Standard Vendor Command Set (ID 0x0701)
 *
 *  maintained by Joris.Lijssens@thomson.net
 *
 * This code is GPL
 *
 * $Id: cfi_cmdset_0701.c,v 0.001 2006/07/05 14:02:32 dwmw2 Exp $
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/init.h>
#include <asm/io.h>
#include <asm/byteorder.h>

#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/mtd/compatmac.h>
#include <linux/mtd/map.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/cfi.h>

#define CFIDEV_BUSWIDTH 2
#define MAX_WORD_RETRIES 3

static int cfi_sststd_read (struct mtd_info *, loff_t, size_t, size_t *, u_char *);
static int cfi_sststd_write(struct mtd_info *, loff_t, size_t, size_t *, const u_char *);
static int cfi_sststd_erase(struct mtd_info *, struct erase_info *);
static void cfi_sststd_sync(struct mtd_info *);
static int cfi_sststd_suspend (struct mtd_info *);
static void cfi_sststd_resume (struct mtd_info *);
static void cfi_sststd_destroy(struct mtd_info *);
struct mtd_info *cfi_cmdset_0701(struct map_info *, int);
static struct mtd_info *cfi_sststd_setup (struct map_info *);


static struct mtd_chip_driver cfi_sststd_chipdrv = {
	.probe		= NULL, /* Not usable directly */
	.destroy	= cfi_sststd_destroy,
	.name		= "cfi_cmdset_0701",
	.module		= THIS_MODULE
};

struct mtd_info *cfi_cmdset_0701(struct map_info *map, int primary)
{
        struct cfi_private *cfi = map->fldrv_priv;
        int ofs_factor = cfi->interleave * cfi->device_type;
        int i;
        __u8 major, minor;
        __u32 base = cfi->chips[0].start;

         if (cfi->cfi_mode==CFI_MODE_CFI){
                 __u16 adr = primary?cfi->cfiq->P_ADR:cfi->cfiq->A_ADR;


		 /*put the chip in read mode*/
		 cfi_send_gen_cmd(0xAA, 0x5555, base, map, cfi, cfi->device_type, NULL);
		 cfi_send_gen_cmd(0x55, 0x2AAA, base, map, cfi, cfi->device_type, NULL);
		 cfi_send_gen_cmd(0xF0, 0x5555, base, map, cfi, cfi->device_type, NULL);

		 /* send query command sequence to chip*/
                 cfi_send_gen_cmd(0xAA, 0x5555, base, map, cfi, cfi->device_type, NULL);
                 cfi_send_gen_cmd(0x55, 0x2AAA, base, map, cfi, cfi->device_type, NULL);
                 cfi_send_gen_cmd(0x98, 0x5555, base, map, cfi, cfi->device_type, NULL);

                 major = cfi_read_query(map, base + (adr+3)*ofs_factor);
                 minor = cfi_read_query(map, base + (adr+4)*ofs_factor);
 
                 printk(" SST Query Table v%c.%c at 0x%4.4X\n",
                 major, minor, adr);
		 
		 /*put the chip in read mode*/
		 cfi_send_gen_cmd(0xAA, 0x5555, base, map, cfi, cfi->device_type, NULL);
		 cfi_send_gen_cmd(0x55, 0x2AAA, base, map, cfi, cfi->device_type, NULL);
                 cfi_send_gen_cmd(0xF0, 0x5555, base, map, cfi, cfi->device_type, NULL);
		 
 		 /* send query command sequence to chip*/
                 cfi_send_gen_cmd(0xAA, 0x5555, base, map, cfi, cfi->device_type, NULL);
                 cfi_send_gen_cmd(0x55, 0x2AAA, base, map, cfi, cfi->device_type, NULL);
                 cfi_send_gen_cmd(0x90, 0x5555, base, map, cfi, cfi->device_type, NULL);
                 cfi->mfr = cfi_read_query(map, base);
                 cfi->id = cfi_read_query(map, base + ofs_factor);

		 DEBUG( MTD_DEBUG_LEVEL3," SST chip with mfr_id : %x device_id : %x \r\n",cfi->mfr,cfi->id);			 
 
		 /*put the chip in read mode*/
		 cfi_send_gen_cmd(0xAA, 0x5555, base, map, cfi, cfi->device_type, NULL);
		 cfi_send_gen_cmd(0x55, 0x2AAA, base, map, cfi, cfi->device_type, NULL);
		 cfi_send_gen_cmd(0xF0, 0x5555, base, map, cfi, cfi->device_type, NULL);
		 
                 switch (cfi->device_type) {
                  case CFI_DEVICETYPE_X16:
                          cfi->addr_unlock1 = 0x555;
                          cfi->addr_unlock2 = 0x2AA;
                          break;
                  default:
                          printk(KERN_NOTICE "Eep. Unknown cfi_cmdset_0701 device type %d\n", cfi->device_type);
                          return NULL;
                 }
         } /* CFI mode */
   
         for (i=0; i< cfi->numchips; i++) {
                   cfi->chips[i].word_write_time = 1<<cfi->cfiq->WordWriteTimeoutTyp;
                   cfi->chips[i].buffer_write_time = 1<<cfi->cfiq->BufWriteTimeoutTyp;
                   cfi->chips[i].erase_time = 1<<cfi->cfiq->BlockEraseTimeoutTyp;
          }
    
           map->fldrv = &cfi_sststd_chipdrv;
    
           return cfi_sststd_setup(map);
}


static struct mtd_info *cfi_sststd_setup(struct map_info *map)
{
	struct cfi_private *cfi = map->fldrv_priv;
	struct mtd_info *mtd;
	unsigned long devsize = (1<<cfi->cfiq->DevSize) * cfi->interleave;
	unsigned long offset = 0;
	int i,j;


	mtd = kmalloc(sizeof(*mtd), GFP_KERNEL);
	printk(KERN_NOTICE "number of %s chips: %d\n", 
	       (cfi->cfi_mode == CFI_MODE_CFI)?"CFI":"JEDEC",cfi->numchips);

	if (!mtd) {
		printk(KERN_EMERG "Failed to allocate memory for MTD device\n");
		return 0;
	}


	memset(mtd, 0, sizeof(*mtd));
	mtd->priv = map;
	mtd->type = MTD_NORFLASH;
	/* Also select the correct geometry setup too */ 
	mtd->size = devsize * cfi->numchips;

	/*actualy there are 2 erase zones but they cover the same address space
	 *so if you try to initialize both erase zones like on the amd chips
	 *you will calculate double the size of the physical flash size
	 *which we don't want of course*/
       	
	//mtd->numeraseregions = cfi->cfiq->NumEraseRegions * cfi->numchips;
	mtd->numeraseregions = 1;
	mtd->eraseregions = kmalloc(sizeof(struct mtd_erase_region_info)
				    * mtd->numeraseregions, GFP_KERNEL);
	if (!mtd->eraseregions) { 
		printk(KERN_WARNING "Failed to allocate memory for MTD erase region info\n");
		return 0;
	}

		
	/* the SST gives back 2 regions of erasezones, but actualy they cover the same area
   	 * this is because of the 2 different blocksizes that are supported by the sst chip
	 * you can have 4Kb blocks or you can have 64kb blocks. Here we choose for the second
	 * option */
	
	for (i=0; i<mtd->numeraseregions; i++) {
		unsigned long ernum, ersize;
		ersize = ((cfi->cfiq->EraseRegionInfo[i+1] >> 8) & ~0xff) * cfi->interleave;
		ernum = (cfi->cfiq->EraseRegionInfo[i+1] & 0xffff) + 1;
		
		if (mtd->erasesize < ersize) {
			mtd->erasesize = ersize;
		}
		for (j=0; j<cfi->numchips; j++) {
			mtd->eraseregions[(j*cfi->cfiq->NumEraseRegions)+i].offset = (j*devsize)+offset;
			mtd->eraseregions[(j*cfi->cfiq->NumEraseRegions)+i].erasesize = ersize;
			mtd->eraseregions[(j*cfi->cfiq->NumEraseRegions)+i].numblocks = ernum;
		}
		offset += (ersize * ernum);
	}

	for(i=0; i<mtd->numeraseregions; i++)
	  	printk("SST %d: offset=0x%x, size=0x%x, blocks=0x%x\r\n",i,mtd->eraseregions[i].offset, mtd->eraseregions[i].erasesize,mtd->eraseregions[i].numblocks); 

	switch (CFIDEV_BUSWIDTH)
            {
             case 1:
             case 2:
             case 4:
                     mtd->erase = cfi_sststd_erase;
                     mtd->read = cfi_sststd_read;
                     mtd->write = cfi_sststd_write;
                     break;
    
             default:
                     printk("Unsupported buswidth\n");
                     kfree(mtd);
                     kfree(cfi->cmdset_priv);
                     return NULL;
                     break;
             }
             mtd->sync = cfi_sststd_sync;
             mtd->suspend = cfi_sststd_suspend;
             mtd->resume = cfi_sststd_resume;
             mtd->flags = MTD_CAP_NORFLASH;
             map->fldrv = &cfi_sststd_chipdrv;
             mtd->name = map->name;
             return mtd;
}

/*
 * Return true if the chip is ready.
 *
 * Ready is one of: read mode, query mode, erase-suspend-read mode (in any
 * non-suspended sector) and is indicated by no toggle bits toggling.
 *
 * Note that anything more complicated than checking if no bits are toggling
 * (including checking DQ5 for an error status) is tricky to get working
 * correctly and is therefore not done	(particulary with interleaved chips
 * as each chip must be checked independantly of the others).
 */
static int chip_ready(struct map_info *map, unsigned long addr)
{
	map_word d, t;

	d = map_read(map, addr);
	t = map_read(map, addr);

	return map_word_equal(map, d, t);
}

static inline int do_read_onechip(struct map_info *map, struct flchip *chip, loff_t adr, size_t len, u_char *buf)
{
  	DECLARE_WAITQUEUE(wait, current);
	unsigned long timeo = jiffies + HZ;

 retry :
	 cfi_spin_lock(chip->mutex);

        if (chip->state != FL_READY){
                set_current_state(TASK_UNINTERRUPTIBLE);
                add_wait_queue(&chip->wq, &wait);

                cfi_spin_unlock(chip->mutex);
                schedule();
                remove_wait_queue(&chip->wq, &wait);
                timeo = jiffies + HZ;

                goto retry;
        }

        adr += chip->start;

        chip->state = FL_READY;

        map->copy_from(map, buf, adr, len);

        wake_up(&chip->wq);
        cfi_spin_unlock(chip->mutex);

	return 0;	

}


static int cfi_sststd_read (struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen, u_char *buf)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	unsigned long ofs;
	int chipnum;
	int ret = 0;

	/* ofs: offset within the first chip that the first read should start */

	chipnum = (from >> cfi->chipshift);
	ofs = from - (chipnum <<  cfi->chipshift);


	*retlen = 0;

	while (len) {
		unsigned long thislen;

		if (chipnum >= cfi->numchips)
			break;

		if ((len + ofs -1) >> cfi->chipshift)
			thislen = (1<<cfi->chipshift) - ofs;
		else
			thislen = len;

		ret = do_read_onechip(map, &cfi->chips[chipnum], ofs, thislen, buf);
		if (ret)
			break;

		*retlen += thislen;
		len -= thislen;
		buf += thislen;

		ofs = 0;
		chipnum++;
	}
	return ret;
}

static int do_write_oneword(struct map_info *map, struct flchip *chip, unsigned long adr, map_word datum)
{

        unsigned long timeo = jiffies + HZ;
        struct cfi_private *cfi = map->fldrv_priv;
        DECLARE_WAITQUEUE(wait, current);
        int ret = 0;
	map_word oldd, curd;
	int retry_cnt = 0;

	/*
         * Check for a NOP for the case when the datum to write is already
         * present - it saves time and works around buggy chips that corrupt
         * data at other locations when 0xff is written to a location that
         * already contains 0xff.
         */
	 cfi_spin_lock(chip->mutex);	
	 adr += chip->start;
         oldd = map_read(map, adr);
         if (map_word_equal(map, oldd, datum)) {
                 DEBUG( MTD_DEBUG_LEVEL3, "MTD %s(): NOP\n",
                        __func__);
                goto op_done;
         }

	 cfi_spin_unlock(chip->mutex);
	 DEBUG( MTD_DEBUG_LEVEL3, "MTD %s(): WRITE 0x%.8lx(0x%.8lx)\n",
	                        __func__, adr, datum.x[0] );

 retry:
        cfi_spin_lock(chip->mutex);

        if (chip->state != FL_READY){
                set_current_state(TASK_UNINTERRUPTIBLE);
                add_wait_queue(&chip->wq, &wait);

                cfi_spin_unlock(chip->mutex);

                schedule();
                remove_wait_queue(&chip->wq, &wait);
                timeo = jiffies + HZ;

                goto retry;
        }

        chip->state = FL_WRITING;

 retry2 :

        ENABLE_VPP(map);
    	cfi_send_gen_cmd(0xAA, cfi->addr_unlock1, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
    	cfi_send_gen_cmd(0x55, cfi->addr_unlock2, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
    	cfi_send_gen_cmd(0xA0, cfi->addr_unlock1, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);

        map_write(map, datum, adr);

        cfi_spin_unlock(chip->mutex);
        cfi_udelay(chip->word_write_time);
        cfi_spin_lock(chip->mutex);

	for(;;)
	{
		if (chip->state != FL_WRITING) {
                         /* Someone's suspended the write. Sleep */
                         DECLARE_WAITQUEUE(wait, current);

                         set_current_state(TASK_UNINTERRUPTIBLE);
                         add_wait_queue(&chip->wq, &wait);
                         cfi_spin_unlock(chip->mutex);
                         schedule();
                         remove_wait_queue(&chip->wq, &wait);
                         timeo = jiffies + (HZ / 2); /* FIXME */
                         cfi_spin_lock(chip->mutex);
                         continue;
                 }
		
		/* Test to see if toggling has stopped. */
         	oldd = map_read(map, adr);
         	curd = map_read(map, adr);
         	if (map_word_equal(map, curd, oldd)) {
         		/* Do we have the correct value? */
              		if (map_word_equal(map, curd, datum)) {
                		goto op_done;
              		}
               		/* Nope something has gone wrong. */
                	break;
		}

		if (time_after(jiffies, timeo)) {
                         DEBUG( MTD_DEBUG_LEVEL3, "MTD %s(): software timeout\n",
                                __func__ );
                         break;
                 }

		/* Latency issues. Drop the lock, wait a while and retry */
                cfi_spin_unlock(chip->mutex);
                cfi_udelay(1);
                cfi_spin_lock(chip->mutex);
          }

	/* reset on all failures. */
        cfi_send_gen_cmd(0xAA, cfi->addr_unlock1, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
	cfi_send_gen_cmd(0x55, cfi->addr_unlock2, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
	cfi_send_gen_cmd(0xF0, cfi->addr_unlock1, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
        /* FIXME - should have reset delay before continuing */
        if (++retry_cnt <= MAX_WORD_RETRIES)
	{
	  	DEBUG( MTD_DEBUG_LEVEL3, "MTD %s(): WRITE_FAIL 0x%.8lx(0x%.8lx)\n",
		                                     __func__, adr, datum.x[0] );
                goto retry2;
	}

        ret = -EIO;

 op_done :
        DISABLE_VPP(map);
        chip->state = FL_READY;
        cfi_spin_unlock(chip->mutex);	
	wake_up(&chip->wq);

        return ret;

}

static int cfi_sststd_write(struct mtd_info *mtd, loff_t to, size_t len,
				  size_t *retlen, const u_char *buf)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int ret = 0;
	int chipnum;
	unsigned long ofs, chipstart;
	DECLARE_WAITQUEUE(wait, current);

	*retlen = 0;
	if (!len)
		return 0;

	chipnum = to >> cfi->chipshift;
	ofs = to  - (chipnum << cfi->chipshift);
	chipstart = cfi->chips[chipnum].start;

	/* If it's not bus-aligned, do the first byte write */
	if (ofs & (map_bankwidth(map)-1)) {
		unsigned long bus_ofs = ofs & ~(map_bankwidth(map)-1);
		int i = ofs - bus_ofs;
		int n = 0;
		map_word tmp_buf;

 retry:
		cfi_spin_lock(cfi->chips[chipnum].mutex);

		if (cfi->chips[chipnum].state != FL_READY) {
			set_current_state(TASK_UNINTERRUPTIBLE);
			add_wait_queue(&cfi->chips[chipnum].wq, &wait);

			cfi_spin_unlock(cfi->chips[chipnum].mutex);
			schedule();
			remove_wait_queue(&cfi->chips[chipnum].wq, &wait);
			goto retry;
		}

		/* Load 'tmp_buf' with old contents of flash */
		tmp_buf = map_read(map, bus_ofs+chipstart);

		cfi_spin_unlock(cfi->chips[chipnum].mutex);

		/* Number of bytes to copy from buffer */
		n = min_t(int, len, map_bankwidth(map)-i);
		
		tmp_buf = map_word_load_partial(map, tmp_buf, buf, i, n);

		ret = do_write_oneword(map, &cfi->chips[chipnum], 
				       bus_ofs, tmp_buf);
		if (ret) 
			return ret;
		
		ofs += n;
		buf += n;
		(*retlen) += n;
		len -= n;

		if (ofs >> cfi->chipshift) {
			chipnum ++; 
			ofs = 0;
			if (chipnum == cfi->numchips)
				return 0;
		}
	}
	
	/* We are now aligned, write as much as possible */
	while(len >= map_bankwidth(map)) {
		map_word datum;

		datum = map_word_load(map, buf);

		ret = do_write_oneword(map, &cfi->chips[chipnum],
				       ofs, datum);
		if (ret)
			return ret;

		ofs += map_bankwidth(map);
		buf += map_bankwidth(map);
		(*retlen) += map_bankwidth(map);
		len -= map_bankwidth(map);

		if (ofs >> cfi->chipshift) {
			chipnum ++; 
			ofs = 0;
			if (chipnum == cfi->numchips)
				return 0;
			chipstart = cfi->chips[chipnum].start;
		}
	}

	/* Write the trailing bytes if any */
	if (len & (map_bankwidth(map)-1)) {
		map_word tmp_buf;

 retry1:
		cfi_spin_lock(cfi->chips[chipnum].mutex);

		if (cfi->chips[chipnum].state != FL_READY) {
			set_current_state(TASK_UNINTERRUPTIBLE);
			add_wait_queue(&cfi->chips[chipnum].wq, &wait);

			cfi_spin_unlock(cfi->chips[chipnum].mutex);
			schedule();
			remove_wait_queue(&cfi->chips[chipnum].wq, &wait);
			goto retry1;
		}

		tmp_buf = map_read(map, ofs + chipstart);

		cfi_spin_unlock(cfi->chips[chipnum].mutex);

		tmp_buf = map_word_load_partial(map, tmp_buf, buf, 0, len);
	
		ret = do_write_oneword(map, &cfi->chips[chipnum], 
				ofs, tmp_buf);
		if (ret) 
			return ret;
		
		(*retlen) += len;
	}

	return 0;
}



typedef int (*frob_t)(struct map_info *map, struct flchip *chip,
		      unsigned long adr, void *thunk);


static int cfi_sststd_varsize_frob(struct mtd_info *mtd, frob_t frob,
				   loff_t ofs, size_t len, void *thunk)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	unsigned long adr;
	int chipnum, ret = 0;
	int i, first;
	struct mtd_erase_region_info *regions = mtd->eraseregions;

	if (ofs > mtd->size)
		return -EINVAL;

	if ((len + ofs) > mtd->size)
		return -EINVAL;

	/* Check that both start and end of the requested erase are
	 * aligned with the erasesize at the appropriate addresses.
	 */

	i = 0;

	/* Skip all erase regions which are ended before the start of 
	   the requested erase. Actually, to save on the calculations,
	   we skip to the first erase region which starts after the
	   start of the requested erase, and then go back one.
	*/
	
	while (i < mtd->numeraseregions && ofs >= regions[i].offset)
	       i++;
	i--;

	/* OK, now i is pointing at the erase region in which this 
	   erase request starts. Check the start of the requested
	   erase range is aligned with the erase size which is in
	   effect here.
	*/

	if (ofs & (regions[i].erasesize-1))
		return -EINVAL;

	/* Remember the erase region we start on */
	first = i;

	/* Next, check that the end of the requested erase is aligned
	 * with the erase region at that address.
	 */

	while (i<mtd->numeraseregions && (ofs + len) >= regions[i].offset)
		i++;

	/* As before, drop back one to point at the region in which
	   the address actually falls
	*/
	i--;
	
	if ((ofs + len) & (regions[i].erasesize-1))
		return -EINVAL;

	chipnum = ofs >> cfi->chipshift;
	adr = ofs - (chipnum << cfi->chipshift);

	i=first;

	while (len) {
		ret = (*frob)(map, &cfi->chips[chipnum], adr, thunk);
		
		if (ret)
			return ret;

		adr += regions[i].erasesize;
		len -= regions[i].erasesize;

		if (adr % (1<< cfi->chipshift) == ((regions[i].offset + (regions[i].erasesize * regions[i].numblocks)) %( 1<< cfi->chipshift)))
			i++;

		if (adr >> cfi->chipshift) {
			adr = 0;
			chipnum++;
			
			if (chipnum >= cfi->numchips)
			break;
		}
	}

	return 0;
}


static inline int do_erase_oneblock(struct map_info *map, struct flchip *chip, unsigned long adr, void *thunk)
{
	struct cfi_private *cfi = map->fldrv_priv;
	unsigned long timeo = jiffies + HZ;
	DECLARE_WAITQUEUE(wait, current);
	int ret = 0;

	adr += chip->start;

retry :	
	cfi_spin_lock(chip->mutex);
	DEBUG( MTD_DEBUG_LEVEL3, "MTD %s(): ERASE 0x%.8lx\n",
	       __func__, adr );

	if (chip->state != FL_READY){
                set_current_state(TASK_UNINTERRUPTIBLE);
                add_wait_queue(&chip->wq, &wait);

                cfi_spin_unlock(chip->mutex);
                schedule();
                remove_wait_queue(&chip->wq, &wait);
                timeo = jiffies + HZ;

                goto retry;
        }


	ENABLE_VPP(map);
	cfi_send_gen_cmd(0xAA, cfi->addr_unlock1, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
	cfi_send_gen_cmd(0x55, cfi->addr_unlock2, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
	cfi_send_gen_cmd(0x80, cfi->addr_unlock1, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
	cfi_send_gen_cmd(0xAA, cfi->addr_unlock1, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
	cfi_send_gen_cmd(0x55, cfi->addr_unlock2, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
	map_write(map, CMD(0x30), adr);

	timeo = jiffies + (HZ*20);

	chip->state = FL_ERASING;
	chip->erase_suspended = 0;
	chip->in_progress_block_addr = adr;

	cfi_spin_unlock(chip->mutex);
	set_current_state(TASK_UNINTERRUPTIBLE);
	schedule_timeout(HZ);
	cfi_spin_lock(chip->mutex);

	for (;;) {
		if (chip->state != FL_ERASING) {
			/* Someone's suspended the erase. Sleep */
			set_current_state(TASK_UNINTERRUPTIBLE);
			add_wait_queue(&chip->wq, &wait);
			cfi_spin_unlock(chip->mutex);
			schedule();
			remove_wait_queue(&chip->wq, &wait);
			cfi_spin_lock(chip->mutex);
			continue;
		}
		if (chip->erase_suspended) {
			/* This erase was suspended and resumed.
			   Adjust the timeout */
			timeo = jiffies + (HZ*20); /* FIXME */
			chip->erase_suspended = 0;
		}

		if (chip_ready(map, adr))
			goto op_done;

		if (time_after(jiffies, timeo))
			break;

		/* Latency issues. Drop the lock, wait a while and retry */
		cfi_spin_unlock(chip->mutex);
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(1);
		cfi_spin_lock(chip->mutex);
	}
	
	DEBUG( MTD_DEBUG_LEVEL3, "MTD %s(): software timeout\n",
	       __func__ );
	
	/* reset on all failures. */
	cfi_send_gen_cmd(0xAA, cfi->addr_unlock1, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
	cfi_send_gen_cmd(0x55, cfi->addr_unlock2, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
	cfi_send_gen_cmd(0xF0, cfi->addr_unlock1, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
	/* FIXME - should have reset delay before continuing */

	ret = -EIO;
 op_done:
	chip->state = FL_READY;
	ENABLE_VPP(map);
	wake_up(&chip->wq);
	cfi_spin_unlock(chip->mutex);
	return ret;
}


int cfi_sststd_erase(struct mtd_info *mtd, struct erase_info *instr)
{
	unsigned long ofs, len;
	int ret;

	ofs = instr->addr;
	len = instr->len;

	ret = cfi_sststd_varsize_frob(mtd, do_erase_oneblock, ofs, len, NULL);
	if (ret)
		return ret;

	instr->state = MTD_ERASE_DONE;
	mtd_erase_callback(instr);
	
	return 0;
}

static void cfi_sststd_sync (struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int i;
	struct flchip *chip;
	int ret = 0;
	DECLARE_WAITQUEUE(wait, current);

	for (i=0; !ret && i<cfi->numchips; i++) {
		chip = &cfi->chips[i];

	retry:
		cfi_spin_lock(chip->mutex);

		switch(chip->state) {
		case FL_READY:
		case FL_STATUS:
		case FL_CFI_QUERY:
		case FL_JEDEC_QUERY:
			chip->oldstate = chip->state;
			chip->state = FL_SYNCING;
			/* No need to wake_up() on this state change - 
			 * as the whole point is that nobody can do anything
			 * with the chip now anyway.
			 */
		case FL_SYNCING:
			cfi_spin_unlock(chip->mutex);
			break;

		default:
			/* Not an idle state */
			set_current_state(TASK_UNINTERRUPTIBLE);
			add_wait_queue(&chip->wq, &wait);
			
			cfi_spin_unlock(chip->mutex);
			schedule();

			remove_wait_queue(&chip->wq, &wait);
			
			goto retry;
		}
	}

	/* Unlock the chips again */

	for (i--; i >=0; i--) {
		chip = &cfi->chips[i];

		cfi_spin_lock(chip->mutex);
		
		if (chip->state == FL_SYNCING) {
			chip->state = chip->oldstate;
			wake_up(&chip->wq);
		}
		cfi_spin_unlock(chip->mutex);
	}
}


static int cfi_sststd_suspend(struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int i;
	struct flchip *chip;
	int ret = 0;

	for (i=0; !ret && i<cfi->numchips; i++) {
		chip = &cfi->chips[i];

		cfi_spin_lock(chip->mutex);

		switch(chip->state) {
		case FL_READY:
		case FL_STATUS:
		case FL_CFI_QUERY:
		case FL_JEDEC_QUERY:
			chip->oldstate = chip->state;
			chip->state = FL_PM_SUSPENDED;
			/* No need to wake_up() on this state change - 
			 * as the whole point is that nobody can do anything
			 * with the chip now anyway.
			 */
		case FL_PM_SUSPENDED:
			break;

		default:
			ret = -EAGAIN;
			break;
		}
		cfi_spin_unlock(chip->mutex);
	}

	/* Unlock the chips again */

	if (ret) {
		for (i--; i >=0; i--) {
			chip = &cfi->chips[i];

			cfi_spin_lock(chip->mutex);
		
			if (chip->state == FL_PM_SUSPENDED) {
				chip->state = chip->oldstate;
				wake_up(&chip->wq);
			}
			cfi_spin_unlock(chip->mutex);
		}
	}
	
	return ret;
}


static void cfi_sststd_resume(struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int i;
	struct flchip *chip;

	for (i=0; i<cfi->numchips; i++) {
	
		chip = &cfi->chips[i];

		cfi_spin_lock(chip->mutex);
		
		if (chip->state == FL_PM_SUSPENDED) {
			chip->state = FL_READY;
		
			/*reset chip */
			cfi_send_gen_cmd(0xAA, cfi->addr_unlock1, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
         		cfi_send_gen_cmd(0x55, cfi->addr_unlock2, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
 		        cfi_send_gen_cmd(0xF0, cfi->addr_unlock1, chip->start, map, cfi, CFI_DEVICETYPE_X16, NULL);
	
			wake_up(&chip->wq);
		}
		else
			DEBUG( MTD_DEBUG_LEVEL3, "Argh. Chip not in PM_SUSPENDED state upon resume()\n");

		cfi_spin_unlock(chip->mutex);
	}
}

static void cfi_sststd_destroy(struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	kfree(cfi->cmdset_priv);
	kfree(cfi->cfiq);
	kfree(cfi);
	kfree(mtd->eraseregions);
}

static char im_name[]="cfi_cmdset_0701";


int __init cfi_sststd_init(void)
{
	inter_module_register(im_name, THIS_MODULE, &cfi_cmdset_0701);
	return 0;
}


static void __exit cfi_sststd_exit(void)
{
	inter_module_unregister(im_name);
}


module_init(cfi_sststd_init);
module_exit(cfi_sststd_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Joris Lijssens <Joris.Lijssens@thomson.net> .");
MODULE_DESCRIPTION("MTD chip driver for SST flash chips");
