/*
 * Inode based directory notifications for Linux.
 *
 * Copyright (C) 2004 John McCutchan
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/spinlock.h>
#include <linux/idr.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/namei.h>
#include <linux/poll.h>
#include <linux/device.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/writeback.h>
#include <linux/inotify.h>

#include <asm/ioctls.h>

static atomic_t inotify_cookie;
static kmem_cache_t *watch_cachep;
static kmem_cache_t *event_cachep;
static kmem_cache_t *inode_data_cachep;

static int sysfs_attrib_max_user_devices;
static int sysfs_attrib_max_user_watches;
static int sysfs_attrib_max_queued_events;

/*
 * struct inotify_device - represents an open instance of an inotify device
 *
 * For each inotify device, we need to keep track of the events queued on it,
 * a list of the inodes that we are watching, and so on.
 *
 * This structure is protected by 'lock'.  Lock ordering:
 *
 * inode->i_lock
 *	dev->lock
 *		dev->wait->lock
 *
 * FIXME: Look at replacing i_lock with i_sem.
 */
struct inotify_device {
	wait_queue_head_t 	wait;
	struct idr		idr;
	struct list_head 	events;
	struct list_head 	watches;
	spinlock_t		lock;
	unsigned int		event_count;
	unsigned int		max_events;
	struct user_struct *	user;
};

struct inotify_watch {
	s32 			wd;	/* watch descriptor */
	u32			mask;
	struct inode *		inode;
	struct inotify_device *	dev;
	struct list_head	d_list;	/* device list */
	struct list_head	i_list; /* inode list */
	struct list_head	u_list; /* unmount list */
};
#define inotify_watch_d_list(pos) list_entry((pos), struct inotify_watch, d_list)
#define inotify_watch_i_list(pos) list_entry((pos), struct inotify_watch, i_list)
#define inotify_watch_u_list(pos) list_entry((pos), struct inotify_watch, u_list)

static ssize_t show_max_queued_events(struct device *dev, char *buf)
{
	sprintf(buf, "%d", sysfs_attrib_max_queued_events);

	return strlen(buf) + 1;
}

static ssize_t store_max_queued_events(struct device *dev, const char *buf,
				      size_t count)
{
	return 0;
}

static ssize_t show_max_user_devices(struct device *dev, char *buf)
{
	sprintf(buf, "%d", sysfs_attrib_max_user_devices);

	return strlen(buf) + 1;
}

static ssize_t store_max_user_devices(struct device *dev, const char *buf,
				      size_t count)
{
	return 0;
}

static ssize_t show_max_user_watches(struct device *dev, char *buf)
{
	sprintf(buf, "%d", sysfs_attrib_max_user_watches);

	return strlen(buf) + 1;
}

static ssize_t store_max_user_watches(struct device *dev, const char *buf,
				      size_t count)
{
	return 0;
}


static DEVICE_ATTR(max_queued_events, S_IRUGO | S_IWUSR, show_max_queued_events,
 store_max_queued_events);
static DEVICE_ATTR(max_user_devices, S_IRUGO | S_IWUSR, show_max_user_devices,
 store_max_user_devices);
static DEVICE_ATTR(max_user_watches, S_IRUGO | S_IWUSR, show_max_user_watches,
 store_max_user_watches);

static struct device_attribute *inotify_device_attrs[] = {
	&dev_attr_max_queued_events,
	&dev_attr_max_user_devices,
	&dev_attr_max_user_watches,
	NULL
};


/*
 * A list of these is attached to each instance of the driver
 * when the drivers read() gets called, this list is walked and
 * all events that can fit in the buffer get delivered
 */
struct inotify_kernel_event {
        struct list_head        list;
	struct inotify_event	event;
};

/*
 * find_inode - resolve a user-given path to a specific inode and iget() it
 */
static struct inode * find_inode(const char __user *dirname)
{
	struct inode *inode;
	struct nameidata nd;
	int error;

	error = __user_walk(dirname, LOOKUP_FOLLOW, &nd);
	if (error) {
		inode = ERR_PTR(error);
		goto out;
	}

	inode = nd.dentry->d_inode;

	/* you can only watch an inode if you have read permissions on it */
	error = vfs_permission(inode, MAY_READ);
	if (error) {
		inode = ERR_PTR(error);
		goto release_and_out;
	}

	__iget(inode);
release_and_out:
	path_release(&nd);
out:
	return inode;
}

static inline void unref_inode(struct inode *inode)
{
	iput(inode);
}

struct inotify_kernel_event *kernel_event(s32 wd, u32 mask, u32 cookie,
					  const char *filename)
{
	struct inotify_kernel_event *kevent;

	kevent = kmem_cache_alloc(event_cachep, GFP_ATOMIC);
	if (!kevent)
		goto out;

	/* we hand this out to user-space, so zero it out just in case */
	memset(kevent, 0, sizeof(struct inotify_kernel_event));

	kevent->event.wd = wd;
	kevent->event.mask = mask;
	kevent->event.cookie = cookie;
	INIT_LIST_HEAD(&kevent->list);

	if (filename) {
		strncpy(kevent->event.filename, filename,
			INOTIFY_FILENAME_MAX);
		kevent->event.filename[INOTIFY_FILENAME_MAX-1] = '\0';
	} else
		kevent->event.filename[0] = '\0';

out:
	return kevent;
}

void delete_kernel_event(struct inotify_kernel_event *kevent)
{
	if (!kevent)
		return;
	kmem_cache_free(event_cachep, kevent);
}

#define list_to_inotify_kernel_event(pos) list_entry((pos), struct inotify_kernel_event, list)
#define inotify_dev_get_event(dev) (list_to_inotify_kernel_event(dev->events.next))
#define inotify_dev_has_events(dev)	(!list_empty(&dev->events))

/* Does this events mask get sent to the watch ? */
#define event_and(event_mask,watches_mask) 	((event_mask == IN_UNMOUNT) || \
						(event_mask == IN_IGNORED) || \
						(event_mask & watches_mask))

/*
 * inotify_dev_queue_event - add a new event to the given device
 *
 * Caller must hold dev->lock.
 */
static void inotify_dev_queue_event(struct inotify_device *dev,
				    struct inotify_watch *watch, u32 mask,
				    u32 cookie, const char *filename)
{
	struct inotify_kernel_event *kevent, *last;

	/* Check if the new event is a duplicate of the last event queued. */
	last = inotify_dev_get_event(dev);
	if (dev->event_count && last->event.mask == mask &&
			last->event.wd == watch->wd) {
		/* Check if the filenames match */
		if (!filename && last->event.filename[0] == '\0')
			return;
		if (filename && !strcmp(last->event.filename, filename))
			return;
	}

	/*
	 * the queue has already overflowed and we have already sent the
	 * Q_OVERFLOW event
	 */
	if (dev->event_count > dev->max_events)
		return;

	/* the queue has just overflowed and we need to notify user space */
	if (dev->event_count == dev->max_events) {
		kevent = kernel_event(-1, IN_Q_OVERFLOW, cookie, NULL);
		goto add_event_to_queue;
	}

	if (!event_and(mask, watch->inode->inotify_data->watch_mask) ||
			!event_and(mask, watch->mask))
		return;

	dev->event_count++;
	kevent = kernel_event(watch->wd, mask, cookie, filename);

add_event_to_queue:
	if (!kevent) {
		dev->event_count--;
		return;
	}

	/* queue the event and wake up anyone waiting */
	list_add_tail(&kevent->list, &dev->events);
	wake_up_interruptible(&dev->wait);
}

/*
 * inotify_dev_event_dequeue - destroy an event on the given device
 *
 * Caller must hold dev->lock.
 */
static void inotify_dev_event_dequeue(struct inotify_device *dev)
{
	struct inotify_kernel_event *kevent;

	if (!inotify_dev_has_events(dev))
		return;

	kevent = inotify_dev_get_event(dev);
	list_del_init(&kevent->list);
	dev->event_count--;
	delete_kernel_event(kevent);

}

/*
 * inotify_dev_get_wd - returns the next WD for use by the given dev
 *
 * This function can sleep.
 */
static int inotify_dev_get_wd(struct inotify_device *dev,
			     struct inotify_watch *watch)
{
	int ret;

	if (atomic_read(&dev->user->inotify_watches) >= sysfs_attrib_max_user_watches)
		return -ENOSPC;

repeat:
	if (!idr_pre_get(&dev->idr, GFP_KERNEL))
		return -ENOSPC;
	spin_lock(&dev->lock);
	ret = idr_get_new(&dev->idr, watch, &watch->wd);
	spin_unlock(&dev->lock);
	if (ret == -EAGAIN) /* more memory is required, try again */
		goto repeat;
	else if (ret)       /* the idr is full! */
		return -ENOSPC;

	atomic_inc(&dev->user->inotify_watches);

	return 0;
}

/*
 * inotify_dev_put_wd - release the given WD on the given device
 *
 * Caller must hold dev->lock.
 */
static int inotify_dev_put_wd(struct inotify_device *dev, s32 wd)
{
	if (!dev || wd < 0)
		return -1;

	atomic_dec(&dev->user->inotify_watches);
	idr_remove(&dev->idr, wd);

	return 0;
}

/*
 * create_watch - creates a watch on the given device.
 *
 * Grabs dev->lock, so the caller must not hold it.
 */
static struct inotify_watch *create_watch(struct inotify_device *dev,
					  u32 mask, struct inode *inode)
{
	struct inotify_watch *watch;

	watch = kmem_cache_alloc(watch_cachep, GFP_KERNEL);
	if (!watch)
		return NULL;

	watch->mask = mask;
	watch->inode = inode;
	watch->dev = dev;
	INIT_LIST_HEAD(&watch->d_list);
	INIT_LIST_HEAD(&watch->i_list);
	INIT_LIST_HEAD(&watch->u_list);

	if (inotify_dev_get_wd(dev, watch)) {
		kmem_cache_free(watch_cachep, watch);
		return NULL;
	}

	return watch;
}

/*
 * delete_watch - removes the given 'watch' from the given 'dev'
 *
 * Caller must hold dev->lock.
 */
static void delete_watch(struct inotify_device *dev,
			 struct inotify_watch *watch)
{
	inotify_dev_put_wd(dev, watch->wd);
	kmem_cache_free(watch_cachep, watch);
}

/*
 * inotify_find_dev - find the watch associated with the given inode and dev
 *
 * Caller must hold dev->lock.
 */
static struct inotify_watch *inode_find_dev(struct inode *inode,
					    struct inotify_device *dev)
{
	struct inotify_watch *watch;

	if (!inode->inotify_data)
		return NULL;

	list_for_each_entry(watch, &inode->inotify_data->watches, i_list) {
		if (watch->dev == dev)
			return watch;
	}

	return NULL;
}

/*
 * dev_find_wd - given a (dev,wd) pair, returns the matching inotify_watcher
 *
 * Returns the results of looking up (dev,wd) in the idr layer.  NULL is
 * returned on error.
 *
 * The caller must hold dev->lock.
 */
static inline struct inotify_watch *dev_find_wd(struct inotify_device *dev,
						u32 wd)
{
	return idr_find(&dev->idr, wd);
}

static int inotify_dev_is_watching_inode(struct inotify_device *dev,
					 struct inode *inode)
{
	struct inotify_watch *watch;

	list_for_each_entry(watch, &dev->watches, d_list) {
		if (watch->inode == inode)
			return 1;
	}

	return 0;
}

/*
 * inotify_dev_add_watcher - add the given watcher to the given device instance
 *
 * Caller must hold dev->lock.
 */
static int inotify_dev_add_watch(struct inotify_device *dev,
				 struct inotify_watch *watch)
{
	if (!dev || !watch)
		return -EINVAL;

	list_add(&watch->d_list, &dev->watches);
	return 0;
}

/*
 * inotify_dev_rm_watch - remove the given watch from the given device
 *
 * Caller must hold dev->lock because we call inotify_dev_queue_event().
 */
static int inotify_dev_rm_watch(struct inotify_device *dev,
				struct inotify_watch *watch)
{
	if (!watch)
		return -EINVAL;

	inotify_dev_queue_event(dev, watch, IN_IGNORED, 0, NULL);
	list_del_init(&watch->d_list);

	return 0;
}

void inode_update_watch_mask(struct inode *inode)
{
	struct inotify_watch *watch;
	u32 new_mask;

	if (!inode->inotify_data)
		return;

	new_mask = 0;
	list_for_each_entry(watch, &inode->inotify_data->watches, i_list)
		new_mask |= watch->mask;

	inode->inotify_data->watch_mask = new_mask;
}

/*
 * inode_add_watch - add a watch to the given inode
 *
 * Callers must hold dev->lock, because we call inode_find_dev().
 */
static int inode_add_watch(struct inode *inode,
			   struct inotify_watch *watch)
{
	if (!inode || !watch)
		return -EINVAL;

	/*
	 * This inode doesn't have an inotify_data structure attached to it
	 */
	if (!inode->inotify_data) {
		inode->inotify_data = kmem_cache_alloc(inode_data_cachep,
						       GFP_ATOMIC);

		if (!inode->inotify_data)
			return -ENOMEM;

		INIT_LIST_HEAD(&inode->inotify_data->watches);
		inode->inotify_data->watch_mask = 0;
		inode->inotify_data->watch_count = 0;
	}

	if (inode_find_dev (inode, watch->dev))
		return -EINVAL;

	list_add(&watch->i_list, &inode->inotify_data->watches);
	inode->inotify_data->watch_count++;
	inode_update_watch_mask(inode);

	return 0;
}

static int inode_rm_watch(struct inode *inode,
			  struct inotify_watch *watch)
{
	if (!inode || !watch || !inode->inotify_data)
		return -EINVAL;

	list_del_init(&watch->i_list);
	inode->inotify_data->watch_count--;

	if (!inode->inotify_data->watch_count) {
		kmem_cache_free(inode_data_cachep, inode->inotify_data);
		inode->inotify_data = NULL;
		return 0;
	}

	inode_update_watch_mask(inode);

	return 0;
}

/* Kernel API */

void inotify_inode_queue_event(struct inode *inode, u32 mask, u32 cookie,
			       const char *filename)
{
	struct inotify_watch *watch;

	if (!inode->inotify_data)
		return;

	spin_lock(&inode->i_lock);

	list_for_each_entry(watch, &inode->inotify_data->watches, i_list) {
		spin_lock(&watch->dev->lock);
		inotify_dev_queue_event(watch->dev, watch, mask, cookie,
					filename);
		spin_unlock(&watch->dev->lock);
	}

	spin_unlock(&inode->i_lock);
}
EXPORT_SYMBOL_GPL(inotify_inode_queue_event);

void inotify_dentry_parent_queue_event(struct dentry *dentry, u32 mask,
				       u32 cookie, const char *filename)
{
	struct dentry *parent;

	parent = dget_parent(dentry);
	inotify_inode_queue_event(parent->d_inode, mask, cookie, filename);
	dput(parent);
}
EXPORT_SYMBOL_GPL(inotify_dentry_parent_queue_event);

u32 inotify_get_cookie(void)
{
	atomic_inc(&inotify_cookie);

	return atomic_read(&inotify_cookie);
}
EXPORT_SYMBOL_GPL(inotify_get_cookie);

static void ignore_helper(struct inotify_watch *watch, int event)
{
	struct inotify_device *dev;
	struct inode *inode;

	inode = watch->inode;
	dev = watch->dev;

	spin_lock(&inode->i_lock);
	spin_lock(&dev->lock);

	if (event)
		inotify_dev_queue_event(dev, watch, event, 0, NULL);

	inode_rm_watch(inode, watch);
	inotify_dev_rm_watch(watch->dev, watch);
	list_del(&watch->u_list);

	delete_watch(dev, watch);
	spin_unlock(&dev->lock);
	spin_unlock(&inode->i_lock);

	unref_inode(inode);
}

static void process_umount_list(struct list_head *umount)
{
	struct inotify_watch *watch, *next;

	list_for_each_entry_safe(watch, next, umount, u_list)
		ignore_helper(watch, IN_UNMOUNT);
}

/*
 * build_umount_list - build a list of watches affected by an unmount.
 *
 * Caller must hold inode_lock.
 */
static void build_umount_list(struct list_head *head, struct super_block *sb,
			      struct list_head *umount)
{
	struct inode *inode;

	list_for_each_entry(inode, head, i_list) {
		struct inotify_watch *watch;

		if (inode->i_sb != sb)
			continue;

		if (!inode->inotify_data)
			continue;

		spin_lock(&inode->i_lock);

		list_for_each_entry(watch, &inode->inotify_data->watches,
				    i_list) {
			list_add(&watch->u_list, umount);
		}

		spin_unlock(&inode->i_lock);
	}
}

void inotify_super_block_umount(struct super_block *sb)
{
	struct list_head umount;

	INIT_LIST_HEAD(&umount);

	spin_lock(&inode_lock);
	build_umount_list(&inode_in_use, sb, &umount);
	spin_unlock(&inode_lock);

	process_umount_list(&umount);
}
EXPORT_SYMBOL_GPL(inotify_super_block_umount);

/*
 * inotify_inode_is_dead - an inode has been deleted, cleanup any watches
 *
 * FIXME: Callers need to always hold inode->i_lock.
 */
void inotify_inode_is_dead(struct inode *inode)
{
	struct inotify_watch *watch, *next;
	struct inotify_inode_data *data;

	data = inode->inotify_data;
	if (!data)
		return;

	list_for_each_entry_safe(watch, next, &data->watches, i_list)
		ignore_helper(watch, 0);
}
EXPORT_SYMBOL_GPL(inotify_inode_is_dead);

/* The driver interface is implemented below */

static unsigned int inotify_poll(struct file *file, poll_table *wait)
{
        struct inotify_device *dev;

        dev = file->private_data;

        poll_wait(file, &dev->wait, wait);

        if (inotify_dev_has_events(dev))
                return POLLIN | POLLRDNORM;

        return 0;
}

static ssize_t inotify_read(struct file *file, char __user *buf,
			    size_t count, loff_t *pos)
{
	size_t event_size;
	struct inotify_device *dev;
	char __user *start;
	DECLARE_WAITQUEUE(wait, current);

	start = buf;
	dev = file->private_data;

	/* We only hand out full inotify events */
	event_size = sizeof(struct inotify_event);
	if (count < event_size)
		return -EINVAL;

	while (1) {
		int has_events;

		spin_lock(&dev->lock);
		has_events = inotify_dev_has_events(dev);
		spin_unlock(&dev->lock);
		if (has_events)
			break;

		if (file->f_flags & O_NONBLOCK)
			return -EAGAIN;

		if (signal_pending(current))
			return -ERESTARTSYS;

		add_wait_queue(&dev->wait, &wait);
		set_current_state(TASK_INTERRUPTIBLE);

		schedule();

		set_current_state(TASK_RUNNING);		
		remove_wait_queue(&dev->wait, &wait);
	}

	while (count >= event_size) {
		struct inotify_kernel_event *kevent;

		spin_lock(&dev->lock);
		if (!inotify_dev_has_events(dev)) {
			spin_unlock(&dev->lock);
			break;
		}
		kevent = inotify_dev_get_event(dev);
		spin_unlock(&dev->lock);
		if (copy_to_user(buf, &kevent->event, event_size))
			return -EFAULT;

		spin_lock(&dev->lock);
		inotify_dev_event_dequeue(dev);
		spin_unlock(&dev->lock);
		count -= event_size;
		buf += event_size;
	}

	return buf - start;
}

static int inotify_open(struct inode *inode, struct file *file)
{
	struct inotify_device *dev;
	struct user_struct *user;

	user = find_user (current->user->uid);

	if (!user)
		return -1;

	if (atomic_read(&user->inotify_devs) >= sysfs_attrib_max_user_devices)
		return -ENOSPC;

	atomic_inc(&current->user->inotify_devs);

	dev = kmalloc(sizeof(struct inotify_device), GFP_KERNEL);

	if (!dev)
		return -ENOMEM;


	idr_init(&dev->idr);

	INIT_LIST_HEAD(&dev->events);
	INIT_LIST_HEAD(&dev->watches);
	init_waitqueue_head(&dev->wait);

	dev->event_count = 0;
	dev->lock = SPIN_LOCK_UNLOCKED;
	dev->max_events = sysfs_attrib_max_queued_events;
	dev->user = user;

	file->private_data = dev;

	return 0;
}

/*
 * inotify_release_all_watches - destroy all watches on a given device
 *
 * FIXME: Do we want a lock here?
 */
static void inotify_release_all_watches(struct inotify_device *dev)
{
	struct inotify_watch *watch,*next;

	list_for_each_entry_safe(watch, next, &dev->watches, d_list)
		ignore_helper(watch, 0);
}

/*
 * inotify_release_all_events - destroy all of the events on a given device
 */
static void inotify_release_all_events(struct inotify_device *dev)
{
	spin_lock(&dev->lock);
	while (inotify_dev_has_events(dev))
		inotify_dev_event_dequeue(dev);
	spin_unlock(&dev->lock);
}

static int inotify_release(struct inode *inode, struct file *file)
{
	struct inotify_device *dev;

	dev = file->private_data;

	inotify_release_all_watches(dev);
	inotify_release_all_events(dev);

	atomic_dec(&dev->user->inotify_devs);
	free_uid (dev->user);

	kfree(dev);

	return 0;
}

static int inotify_watch(struct inotify_device *dev,
			 struct inotify_watch_request *request)
{
	struct inode *inode;
	struct inotify_watch *watch;
	int ret;

	inode = find_inode((const char __user*)request->dirname);
	if (IS_ERR(inode))
		return PTR_ERR(inode);

	spin_lock(&inode->i_lock);
	spin_lock(&dev->lock);

	/*
	 * This handles the case of re-adding a directory we are already
	 * watching, we just update the mask and return 0
	 */
	if (inotify_dev_is_watching_inode(dev, inode)) {
		struct inotify_watch *owatch;	/* the old watch */

		owatch = inode_find_dev(inode, dev);
		owatch->mask = request->mask;
		inode_update_watch_mask(inode);
		spin_unlock(&dev->lock);
		spin_unlock(&inode->i_lock);		
		unref_inode(inode);

		return owatch->wd;
	}

	spin_unlock(&dev->lock);
	spin_unlock(&inode->i_lock);	

	watch = create_watch(dev, request->mask, inode);
	if (!watch) {
		unref_inode(inode);
		return -ENOSPC;
	}

	spin_lock(&inode->i_lock);
	spin_lock(&dev->lock);

	/* We can't add anymore watches to this device */
	if (inotify_dev_add_watch(dev, watch)) {
		delete_watch(dev, watch);
		spin_unlock(&dev->lock);
		spin_unlock(&inode->i_lock);		
		unref_inode(inode);
		return -EINVAL;
	}

	ret = inode_add_watch(inode, watch);
	if(ret < 0) {
		list_del_init(&watch->d_list); /* inotify_dev_rm_watch w/o event */
		delete_watch(dev, watch);
		spin_unlock(&dev->lock);
		spin_unlock(&inode->i_lock);
		unref_inode(inode);
		return ret;
	}

	spin_unlock(&dev->lock);
	spin_unlock(&inode->i_lock);

	return watch->wd;
}

static int inotify_ignore(struct inotify_device *dev, s32 wd)
{
	struct inotify_watch *watch;

	/*
	 * FIXME: Silly to grab dev->lock here and then drop it, when
	 * ignore_helper() grabs it anyway a few lines down.
	 */
	spin_lock(&dev->lock);
	watch = dev_find_wd(dev, wd);
	spin_unlock(&dev->lock);
	if (!watch)
		return -EINVAL;
	ignore_helper(watch, 0);

	return 0;
}

/*
 * inotify_ioctl() - our device file's ioctl method
 *
 * The VFS serializes all of our calls via the BKL and we rely on that.  We
 * could, alternatively, grab dev->lock.  Right now lower levels grab that
 * where needed.
 */
static int inotify_ioctl(struct inode *ip, struct file *fp,
			 unsigned int cmd, unsigned long arg)
{
	struct inotify_device *dev;
	struct inotify_watch_request request;
	void __user *p;
	int bytes;
	s32 wd;

	dev = fp->private_data;
	p = (void __user *) arg;

	switch (cmd) {
	case INOTIFY_WATCH:
		if (copy_from_user(&request, p, sizeof (request)))
			return -EFAULT;
		return inotify_watch(dev, &request);
	case INOTIFY_IGNORE:
		if (copy_from_user(&wd, p, sizeof (wd)))
			return -EFAULT;
		return inotify_ignore(dev, wd);
	case FIONREAD:
		bytes = dev->event_count * sizeof(struct inotify_event);
		return put_user(bytes, (int __user *) p);
	default:
		return -ENOTTY;
	}
}

static struct file_operations inotify_fops = {
	.owner		= THIS_MODULE,
	.poll		= inotify_poll,
	.read		= inotify_read,
	.open		= inotify_open,
	.release	= inotify_release,
	.ioctl		= inotify_ioctl,
};

struct miscdevice inotify_device = {
	.minor  = MISC_DYNAMIC_MINOR,
	.name	= "inotify",
	.fops	= &inotify_fops,
};

static int __init inotify_init(void)
{
	int ret,i;

	ret = misc_register(&inotify_device);
	if (ret)
		return ret;

	sysfs_attrib_max_queued_events = 512;
	sysfs_attrib_max_user_devices = 64;
	sysfs_attrib_max_user_watches = 16384;

	for (i = 0; inotify_device_attrs[i]; i++) {
		int res;
		res = device_create_file (inotify_device.dev, inotify_device_attrs[i]);
		printk(KERN_INFO "Inotify sysfs create file = %d\n", res);
	}

	atomic_set(&inotify_cookie, 0);

	watch_cachep = kmem_cache_create("inotify_watch_cache",
			sizeof(struct inotify_watch), 0, SLAB_PANIC,
			NULL, NULL);

	event_cachep = kmem_cache_create("inotify_event_cache",
			sizeof(struct inotify_kernel_event), 0,
			SLAB_PANIC, NULL, NULL);

	inode_data_cachep = kmem_cache_create("inotify_inode_data_cache",
			sizeof(struct inotify_inode_data), 0, SLAB_PANIC,
			NULL, NULL);

	printk(KERN_INFO "inotify device minor=%d\n", inotify_device.minor);

	return 0;
}

module_init(inotify_init);
