/*
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 1995 - 2000 by Ralf Baechle
 */
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/types.h>
#include <linux/ptrace.h>
#include <linux/mman.h>
#include <linux/mm.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>
#include <linux/vt_kern.h>		/* For unblank_screen() */
#include <linux/module.h>

#include <asm/branch.h>
#include <asm/mmu_context.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/ptrace.h>
#include <asm/highmem.h>		/* For VMALLOC_END */

/* --dimm. Additional debugging info */
void print_extended_info(int signum, struct pt_regs *regs,
			 unsigned long write, unsigned long address)
{
	/* let's hope we are not short of staaaaack :) */
	struct mm_struct *mm = get_task_mm(current);
	struct vm_area_struct *epc_map, *ra_map;
	unsigned long epc_vma_start, epc_vma_end,
		      ra_vma_start, ra_vma_end;
	unsigned long epc, ra;
	char epc_module[64], ra_module[64];
	char *epc_path = NULL, *ra_path = NULL;
	int orig_level = console_loglevel;
	const int field = sizeof(unsigned long) * 2;
	int i;

	epc_vma_start = epc_vma_end = 0;
	ra_vma_start = ra_vma_end = 0;
	
	epc = regs->cp0_epc;
	ra = regs->regs[31];
	
	if (!mm)
		goto print_info;

	down_read(&mm->mmap_sem);

	epc_map = find_vma_intersection(mm, epc, epc);
	ra_map = find_vma_intersection(mm, ra, ra);

	if (epc_map) {
		epc_vma_start = epc_map->vm_start;
		epc_vma_end = epc_map->vm_end;
	}
	if (ra_map) {
		ra_vma_start = ra_map->vm_start;
		ra_vma_end = ra_map->vm_end;
	}
	
	if (epc_map && epc_map->vm_file)
		epc_path = d_path(epc_map->vm_file->f_dentry, epc_map->vm_file->f_vfsmnt,
				epc_module, sizeof(epc_module));
	
	if (ra_map && ra_map->vm_file)
		ra_path = d_path(ra_map->vm_file->f_dentry, ra_map->vm_file->f_vfsmnt,
				ra_module, sizeof(ra_module));

	up_read(&mm->mmap_sem);
	mmput(mm);

print_info:
	console_loglevel = 15;
	printk(	"\n"
		"\t|====================================================|\n");
	printk(	"\t|                Segmentation fault                  |\n");
	printk(	"\t|====================================================|\n");
	printk(	"\t  do_page_fault() sending SIGNAL #%d to [ %s ]\n", signum, current->comm);
	printk(	"\t  reason: invalid %s %0*lx\n"
		"\t|----------------------------------------------------|\n"
		"\t  <epc> : %0*lx in (%08lx-%08lx) %s\n"
		"\t  <ra>  : %0*lx in (%08lx-%08lx) %s\n"
		"\t  <sp>  : %0*lx\n"
		"\t|====================================================|\n\n",
		write ? "write access to" : "read access from",
		field, address,
		field, (unsigned long) regs->cp0_epc, epc_vma_start, epc_vma_end, (epc_path? epc_path : "unknown"),
		field, (unsigned long) regs->regs[31], ra_vma_start, ra_vma_end, (ra_path? ra_path  : "unknown"),
		field, (unsigned long) regs->regs[29]);

	printk("Registers:\n");
	for (i = 0; i < 32; i += 4)
		printk("  %08lx %08lx %08lx %08lx\n",
			regs->regs[i], regs->regs[i+1],
			regs->regs[i+2], regs->regs[i+3]);

	console_loglevel = orig_level;
}



/*
 * This routine handles page faults.  It determines the address,
 * and the problem, and then passes it off to one of the appropriate
 * routines.
 */
asmlinkage void do_page_fault(struct pt_regs *regs, unsigned long write,
			      unsigned long address)
{
	struct vm_area_struct * vma = NULL;
	struct task_struct *tsk = current;
	struct mm_struct *mm = tsk->mm;
	const int field = sizeof(unsigned long) * 2;
	siginfo_t info;

#if 0
	printk("Cpu%d[%s:%d:%0*lx:%ld:%0*lx]\n", smp_processor_id(),
	       current->comm, current->pid, field, address, write,
	       field, regs->cp0_epc);
#endif

	info.si_code = SEGV_MAPERR;

	/*
	 * We fault-in kernel-space virtual memory on-demand. The
	 * 'reference' page table is init_mm.pgd.
	 *
	 * NOTE! We MUST NOT take any locks for this case. We may
	 * be in an interrupt or a critical region, and should
	 * only copy the information from the master page table,
	 * nothing more.
	 */
	if (unlikely(address >= VMALLOC_START && address <= VMALLOC_END))
		goto vmalloc_fault;

	/*
	 * If we're in an interrupt or have no user
	 * context, we must not take the fault..
	 */
	if (in_atomic() || !mm)
		goto bad_area_nosemaphore;

	down_read(&mm->mmap_sem);
	vma = find_vma(mm, address);
	if (!vma)
		goto bad_area;
	if (vma->vm_start <= address)
		goto good_area;
	if (!(vma->vm_flags & VM_GROWSDOWN))
		goto bad_area;
	if (expand_stack(vma, address))
		goto bad_area;
/*
 * Ok, we have a good vm_area for this memory access, so
 * we can handle it..
 */
good_area:
	info.si_code = SEGV_ACCERR;

	if (write) {
		if (!(vma->vm_flags & VM_WRITE))
			goto bad_area;
	} else {
		if (!(vma->vm_flags & (VM_READ | VM_EXEC)))
			goto bad_area;
	}

survive:
	/*
	 * If for any reason at all we couldn't handle the fault,
	 * make sure we exit gracefully rather than endlessly redo
	 * the fault.
	 */
	switch (handle_mm_fault(mm, vma, address, write)) {
	case VM_FAULT_MINOR:
		tsk->min_flt++;
		break;
	case VM_FAULT_MAJOR:
		tsk->maj_flt++;
		break;
	case VM_FAULT_SIGBUS:
		goto do_sigbus;
	case VM_FAULT_OOM:
		goto out_of_memory;
	default:
		BUG();
	}

	up_read(&mm->mmap_sem);
	return;

/*
 * Something tried to access memory that isn't in our memory map..
 * Fix it, but check if it's kernel or user first..
 */
bad_area:
	up_read(&mm->mmap_sem);

bad_area_nosemaphore:
	/* User mode accesses just cause a SIGSEGV */
	if (user_mode(regs)) {
		tsk->thread.cp0_badvaddr = address;
		tsk->thread.error_code = write;

/* --dimm. Extended debugging info for Livebox. */
#if 1
		print_extended_info(SIGSEGV, regs, write, address);
#endif /* --dimm */

		printk("do_page_fault() #2: sending SIGSEGV to %s for "
		       "invalid %s\n%0*lx (epc == %0*lx, ra == %0*lx)\n",
		       tsk->comm,
		       write ? "write access to" : "read access from",
		       field, address,
		       field, (unsigned long) regs->cp0_epc,
		       field, (unsigned long) regs->regs[31]);

		info.si_signo = SIGSEGV;
		info.si_errno = 0;
		/* info.si_code has been set above */
		info.si_addr = (void __user *) address;
		force_sig_info(SIGSEGV, &info, tsk);
		return;
	}

no_context:
	/* Are we prepared to handle this kernel fault?  */
	if (fixup_exception(regs)) {
		current->thread.cp0_baduaddr = address;
		return;
	}

	/*
	 * Oops. The kernel tried to access some bad page. We'll have to
	 * terminate things with extreme prejudice.
	 */

	bust_spinlocks(1);

	printk(KERN_ALERT "CPU %d Unable to handle kernel paging request at "
	       "virtual address %0*lx, epc == %0*lx, ra == %0*lx\n",
	       smp_processor_id(), field, address, field, regs->cp0_epc,
	       field,  regs->regs[31]);
	die("Oops", regs);

/*
 * We ran out of memory, or some other thing happened to us that made
 * us unable to handle the page fault gracefully.
 */
out_of_memory:
	up_read(&mm->mmap_sem);
	if (tsk->pid == 1) {
		yield();
		down_read(&mm->mmap_sem);
		goto survive;
	}
	printk("VM: killing process %s\n", tsk->comm);
	if (user_mode(regs))
		do_exit(SIGKILL);
	goto no_context;

do_sigbus:
	up_read(&mm->mmap_sem);

	/* Kernel mode? Handle exceptions or die */
	if (!user_mode(regs))
		goto no_context;

	/*
	 * Send a sigbus, regardless of whether we were in kernel
	 * or user mode.
	 */

/* --dimm. Extended debugging info for Livebox. */
#if 1
	print_extended_info(SIGBUS, regs, write, address);
#endif /* --dimm */

	tsk->thread.cp0_badvaddr = address;
	info.si_signo = SIGBUS;
	info.si_errno = 0;
	info.si_code = BUS_ADRERR;
	info.si_addr = (void __user *) address;
	force_sig_info(SIGBUS, &info, tsk);

	return;

vmalloc_fault:
	{
		/*
		 * Synchronize this task's top level page-table
		 * with the 'reference' page table.
		 *
		 * Do _not_ use "tsk" here. We might be inside
		 * an interrupt in the middle of a task switch..
		 */
		int offset = __pgd_offset(address);
		pgd_t *pgd, *pgd_k;
		pmd_t *pmd, *pmd_k;
		pte_t *pte_k;

		pgd = (pgd_t *) pgd_current[smp_processor_id()] + offset;
		pgd_k = init_mm.pgd + offset;

		if (!pgd_present(*pgd_k))
			goto no_context;
		set_pgd(pgd, *pgd_k);

		pmd = pmd_offset(pgd, address);
		pmd_k = pmd_offset(pgd_k, address);
		if (!pmd_present(*pmd_k))
			goto no_context;
		set_pmd(pmd, *pmd_k);

		pte_k = pte_offset_kernel(pmd_k, address);
		if (!pte_present(*pte_k))
			goto no_context;
		return;
	}
}
