/**
 * @file   usb_printer.c
 * @author Thibault Gueslin <thibault.gueslin@inventel.fr>
 * @date   2004-2005
 * 
 * @brief  Gestion des imprimantes sur port usb.
 * 
 * Copyright Inventel 2004
 */
 
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <pthread.h>
#include <termios.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#include <linux/lp.h>
//#include <linux/delay.h>

#define NUM_THREADS 1
pthread_t tid[NUM_THREADS];      /* array of thread IDs */
pthread_mutex_t mut;

#define		PRINTER_FILE	"/dev/usb/lp%d"
#define         PRINTER_STATUS_FILE "/var/run/printer%d"

#define LPGETSTATUS		0x060b		/* same as in drivers/char/lp.c*/
#define IOCNR_GET_DEVICE_ID	1
#define LPIOC_GET_DEVICE_ID(len) _IOC(_IOC_READ, 'P', IOCNR_GET_DEVICE_ID, len)	/* get device_id string */

void * printer_report(void * parm);


int total_written;

/////// on rcupre l'tat de l'imprimante ////////////
int GetString (char *dest, char *source, char *pattern, int maxSize)
{
   char    *start;
   char    *end;
   int len;
    
   start = strstr (source, pattern);
   if (start == NULL)
   {
      dest[0]= 0x00;
      return 0;
   }
   start += strlen(pattern);
   len=maxSize;
   
   end = strstr (start, ";");
   if (end != NULL)
   {
      len = end-start;
      if (len> (maxSize-1)) len=maxSize-1;
   }
 //  fprintf(stderr, "Length: %d\n", len);
   strncpy (dest, start, len);
   dest[len]=0x0;
   return len;
}

int show_printer_status(int lpnumber, int fd, int operation)
{
   FILE* fo;
   char filename[255];
   int status=0;
   unsigned char argp[1024];
   int length;
   char buffer[512];
   
   if (fd < 0 )
   {
      fprintf(stderr, "Show_status: Printer is not opened\n"); 
      return 1;
   }
   
   if (ioctl (fd, LPIOC_GET_DEVICE_ID(sizeof(argp)), argp) < 0)
   {
      fprintf (stderr, "Error doing ioctl: %s\n", strerror (errno));
      return 1;
   }
   length = (argp[0] << 8) + argp[1] - 2;
    
   if (GetString (buffer, argp+2, "MDL:", 255) == 0)
      if (GetString (buffer, argp+2, "MODEL:", 255) == 0)
         sprintf(buffer, "????");
   
   if (ioctl (fd, LPGETSTATUS, &status) < 0)
   {
      fprintf (stderr, "Error doing ioctl: %s\n", strerror (errno));
      return 1;
   }
  // fprintf(stderr, "*******Status from printer: 0x%02x***********\n", status);
   sprintf(filename,  PRINTER_STATUS_FILE, lpnumber);
   fo = fopen(filename, "w");
   if (fo == NULL)
   {
      fprintf(stderr, "Can't open printer status file / %s\n", filename);
      return 2;
   }
   fprintf(fo, "STATUS=0x%2x\n", status);
   fprintf(fo, "NAME=%s\n", buffer);
   fprintf(fo, "ACTION=%d\n", operation);
   
   fclose(fo);
   return 0;
      
}

int remove_printer_status(int lpnumber)
{
   char filename[255];
   sprintf(filename,  PRINTER_STATUS_FILE, lpnumber);
   unlink(filename);
   return 0;
}


/////// niveau d'encre pour les HPs
#  define kMaxInkColors                     16

//void parseHPDeviceIDForStatusInfoForContext(TBUSBInterfaceContext * tbContext)
void parseHPDeviceIDForStatus(char * deviceID, int deviceIDLength)
{
    //note from http://www.hpdevelopersolutions.com/FAQpage.cfm?FAQID=222&CATEGORYID=84
    //Some early HP printers tended to exhibit issues with deviceID/status byte requests when the printer's buffer was full. A full buffer is a common occurrence during the course of a print-job and can be forced with an out-of-paper condition or by opening the top cover. The APSDK printer driver specifically handles these printers (640/810/830/880/895) internally by 'turning off' it's status checking after discovering one of these printers. However, it has been discovered that some implementers of the USB Printer Class Driver (the interface by which the APSDK or any other printer driver communicate over USB to the printer) incorporate their own deviceID or status byte checks during the course of a print job. Namely, Linux kernels previous to 2.4.0 performed this action and consequently exposed the issue. It is our recommendation that the USB Printer Class Driver not perform these device ID or status byte checks until the printer driver makes the request.
    // usage:
    //parseHPDeviceIDForStatus(tbContext->deviceIDBuffer, tbContext->deviceIDBufferSize);
    // now we should have a valid return within an adequately sized buffer.
    
   fprintf(stderr, "attempting to parse an HP device ID ...\n");
    // check for a valid device ID buffer ...
   if (NULL == deviceID || 0 == deviceIDLength)
   {
      fprintf(stderr, "* attempt to parse an invalid device ID");
      return;
   }
   char * status, *field, *statusEnd;
   char * hpColors[]={"Black","Color","Photo",0};
   int i, statusLength=0, cartridgeCount=0, val, level[kMaxInkColors];
    // parse it ...
    
    // here I changed the code to use multiple calls to strstr() instead of strcasestr()
    // because jaguar apparently has that particular function in a different shared
    // library than the one it resides in on Panther, so there are errors running on jaguar
    // and I have not cross-compiled for jaguar...
   if ((status = strstr(deviceID, "VSTATUS:")) != NULL||(status = strstr(deviceID, "vstatus:")) != NULL)
   {
      status += 8;
        // look for the ending `;'
      if ((statusEnd = strstr(status, ";")) != NULL)
         statusLength = statusEnd - status;
        // check for ink info by string length
      if (40 < statusLength)
      {
         cartridgeCount = '2' - 48;
            // look for K#,C# and KP###,CP### where # is a decimal digit
         level[0] = -2;  // set the default error condition
         if (0 == strncasecmp(status+23, ",K", 2))
         {
            if (0 == strncasecmp(status+25, "0", 1))
            {   // 0 signals an installed cartridge
                    // get the percent remaining
               if (0 == strncasecmp(status+35, ",KP", 3))
                  level[0]= (status[38] - 48)*100 + (status[39] - 48)*10 + (status[40] - 48);
            }
            else if (0 == strncasecmp(status+25, "1", 1))
               level[0] = -1;  // cartridge not installed
         }
         level[1] = -2;  // set the default error condition
         if (0 == strncasecmp(status+26, ",C", 2))
         {
            if (0 == strncasecmp(status+28, "0", 1))
            {   // 0 signals an installed cartridge
                    // get the percent remaining
               if (0 == strncasecmp(status+41, ",CP", 3))
                  level[1]= (status[44] - 48)*100 + (status[45] - 48)*10 + (status[46] - 48);
            }
            else if (0 == strncasecmp(status+28, "1", 1))
               level[1] = -1;  // cartridge not installed
         }
      }
   }
   else if ((status = strstr(deviceID, ";S:")) != NULL || (status = strstr(deviceID, ";s:")) != NULL)
   {
      status += 3;
        // check the version to determine status string length
        // the version is hex-encoded in the first two bytes
        // int version = (status[0] - 48)* 16 + status[1] - 48;
        // look for #c1......c2 where # is a decimal digit indicating the number of cartridges
      if (((field = strstr(status, "c1")) != NULL|| (field = strstr(status, "C1")) != NULL) && (strncasecmp(field+8, "c2",2) == 0))
      {
         cartridgeCount = *(field -1) - 48;
                
         for (i=0;i < cartridgeCount && i < kMaxInkColors ;i++)
         {
                // the following line deserves some explanation (primarily so that I can
                // have a chance to figure out what in the world I was thinking when I wrote it!)
                // we get the device ID from the printer as a raw string without modification.
                // HP printer's (this is an assumption) always use UTF8 encoding for the
                // hexidecimal ink-level percentage, thus we can decode the raw values even if
                // the host's native character encoding were to change. The first part takes
                // care of the high byte (it's big-endian) and we don't have to worry about any
                // hex values > 6 because if the percentage is ever over 100 then we have an error.
                // the second part (Starting with (val=...) decodes the low byte, and here we need
                // to be prepared to handle hexidecimal values so we subtract 48 if it's a decimal
                // digit, or we subtract 55 if it's an upper-case A-F (leaving 10 for `A' ...), or 
                // we subtract 87 if it's a-f (leaving 10 for `a' ...), else we add 1000 in error.
                // We check for values > 100 later and report an error if true.
                // the raw string looks like this:  3c148005fc2500061c3580061
                // so here cartridge 1 has a hex value of 5f = 95%, cartridge 2 has a hex value
                // of 61 = 97%, and cartridge 3 has the same value as cartridge 2.
                            
            level[i] = (field[8*i + 6] - 48)*16 + (val=field[8*i + 7]) - ((47 < val && 58 > val) ? 48 : ((64 < val && 71 > val) ? 55 : ((96 < val && 103 > val) ? 87 : -1000)));
         }
      }
   }
	
    // print the results on stderr INFO
   if (0 < cartridgeCount)
   {
      fprintf(stderr,"INFO: Ink levels:");
      for (i=0;i < cartridgeCount;i++)
      {
         if (-1 < level[i] && 101 > level[i])
            fprintf(stderr," %s=%d%%",hpColors[i],level[i]);
         else if ( -1 == level[i] )
            fprintf(stderr," %s=%s",hpColors[i],"missing");
         else
            fprintf(stderr," %s=%s",hpColors[i],"error");
      }
      fprintf(stderr,"\n");
   }
   else
      fprintf(stderr,"ERROR: Could not read ink levels\n");

}

//////////////////////////////////////////////////////////////////////

// maj du fichier d'etat de l'imprimante sans rien faire d'autre
int one_job_printer_status(int lpnumber)
{
   
   int lpfd=-1;
   char PrinterName[255];
   
   sprintf(PrinterName, PRINTER_FILE, lpnumber);
   fprintf(stderr, "PrinterName: %s (lp: %d)\n", PrinterName, lpnumber);
   lpfd = open(PrinterName,  O_RDWR);
   if (lpfd < 0)
   {
      fprintf(stderr, "Error printer: %s (%d)\n", strerror(errno), errno);
      if (errno == ENODEV) //l'imprimante est debranchee
         remove_printer_status(lpnumber);
      return 1;
   }
   
   show_printer_status(lpnumber, lpfd, 0);
   close(lpfd);
   return 0;
}

///////// write async (need to open printer with O_NDELAY)
int async_write(int lpfd, char* buffer, int len)
{
   int nwritten=0;
   int res, retval;
   struct timeval timeout;
   fd_set writefds;
   
  
   if (len==0) return 0; //on n'crit rien !
   
   while (1)
   {
      timeout.tv_sec = 10;
      timeout.tv_usec = 0;
 
      FD_ZERO(&writefds);
      FD_SET(lpfd, &writefds);
      
      retval = select(lpfd+1, NULL, &writefds, NULL, &timeout);
      if (retval == -1) {
         perror("select()");
         continue;
      }
      else if (retval==0) {
            fprintf(stderr, "timeout\n");
            show_printer_status(0, lpfd, 2);
            continue;
      }
      if (!FD_ISSET(lpfd, &writefds)) {
         fprintf(stderr, "something else\n");
         continue;
      }
      fprintf(stderr, "Writing ");
      res = write(lpfd, buffer+nwritten, len-nwritten );
      
      fprintf(stderr, "-> result: %d\n", res);
      
      if (res > 0 ) 
      {
         nwritten += res;
	if (nwritten >= len)	//on a crit le compte d'octets demands
	{
		total_written += nwritten;
	 	return (nwritten);
	}
      } 
      else
      {
         if (show_printer_status(0, lpfd, 1) > 0) 
            return (-1);
      }
      fprintf(stderr, "Written: %d/%d\r", nwritten, len);
      //fprintf(stderr, "Sleep now\n");
      usleep(10000);
   }
   return nwritten;
} 

//fonction pour ouvrir l'imprimante
int open_printer (char* PrinterName)
{
	int lpfd;           
	
	total_written =0;
	fprintf(stderr, "Opening printer _\n");
	lpfd = open(PrinterName, O_RDWR | O_EXCL| O_APPEND| O_NONBLOCK);
	

	return lpfd;
}

//pour rajouter du debug au besoin
void close_printer(int lpfd)
{
	int res;
	
	fprintf(stderr, "Closing printer: ");
	res=close(lpfd);
	fprintf(stderr, "%d\n", res);
	fprintf(stderr, "Total written %d\n", total_written);
}

