/* ----------------------------------
 * SET YOUR TAB STOPS TO 4 CHARACTERS
 * ----------------------------------
 * Non Spooling LPD server (RFC1179) V1.4
 * http://www.cis.ohio-state.edu/htbin/rfc/rfc1179.html
 *
 * V1.0 - 1.2, 1.4 Written by Steve Flynn
 * Based on Sigma Designs "How to write LPD programs"
 * http://www.dd.sigma.se/network/lpd.html
 * For Final Semester WebGate Project
 * June 1999
 *
 * V1.3 Modified by Simon Byrnand (sbyrnand@xtra.co.nz)
 * for project freesco. (http://www.freesco.org)
 * Changes since V1.2:
 *
 * Fixed #includes so it compiles under both
 * 2.0/libc5 and 2.2/glibc2.1. Increased receive timeout
 * to 300 seconds, (-t command line option to override) to
 * prevent large jobs timing out. Added a function to
 * reset the printer on unknown subcommands or timeouts
 * which usually occur if someone tries to abort the
 * print job. This makes sure the printer is back in a
 * known state so manual intervention isnt required.
 * More logging of errors and job state. Printing to an
 * invalid queue name is logged, timeouts waiting for
 * data are logged, the size of each print job is logged
 * and the connection closing is logged. Subcommand 1
 * (abort job) is now logged and also resets the printer.
 * Tidied up logging by using openlog() to specify process
 * name and we now log PID in case someone runs multiple
 * instances of lpd. Placed accept() call within a while
 * loop to retry in the case of transient errors. Some
 * error returns are vaild for example a socket error on
 * a queued socket, and we dont want to exit when receiving
 * one. Added command line options -h and --help to print
 * a usage summary.
 * May 2000 */

/* V1.4 June 2000 by Steve Flynn
 * Fixed incorrect dimension of Printer name character array
 * and set max Printer name length to 256 characters
 * Allow Netmasks up to 15 characters long (for people with subnetted Class C)
 * Added support for multiple subnets
 * Added support for multiple queue names
 * Added support for multiple printers
 * Added verbose logging command line switch
 * Added Request queue length command line switch
 *
 * Printer Device/File error no longer causes the program to exit. This
 * is an attempt to stop a faulty printer blocking others, since multpile
 * queue devices are now supported in sequential fashion.
*/

/* Usage: lpdsrv -v -n netmask -o output -p port -q len -r rq[:dev] -f fq[:dev] -t timeout
 * -v       : Enable verbose status logging
 * len      : Length of request queue									- default: 5
 * netmask  : Network mask for job acceptance			(max 15 char)	- default: 192.168.0
 *            Multiple occurences of -n are allowed (V1.4+) up to ten subnets
 * output   : Default name of Output File/Device name	(max 256 char)	- default: /dev/lp
 *            Sets the default queue name for following -r or -f parameters 
 * port     : TCP/IP port in decimal									- default: 515
 * rq[:dev] : Name of Raw print queue name			(max 32:256 char)	- default: nt:/dev/lp
 *            Optional device name for queue allowed (V1.4+)
 * fq[:dev] : Name of FormFeed print queue name		(max 32:256 char)	- default: lp:/dev/lp
 *            Optional device name for queue allowed (V1.4+)
 * timeout  : Seconds of no data from a client before timing out.		- default: 300 seconds
 *
 * Up to ten subnets may be included
 * Up to ten name:device pairs are supported for each queue type
 */

#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <syslog.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <linux/lp.h>

#define TBUF_LEN 2048
#define QNAME_LEN 32
#define DNAME_LEN 256
#define PNAME_LEN DNAME_LEN
#define MAXSUBNETS 10
#define SUBNET_LEN 15
#define MAXQUES 10
#define DEFAULTDEV "/dev/usb/lp0"

#define PIDFILE "/var/run/lpdsrv.pid"

char Printer[PNAME_LEN+1] = DEFAULTDEV;

extern int one_job_printer_status(int lpnumber);
extern int async_write(int lpfd, char* buffer, int len);
extern int open_printer (char* PrinterName);
extern void close_printer(int lpfd);

static inline int my__select( int line, int n, fd_set *readfds, fd_set *writefds, fd_set *exceptfds, struct timeval *timeout ) {
	int r;
	printf("in select line %d\n", line);
	r= select( n, readfds, writefds, exceptfds, timeout );
	printf("out select result %d\n", r);
	return r;
}

//#define select( n, readfds, writefds, exceptfds, timeout )  (printf("%d\n", __LINE__),  __select( n, readfds, writefds, exceptfds, timeout ))
#define select( n, readfds, writefds, exceptfds, timeout ) my__select( __LINE__ , n, readfds, writefds, exceptfds, timeout )


int write_pid_file()
{
	FILE* fo;

	if ((fo = fopen(PIDFILE, "w")) == NULL)
	{
		exit(1);
	}
	fprintf(fo, "%d\n", getpid());
	fclose(fo);
	return 0;
}

void reset_printer(Printer)
char *Printer;
	{
	/* We're assuming the printer is not open here, so dont call this function when the printer is already open. */
	/* No harm should come if we do however. We reset the printer after a print job is aborted or an error occurs */
	/* to clear its buffer, eject any half printed sheets and make sure it's in a known state for the next job. */
	int lpfd = -1;
	sleep(15);
	fprintf(stderr, "Resetting printer %s\n", Printer);
	lpfd = open(Printer, O_RDWR | O_NONBLOCK);
	if (lpfd >= 0)
		{
		ioctl(lpfd, LPRESET);
		close(lpfd);
		}
	sleep(5);
	}

/* Seperate a string (argv[i]) into queue name and device name */
void namedev(char *s, char* n, char* d)
	{
	char *ss = s;
	while (*ss++)
		if (*ss == ':')
			{
			*ss++ = '\0';
			strncpy(n, s, QNAME_LEN);
			strncpy(d, ss, DNAME_LEN);
			if (!strlen(d))
				strncpy(d, Printer, DNAME_LEN);
			return;
			}
	strncpy(n, s, QNAME_LEN);
	strncpy(d, Printer, DNAME_LEN);
	}

/* LPD server program. */

int main(argc, argv)
int argc;
char *argv[];
	{
	int i, j, sockfd, newsockfd, clilen, RejectNet;
	struct sockaddr_in CliAddr, ServAddr;
	int TCPPort = 515;
	int Timeout = /* amo 300*/ 10;
	int rawqcount = 0;
	int feedqcount = 0;
	int subnetcount = 0;
	struct timeval timeout, long_timeout;
	fd_set readfds, writefds;
	int reqlen, totreqlen, formfeed, controlfile;
	char tbuf[TBUF_LEN] = "by Steve Flynn & Simon Byrnand June 2000 sflynn@dingoblue.net.au sbyrnand@xtra.co.nz";
	char rbuf[TBUF_LEN];
	char *cli_ip;
	char ValidNets[MAXSUBNETS][SUBNET_LEN+1] = {"192.168"};
	char FeedQues[MAXQUES][QNAME_LEN+1] = {"lp"};
	char FeedDevs[MAXQUES][DNAME_LEN+1] = {DEFAULTDEV};
	char RawQues[MAXQUES][QNAME_LEN+1]  = {"nt"};
	char RawDevs[MAXQUES][DNAME_LEN+1]  = {DEFAULTDEV};
	char *PageFeed = "\f";
	int file_length = 0;
	int PrinterOpened, Completed;
	int lpfd = -1;
	int Verbose = 0;
	int QueLen = 5;
		
	int write_zero = 0;
	/* check for -h or --help as the first argument, print a usage summary and exit */

	if(argc > 1)
		{
		if(!strcasecmp(argv[1], "-h") || !strcasecmp(argv[1], "--help"))
			{
			printf(
			"Usage:\n"
			"%s -v -n netmask -o output -p port -q len -r rq[:dev] -f fq[:dev] -t timeout\n"
			"-v      : Enable verbose status logging\n"
			"len     : Length of request queue               - default: 5\n"
 			"netmask : Network mask for job acceptance       - default: 192.168.0\n"
			"port    : TCP/IP port in decimal                - default: 515\n"
			"rq[:dev]: Raw print queue name and device       - default: nt:/dev/lp\n"
			"fq[:dev]: FormFeed print queue name and device  - default: lp:/dev/lp\n"
			"output  : Default output File/Device name       - default: /dev/lp\n"
			"timeout : Seconds of no data received from a    - default: 300 seconds\n"
			"          client before timing out.\n"
			"Notes:\n"
			"1) Up to ten subnets are supported using multiple -n commands\n"
			"2) Up to ten name:device pairs are supported for each queue type\n"
			"3) -o Sets the default device for all following -r and -f parameters\n", argv[0]);
			exit(0);
			}
		}
                if (strcmp(argv[1], "-status")==0)
                {
                   return (one_job_printer_status(0));
                }
		
		if (strcmp(argv[1], "-reset")==0)
		{
			reset_printer(DEFAULTDEV);
			return 0;
		}
	fprintf(stderr, "starting lpdsrv\n");
	openlog(argv[0], LOG_PID, LOG_LPR);
	syslog(LOG_INFO, "%s started.", argv[0]);

	for (i = 1; i < argc; i++)
		{
		/* -v */
		if (!strncasecmp(argv[i], "-v", 2))
			{
			fprintf(stderr, "Verbose on\n");
			Verbose = 1;
			continue;
			}

		/* -n netmask */
		if (!strncasecmp(argv[i], "-n", 2) && (++i < argc))
			{
			if (subnetcount < MAXSUBNETS)
				strncpy(ValidNets[subnetcount++], argv[i], SUBNET_LEN);
			else
				syslog(LOG_ERR, "Subnet count exceeded!");
			continue;
			}

		/* -p port */
		if (!strncasecmp(argv[i], "-p", 2) && (++i < argc))
			{
			TCPPort = atoi(argv[i]);
			continue;
			}

		/* -r rq[:dev] */
		if (!strncasecmp(argv[i], "-r", 2) && (++i < argc))
			{
			if (rawqcount < MAXQUES)
				{
				namedev(argv[i], RawQues[rawqcount], RawDevs[rawqcount]);
				rawqcount++;
				}
			else
				syslog(LOG_ERR, "Raw queue name count exceeded!");
			continue;
			}

		/* -f fq[:dev] */
		if (!strncasecmp(argv[i], "-f", 2) && (++i < argc))
			{
			if (feedqcount < MAXQUES)
				{
				namedev(argv[i], FeedQues[feedqcount], FeedDevs[feedqcount]);
				feedqcount++;
				}
			else
				syslog(LOG_ERR, "Feed queue name count exceeded!");
			continue;
			}

		/* -o output */
		if (!strncasecmp(argv[i], "-o", 2) && (++i < argc))
			{
			strncpy(Printer, argv[i], PNAME_LEN);
			continue;
			}

		/* -t timeout */
		if (!strncasecmp(argv[i], "-t", 2) && (++i < argc))
			{
			Timeout = atoi(argv[i]);
			continue;
			}
		/* -q len */
		if (!strncasecmp(argv[i], "-q", 2) && (++i < argc))
			{
			QueLen = atoi(argv[i]);
			continue;
			}
		}

	/* Set default queue and subnet counts */
	if (!subnetcount)
		subnetcount++;
	if (!feedqcount)
		feedqcount++;
	if (!rawqcount)
		rawqcount++;

	/* Tell them who made it if they are interested */
	if (Verbose)
		syslog(LOG_INFO, "%s %s", argv[0], tbuf);

	/* Make sure all queues have different names */
	for (i = 0; i < feedqcount; i++)
		for (j = 0; j < rawqcount; j++)
			if (!strcasecmp(FeedQues[i], RawQues[j]))
				{
				fprintf(stderr, "Duplicate queue name %s, exiting!\n", FeedQues[i]);
				exit(1);
				}

	/* Open a TCP stream socket. */
	if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
		{
		syslog(LOG_ERR, "Can't open stream socket, exiting!");
		exit(1);
		}

	/* Bind our local address so that the client can send to us. */
	memset((char *)&ServAddr, 0, sizeof(ServAddr));
	ServAddr.sin_family      = AF_INET;
	ServAddr.sin_addr.s_addr = htonl(INADDR_ANY);
	ServAddr.sin_port        = htons((u_short)TCPPort);
	if (bind(sockfd, (struct sockaddr *)&ServAddr, sizeof(ServAddr)) < 0)
		{
		syslog(LOG_ERR, "Can't bind local port or address, exiting! (%s already running ?)", argv[0]);
		exit(1);
		}

	/* Detach from parent */
	
	if (!Verbose)
		daemon(0, 0);

	//if (!Verbose)
	//	daemon(0, 1);
	if (write_pid_file() != 0)
		exit(1);
 	
	/* Informative stuff for interested folk */
	if (Verbose)
		{
		syslog(LOG_INFO, "TCP/IP Port : %d", TCPPort);
		syslog(LOG_INFO, "Output      : %s", Printer);
		syslog(LOG_INFO, "Timeout     : %d", Timeout);
		syslog(LOG_INFO, "Job Que Len : %d", QueLen);
		for (i = 0; i < subnetcount; i++)
			fprintf(stderr, "Valid Subnet: %s\n", ValidNets[i]);
		for (i = 0; i < feedqcount; i++)
			fprintf(stderr, "Feed Que:Dev: %s:%s\n", FeedQues[i], FeedDevs[i]);
		for (i = 0; i < rawqcount; i++)
			fprintf(stderr, "Raw Que:Dev : %s:%s\n", RawQues[i], RawDevs[i]);
		}

	/* Queue requests */
	listen(sockfd, QueLen);

	/* Main server infinite loop processing client commands */
	for (;;)
		{
		/* Wait for a connection from a client process. */
		PrinterOpened = 0;
		Completed = 0;
		clilen = sizeof(CliAddr);
		while ((newsockfd = accept(sockfd, (struct sockaddr *)&CliAddr, &clilen)) < 0)
			if (Verbose)
				syslog(LOG_ERR, "Network socket error %d, retrying.", newsockfd);

		/* Identify connection client */
		if ((cli_ip = inet_ntoa(CliAddr.sin_addr)) == NULL)
			{
			if (Verbose)
				syslog(LOG_INFO, "Rejected connection with bad IP address.");
			close(newsockfd);
			continue;
			}
		for (i = 0; i < subnetcount; i++)
			{
			if ((RejectNet = strncmp(cli_ip, ValidNets[i], strlen(ValidNets[i]))) == 0)
				break;
			}
		if (RejectNet != 0)
			{

			if (Verbose)
				syslog(LOG_INFO, "Rejected connection from %s:%d.", cli_ip, ntohs(CliAddr.sin_port));
			close(newsockfd);
			continue;
			}
		if (Verbose)
		{
			fprintf(stderr,"Accepted connection from %s:%d\n", cli_ip, ntohs(CliAddr.sin_port));
			syslog(LOG_INFO, "Accepted connection from %s:%d.", cli_ip, ntohs(CliAddr.sin_port));
		}
		/* Setup a timeout for read and write, use a longer timeout when waiting on commands or subcommands */
		timeout.tv_sec = 16;
		timeout.tv_usec = 0;
		long_timeout.tv_sec = Timeout;
		long_timeout.tv_usec = 0;
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);		
		FD_SET(newsockfd, &readfds);
		FD_SET(newsockfd, &writefds);

		printf("newsockfd=%d\n", newsockfd );
		/* Read a complete LPD daemon command from the connected client */
		if ((select(FD_SETSIZE, &readfds, NULL, NULL, &long_timeout) == 1)
			&& ((reqlen = recv(newsockfd, tbuf, TBUF_LEN-1, 0)) > 0) )
			{ 
			/* Interpret daemon command acording to LPD RFC1179 */
			switch (tbuf[0])
				{
				case 1:	/* Print any waiting jobs - do nothing */
					break;
				case 2: /* Receive subcommand */
					/* Accept connections to named printers */
					rbuf[0] = 0x01;
					formfeed = 0;
					for (i = 0; i < feedqcount; i++)
						if (!strncasecmp(&tbuf[1], FeedQues[i], strlen(FeedQues[i])))
							{
							strcpy(Printer, FeedDevs[i]);
							formfeed = 1;
							rbuf[0] = 0x00;
							}
					for (i = 0; i < rawqcount; i++)
						if (!strncasecmp(&tbuf[1], RawQues[i], strlen(RawQues[i])))
							{
							strcpy(Printer, RawDevs[i]);
							formfeed = 0;
							rbuf[0] = 0x00;
							}
					/* Send ack/nak */
					if (select(FD_SETSIZE, NULL, &writefds, NULL, &timeout) == 1)
						send(newsockfd, rbuf, 1, 0);
					else
						break;
					/* Check for invalid printer name request */
					if (rbuf[0] == 0x01)
						{
						syslog(LOG_INFO, "Client %s attempted connection to non-existant queue name %s.", cli_ip, &tbuf[1]);
						break;
						}
					/* Deal with all job subcommands */
					while (select(FD_SETSIZE, &readfds, NULL, NULL, &long_timeout) == 1
						&& ((reqlen = recv(newsockfd, tbuf, TBUF_LEN-1, 0)) > 0) )
						{
						controlfile = 0;
						switch (tbuf[0])
							{
							case 1: /* Abort job */
								/* Send subcommand ack and reset printer if we've printed anything */
								rbuf[0] = 0x00;
								if (select(FD_SETSIZE, NULL, &writefds, NULL, &timeout) == 1)
									send(newsockfd, rbuf, 1, 0);
								if (PrinterOpened)
									{
									if (Verbose)
										syslog(LOG_INFO, "Client %s sent subcommand 1, (abort job) resetting printer in 15 seconds.", cli_ip);
									reset_printer(Printer);
									}
								else
									if (Verbose)
										syslog(LOG_INFO, "Client %s sent subcommand 1, (abort job) ignoring.", cli_ip);
								goto disconnect;
							case 2: /* ReceiveReceive control file */
								controlfile = 1;
							case 3: /* Receive data file */
								/* Send subcommand ack */
								rbuf[0] = 0x00;
								if (select(FD_SETSIZE, NULL, &writefds, NULL, &timeout) == 1)
									send(newsockfd, rbuf, 1, 0);
								else
									break;
								/* Get file length - forget filename */
								tbuf[reqlen] = '\0';
								file_length = 0;
								if (!sscanf(&tbuf[1], "%d", &file_length))
									file_length = 0;
								if (lpfd >= 0)
									close_printer(lpfd);
								lpfd = -1;
								/* Open Output Device/File with Wait Timeout */
								if (!controlfile)
									{
									if (Verbose)
										if(file_length == 0)
											fprintf(stderr, "Job received from %s has unspecified size.\n", cli_ip);
										else
											if(file_length < 10240)
												fprintf(stderr, "Job received from %s is %d bytes.\n", cli_ip, file_length);
											else
												fprintf(stderr, "Job received from %s is %d Kb.\n", cli_ip, file_length / 1024);
									for (i = 1; ; i = i < 32 ? i << 1 : i)
										{
										lpfd = open_printer(Printer);
										if (lpfd >= 0)
											{
											/* Flag that we've opened the printer and are about to print */
											/* so we know to reset the printer if an error occurs */
											PrinterOpened = 1;
											write_zero=0;
											break;
											}
										if (errno == ENOENT)
											{
											fprintf(stderr, "Can't open %s\n", Printer);
											/*exit(1);*/
											goto disconnect;
											}
										if (i == 1)
											fprintf(stderr, "Waiting for %s to become ready (offline ?)\n", Printer);
										sleep(i);
										}
									}
								/* Read and Print Data */
								totreqlen = 0;
								while (select(FD_SETSIZE, &readfds, NULL, NULL, &timeout) == 1
									&& ((reqlen = recv(newsockfd, tbuf, TBUF_LEN-1, 0)) > 0))
									{
                                                                           if (Verbose)
                                                                              fprintf(stderr, "Got %d from network\n", reqlen);
                                                                           if (lpfd >0)
                                                                                show_printer_status(0, lpfd, 1);
									int tmp_len=reqlen;
									if ((!controlfile) && (write_zero != 0))
									{
										fprintf(stderr, "Written zero\n");
										async_write(lpfd, "\0", 1); 
									}
									if ((!controlfile) && (tbuf[reqlen-1] == 0x00))
									{
										fprintf(stderr, "Last byte is null\n");
										tmp_len =tmp_len-1;
										write_zero=1;
									}
									else
										write_zero=0;
                                                                        if (!controlfile && (lpfd >= 0) && (async_write(lpfd, tbuf, tmp_len) == -1))
										{
										fprintf(stderr, "Error writing to %s\n", Printer);
										if (lpfd >= 0)
											close_printer(lpfd);
										lpfd = -1;
										/*close(newsockfd);*/
										/*exit(1);*/
										goto disconnect;
										}
									totreqlen += reqlen;
									/* Look for EOF */
									if (file_length <= 0)
										{
										/* No real file length available */
										if (reqlen > 0 && tbuf[reqlen-1] == 0x00)
											{
											/* Send ack for received complete file */
											rbuf[0] = 0x00;
											if (select(FD_SETSIZE, NULL, &writefds, NULL, &timeout) == 1)
												send(newsockfd, rbuf, 1, 0);
											if (formfeed && !controlfile && (lpfd >= 0) && (async_write(lpfd, PageFeed, 1) == -1))
												{
												fprintf(stderr, "Error writing to %s\n", Printer);
												if (lpfd >= 0)
													close_printer(lpfd);
												lpfd = -1;
												/*close(newsockfd);*/
												/*exit(1);*/
												goto disconnect;
												}
											if(!controlfile)
												Completed = 1;
											break;
											}
										}
									else
										{
										/* Real file length available */
										if ((totreqlen > file_length)
											&& (reqlen > 0)
											&& (tbuf[reqlen-1] == 0x00))
											{
											/* Send ack for received complete file */
											rbuf[0] = 0x00;
											if (select(FD_SETSIZE, NULL, &writefds, NULL, &timeout) == 1)
												send(newsockfd, rbuf, 1, 0);
											if (formfeed && !controlfile && (lpfd >= 0) && (async_write(lpfd, PageFeed, 1) == -1))
												{
												fprintf(stderr, "Error writing to %s\n", Printer);
												if (lpfd >= 0)
													close_printer(lpfd);
												lpfd = -1;
												/*close(newsockfd);*/
												/*exit(1);*/
												goto disconnect;
												}
											if(!controlfile)
												Completed = 1;
											break;
											}
										}
									} /* End of read file loop */
									if (lpfd >= 0)
										close_printer(lpfd);
									lpfd = -1;
									break;
							default:
								/* Send subcommand nak */
								rbuf[0] = 0x01;
								if (select(FD_SETSIZE, NULL, &writefds, NULL, &timeout) == 1)
									send(newsockfd, rbuf, 1, 0);
								if(PrinterOpened)
									{
									fprintf(stderr, "Unknown subcommand %d received from %s, resetting printer in 15 seconds\n", tbuf[0], cli_ip);
									reset_printer(Printer);
									}
								else
									fprintf(stderr, "Unknown subcommand %d received from %s\n", tbuf[0], cli_ip);
								goto disconnect;
								break;
								}
							}
						if(!Completed)
							{
							if (PrinterOpened)
								{
								fprintf(stderr, "Timeout waiting for data from %s, resetting printer in 15 seconds\n", cli_ip);
								reset_printer(Printer);
								}
							else
								fprintf(stderr, "Timeout waiting for data from %s\n", cli_ip);
							}
					break;
			case 3: /* Send queue state (short) */
			case 4:	/* Send queue state (long) */
				strcpy(rbuf, "no entries\n");
				if (select(FD_SETSIZE, NULL, &writefds, NULL, &timeout) == 1)
					send(newsockfd, rbuf, strlen(rbuf), 0);
				break;
			case 5:	/* Remove jobs - do nothing */
				break;
			default: /* Unknown command */
				fprintf(stderr, "Unknown command %d received from %s\n", tbuf[0], cli_ip);
				/* Send command nak */
				rbuf[0] = 0x01;
				if (select(FD_SETSIZE, NULL, &writefds, NULL, &timeout) == 1)
					send(newsockfd, rbuf, 1, 0);
				goto disconnect;
				break;
			}
		}
disconnect:
		/* Close connection to client process */
		if (Verbose)
			fprintf(stderr, "Connection from %s closed\n", cli_ip);
		if (lpfd >= 0)
			close_printer(lpfd);
		lpfd = -1;
                remove_printer_status(0);
		close(newsockfd);
		} /* End of main server infinite loop */

	/* This point should never be reached! */
	fprintf(stderr, "Abnormal exit\n");
	if (lpfd >= 0)
		close_printer(lpfd);
	}
