/*
 * Broadcom mapping driver for thomson modems
 * based on drivers/mtd/maps/pnc2000.c
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#include <asm/io.h>

#define BT_OFFSET	0x400000

static map_word physmap_read16(struct map_info *map, unsigned long ofs)
{
        map_word val;
	unsigned long phys;

	phys = map->phys - BT_OFFSET;
        val.x[0] = __raw_readw(phys + ofs);

        return val;
}

static void physmap_write16(struct map_info *map, map_word data, unsigned long ofs)
{
  	unsigned long phys;
	unsigned long phys_ofs;

	phys = map->phys - BT_OFFSET;
	phys_ofs = phys + ofs;

	__raw_writew(data.x[0], phys_ofs);
}


static void physmap_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
  	unsigned long phys;
	
	phys = map->phys - BT_OFFSET;
        memcpy_fromio(to, phys + from, len);
}


/* 
 * MAP DRIVER STUFF
 */


struct map_info bcm_map = {
	.name = "flash",
	.size = CONFIG_BCM_PHYS_SIZE,
	.phys = CONFIG_BCM_FLASH_START,
	.bankwidth = 2,
	.read = physmap_read16,
	.write = physmap_write16,
	.copy_from = physmap_copy_from,
};

/*
 * MTD 'PARTITIONING' STUFF 
*/
static struct mtd_partition bcm_partitions[3] = {
	{	
		.name = CONFIG_BCM_PART1_NAME,
		.size = CONFIG_BCM_PART1_SIZE,
		.offset = CONFIG_BCM_PART1_OFFSET
	},
	{
		.name = CONFIG_BCM_PART2_NAME,
		.size = CONFIG_BCM_PART2_SIZE,
		.offset = CONFIG_BCM_PART2_OFFSET
	},
	{
		.name = CONFIG_BCM_PART3_NAME,
		.size =  CONFIG_BCM_PART3_SIZE,
		.offset = CONFIG_BCM_PART3_OFFSET
	}
};

/* 
 * This is the master MTD device for which all the others are just
 * auto-relocating aliases.
 */
static struct mtd_info *mymtd;

int __init init_bcm(void)
{
	printk(KERN_NOTICE "Thomson BCM963xx flash mapping for BTHub\n");

	mymtd = do_map_probe("cfi_probe", &bcm_map);
	printk(KERN_NOTICE "flash mapping initialized\n");
	if (mymtd) {
		mymtd->owner = THIS_MODULE;
		return add_mtd_partitions(mymtd, bcm_partitions, 3);
		return 0;	
	}

	return -ENXIO;
}

static void __exit cleanup_bcm(void)
{
	if (mymtd) {
		del_mtd_partitions(mymtd);
		map_destroy(mymtd);
	}
}

module_init(init_bcm);
module_exit(cleanup_bcm);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Philippe De Swert <philippe.deswert@thomson.net");
MODULE_DESCRIPTION("MTD map driver for 516/536 board");
