/* dhcpc.c
 *
 * udhcp DHCP client
 *
 * Russ Dill <Russ.Dill@asu.edu> July 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/file.h>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <time.h>
#include <string.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <errno.h>

#include "dhcpd.h"
#include "dhcpc.h"
#include "options.h"
#include "clientpacket.h"
#include "packet.h"
#include "script.h"
#include "socket.h"
#include "debug.h"
#include "pidfile.h"
#ifdef CONFIG_VENDOR_BCM
#include "bcmnvram.h"
#else
#include "nvram.h"
#endif

#ifdef CONFIG_SYS_UPTIME
#include <sys/sysinfo.h>
struct sysinfo info;
#endif

static int state;
static unsigned long requested_ip; /* = 0 */
static unsigned long server_addr;
static unsigned long timeout;
static int packet_num; /* = 0 */
static int fd;
static int signal_pipe[2];
static int bootp_cnt_offer = 0/*, bootp_cnt_ack = 0*/;
#define LISTEN_NONE 0
#define LISTEN_KERNEL 1
#define LISTEN_RAW 2
static int listen_mode;

#define LOG_NUM 100
#define DHCPLOG		"/tmp/log/dhcpc_new.log"
int log_dhcpc_cnt = 0;
unsigned char dhcpc_buf[512];

	

#define DEFAULT_SCRIPT	"/usr/share/udhcpc/default.script"

extern int detect_ACK_options(struct dhcpMessage *packet);

struct client_config_t client_config = {
	/* Default options. */
	abort_if_no_lease: 0,
	foreground: 0,
	quit_after_lease: 0,
	background_if_no_lease: 0,
	interface: "eth0",
	pidfile: NULL,
	script: DEFAULT_SCRIPT,
	clientid: NULL,
	hostname: NULL,
	ifindex: 0,
	arp: "\0\0\0\0\0\0",		/* appease gcc-3.0 */
};

#ifndef BB_VER
static void show_usage(void)
{
	printf(
"Usage: udhcpc [OPTIONS]\n\n"
"  -c, --clientid=CLIENTID         Client identifier\n"
"  -H, --hostname=HOSTNAME         Client hostname\n"
"  -h                              Alias for -H\n"
"  -f, --foreground                Do not fork after getting lease\n"
"  -b, --background                Fork to background if lease cannot be\n"
"                                  immediately negotiated.\n"
"  -i, --interface=INTERFACE       Interface to use (default: eth0)\n"
"  -n, --now                       Exit with failure if lease cannot be\n"
"                                  immediately negotiated.\n"
"  -p, --pidfile=file              Store process ID of daemon in file\n"
"  -q, --quit                      Quit after obtaining lease\n"
"  -r, --request=IP                IP address to request (default: none)\n"
"  -s, --script=file               Run file at dhcp events (default:\n"
"                                  " DEFAULT_SCRIPT ")\n"
"  -v, --version                   Display version\n"
	);
	exit(0);
}
#endif

void write_dhcpc_msg(char *msg)
{
  int i;
  FILE *dhcp_fp=NULL;
	char cat_data[128]="";
	
  log_dhcpc_cnt=atoi(nvram_safe_get("Dhcp_Log_count"));
  if(log_dhcpc_cnt >= LOG_NUM)
	     {
      	  sprintf(cat_data, "sed -i '/^$/d' %s ; sed -i '1d' %s", DHCPLOG, DHCPLOG);
          system(cat_data);
          log_dhcpc_cnt--;
          sprintf(cat_data, "%d", log_dhcpc_cnt);
          nvram_set("Dhcp_Log_count", cat_data);  
	      }
	
	dhcp_fp = fopen (DHCPLOG,"a");
	if( dhcp_fp == NULL ){
		fprintf(stderr,"open %s error !!! \n",DHCPLOG);
		return ;
	}
	else
	 { 		    
          fprintf(dhcp_fp, "%s\n", msg);
          fclose(dhcp_fp);  
          log_dhcpc_cnt++;
          sprintf(cat_data, "%d", log_dhcpc_cnt);
          nvram_set("Dhcp_Log_count", cat_data);                 
   }
   return;
}


/* just a little helper */
static void change_mode(int new_mode)
{
	DEBUG(LOG_INFO, "entering %s listen mode",
		new_mode ? (new_mode == 1 ? "kernel" : "raw") : "none");
	close(fd);
	fd = -1;
	listen_mode = new_mode;
}


/* perform a renew */
static void perform_renew(void)
{
	FILE *fp;
	clock_t t;
	time(&t);
#ifdef CONFIG_LOG	
  fp = fopen (GTK_LOG,"a");
  fprintf(fp ,"Renewing Wan Connection => %s\n", ctime(&t));
  fclose(fp);
#endif
	LOG(LOG_INFO, "Performing a DHCP renew");
	switch (state) {
	case BOUND:
		change_mode(LISTEN_KERNEL);
		//state = RENEW_REQUESTED;	//GMTK add for coverity 20090901
                //break;				//GMTK add for coverity 20090901
	case RENEWING:
	case REBINDING:
		state = RENEW_REQUESTED;
		break;
	case RENEW_REQUESTED: /* impatient are we? fine, square 1 */
		run_script(NULL, "deconfig");
		//change_mode(LISTEN_RAW);	//GMTK add for coverity 20090901
                //state = INIT_SELECTING;		//GMTK add for coverity 20090901
                //break;
	case REQUESTING:
	case RELEASED:
		change_mode(LISTEN_RAW);
		state = INIT_SELECTING;
		break;
	case INIT_SELECTING:
		break;
	}

	/* start things over */
	packet_num = 0;

	/* Kill any timeouts because the user wants this to hurry along */
	timeout = 0;
}


/* perform a release */
static void perform_release(void)
{
	char buffer[16];
	struct in_addr temp_addr;
	FILE *fp;
	clock_t t;
	time(&t);
#ifdef CONFIG_LOG	
  fp = fopen (GTK_LOG,"a");
  fprintf(fp ,"Wan IP Address Expired => %s\n", ctime(&t));
  fclose(fp);
#endif

	/* send release packet */
	if (state == BOUND || state == RENEWING || state == REBINDING) {
		temp_addr.s_addr = server_addr;
		sprintf(buffer, "%s", inet_ntoa(temp_addr));
		temp_addr.s_addr = requested_ip;
		LOG(LOG_INFO, "Unicasting a release of %s to %s", 
				inet_ntoa(temp_addr), buffer);
		send_release(server_addr, requested_ip); /* unicast */
		run_script(NULL, "deconfig");
	}
	LOG(LOG_INFO, "Entering released state");

	change_mode(LISTEN_NONE);
	state = RELEASED;
	timeout = 0x7fffffff;
}


/* Exit and cleanup */
static void exit_client(int retval)
{
	char buf[10] = "";
	
	sprintf(buf, "%d", bootp_cnt_offer);
	nvram_set("bootp_cnt_offer", buf);
	//nvram_set("bootp_cnt_ack", bootp_cnt_ack);
	pidfile_delete(client_config.pidfile);
	CLOSE_LOG();
	exit(retval);
}


/* Signal handler */
static void signal_handler(int sig)
{
	if (write(signal_pipe[1], &sig, sizeof(sig)) < 0) {
		LOG(LOG_ERR, "Could not send signal: %s",
			strerror(errno));
	}
}


static void background(void)
{
	int pid_fd;

	pid_fd = pidfile_acquire(client_config.pidfile); /* hold lock during fork. */
	while (pid_fd >= 0 && pid_fd < 3) pid_fd = dup(pid_fd); /* don't let daemon close it */
	if (daemon(0, 0) == -1) {
		perror("fork");
		exit_client(1);
	}
	client_config.foreground = 1; /* Do not fork again. */
	pidfile_write_release(pid_fd);
}


#ifdef COMBINED_BINARY
int udhcpc_main(int argc, char *argv[])
#else
int main(int argc, char *argv[])
#endif
{
	unsigned char *temp, *message;
	unsigned long t1 = 0, t2 = 0, xid = 0;
	unsigned long start = 0, lease = 0;
	fd_set rfds;
	int retval;
	struct timeval tv;
	int c, len;
	struct dhcpMessage packet;
	struct in_addr temp_addr;
	int pid_fd;
	time_t now;
	int max_fd;
	int sig;
	FILE *fp;
	clock_t t;
  time(&t);
  memset(dhcpc_buf, 0, 512);

	static struct option arg_options[] = {
		{"clientid",	required_argument,	0, 'c'},
		{"foreground",	no_argument,		0, 'f'},
		{"background",	no_argument,		0, 'b'},
		{"hostname",	required_argument,	0, 'H'},
		{"hostname",    required_argument,      0, 'h'},
		{"interface",	required_argument,	0, 'i'},
		{"now", 	no_argument,		0, 'n'},
		{"pidfile",	required_argument,	0, 'p'},
		{"quit",	no_argument,		0, 'q'},
		{"request",	required_argument,	0, 'r'},
		{"script",	required_argument,	0, 's'},
		{"version",	no_argument,		0, 'v'},
		{"help",	no_argument,		0, '?'},
		{0, 0, 0, 0}
	};

	/* get options */
	while (1) {
		int option_index = 0;
		c = getopt_long(argc, argv, "c:fbH:h:i:np:qr:s:v", arg_options, &option_index);
		if (c == -1) break;
		
		switch (c) {
		case 'c':
			len = strlen(optarg) > 255 ? 255 : strlen(optarg);
			if (client_config.clientid) free(client_config.clientid);
			client_config.clientid = xmalloc(len + 2);
			client_config.clientid[OPT_CODE] = DHCP_CLIENT_ID;
			client_config.clientid[OPT_LEN] = len;
			client_config.clientid[OPT_DATA] = '\0';
			strncpy(client_config.clientid + OPT_DATA, optarg, len);
			break;
		case 'f':
			client_config.foreground = 1;
			break;
		case 'b':
			client_config.background_if_no_lease = 1;
			break;
		case 'h':
		case 'H':
			len = strlen(optarg) > 255 ? 255 : strlen(optarg);
			if (client_config.hostname) free(client_config.hostname);
			client_config.hostname = xmalloc(len + 2);
			client_config.hostname[OPT_CODE] = DHCP_HOST_NAME;
			client_config.hostname[OPT_LEN] = len;
			strncpy(client_config.hostname + 2, optarg, len);
			break;
		case 'i':
			client_config.interface =  optarg;
			break;
		case 'n':
			client_config.abort_if_no_lease = 1;
			break;
		case 'p':
			client_config.pidfile = optarg;
			break;
		case 'q':
			client_config.quit_after_lease = 1;
			break;
		case 'r':
			requested_ip = inet_addr(optarg);
			break;
		case 's':
			client_config.script = optarg;
			break;
		case 'v':
			printf("udhcpcd, version %s\n\n", VERSION);
			exit_client(0);
			break;
		default:
			show_usage();
		}
	}
	


	if(nvram_get("bootp_cnt_offer") == NULL)
	{
		bootp_cnt_offer = 0;
	}
	else
	{
		bootp_cnt_offer = atoi(nvram_get("bootp_cnt_offer"));
	}

//	if(nvram_get("bootp_cnt_ack") == NULL)
//	{
//		bootp_cnt_ack = 0;
//	}
//	else
//	{
//		bootp_cnt_ack = atoi(nvram_get("bootp_cnt_ack"));
//	}

	OPEN_LOG("udhcpc");
	LOG(LOG_INFO, "udhcp client (v%s) started", VERSION);

	pid_fd = pidfile_acquire(client_config.pidfile);
	pidfile_write_release(pid_fd);

	if (read_interface(client_config.interface, &client_config.ifindex, 
			   NULL, client_config.arp) < 0)
		exit_client(1);
		
	if (!client_config.clientid) {
		client_config.clientid = xmalloc(6 + 3);
		client_config.clientid[OPT_CODE] = DHCP_CLIENT_ID;
		client_config.clientid[OPT_LEN] = 7;
		client_config.clientid[OPT_DATA] = 1;
		memcpy(client_config.clientid + 3, client_config.arp, 6);
	}

	/* ensure that stdin/stdout/stderr are never returned by pipe() */
	if (fcntl(STDIN_FILENO, F_GETFL) == -1)
		(void) open("/dev/null", O_RDONLY);
	if (fcntl(STDOUT_FILENO, F_GETFL) == -1)
		(void) open("/dev/null", O_WRONLY);
	if (fcntl(STDERR_FILENO, F_GETFL) == -1)
		(void) open("/dev/null", O_WRONLY);	

	/* setup signal handlers */
	if (pipe(signal_pipe) != 0)
		fprintf(stderr,"Can't create pipe\n");
	signal(SIGUSR1, signal_handler);
	signal(SIGUSR2, signal_handler);
	signal(SIGTERM, signal_handler);
	
	state = INIT_SELECTING;
	run_script(NULL, "deconfig");
	change_mode(LISTEN_RAW);

	for (;;) 
	{
#ifdef CONFIG_SYS_UPTIME		
		sysinfo(&info);
		tv.tv_sec = timeout - info.uptime;
#else		*/
		tv.tv_sec = timeout - time(0);
#endif	
		tv.tv_usec = 0;
		FD_ZERO(&rfds);

		if (listen_mode != LISTEN_NONE && fd < 0) {
			if (listen_mode == LISTEN_KERNEL)
				fd = listen_socket(INADDR_ANY, CLIENT_PORT, client_config.interface);
			else
				fd = raw_socket(client_config.ifindex);
			if (fd < 0) {
				LOG(LOG_ERR, "FATAL: couldn't listen on socket, %s", strerror(errno));
            sprintf(dhcpc_buf, "couldn't listen on socket => %s", ctime(&t));
				    write_dhcpc_msg(dhcpc_buf);
				exit_client(0);
			}
		}
		if (fd >= 0) 
			FD_SET(fd, &rfds);
		FD_SET(signal_pipe[0], &rfds);		

		if (tv.tv_sec > 0) {
			DEBUG(LOG_INFO, "Waiting on select...\n");
			max_fd = signal_pipe[0] > fd ? signal_pipe[0] : fd;
			retval = select(max_fd + 1, &rfds, NULL, NULL, &tv);
		} 
		else 
			retval = 0; /* If we already timed out, fall through */
#ifdef CONFIG_SYS_UPTIME
		sysinfo(&info);
		now = info.uptime;
#else
		now = time(0);
#endif
		if (retval == 0) {
			/* timeout dropped to zero */
			switch (state) {
			case INIT_SELECTING:
				if (packet_num < 3) {
					if (packet_num == 0)
						xid = random_xid();

					/* send discover packet */
					send_discover(xid, requested_ip); /* broadcast */
					printf("Send DHCP DISCOVER !!\n");
					system("date > /dev/console");
					int ii=0;
					for(ii=1 ; ii<100 ; ii++){}
					send_discover(xid, requested_ip);
					printf("Send DHCP DISCOVER !!\n");
					system("date > /dev/console");
					for (ii=1 ; ii<100 ; ii++){}
					send_discover(xid, requested_ip);
					printf("Send DHCP DISCOVER !!\n");
					system("date > /dev/console");
					for (ii=1 ; ii<100 ; ii++){}
					send_discover(xid, requested_ip);
					printf("Send DHCP DISCOVER !!\n");
					system("date > /dev/console");
#ifdef CONFIG_LOG
					fp = fopen (GTK_LOG,"a");
  				fprintf(fp ,"Send Discover Packet => %s\n", ctime(&t));
  				fclose(fp);
#endif        

            sprintf(dhcpc_buf, "Send Discover Packet. => %s", ctime(&t));
  				  write_dhcpc_msg(dhcpc_buf);
                       				  
					timeout = now + ((packet_num == 2) ? 4 : 2);
					packet_num++;
#ifdef WL_5VT
					/* Never leave init state if not lease */
					packet_num = 2;
#endif
				} else {
					if (client_config.background_if_no_lease) {
						LOG(LOG_INFO, "No lease, forking to background.");

					  sprintf(dhcpc_buf, "No lease, forking to background. => %s", ctime(&t));
				    write_dhcpc_msg(dhcpc_buf);

				    
						background();
					} else if (client_config.abort_if_no_lease) {
						LOG(LOG_INFO, "No lease, failing.");
            				    

						sprintf(dhcpc_buf, "No lease, failing. => %s", ctime(&t));
				    write_dhcpc_msg(dhcpc_buf);
				    
						exit_client(1);
				  	}
					/* wait to try again */
					packet_num = 0;
					timeout = now + 60;
				}
				break;
			case RENEW_REQUESTED:
			case REQUESTING:
				if (packet_num < 3) {
					/* send request packet */
					if (state == RENEW_REQUESTED)
						send_renew(xid, server_addr, requested_ip); /* unicast */
					else send_selecting(xid, server_addr, requested_ip); /* broadcast */
#ifdef CONFIG_LOG
					fp = fopen (GTK_LOG,"a");
  				fprintf(fp ,"Send Request Packet => %s\n", ctime(&t));
  				fclose(fp);
#endif

          sprintf(dhcpc_buf, "Send Request Packet. => %s", ctime(&t));
				  write_dhcpc_msg(dhcpc_buf);
				  
					timeout = now + ((packet_num == 2) ? 10 : 2);
					packet_num++;
				} else {
					/* timed out, go back to init state */
					if (state == RENEW_REQUESTED) run_script(NULL, "deconfig");
					state = INIT_SELECTING;
					timeout = now;
					packet_num = 0;
					change_mode(LISTEN_RAW);
				}
				break;
			case BOUND:
				/* Lease is starting to run out, time to enter renewing state */
				state = RENEWING;
				change_mode(LISTEN_KERNEL);
				DEBUG(LOG_INFO, "Entering renew state");
				
          	sprintf(dhcpc_buf, "Entering renew state. => %s", ctime(&t));
    				write_dhcpc_msg(dhcpc_buf);
				
				/* fall right through */
			case RENEWING:
				/* Either set a new T1, or enter REBINDING state */
				if ((t2 - t1) <= (lease / 14400 + 1)) {
					/* timed out, enter rebinding state */
					state = REBINDING;
					timeout = now + (t2 - t1);
					DEBUG(LOG_INFO, "Entering rebinding state");


        	sprintf(dhcpc_buf, "Entering rebinding state. => %s", ctime(&t));
				  write_dhcpc_msg(dhcpc_buf);
				  
				} else {
					/* send a request packet */
					send_renew(xid, server_addr, requested_ip); /* unicast */
					
					t1 = (t2 - t1) / 2 + t1;
					timeout = t1 + start;
				}
				break;
			case REBINDING:
				/* Either set a new T2, or enter INIT state */
				if ((lease - t2) <= (lease / 14400 + 1)) {
					/* timed out, enter init state */
					state = INIT_SELECTING;
					LOG(LOG_INFO, "Lease lost, entering init state");

					sprintf(dhcpc_buf, "Lease lost, entering init state. => %s", ctime(&t));
				  write_dhcpc_msg(dhcpc_buf);
					  
					run_script(NULL, "deconfig");
					timeout = now;
					packet_num = 0;
					change_mode(LISTEN_RAW);
				} else {
					/* send a request packet */
					send_renew(xid, 0, requested_ip); /* broadcast */

					t2 = (lease - t2) / 2 + t2;
					timeout = t2 + start;
				}
				break;
			case RELEASED:
				/* yah, I know, *you* say it would never happen */
				timeout = 0x7fffffff;
				break;
			}
		} else if (retval > 0 && listen_mode != LISTEN_NONE && FD_ISSET(fd, &rfds)) {
			/* a packet is ready, read it */
			
			if (listen_mode == LISTEN_KERNEL)
				len = get_packet(&packet, fd);
			else len = get_raw_packet(&packet, fd);
			
			if (len == -1 && errno != EINTR) {
				DEBUG(LOG_INFO, "error on read, %s, reopening socket", strerror(errno));
				change_mode(listen_mode); /* just close and reopen */
			}
			if (len < 0) continue;
			
			if (packet.xid != xid) {
				DEBUG(LOG_INFO, "Ignoring XID %lx (our xid is %lx)",
					(unsigned long) packet.xid, xid);
				continue;
			}
			
			if ((message = get_option(&packet, DHCP_MESSAGE_TYPE)) == NULL) {
				DEBUG(LOG_ERR, "couldnt get option from packet -- ignoring");
				continue;
			}
			
			switch (state) {
			case INIT_SELECTING:
				/* Must be a DHCPOFFER to one of our xid's */
				if (*message == DHCPOFFER) {
					
					//LOG(LOG_INFO, "udhcpc.c[%d] : [%d] [%d]\n", __LINE__, packet.sname[0], packet.file[0]);
					
					if(!memcmp(packet.chaddr, client_config.arp,6))
					{
						LOG(LOG_INFO, "Get DHCP OFFER packet!!!\n");
						system("date > /dev/console");
    					  sprintf(dhcpc_buf, "Get DHCP OFFER acket!!! => %s", ctime(&t));
    				    write_dhcpc_msg(dhcpc_buf);	

#ifdef CONFIG_LOG
						fp = fopen (GTK_LOG,"a");
  					fprintf(fp ,"Get DHCP OFFER Packet => %s\n", ctime(&t));
  					fclose(fp);
#endif
					}
					else
					{
						LOG(LOG_INFO, "Get DHCP OFFER packet --- not for us!!!\n");
					
    					  sprintf(dhcpc_buf, "Get DHCP OFFER packet --- not for us!!! => %s", ctime(&t));
    				    write_dhcpc_msg(dhcpc_buf);		

						break;
					}
#if 0					
					if((strlen(packet.sname) != 0) || (strlen(packet.file) != 0))
					{
/*
						if( packet.file[0] < 128 )
						{
								LOG(LOG_INFO, "Get BOOTP data...ignore\n");
								break;
						}		
*/
						bootp_cnt_offer++;	

						if(bootp_cnt_offer >= 2)
						{
							LOG(LOG_INFO, "DHCP Offer : Get BOOTP data for %d times. Pass!!!\n", bootp_cnt_offer);
    					    sprintf(dhcpc_buf, "DHCP Offer : Get BOOTP data for %d times. Pass!!! => %s", ctime(&t));
    				      write_dhcpc_msg(dhcpc_buf);
								
						}
						else
						{
							LOG(LOG_INFO, "DHCP Offer : Get BOOTP data for %d times. Ignore!!!\n", bootp_cnt_offer);							
    					    sprintf(dhcpc_buf, "DHCP Offer : Get BOOTP data for %d times. Ignore!!! => %s", ctime(&t));
    				      write_dhcpc_msg(dhcpc_buf);		
  
							
							timeout += 2;
							break;
						}
					}
#endif					
					bootp_cnt_offer = 0;
								
					if ((temp = get_option(&packet, DHCP_SERVER_ID))) {
						memcpy(&server_addr, temp, 4);
						xid = packet.xid;
						requested_ip = packet.yiaddr;
						
						/* enter requesting state */
						state = REQUESTING;
						timeout = now;
						packet_num = 0;
					} else {
						DEBUG(LOG_ERR, "No server ID in message");
					}
				}
				break;
			case RENEW_REQUESTED:
			case REQUESTING:
			case RENEWING:
			case REBINDING:
				if (*message == DHCPACK) {

					//LOG(LOG_INFO, "udhcpc.c[%d] : [%d] [d]\n", __LINE__, packet.sname[0], packet.file[0]);
					
					if(!memcmp(packet.chaddr, client_config.arp,6))
					{
						LOG(LOG_INFO, "Get DHCP ACK packet!!!\n");
                                                system("date > /dev/console");
    					  sprintf(dhcpc_buf, "Get DHCP ACK packet!!! => %s", ctime(&t));
    				    write_dhcpc_msg(dhcpc_buf);	

#ifdef CONFIG_LOG
						fp = fopen (GTK_LOG,"a");
  					fprintf(fp ,"Get DHCP ACK Packet => %s\n", ctime(&t));
  					fclose(fp);
#endif
					}
					else
					{

					  sprintf(dhcpc_buf, "Get DHCP ACK packet --- not for us!!! => %s", ctime(&t));
				    write_dhcpc_msg(dhcpc_buf);
         						
						LOG(LOG_INFO, "Get DHCP ACK packet --- not for us!!!\n");
						break;
					}					
					
//					if((strlen(packet.sname) != 0) || (strlen(packet.file) != 0))
//					{
///*						
//						if( packet.file[0] < 128 )
//						{
//								LOG(LOG_INFO, "Get BOOTP data...ignore\n");
//								break;
//						}		
//*/
//
//						bootp_cnt_ack++;	
//						
//						if(bootp_cnt_ack == 3)
//						{
//							LOG(LOG_INFO, "DHCP ACK : Get BOOTP data for 3 times. Pass!!!\n");
//						}
//						else
//						{
//							LOG(LOG_INFO, "DHCP ACK : Get BOOTP data for %d times. Ignore!!!\n", bootp_cnt_ack);
//							timeout += 2;
//							break;
//						}						
//					}
//					
//					bootp_cnt_ack = 0;
					
					if (!(temp = get_option(&packet, DHCP_LEASE_TIME))) {
						LOG(LOG_ERR, "No lease time with ACK, using 1 hour lease");
						
					  sprintf(dhcpc_buf, "No lease time with ACK, using 1 hour lease => %s", ctime(&t));
				    write_dhcpc_msg(dhcpc_buf);	
          						
						lease = 60 * 60;
					} else {
						memcpy(&lease, temp, 4);
						lease = ntohl(lease);
					}
						
					/* enter bound state */
					t1 = lease / 2;
					
					/* little fixed point for n * .875 */
					t2 = (lease * 0x7) >> 3;
					temp_addr.s_addr = packet.yiaddr;
					LOG(LOG_INFO, "Lease of %s obtained, lease time %ld", 
						inet_ntoa(temp_addr), lease);
					
					sprintf(dhcpc_buf, "Lease of %s obtained => %s",inet_ntoa(temp_addr), ctime(&t));
				  write_dhcpc_msg(dhcpc_buf);	
          						
					start = now;
					timeout = t1 + start;
					requested_ip = packet.yiaddr;
					run_script(&packet,
						   ((state == RENEWING || state == REBINDING) ? "renew" : "bound"));

					state = BOUND;
					change_mode(LISTEN_NONE);
					if (client_config.quit_after_lease) 
						exit_client(0);
					if (!client_config.foreground)
						background();

				} else if (*message == DHCPNAK) {
					/* return to init state */
					LOG(LOG_INFO, "Received DHCP NAK");
					
  					sprintf(dhcpc_buf, "Received DHCP NAK => %s", ctime(&t));
  				  write_dhcpc_msg(dhcpc_buf);	


#ifdef CONFIG_LOG
					fp = fopen (GTK_LOG,"a");
  				fprintf(fp ,"Received DHCP NAK => %s\n", ctime(&t));
  				fclose(fp);
#endif
					run_script(&packet, "nak");
					if (state != REQUESTING)
						run_script(NULL, "deconfig");
					state = INIT_SELECTING;
					timeout = now;
					requested_ip = 0;
					packet_num = 0;
					change_mode(LISTEN_RAW);
					sleep(3); /* avoid excessive network traffic */
				}
				break;
			/* case BOUND, RELEASED: - ignore all packets */
			}	
		} else if (retval > 0 && FD_ISSET(signal_pipe[0], &rfds)) {
			if (read(signal_pipe[0], &sig, sizeof(sig)) < 0) {
				DEBUG(LOG_ERR, "Could not read signal: %s", 
					strerror(errno));
				continue; /* probably just EINTR */
			}
			switch (sig) {
			case SIGUSR1: 
				perform_renew();
				break;
			case SIGUSR2:
				perform_release();
				break;
			case SIGTERM:
				//	perform_release();
				LOG(LOG_INFO, "Received SIGTERM");

    				sprintf(dhcpc_buf, "Received SIGTERM => %s", ctime(&t));
    				write_dhcpc_msg(dhcpc_buf);	      
				
				exit_client(0);
				//	break;
				//default:
				//	break;
			}
		} else if (retval == -1 && errno == EINTR) {
			/* a signal was caught */		
		} else {
			/* An error occured */
			DEBUG(LOG_ERR, "Error on select");
		}
	}
	return 0;
}






