/*
<:copyright-gpl
 Copyright 2010 Broadcom Corp. All Rights Reserved.

 This program is free software; you can distribute it and/or modify it
 under the terms of the GNU General Public License (Version 2) as
 published by the Free Software Foundation.

 This program is distributed in the hope it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 for more details.

 You should have received a copy of the GNU General Public License along
 with this program; if not, write to the Free Software Foundation, Inc.,
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
:>
*/

#include <linux/module.h>
#include <linux/udp.h>
#include <linux/ip.h>

#include <net/netfilter/nf_nat.h>
#include <net/netfilter/nf_nat_helper.h>
#include <net/netfilter/nf_nat_rule.h>
#include <net/netfilter/nf_conntrack_helper.h>
#include <net/netfilter/nf_conntrack_expect.h>
#include <linux/netfilter/nf_conntrack_ipsec.h>
#include <net/netfilter/nf_conntrack_core.h>
#include <net/ip.h>

MODULE_AUTHOR("Pavan Kumar <pavank@broadcom.com>");
MODULE_DESCRIPTION("Netfilter connection tracking module for ipsec");
MODULE_LICENSE("GPL");
MODULE_ALIAS("nf_nat_ipsec");

#if 0
#define DEBUGP(format, args...) printk(KERN_DEBUG "%s:%s: " format, __FILE__, \
				       __FUNCTION__, ## args)
#else
#define DEBUGP(format, args...)
#endif


/* outbound packets == from LAN to WAN */
static int
ipsec_outbound_pkt(struct sk_buff *skb,
                   struct nf_conn *ct, enum ip_conntrack_info ctinfo)

{
   struct iphdr *iph = ip_hdr(skb);
   struct udphdr *udph = (void *)iph + iph->ihl * 4;

#if defined(ARCADYAN)
   /*
       what we do now is
       1.when sending, change the source port, that's here
       2.when receiving, change the destination port, that's in xt_ipsec_target
       what 1 doing is to keep IPSec source port to 500, what 2 doing is to make
       system can find correct conntrack
   */
   if (ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple.src.u.udp.port == IPSEC_PORT)
       udph->source = htons(IPSEC_PORT);
#else
   /* make sure source port is 500 */
   udph->source = htons(IPSEC_PORT);
#endif
   udph->check = 0;
   
   return NF_ACCEPT;
}


/* inbound packets == from WAN to LAN */
static int
ipsec_inbound_pkt(struct sk_buff *skb, struct nf_conn *ct,
                  enum ip_conntrack_info ctinfo, __be32 lan_ip)
{
   struct iphdr *iph = ip_hdr(skb);
   struct udphdr *udph = (void *)iph + iph->ihl * 4;

   iph->daddr = lan_ip;
   udph->check = 0;
   iph->check = 0;
   iph->check = ip_fast_csum((unsigned char *)iph, iph->ihl);
   
   return NF_ACCEPT;
}

#if defined(ARCADYAN) && defined(CONFIG_IP_NF_RAW_MODULE)
static unsigned int
xt_ipsec_target(struct sk_buff *skb, const struct xt_target_param *par)
{
	__be32 ip;
	__be16 port;
	struct iphdr *iph = ip_hdr(skb);
	struct udphdr *udph = NULL;

	if (iph->protocol != IPPROTO_UDP)
		return XT_CONTINUE;
	udph = (void *)iph + iph->ihl * 4;
	if ((udph->dest != htons(IPSEC_PORT)) || (udph->source != htons(IPSEC_PORT)))
		return XT_CONTINUE;

	if (get_ipsec_src(skb, &ip, &port)) /* can not found related ipsec entry, just return */
		return XT_CONTINUE;
	/* change destination */
	udph->dest = port;
	udph->check = 0;
	skb->csum = skb_checksum(skb, ip_hdrlen(skb), skb->len - ip_hdrlen(skb), 0);
	udph->check = csum_tcpudp_magic(iph->saddr,
							iph->daddr,
							skb->len - ip_hdrlen(skb),
							IPPROTO_UDP,
							skb->csum);
	iph->check = 0;
	iph->check = ip_fast_csum((unsigned char *)iph, iph->ihl);
	return NF_ACCEPT;
}

static struct xt_target xt_ipsec_reg __read_mostly = {
	.name		= "IPSEC",
	.target		= xt_ipsec_target,
	.table		= "raw",
	.hooks		= (1 << NF_INET_PRE_ROUTING),
	.family		= AF_INET,
};
#endif

static int __init nf_nat_helper_ipsec_init(void)
{
   BUG_ON(rcu_dereference(nf_nat_ipsec_hook_outbound));
   rcu_assign_pointer(nf_nat_ipsec_hook_outbound, ipsec_outbound_pkt);

   BUG_ON(rcu_dereference(nf_nat_ipsec_hook_inbound));
   rcu_assign_pointer(nf_nat_ipsec_hook_inbound, ipsec_inbound_pkt);

#if defined(ARCADYAN) && defined(CONFIG_IP_NF_RAW_MODULE)
   xt_register_target(&xt_ipsec_reg);
#endif

   return 0;
}

static void __exit nf_nat_helper_ipsec_fini(void)
{
#if defined(ARCADYAN) && defined(CONFIG_IP_NF_RAW_MODULE)
   xt_unregister_target(&xt_ipsec_reg);
#endif

   rcu_assign_pointer(nf_nat_ipsec_hook_inbound, NULL);
   rcu_assign_pointer(nf_nat_ipsec_hook_outbound, NULL);
   synchronize_rcu();
}

module_init(nf_nat_helper_ipsec_init);
module_exit(nf_nat_helper_ipsec_fini);
