/*
 *	Forwarding database
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/rculist.h>
#include <linux/spinlock.h>
#include <linux/times.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/jhash.h>
#include <linux/random.h>
#include <asm/atomic.h>
#include <asm/unaligned.h>
#include "br_private.h"
#if defined(CONFIG_BRCM_KERNEL_ENHANCE) && defined(CONFIG_BR_IGMP_SNOOP)
#include "br_igmp.h"
#endif /* CONFIG_BRCM_KERNEL_ENHANCE && CONFIG_BR_IGMP_SNOOP */
#if defined(CONFIG_BRCM_KERNEL_ENHANCE) && defined(CONFIG_BLOG)
#include <linux/blog.h>
#endif /* CONFIG_BRCM_KERNEL_ENHANCE && CONFIG_BLOG */
#ifdef CONFIG_BR_FP
#include "br_fp.h"
#include "br_fp_hooks.h"
#endif /* CONFIG_BR_FP */
#if defined(CONFIG_BCM_CTF) || defined(CONFIG_BCM_CTF_MODULE)
#include <linux/if.h>
#include <linux/if_vlan.h>

#include <typedefs.h>
#include <osl.h>
#include <ctf/hndctf.h>
#endif /* CONFIG_BCM_CTF || CONFIG_BCM_CTF_MODULE */

#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
int (*fdb_check_expired_wl_hook)(unsigned char *addr) = NULL;
EXPORT_SYMBOL(fdb_check_expired_wl_hook);

int (*fdb_clear_enet_arl_hook)(void *ptr) = NULL;
EXPORT_SYMBOL(fdb_clear_enet_arl_hook);
#endif

static struct kmem_cache *br_fdb_cache __read_mostly;
static int fdb_insert(struct net_bridge *br, struct net_bridge_port *source,
		      const unsigned char *addr);

static u32 fdb_salt __read_mostly;

#if defined(CONFIG_BCM_CTF) || defined(CONFIG_BCM_CTF_MODULE)
static void bcm_ctf_br_cache_init(ctf_brc_t *brc, unsigned char *ea, struct net_device *rxdev)
{
	memset(brc, 0, sizeof(ctf_brc_t));

	memcpy(brc->dhost.octet, ea, ETH_ALEN);

        if (rxdev->priv_flags & IFF_802_1Q_VLAN) {
		brc->txifp = (void *)vlan_dev_real_dev(rxdev);
		brc->vid = vlan_dev_vlan_id(rxdev);
		brc->action = (vlan_dev_flags(rxdev) & 1) ? CTF_ACTION_TAG : CTF_ACTION_UNTAG;
	} else {
		brc->txifp = (void *)rxdev;
		brc->action = CTF_ACTION_UNTAG;
	}

#ifdef BCM_CTF_DEBUG
	printk("mac %02x:%02x:%02x:%02x:%02x:%02x\n",
	       brc->dhost.octet[0], brc->dhost.octet[1],
	       brc->dhost.octet[2], brc->dhost.octet[3],
	       brc->dhost.octet[4], brc->dhost.octet[5]);
	printk("vid: %d action %x\n", brc->vid, brc->action);
	printk("txif: %s\n", ((struct net_device *)brc->txifp)->name);
#endif
}

void
bcm_ctf_br_cache_add(unsigned char *ea, struct net_device *rxdev)
{
	ctf_brc_t brc_entry;

	/* Add brc entry only if packet is received on ctf 
	 * enabled interface
	 */
	if (!ctf_isenabled(kcih, ((rxdev->priv_flags & IFF_802_1Q_VLAN) ?
	                   vlan_dev_real_dev(rxdev) : rxdev)))
		return;

	bcm_ctf_br_cache_init(&brc_entry, ea, rxdev);

#ifdef BCM_CTF_DEBUG
	printk("%s: Adding bridge cache entry\n", __FUNCTION__);
#endif

	/* Add the bridge cache entry */
	if (ctf_brc_lkup(kcih, ea) == NULL)
		ctf_brc_add(kcih, &brc_entry);
	else
		ctf_brc_update(kcih, &brc_entry);
}

void
bcm_ctf_br_cache_update(unsigned char *ea, struct net_device *rxdev)
{
	ctf_brc_t brc_entry;

	/* Update brc entry only if packet is received on ctf 
	 * enabled interface
	 */
	if (!ctf_isenabled(kcih, ((rxdev->priv_flags & IFF_802_1Q_VLAN) ?
	                   vlan_dev_real_dev(rxdev) : rxdev)))
		return;

	/* Initialize the new device and/or vlan info */
	bcm_ctf_br_cache_init(&brc_entry, ea, rxdev);

#ifdef BCM_CTF_DEBUG
	printk("%s: Updating CTF bridge cache entry\n", __FUNCTION__);
#endif

	/* Update the bridge cache entry */
	ctf_brc_update(kcih, &brc_entry);

	return;
}
#endif /* CONFIG_BCM_CTF || CONFIG_BCM_CTF_MODULE */

int __init br_fdb_init(void)
{
	br_fdb_cache = kmem_cache_create("bridge_fdb_cache",
					 sizeof(struct net_bridge_fdb_entry),
					 0,
					 SLAB_HWCACHE_ALIGN, NULL);
	if (!br_fdb_cache)
		return -ENOMEM;

	get_random_bytes(&fdb_salt, sizeof(fdb_salt));
	return 0;
}

void br_fdb_fini(void)
{
	kmem_cache_destroy(br_fdb_cache);
}


/* if topology_changing then use forward_delay (default 15 sec)
 * otherwise keep longer (default 5 minutes)
 */
static inline unsigned long hold_time(const struct net_bridge *br)
{
#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
	/* Seems one timer constant in bridge code can serve several different purposes. As we use forward_delay=0,
	if the code left unchanged, every entry in fdb will expire immidately after a topology change and every packet
	will flood the local ports for a period of bridge_max_age. This will result in low throughput after boot up. 
	So we decoulpe this timer from forward_delay. */
	return br->topology_change ? (15*HZ) : br->ageing_time;
#else
	return br->topology_change ? br->forward_delay : br->ageing_time;
#endif
}

static inline int has_expired(const struct net_bridge *br,
				  const struct net_bridge_fdb_entry *fdb)
{
	return !fdb->is_static 
		&& time_before_eq(fdb->ageing_timer + hold_time(br), jiffies);
}

static inline int br_mac_hash(const unsigned char *mac)
{
	/* use 1 byte of OUI cnd 3 bytes of NIC */
	u32 key = get_unaligned((u32 *)(mac + 2));
	return jhash_1word(key, fdb_salt) & (BR_HASH_SIZE - 1);
}

#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
static inline void fdb_delete(struct net_bridge *br, 
	                          struct net_bridge_fdb_entry *f)
#else
static inline void fdb_delete(struct net_bridge_fdb_entry *f)
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
{
#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
	br->num_fdb_entries--;
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
	hlist_del_rcu(&f->hlist);
#if defined(CONFIG_BRCM_KERNEL_ENHANCE) && defined(CONFIG_BLOG)
	blog_notify(DESTROY_BRIDGEFDB, (void*)f, 0, 0);
#endif /* CONFIG_BRCM_KERNEL_ENHANCE && CONFIG_BLOG */
	br_fdb_put(f);
}

void br_fdb_changeaddr(struct net_bridge_port *p, const unsigned char *newaddr)
{
	struct net_bridge *br = p->br;
	int i;
	
	spin_lock_bh(&br->hash_lock);

	/* Search all chains since old address/hash is unknown */
	for (i = 0; i < BR_HASH_SIZE; i++) {
		struct hlist_node *h;
		hlist_for_each(h, &br->hash[i]) {
			struct net_bridge_fdb_entry *f;

			f = hlist_entry(h, struct net_bridge_fdb_entry, hlist);
			if (f->dst == p && f->is_local) {
				/* maybe another port has same hw addr? */
				struct net_bridge_port *op;
				list_for_each_entry(op, &br->port_list, list) {
					if (op != p && 
					    !compare_ether_addr(op->dev->dev_addr,
								f->addr.addr)) {
						f->dst = op;
						goto insert;
					}
				}

				/* delete old one */
#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
				fdb_delete(br, f);
#else
				fdb_delete(f);
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
				goto insert;
			}
		}
	}
 insert:
	/* insert new address,  may fail if invalid address or dup. */
	fdb_insert(br, p, newaddr);

	spin_unlock_bh(&br->hash_lock);
}

void br_fdb_cleanup(unsigned long _data)
{
	struct net_bridge *br = (struct net_bridge *)_data;
	unsigned long delay = hold_time(br);
	unsigned long next_timer = jiffies + br->forward_delay;
	int i;

	spin_lock_bh(&br->hash_lock);
	for (i = 0; i < BR_HASH_SIZE; i++) {
		struct net_bridge_fdb_entry *f;
		struct hlist_node *h, *n;

		hlist_for_each_entry_safe(f, h, n, &br->hash[i], hlist) {
			unsigned long this_timer;
			if (f->is_static)
				continue;
			this_timer = f->ageing_timer + delay;
			if (time_before_eq(this_timer, jiffies))
#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
#ifdef CONFIG_BR_FP
			{
				u8 flag = 0;
				BR_FP_HOOK_TIMERS(f->dst, f->addr.addr, &flag, BR_FP_FDB_CHECK_AGE);
				if (flag) {
					f->ageing_timer = jiffies;
				} else {
					fdb_delete(br, f);
				}
			}
#elif defined(CONFIG_BCM_CTF) || defined(CONFIG_BCM_CTF_MODULE)
			{
				ctf_brc_t *brcp;

				/* Before expiring the fdb entry check the brc
				 * live counter to make sure there are no frames
				 * on this connection for timeout period.
				 */
				brcp = ctf_brc_lkup(kcih, f->addr.addr);
				if ((brcp != NULL) && (brcp->live > 0)) {
					brcp->live = 0;
					f->ageing_timer = jiffies;
					continue;
				}

				fdb_delete(br, f);
			}
#else
			{
				if (fdb_check_expired_wl_hook && (fdb_check_expired_wl_hook(f->addr.addr) == 0))
					f->ageing_timer = jiffies;  /* refresh */
				else
					fdb_delete(br, f);
			}
#endif /* CONFIG_BR_FP */
#else
				fdb_delete(f);
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
			else if (time_before(this_timer, next_timer))
				next_timer = this_timer;
		}
	}
	spin_unlock_bh(&br->hash_lock);

	/* Add HZ/4 to ensure we round the jiffies upwards to be after the next
	 * timer, otherwise we might round down and will have no-op run. */
	mod_timer(&br->gc_timer, round_jiffies(next_timer + HZ/4));
}

/* Completely flush all dynamic entries in forwarding database.*/
void br_fdb_flush(struct net_bridge *br)
{
	int i;

	spin_lock_bh(&br->hash_lock);
	for (i = 0; i < BR_HASH_SIZE; i++) {
		struct net_bridge_fdb_entry *f;
		struct hlist_node *h, *n;
		hlist_for_each_entry_safe(f, h, n, &br->hash[i], hlist) {
			if (!f->is_static)
#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
				fdb_delete(br, f);
#else
				fdb_delete(f);
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
		}
	}
	spin_unlock_bh(&br->hash_lock);
}

/* Flush all entries refering to a specific port.
 * if do_all is set also flush static entries
 */
void br_fdb_delete_by_port(struct net_bridge *br,
			   const struct net_bridge_port *p,
			   int do_all)
{
	int i;

	spin_lock_bh(&br->hash_lock);
	for (i = 0; i < BR_HASH_SIZE; i++) {
		struct hlist_node *h, *g;
		
		hlist_for_each_safe(h, g, &br->hash[i]) {
			struct net_bridge_fdb_entry *f
				= hlist_entry(h, struct net_bridge_fdb_entry, hlist);
			if (f->dst != p) 
				continue;

			if (f->is_static && !do_all)
				continue;
			/*
			 * if multiple ports all have the same device address
			 * then when one port is deleted, assign
			 * the local entry to other port
			 */
			if (f->is_local) {
				struct net_bridge_port *op;
				list_for_each_entry(op, &br->port_list, list) {
					if (op != p && 
					    !compare_ether_addr(op->dev->dev_addr,
								f->addr.addr)) {
						f->dst = op;
						goto skip_delete;
					}
				}
			}
#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
			fdb_delete(br, f);
#else
			fdb_delete(f);
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
		skip_delete: ;
		}
	}
	spin_unlock_bh(&br->hash_lock);
}

/* No locking or refcounting, assumes caller has no preempt (rcu_read_lock) */
struct net_bridge_fdb_entry *__br_fdb_get(struct net_bridge *br,
					  const unsigned char *addr)
{
	struct hlist_node *h;
	struct net_bridge_fdb_entry *fdb;

	hlist_for_each_entry_rcu(fdb, h, &br->hash[br_mac_hash(addr)], hlist) {
		if (!compare_ether_addr(fdb->addr.addr, addr)) {
			if (unlikely(has_expired(br, fdb)))
				break;
			return fdb;
		}
	}

	return NULL;
}

/* Interface used by ATM hook that keeps a ref count */
struct net_bridge_fdb_entry *br_fdb_get(struct net_bridge *br, 
					unsigned char *addr)
{
	struct net_bridge_fdb_entry *fdb;

	rcu_read_lock();
	fdb = __br_fdb_get(br, addr);
	if (fdb && !atomic_inc_not_zero(&fdb->use_count))
		fdb = NULL;
	rcu_read_unlock();
	return fdb;
}

#ifdef CONFIG_ARCADYAN_KERNEL_ENHANCE
struct net_device *br_fdb_get_device(struct net_device *dev,
					  const unsigned char *addr)
{
	struct net_bridge *br = netdev_priv(dev);
	struct hlist_node *h;
	struct net_bridge_fdb_entry *fdb;

	if (br == NULL)
		return NULL;

	hlist_for_each_entry_rcu(fdb, h, &br->hash[br_mac_hash(addr)], hlist) {
		if (!compare_ether_addr(fdb->addr.addr, addr)) {
			if (unlikely(has_expired(br, fdb)))
				break;
			return fdb->dst->dev;
		}
	}

	return NULL;
}
#endif /* CONFIG_ARCADYAN_KERNEL_ENHANCE */

static void fdb_rcu_free(struct rcu_head *head)
{
	struct net_bridge_fdb_entry *ent
		= container_of(head, struct net_bridge_fdb_entry, rcu);
	kmem_cache_free(br_fdb_cache, ent);
}

/* Set entry up for deletion with RCU  */
void br_fdb_put(struct net_bridge_fdb_entry *ent)
{
#ifdef CONFIG_BR_FP
    BR_FP_HOOK_FDB(ent->dst, ent->addr.addr, BR_FP_FDB_REMOVE);
#endif /* CONFIG_BR_FP */
	if (atomic_dec_and_test(&ent->use_count)) {
#if defined(CONFIG_BCM_CTF) || defined(CONFIG_BCM_CTF_MODULE)
		/* Delete the corresponding brc entry when it expires
		 * or deleted by user.
		 */
		ctf_brc_delete(kcih, ent->addr.addr);
#endif /* CONFIG_BCM_CTF || CONFIG_BCM_CTF_MODULE */
		call_rcu(&ent->rcu, fdb_rcu_free);
	}
}

/*
 * Fill buffer with forwarding table records in 
 * the API format.
 */
int br_fdb_fillbuf(struct net_bridge *br, void *buf,
		   unsigned long maxnum, unsigned long skip)
{
	struct __fdb_entry *fe = buf;
	int i, num = 0;
	struct hlist_node *h;
	struct net_bridge_fdb_entry *f;

	memset(buf, 0, maxnum*sizeof(struct __fdb_entry));

	rcu_read_lock();
	for (i = 0; i < BR_HASH_SIZE; i++) {
		hlist_for_each_entry_rcu(f, h, &br->hash[i], hlist) {
			if (num >= maxnum)
				goto out;

			if (has_expired(br, f)) 
				continue;

			if (skip) {
				--skip;
				continue;
			}

			/* convert from internal format to API */
			memcpy(fe->mac_addr, f->addr.addr, ETH_ALEN);

			/* due to ABI compat need to split into hi/lo */
			fe->port_no = f->dst->port_no;
			fe->port_hi = f->dst->port_no >> 8;

			fe->is_local = f->is_local;
			if (!f->is_static)
				fe->ageing_timer_value = jiffies_to_clock_t(jiffies - f->ageing_timer);
			++fe;
			++num;
		}
	}

 out:
	rcu_read_unlock();

	return num;
}

static inline struct net_bridge_fdb_entry *fdb_find(struct hlist_head *head,
						    const unsigned char *addr)
{
	struct hlist_node *h;
	struct net_bridge_fdb_entry *fdb;

	hlist_for_each_entry_rcu(fdb, h, head, hlist) {
		if (!compare_ether_addr(fdb->addr.addr, addr))
			return fdb;
	}
	return NULL;
}

#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
static struct net_bridge_fdb_entry *fdb_create(struct net_bridge *br, 
					       struct hlist_head *head,
					       struct net_bridge_port *source,
					       const unsigned char *addr,
					       int is_local,
					       int is_static)
#else
static struct net_bridge_fdb_entry *fdb_create(struct hlist_head *head,
					       struct net_bridge_port *source,
					       const unsigned char *addr,
					       int is_local)
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
{
	struct net_bridge_fdb_entry *fdb;

#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
	if(br->num_fdb_entries >= BR_MAX_FDB_ENTRIES)
		return NULL;

	/* some users want to always flood. */
	if (hold_time(br) == 0 && !is_local && !is_static)
		return NULL;
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */

	fdb = kmem_cache_alloc(br_fdb_cache, GFP_ATOMIC);
	if (fdb) {
		memcpy(fdb->addr.addr, addr, ETH_ALEN);
		atomic_set(&fdb->use_count, 1);
		hlist_add_head_rcu(&fdb->hlist, head);

		fdb->dst = source;
		fdb->is_local = is_local;
#ifdef CONFIG_BRCM_KERNEL_ENHANCE
		fdb->is_static = is_static;
#else
		fdb->is_static = is_local;
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
		fdb->ageing_timer = jiffies;
#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
		br->num_fdb_entries++;
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
#ifdef CONFIG_BR_FP
        BR_FP_HOOK_FDB(source, fdb->addr.addr, BR_FP_FDB_ADD);
#endif /* CONFIG_BR_FP */
#if defined(CONFIG_BCM_CTF) || defined(CONFIG_BCM_CTF_MODULE)
		/* Add bridge cache entry for non local hosts */
		if (!is_local && (source->state == BR_STATE_FORWARDING))
			bcm_ctf_br_cache_add((unsigned char *)addr, source->dev);
#endif /* CONFIG_BCM_CTF || CONFIG_BCM_CTF_MODULE */
	}
	return fdb;
}

#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
static int fdb_adddel_static(struct net_bridge *br,
                             struct net_bridge_port *source,
                             const unsigned char *addr, 
                             int addEntry)
{
	struct hlist_head *head;
	struct net_bridge_fdb_entry *fdb;

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	head = &br->hash[br_mac_hash(addr)];

	rcu_read_lock();
	fdb = fdb_find(head, addr);
	if (fdb)
	{
		/* if the entry exists and it is not static then we will delete it
		   and then add it back as static. If we are not adding an entry
		   then just delete it */
		if ( (0 == addEntry) || (0 == fdb->is_static) )
		{
			fdb_delete(br, fdb);
		}
	}
	rcu_read_unlock();
   
	if ( 1 == addEntry )
	{
		if (!fdb_create(br, head, source, addr, 0, 1))
			return -ENOMEM;
	}

	return 0;
}
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */

static int fdb_insert(struct net_bridge *br, struct net_bridge_port *source,
		  const unsigned char *addr)
{
	struct hlist_head *head = &br->hash[br_mac_hash(addr)];
	struct net_bridge_fdb_entry *fdb;

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	fdb = fdb_find(head, addr);
	if (fdb) {
		/* it is okay to have multiple ports with same
		 * address, just use the first one.
				 */
		if (fdb->is_local)
			return 0;

		printk(KERN_WARNING "%s adding interface with same address "
	      "as a received packet\n", source->dev->name);
#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
		fdb_delete(br, fdb);
#else
		fdb_delete(fdb);
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
	}

#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
	if (!fdb_create(br, head, source, addr, 1, 1))
#else
	if (!fdb_create(head, source, addr, 1))
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
		return -ENOMEM;

	return 0;
}

int br_fdb_insert(struct net_bridge *br, struct net_bridge_port *source,
		  const unsigned char *addr)
{
	int ret;

	spin_lock_bh(&br->hash_lock);
	ret = fdb_insert(br, source, addr);
	spin_unlock_bh(&br->hash_lock);
	return ret;
}

void br_fdb_update(struct net_bridge *br, struct net_bridge_port *source,
		   const unsigned char *addr)
{
	struct hlist_head *head = &br->hash[br_mac_hash(addr)];
	struct net_bridge_fdb_entry *fdb;

	/* some users want to always flood. */
	if (hold_time(br) == 0)
		return;

	/* ignore packets unless we are using this port */
	if (!(source->state == BR_STATE_LEARNING ||
	      source->state == BR_STATE_FORWARDING))
		return;

	fdb = fdb_find(head, addr);
	if (likely(fdb)) {
		/* attempt to update an entry for a local interface */
		if (unlikely(fdb->is_local)) {
			if (net_ratelimit())
				printk(KERN_WARNING "%s: received packet with "
				       " own address as source address\n",
				       source->dev->name);
		} else {
#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
            /* In case of MAC move - let ethernet driver clear switch ARL */
			if (fdb->dst && fdb->dst->port_no != source->port_no) {
#ifdef ARCADYAN
				if (strncmp(fdb->dst->dev->name, "eth", 3) == 0 &&
					strncmp(source->dev->name, "wl", 2) == 0 &&
					fdb_clear_enet_arl_hook) {
					fdb_clear_enet_arl_hook((void *)addr);
				}
#else
				bcmFun_t *ethswClearArlFun;
				/* Get the switch clear ARL function pointer */
				ethswClearArlFun = bcmFun_get(BCM_FUN_IN_ENET_CLEAR_ARL_ENTRY);
				if (ethswClearArlFun) {
					ethswClearArlFun((void*)addr);
				}
#endif /* ARCADYAN */
#if defined(CONFIG_BLOG)
				/* Also flush the associated entries in accelerators */
				blog_notify(DESTROY_BRIDGEFDB, (void*)fdb, 0, 0);
#endif
			}
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
#ifdef CONFIG_BR_FP
            BR_FP_HOOK_FDB(fdb->dst, fdb->addr.addr, BR_FP_FDB_MODIFY);
#endif /* CONFIG_BR_FP */
#if defined(CONFIG_BCM_CTF) || defined(CONFIG_BCM_CTF_MODULE)
			/* 
			 * update the brc entry incase the host moved from
			 * one bridge to another or to a different port under
			 * the same bridge.
			 */
			if (source->state == BR_STATE_FORWARDING)
				bcm_ctf_br_cache_update((unsigned char *)addr, source->dev);
#endif /* CONFIG_BCM_CTF || CONFIG_BCM_CTF_MODULE */
            /* fastpath: update of existing entry */
			fdb->dst = source;
			fdb->ageing_timer = jiffies;
		}
	} else {
		spin_lock(&br->hash_lock);
		if (!fdb_find(head, addr))
#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
			fdb_create(br, head, source, addr, 0, 0);
#else
			fdb_create(head, source, addr, 0);
#endif /* CONFIG_BRCM_KERNEL_ENHANCE */
		/* else  we lose race and someone else inserts
		 * it first, don't bother updating
		 */
		spin_unlock(&br->hash_lock);
	}
}

#if defined(CONFIG_BRCM_KERNEL_ENHANCE) && defined(CONFIG_BLOG)
extern void br_fdb_refresh( struct net_bridge_fdb_entry *fdb );
void br_fdb_refresh( struct net_bridge_fdb_entry *fdb )
{
	fdb->ageing_timer = jiffies;
	return;
}
#endif


#if defined(CONFIG_BRCM_KERNEL_ENHANCE)
int br_fdb_adddel_static(struct net_bridge *br, struct net_bridge_port *source,
                         const unsigned char *addr, int bInsert)
{
	int ret = 0;

	spin_lock_bh(&br->hash_lock);

	ret = fdb_adddel_static(br, source, addr, bInsert);

	spin_unlock_bh(&br->hash_lock);
   
	return ret;
}
#endif

