/*
<:copyright-gpl 
 Copyright 2013 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/bitops.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/cpumask.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/reboot.h>
#include <linux/types.h>
#include <linux/uaccess.h>
#include <linux/watchdog.h>
#include <linux/timer.h>
#include <linux/jiffies.h>

#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/clk.h>
#include <linux/err.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <linux/completion.h>

#include <bcm63268_io.h>
#include <bcm63268_regs.h>
#include <bcm63268_dev_nand.h>

//#define SMP_DEBUG

#define DRV_NAME		"bcm63268-wdt"
#define DRV_AUTHOR		"Arcadyan Technology"
#define DRV_DESC		"Broadcom on-chip Watchdog Driver"
#define DRV_VER			"1.0"

#define BCM63268_WDT_DEFAULT_TIME	60	/* seconds */
#define BCM63268_WDT_MAX_TIME		255	/* seconds */
#define BCM63268_WDT_UPDATE_TIMER	(HZ * 10)

#define BCM63268_WDT_CLOCK_RATE		50000000

static int bcm63268_wdt_time = BCM63268_WDT_DEFAULT_TIME;
static int bcm63268_wdt_nowayout = WATCHDOG_NOWAYOUT;

module_param(bcm63268_wdt_time, int, 0);
MODULE_PARM_DESC(bcm63268_wdt_time, "Watchdog time in seconds. (default="
				__MODULE_STRING(BCM63268_WDT_DEFAULT_TIME) ")");

#ifdef CONFIG_WATCHDOG_NOWAYOUT
module_param(bcm63268_wdt_nowayout, int, 0);
MODULE_PARM_DESC(bcm63268_wdt_nowayout,
		"Watchdog cannot be stopped once started (default="
				__MODULE_STRING(WATCHDOG_NOWAYOUT) ")");
#endif

static unsigned long bcm63268_wdt_busy;
static char bcm63268_wdt_expect_release;
static struct timer_list bcm63268_wdt_timer;
static atomic_t bcm63268_wdt_ticks;

static inline void bcm63268_wdt_hw_start(void)
{
	bcm_wdt_writel((BCM63268_WDT_CLOCK_RATE * bcm63268_wdt_time), WDT_DEFVAL_REG);
	bcm_wdt_writel(WDT_START_1, WDT_CTL_REG);
	bcm_wdt_writel(WDT_START_2, WDT_CTL_REG);
}

static inline int bcm63268_wdt_hw_stop(void)
{
	bcm_wdt_writel(BCM63268_WDT_CLOCK_RATE, WDT_DEFVAL_REG);
	bcm_wdt_writel(WDT_STOP_1, WDT_CTL_REG);
	bcm_wdt_writel(WDT_STOP_2, WDT_CTL_REG);
	return 0;
}

static inline void bcm63268_wdt_pet(void)
{
	atomic_set(&bcm63268_wdt_ticks, bcm63268_wdt_time);
}

static void bcm63268_timer_tick(unsigned long unused)
{
#if 0
	static int debug_cnt = 0;
#endif
	unsigned long expire = BCM63268_WDT_UPDATE_TIMER;
	unsigned long now = jiffies;
	
	if (!atomic_dec_and_test(&bcm63268_wdt_ticks)) {
		bcm63268_wdt_hw_stop();
		bcm63268_wdt_hw_start();
	} else {
		bcm63268_wdt_pet();
	}

	mod_timer(&bcm63268_wdt_timer, round_jiffies(now + expire));

#if 0
	if (debug_cnt++ >= 12) {
		printk("WATCHDOG alive...\n");
		debug_cnt = 0;
	}
#endif
}

static void bcm63268_wdt_start(void)
{
	bcm63268_wdt_pet();
	bcm63268_timer_tick(0);
}

static void bcm63268_wdt_pause(void)
{
	del_timer_sync(&bcm63268_wdt_timer);
	bcm63268_wdt_hw_stop();
}

static void bcm63268_wdt_stop(void)
{
	bcm63268_wdt_pause();
}

static int bcm63268_wdt_settimeout(int new_time)
{
	if ((new_time <= 0) || (new_time > BCM63268_WDT_MAX_TIME))
		return -EINVAL;

	bcm63268_wdt_time = new_time;
	return 0;
}

#ifdef SMP_DEBUG
static void bcm63268_smp_debug(struct work_struct *work);

static DECLARE_DELAYED_WORK(bcm63268_smp_debug_cpu0_work, bcm63268_smp_debug);
static DECLARE_DELAYED_WORK(bcm63268_smp_debug_cpu1_work, bcm63268_smp_debug);

static void bcm63268_smp_debug(struct work_struct *work)
{
	int cpu = raw_smp_processor_id();

	printk("%s: CPU %d alive\n", __FUNCTION__, cpu);
	schedule_delayed_work_on(cpu, 
				cpu ? &bcm63268_smp_debug_cpu1_work : &bcm63268_smp_debug_cpu0_work, msecs_to_jiffies(300000));
}
#endif

static int bcm63268_wdt_open(struct inode *inode, struct file *file)
{
	if (test_and_set_bit(0, &bcm63268_wdt_busy))
		return -EBUSY;

#ifdef SMP_DEBUG
	schedule_delayed_work_on(0, &bcm63268_smp_debug_cpu0_work, msecs_to_jiffies(300000));
	schedule_delayed_work_on(1, &bcm63268_smp_debug_cpu1_work, msecs_to_jiffies(300000));
#endif

	bcm63268_wdt_start();
	return nonseekable_open(inode, file);
}

static int bcm63268_wdt_release(struct inode *inode, struct file *file)
{
	if (bcm63268_wdt_expect_release == 42) {
		bcm63268_wdt_stop();
	} else {
		printk(KERN_CRIT DRV_NAME
			": Unexpected close, not stopping watchdog!\n");
		bcm63268_wdt_start();
	}

	clear_bit(0, &bcm63268_wdt_busy);
	bcm63268_wdt_expect_release = 0;
	return 0;
}

static ssize_t bcm63268_wdt_write(struct file *file, const char __user *data,
				size_t len, loff_t *ppos)
{
	if (len) {
		if (!bcm63268_wdt_nowayout) {
			size_t i;

			bcm63268_wdt_expect_release = 0;

			for (i = 0; i != len; i++) {
				char c;
				if (get_user(c, data + i))
					return -EFAULT;
				if (c == 'V')
					bcm63268_wdt_expect_release = 42;
			}
		}
		bcm63268_wdt_pet();
	}
	return len;
}

static struct watchdog_info bcm63268_wdt_info = {
	.identity 	= DRV_NAME,
	.options 	= WDIOF_SETTIMEOUT |
				WDIOF_KEEPALIVEPING |
				WDIOF_MAGICCLOSE,
};

static long bcm63268_wdt_ioctl(struct file *file,
					unsigned int cmd, unsigned long arg)
{
	void __user *argp = (void __user *)arg;
	int __user *p = argp;
	int new_value, retval = -EINVAL;

	switch (cmd) {
	case WDIOC_GETSUPPORT:
		return copy_to_user(argp, &bcm63268_wdt_info,
				sizeof(bcm63268_wdt_info)) ? -EFAULT : 0;

	case WDIOC_GETSTATUS:
	case WDIOC_GETBOOTSTATUS:
		return put_user(0, p);

	case WDIOC_SETOPTIONS:
		if (get_user(new_value, p))
			return -EFAULT;

		if (new_value & WDIOS_DISABLECARD) {
			bcm63268_wdt_stop();
			retval = 0;
		}

		if (new_value & WDIOS_ENABLECARD) {
			bcm63268_wdt_start();
			retval = 0;
		}

		return retval;

	case WDIOC_KEEPALIVE:
		bcm63268_wdt_pet();
		return 0;

	case WDIOC_SETTIMEOUT:
		if (get_user(new_value, p))
			return -EFAULT;

		if (bcm63268_wdt_settimeout(new_value))
			return -EINVAL;

		bcm63268_wdt_pet();

	case WDIOC_GETTIMEOUT:
		return put_user(bcm63268_wdt_time, p);

	default:
		return -ENOTTY;
	}
}

static int bcm63268_wdt_notify_sys(struct notifier_block *this,
	unsigned long code, void *unused)
{
	if (code == SYS_DOWN || code == SYS_HALT)
		bcm63268_wdt_stop();
	return NOTIFY_DONE;
}

static const struct file_operations bcm63268_wdt_fops = {
	.owner		= THIS_MODULE,
	.llseek		= no_llseek,
	.unlocked_ioctl	= bcm63268_wdt_ioctl,
	.open		= bcm63268_wdt_open,
	.release	= bcm63268_wdt_release,
	.write		= bcm63268_wdt_write,
};

static struct miscdevice bcm63268_wdt_miscdev = {
	.minor		= WATCHDOG_MINOR,
	.name		= "watchdog",
	.fops		= &bcm63268_wdt_fops,
};

static struct notifier_block bcm63268_wdt_notifier = {
	.notifier_call = bcm63268_wdt_notify_sys,
};

static int __init bcm63268_wdt_init(void)
{
	int ret;

	if (bcm63268_wdt_hw_stop() < 0)
		return -ENODEV;

	setup_timer(&bcm63268_wdt_timer, bcm63268_timer_tick, 0L);

	if (bcm63268_wdt_settimeout(bcm63268_wdt_time)) {
		bcm63268_wdt_settimeout(BCM63268_WDT_DEFAULT_TIME);
		printk(KERN_INFO DRV_NAME ": "
			"wdt_time value must be 0 < wdt_time < %d, using %d\n",
			(BCM63268_WDT_MAX_TIME + 1), bcm63268_wdt_time);
	}

	ret = register_reboot_notifier(&bcm63268_wdt_notifier);
	if (ret)
		return ret;

	ret = misc_register(&bcm63268_wdt_miscdev);
	if (ret) {
		unregister_reboot_notifier(&bcm63268_wdt_notifier);
		return ret;
	}

	printk(KERN_INFO "BCM63268 Watchdog Timer enabled (%d seconds%s)\n",
				bcm63268_wdt_time, bcm63268_wdt_nowayout ? ", nowayout" : "");
	return 0;
}

static void __exit bcm63268_wdt_exit(void)
{
	if (!bcm63268_wdt_nowayout)
		bcm63268_wdt_stop();

	misc_deregister(&bcm63268_wdt_miscdev);

	unregister_reboot_notifier(&bcm63268_wdt_notifier);
}

module_init(bcm63268_wdt_init);
module_exit(bcm63268_wdt_exit);

MODULE_AUTHOR(DRV_AUTHOR);
MODULE_DESCRIPTION(DRV_DESC);
MODULE_LICENSE("GPL");
MODULE_VERSION(DRV_VER);

