/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/clk.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/spi/spi.h>
#include <linux/completion.h>
#include <linux/err.h>

#include <bcm63268_io.h>
#include <bcm63268_regs.h>
#include <bcm63268_dev_hsspi.h>

#define DRV_NAME	"bcm63268-hsspi"
#define DRV_AUTHOR	"Arcadyan Technology"
#define DRV_DESC	"Broadcom on-chip high-speed SPI Controller Driver"
#define DRV_VER		"1.0"

#define START_STATE	((void *)0)
#define RUNNING_STATE	((void *)1)
#define DONE_STATE	((void *)2)
#define ERROR_STATE	((void *)-1)
#define QUEUE_RUNNING	0
#define QUEUE_STOPPED	1

#define BCM63268_HSSPI_PLL_FREQ			400000000
#define BCM63268_HSSPI_MSG_TYPE_LEN		2
#define BCM63268_HSSPI_RWMODE_FD_RW		1
#define BCM63268_HSSPI_RWMODE_HD_R		2
#define BCM63268_HSSPI_RWMODE_HD_W		3

enum bcm63268_rset_hsspi {
        HSSPI_GLOBAL_CTL,
        HSSPI_INT_STATUS,
        HSSPI_MASK_INT_STATUS,
        HSSPI_INT_MASK,
        HSSPI_FLASH_CTL,
        HSSPI_FLASH_ADDR_BASE,
        HSSPI_PINGPONG0_CMD,
        HSSPI_PINGPONG0_STATUS,
        HSSPI_PINGPONG0_FIFO_STATUS,
        HSSPI_PINGPONG0_CTL,
        HSSPI_PROFILE0_CLK_CTL,
        HSSPI_PROFILE0_MODE_CTL,
        HSSPI_FIFO0,
};

/*
 * register offsets
 */
static const unsigned long bcm63268_regs_hsspi[] = {
	[HSSPI_GLOBAL_CTL]				= HSSPI_BCM_63268_GLOBAL_CTL_REG,
	[HSSPI_INT_STATUS]				= HSSPI_BCM_63268_INT_ST_REG,
	[HSSPI_MASK_INT_STATUS]			= HSSPI_BCM_63268_INT_MASK_ST_REG,
	[HSSPI_INT_MASK]				= HSSPI_BCM_63268_INT_MASK_REG,
	[HSSPI_FLASH_CTL]				= HSSPI_BCM_63268_FLASH_CTL_REG,
	[HSSPI_FLASH_ADDR_BASE]			= HSSPI_BCM_63268_FLASH_ADDR_BASE_REG,
	[HSSPI_PINGPONG0_CMD]			= HSSPI_BCM_63268_PINGPONG0_CMD_REG,
	[HSSPI_PINGPONG0_STATUS]		= HSSPI_BCM_63268_PINGPONG0_STS_REG,
	[HSSPI_PINGPONG0_FIFO_STATUS]	= HSSPI_BCM_63268_PINGPONG0_FIFO_STS_REG,
	[HSSPI_PINGPONG0_CTL]			= HSSPI_BCM_63268_PINGPONG0_CTL_REG,
	[HSSPI_PROFILE0_CLK_CTL]		= HSSPI_BCM_63268_PROFILES0_CLK_CTL_REG,
	[HSSPI_PROFILE0_MODE_CTL]		= HSSPI_BCM_63268_PROFILES0_MODE_CTL_REG,
	[HSSPI_FIFO0]					= HSSPI_BCM_63268_FIFO0,
};

#ifdef BCMCPU_RUNTIME_DETECT
static const unsigned long *bcm63268_regs_hsspi;

static __init void bcm63268_hsspi_regs_init(void)
{
	if (BCMCPU_IS_63268())
		bcm63268_regs_hsspi = bcm63268_regs_hsspi;
}
#else
static __init void bcm63268_hsspi_regs_init(void) { }
#endif /* BCMCPU_RUNTIME_DETECT */

static inline unsigned long bcm63268_hsspireg(enum bcm63268_rset_hsspi reg)
{
#ifdef BCMCPU_RUNTIME_DETECT
        return bcm63268_regs_hsspi[reg];
#else
#ifdef CONFIG_BCM63268_CPU_63268
switch (reg) {
	case HSSPI_GLOBAL_CTL:
		return HSSPI_BCM_63268_GLOBAL_CTL_REG;
	case HSSPI_INT_STATUS:
		return HSSPI_BCM_63268_INT_ST_REG;
	case HSSPI_MASK_INT_STATUS:
		return HSSPI_BCM_63268_INT_MASK_ST_REG;
	case HSSPI_INT_MASK:
		return HSSPI_BCM_63268_INT_MASK_REG;
	case HSSPI_FLASH_CTL:
		return HSSPI_BCM_63268_FLASH_CTL_REG;
	case HSSPI_FLASH_ADDR_BASE:
		return HSSPI_BCM_63268_FLASH_ADDR_BASE_REG;
	case HSSPI_PINGPONG0_CMD:
		return HSSPI_BCM_63268_PINGPONG0_CMD_REG;
	case HSSPI_PINGPONG0_STATUS:
		return HSSPI_BCM_63268_PINGPONG0_STS_REG;
	case HSSPI_PINGPONG0_FIFO_STATUS:
		return HSSPI_BCM_63268_PINGPONG0_FIFO_STS_REG;
	case HSSPI_PINGPONG0_CTL:
		return HSSPI_BCM_63268_PINGPONG0_CTL_REG;
	case HSSPI_PROFILE0_CLK_CTL:
		return HSSPI_BCM_63268_PROFILES0_CLK_CTL_REG;
	case HSSPI_PROFILE0_MODE_CTL:
		return HSSPI_BCM_63268_PROFILES0_MODE_CTL_REG;
	case HSSPI_FIFO0:
		return HSSPI_BCM_63268_FIFO0;
}
#elif defined(CONFIG_BCM63268_CPU_63168)
switch (reg) {
	case HSSPI_GLOBAL_CTL:
		return HSSPI_BCM_63268_GLOBAL_CTL_REG;
	case HSSPI_INT_STATUS:
		return HSSPI_BCM_63268_INT_ST_REG;
	case HSSPI_MASK_INT_STATUS:
		return HSSPI_BCM_63268_INT_MASK_ST_REG;
	case HSSPI_INT_MASK:
		return HSSPI_BCM_63268_INT_MASK_REG;
	case HSSPI_FLASH_CTL:
		return HSSPI_BCM_63268_FLASH_CTL_REG;
	case HSSPI_FLASH_ADDR_BASE:
		return HSSPI_BCM_63268_FLASH_ADDR_BASE_REG;
	case HSSPI_PINGPONG0_CMD:
		return HSSPI_BCM_63268_PINGPONG0_CMD_REG;
	case HSSPI_PINGPONG0_STATUS:
		return HSSPI_BCM_63268_PINGPONG0_STS_REG;
	case HSSPI_PINGPONG0_FIFO_STATUS:
		return HSSPI_BCM_63268_PINGPONG0_FIFO_STS_REG;
	case HSSPI_PINGPONG0_CTL:
		return HSSPI_BCM_63268_PINGPONG0_CTL_REG;
	case HSSPI_PROFILE0_CLK_CTL:
		return HSSPI_BCM_63268_PROFILES0_CLK_CTL_REG;
	case HSSPI_PROFILE0_MODE_CTL:
		return HSSPI_BCM_63268_PROFILES0_MODE_CTL_REG;
	case HSSPI_FIFO0:
		return HSSPI_BCM_63268_FIFO0;
}
#elif defined(CONFIG_BCM63268_CPU_63167)
switch (reg) {
	case HSSPI_GLOBAL_CTL:
		return HSSPI_BCM_63268_GLOBAL_CTL_REG;
	case HSSPI_INT_STATUS:
		return HSSPI_BCM_63268_INT_ST_REG;
	case HSSPI_MASK_INT_STATUS:
		return HSSPI_BCM_63268_INT_MASK_ST_REG;
	case HSSPI_INT_MASK:
		return HSSPI_BCM_63268_INT_MASK_REG;
	case HSSPI_FLASH_CTL:
		return HSSPI_BCM_63268_FLASH_CTL_REG;
	case HSSPI_FLASH_ADDR_BASE:
		return HSSPI_BCM_63268_FLASH_ADDR_BASE_REG;
	case HSSPI_PINGPONG0_CMD:
		return HSSPI_BCM_63268_PINGPONG0_CMD_REG;
	case HSSPI_PINGPONG0_STATUS:
		return HSSPI_BCM_63268_PINGPONG0_STS_REG;
	case HSSPI_PINGPONG0_FIFO_STATUS:
		return HSSPI_BCM_63268_PINGPONG0_FIFO_STS_REG;
	case HSSPI_PINGPONG0_CTL:
		return HSSPI_BCM_63268_PINGPONG0_CTL_REG;
	case HSSPI_PROFILE0_CLK_CTL:
		return HSSPI_BCM_63268_PROFILES0_CLK_CTL_REG;
	case HSSPI_PROFILE0_MODE_CTL:
		return HSSPI_BCM_63268_PROFILES0_MODE_CTL_REG;
	case HSSPI_FIFO0:
		return HSSPI_BCM_63268_FIFO0;
}
#endif /* CONFIG_BCM63268_CPU_63268 || CONFIG_BCM63268_CPU_63168 || CONFIG_BCM63268_CPU_63167 */
#endif /* BCMCPU_RUNTIME_DETECT */
	return 0;
}

/*
 * helpers for the SPI register sets
 */
#define bcm_hsspi_readl(b,o)      bcm_readl((b) + bcm63268_hsspireg(o))
#define bcm_hsspi_readw(b,o)      bcm_readw((b) + bcm63268_hsspireg(o))
#define bcm_hsspi_writel(v,b,o)   bcm_writel((v), (b) + bcm63268_hsspireg(o))
#define bcm_hsspi_writew(v,b,o)   bcm_writew((v), (b) + bcm63268_hsspireg(o))

struct bcm63268_spi {
	/* bitbang has to be first */
	struct spi_master *master;

	resource_size_t mapbase;
	void __iomem	*membase;
	int				irq;

	/* Platform data */
	struct bcm63268_hsspi_pdata *pdata;

	/* Driver message queue */
	struct workqueue_struct *workqueue;
	struct work_struct pump_messages;
	spinlock_t lock;
	struct list_head queue;
	int run;
	int busy;
	int msg_type;

	/* Message Transfer pump */
	struct tasklet_struct pump_transfers;

	/* Current message transfer state info */
	struct spi_message *cur_msg;
	struct spi_transfer *cur_transfer;

	/* Data Buffers */
	u8 *cur_tx_buf;
	u8 *next_tx_buf;
	u8 *rx_buf;
	int cur_tx_len;		/* expected TX length */
	int	cur_tx_cnt;		/* actual TX length */
	int next_tx_len;
	int	next_tx_cnt;
	int rx_len;
	int	rx_cnt;

	struct clk	*clk;
	struct platform_device	*pdev;
};

static int bcm63268_spi_setup_transfer(struct bcm63268_spi *drv_data)
{
	struct spi_device *spi = drv_data->cur_msg->spi;
	u32 clk_cfg;
	int divisor, fcw;

	if (spi->bits_per_word != 8) {
		dev_err(&spi->dev, "%s, unsupported bits_per_word=%d\n",
			__func__, spi->bits_per_word);
		return -EINVAL;
        }

	if (spi->chip_select > spi->master->num_chipselect) {
		dev_err(&spi->dev, "%s, unsupported slave %d\n",
			__func__, spi->chip_select);
		return -EINVAL;
	}

	/* Check clock setting */
	divisor = BCM63268_HSSPI_PLL_FREQ / spi->max_speed_hz;
	if ((BCM63268_HSSPI_PLL_FREQ % spi->max_speed_hz) 
		>= (spi->max_speed_hz / 2)) {
		/* round-up divisor */
		divisor++;
	}

	fcw = 2048 / divisor;
	if (2048 % divisor) {
		/* ceil fcw */
		fcw++;
	}
	
	clk_cfg = V_HSSPI_PROFILES_CLOCK_CTL_FREQ_CTL_WORD(fcw) 
						| M_HSSPI_PROFILES_CLOCK_CTL_ACCUM_RST_ON_LOOP;;
	bcm_hsspi_writel(clk_cfg, drv_data->mapbase, HSSPI_PROFILE0_CLK_CTL);

	dev_dbg(&spi->dev, "Setting clock register to 0x%x (hz %d)\n",
								clk_cfg, spi->max_speed_hz);

	return 0;
}

static void bcm63268_spi_setup_msgtype(struct bcm63268_spi *drv_data, u16 byte_cnt)
{
	u16 reg;
	
	/* Fill in the Message control register */
	reg = V_HSSPI_MSG_CTL_MSG_TYPE(drv_data->msg_type)
		| V_HSSPI_MSG_CTL_BYTE_CNT(byte_cnt);

	bcm_hsspi_writew(reg, drv_data->mapbase, HSSPI_FIFO0);
}

/* Fill the TX FIFO with as many bytes as possible */
static void bcm63268_spi_fill_tx_fifo(struct bcm63268_spi *drv_data)
{
	struct bcm63268_hsspi_pdata *pdata = drv_data->pdata;
	struct spi_device *spi = drv_data->cur_msg->spi;
	u16 tail;
	int copy_len;
	int fifo_avail;
	int msg_left;
	u8 *tx_src, *tx_dst;

	/* Fill the Tx FIFO with as many bytes as possible */
	if (drv_data->cur_tx_buf == NULL) {
		dev_dbg(&spi->dev, "no current Tx data\n");
		return;
	}
	
	/* check if Tx FIFO is full */
	tail = (bcm_hsspi_readl(drv_data->mapbase, HSSPI_PINGPONG0_FIFO_STATUS) &
			M_HSSPI_PINGPONG_FIFO_STATUS_MSG_TAIL) >> S_HSSPI_PINGPONG_FIFO_STATUS_MSG_TAIL;
	if (tail >= pdata->fifo_size) {
		dev_dbg(&spi->dev, "Tx FIFO is full now\n");
		return;
	}
	
	if (drv_data->cur_tx_cnt < drv_data->cur_tx_len) {
		msg_left = drv_data->cur_tx_len - drv_data->cur_tx_cnt;
		fifo_avail = pdata->fifo_size - tail;
		copy_len = msg_left > fifo_avail ? fifo_avail : msg_left;
		tx_src = drv_data->cur_tx_buf + drv_data->cur_tx_cnt;
		tx_dst = (u8 *)(drv_data->mapbase + bcm63268_hsspireg(HSSPI_FIFO0) + 
											BCM63268_HSSPI_MSG_TYPE_LEN + tail);
		memcpy(tx_dst, tx_src, copy_len);
		drv_data->cur_tx_cnt += copy_len;
	}

	if (drv_data->next_tx_buf == NULL) {
		dev_dbg(&spi->dev, "no next Tx data\n");
		return;
	}

	/* check if Tx FIFO is full */
	tail = (bcm_hsspi_readl(drv_data->mapbase, HSSPI_PINGPONG0_FIFO_STATUS) &
			M_HSSPI_PINGPONG_FIFO_STATUS_MSG_TAIL) >> S_HSSPI_PINGPONG_FIFO_STATUS_MSG_TAIL;
	if (tail >= pdata->fifo_size) {
		dev_dbg(&spi->dev, "Tx FIFO is full now\n");
		return;
	}
	
	if (drv_data->next_tx_cnt < drv_data->next_tx_len) {
		msg_left = drv_data->next_tx_len - drv_data->next_tx_cnt;
		fifo_avail = pdata->fifo_size - tail;
		copy_len = msg_left > fifo_avail ? fifo_avail : msg_left;
		tx_src = drv_data->next_tx_buf + drv_data->next_tx_cnt;
		tx_dst = (u8 *)(drv_data->mapbase + bcm63268_hsspireg(HSSPI_FIFO0) + 
											BCM63268_HSSPI_MSG_TYPE_LEN + tail);
		memcpy(tx_dst, tx_src, copy_len);
		drv_data->next_tx_cnt += copy_len;
	}
}

static void bcm63268_spi_issue_transfer(struct bcm63268_spi *drv_data, u16 prepend_byte_cnt)
{
	struct spi_device *spi = drv_data->cur_msg->spi;
	u32 reg;

	/* Enable the command done interrupt which
	 * we use to determine completion of a command */
	bcm_hsspi_writel(M_HSSPI_INTR_PING0_CMD_DONE, drv_data->mapbase, HSSPI_INT_MASK);

	reg = (V_HSSPI_PROFILES_MODE_CTL_PREPENDBYTE_CNT(prepend_byte_cnt) 
		& M_HSSPI_PROFILES_MODE_CTL_PREPENDBYTE_CNT);
	reg |= V_HSSPI_PROFILES_MODE_CTL_MULTIDATA_WR_STRT(2)
			| V_HSSPI_PROFILES_MODE_CTL_MULTIDATA_RD_STRT(2)
			| V_HSSPI_PROFILES_MODE_CTL_FILLBYTE(0xff);
	bcm_hsspi_writel(reg, drv_data->mapbase, HSSPI_PROFILE0_MODE_CTL);

	reg = V_HSSPI_PINGPONG_CMD_SS_NUM(spi->chip_select)
		| V_HSSPI_PINGPONG_CMD_PROFILE_NUM(drv_data->pdata->profile)
		| V_HSSPI_PINGPONG_CMD_TRIGGER_NUM(0)
		| V_HSSPI_PINGPONG_CMD_VALUE(HSSPI_PINGPONG_CMD_START_NOW);

	bcm_hsspi_writel(reg, drv_data->mapbase, HSSPI_PINGPONG0_CMD);
}

/*
 * caller already set message->status;
 * dma and pio irqs are blocked give finished message back
 */
static void bcm63268_spi_giveback(struct bcm63268_spi *drv_data)
{
	struct spi_transfer *last_transfer;
	unsigned long flags;
	struct spi_message *msg;

	spin_lock_irqsave(&drv_data->lock, flags);
	msg = drv_data->cur_msg;
	drv_data->cur_msg = NULL;
	drv_data->cur_transfer = NULL;
	queue_work(drv_data->workqueue, &drv_data->pump_messages);
	spin_unlock_irqrestore(&drv_data->lock, flags);

	last_transfer = list_entry(msg->transfers.prev,
				   struct spi_transfer, transfer_list);

	msg->state = NULL;

	if (msg->complete)
		msg->complete(msg->context);
}

/* test if ther is more transfer to be done */
static void *bcm63268_spi_next_transfer(struct bcm63268_spi *drv_data)
{
	struct spi_message *msg = drv_data->cur_msg;
	struct spi_transfer *trans = drv_data->cur_transfer;

	/* Move to next transfer */
	if (trans->transfer_list.next != &msg->transfers) {
		drv_data->cur_transfer =
		    list_entry(trans->transfer_list.next,
			       struct spi_transfer, transfer_list);
		return RUNNING_STATE;
	} else
		return DONE_STATE;
}

/* This driver supports single master mode only. Hence
 * CMD_DONE is the only interrupt we care about
 */
static irqreturn_t bcm63268_spi_interrupt_transfer(struct bcm63268_spi *drv_data)
{
	unsigned long limit = loops_per_jiffy << 1;
	int transfer_success;
	u8 *rx_src, *rx_dst;
	u16 copy_len;

	rx_dst = NULL;
	switch (drv_data->msg_type) {
		case HSSPI_MSG_CTL_MSG_TYPE_READ_WRITE:
			rx_dst = drv_data->rx_buf + drv_data->rx_cnt;
			break;
			
		case HSSPI_MSG_CTL_MSG_TYPE_WRITE:
			rx_dst = NULL;
			break;

		case HSSPI_MSG_CTL_MSG_TYPE_READ:
			rx_dst = drv_data->rx_buf + drv_data->rx_cnt;
			break;
	}
	
	if (rx_dst != NULL) {
		while ((bcm_hsspi_readl(drv_data->mapbase, HSSPI_PINGPONG0_FIFO_STATUS) & 
			M_HSSPI_PINGPONG_FIFO_STATUS_RX_EMPTY) && --limit) {
			cpu_relax();
		}

		rx_src = (u8 *)(drv_data->mapbase + bcm63268_hsspireg(HSSPI_FIFO0));
		copy_len = (bcm_hsspi_readl(drv_data->mapbase, HSSPI_PINGPONG0_FIFO_STATUS) &
			M_HSSPI_PINGPONG_FIFO_STATUS_RX_TAIL) >> S_HSSPI_PINGPONG_FIFO_STATUS_RX_TAIL;;
		memcpy(rx_dst, rx_src, copy_len);
		drv_data->rx_cnt += copy_len;
	}

	if ((drv_data->cur_tx_cnt != drv_data->cur_tx_len) || 
		(drv_data->next_tx_cnt != drv_data->next_tx_len)) {
		bcm63268_spi_fill_tx_fifo(drv_data);
		bcm63268_spi_issue_transfer(drv_data, 0);
		return IRQ_HANDLED;
	}

	if (drv_data->rx_cnt == drv_data->rx_len) {
		transfer_success = 1;
	}

	if (!transfer_success) {
		dev_dbg(&drv_data->pdev->dev,
			"IO write error!\n");
		drv_data->cur_msg->state = ERROR_STATE;
	} else {
		/* Update total byte transfered */
		drv_data->cur_msg->actual_length += drv_data->cur_tx_cnt;
		drv_data->cur_msg->actual_length += drv_data->next_tx_cnt;
		drv_data->cur_msg->actual_length += drv_data->rx_cnt;
		
		/* Move to next transfer of this msg */
		drv_data->cur_msg->state = bcm63268_spi_next_transfer(drv_data);
	}

	/* Schedule next transfer tasklet */
	tasklet_schedule(&drv_data->pump_transfers);

	return IRQ_HANDLED;
}

static irqreturn_t bcm63268_spi_interrupt(int irq, void *dev_id)
{
	struct spi_master *master = (struct spi_master *)dev_id;
	struct bcm63268_spi *drv_data = spi_master_get_devdata(master);
	u8 intr;

	if (!drv_data->cur_msg)
		return IRQ_HANDLED;

	/* Read interupts and clear them immediately */
	intr = bcm_hsspi_readl(drv_data->mapbase, HSSPI_INT_STATUS);
	bcm_hsspi_writel(intr, drv_data->mapbase, HSSPI_INT_STATUS);

	bcm_hsspi_writel(0, drv_data->mapbase, HSSPI_INT_MASK);

	if (!(intr & M_HSSPI_INTR_PING0_CMD_DONE)) {
		drv_data->cur_msg->state = ERROR_STATE;
		return IRQ_HANDLED;
	}
	
	return bcm63268_spi_interrupt_transfer(drv_data);
}

static void bcm63268_spi_pump_transfers(unsigned long data)
{
	struct bcm63268_spi *drv_data = (struct bcm63268_spi *)data;
	struct spi_message *message;
	struct spi_transfer *cur_transfer;
	struct spi_transfer *next_transfer;
	struct spi_transfer *prev_transfer;
	struct spi_device *spi;
	u16 byte_cnt, prepend_byte_cnt;

	/* Get current state information */
	message = drv_data->cur_msg;
	cur_transfer = drv_data->cur_transfer;
	spi = message->spi;

	/*
	 * if msg is error or done, report it back using complete() callback
	 */

	 /* Handle for abort */
	if (message->state == ERROR_STATE) {
		dev_dbg(&drv_data->pdev->dev, "transfer: we've hit an error\n");
		message->status = -EIO;
		bcm63268_spi_giveback(drv_data);
		return;
	}

	/* Handle end of message */
	if (message->state == DONE_STATE) {
		dev_dbg(&drv_data->pdev->dev, "transfer: all done!\n");
		message->status = 0;
		bcm63268_spi_giveback(drv_data);
		return;
	}

	/* Delay if requested at end of transfer */
	if (message->state == RUNNING_STATE) {
		dev_dbg(&drv_data->pdev->dev, "transfer: still running ...\n");
		prev_transfer = list_entry(cur_transfer->transfer_list.prev,
				      struct spi_transfer, transfer_list);
		if (prev_transfer->delay_usecs)
			udelay(prev_transfer->delay_usecs);
	}

	if (cur_transfer->len == 0) {
		/* Move to next transfer of this msg */
		message->state = bcm63268_spi_next_transfer(drv_data);
		/* Schedule next transfer tasklet */
		tasklet_schedule(&drv_data->pump_transfers);
	}

	if (cur_transfer->tx_buf == NULL && cur_transfer->rx_buf == NULL) {
		/* Move to next transfer of this msg */
		message->state = bcm63268_spi_next_transfer(drv_data);
		/* Schedule next transfer tasklet */
		tasklet_schedule(&drv_data->pump_transfers);
	}

	/* this controller does not support keeping the chip select active between
       transfers. If a message is detected with a write transfer followed by a 
       read transfer and cs_change is set to 0 then the two transfers need to be
       combined. The message state is used to indicate that the transfers 
       need to be combined */
	drv_data->cur_tx_buf = NULL;
	drv_data->next_tx_buf = NULL;
	drv_data->rx_buf = NULL;
	drv_data->cur_tx_len = 0;
	drv_data->cur_tx_cnt = 0;
	drv_data->next_tx_len = 0;
	drv_data->next_tx_cnt = 0;
	drv_data->rx_len = 0;
	drv_data->rx_cnt = 0;
	byte_cnt = 0;
	prepend_byte_cnt = 0;
	if ((cur_transfer->tx_buf != NULL) && (cur_transfer->rx_buf != NULL)) {
		/* full-duplex read/write */
		drv_data->cur_tx_buf = (u8 *)cur_transfer->tx_buf;
		drv_data->rx_buf = (u8 *)cur_transfer->rx_buf;
		drv_data->cur_tx_len = cur_transfer->len;
		drv_data->rx_len = cur_transfer->len;
		drv_data->msg_type = HSSPI_MSG_CTL_MSG_TYPE_READ_WRITE;
		byte_cnt = cur_transfer->len;
		prepend_byte_cnt = 0;
	}
	else if (cur_transfer->tx_buf != NULL) {
		/* half-duplex */
		next_transfer = NULL;
		if (cur_transfer->transfer_list.next != &message->transfers) {
			next_transfer =
		    	list_entry(cur_transfer->transfer_list.next,
			       struct spi_transfer, transfer_list);
		}

		if (next_transfer != NULL) {
			if (next_transfer->tx_buf != NULL) {
				/* half-duplex write */
				drv_data->cur_tx_buf = (u8 *)cur_transfer->tx_buf;
				drv_data->next_tx_buf = (u8 *)next_transfer->tx_buf;
				drv_data->cur_tx_len = cur_transfer->len;
				drv_data->next_tx_len = next_transfer->len;
				drv_data->msg_type = HSSPI_MSG_CTL_MSG_TYPE_WRITE;
				byte_cnt = cur_transfer->len + next_transfer->len;
				prepend_byte_cnt = 0;
			}
			else if (next_transfer->rx_buf != NULL) {
				/* half-duplex read */
				drv_data->cur_tx_buf = (u8 *)cur_transfer->tx_buf;
				drv_data->rx_buf = (u8 *)next_transfer->rx_buf;
				drv_data->cur_tx_len = cur_transfer->len;
				drv_data->rx_len = next_transfer->len;
				drv_data->msg_type = HSSPI_MSG_CTL_MSG_TYPE_READ;
				byte_cnt = next_transfer->len;
				prepend_byte_cnt = cur_transfer->len;
			}
			drv_data->cur_transfer = next_transfer;
		}
		else {
			/* half-duplex write */
			drv_data->cur_tx_buf = (u8 *)cur_transfer->tx_buf;
			drv_data->cur_tx_len = cur_transfer->len;
			drv_data->msg_type = HSSPI_MSG_CTL_MSG_TYPE_WRITE;
			byte_cnt = cur_transfer->len;
			prepend_byte_cnt = 0;
		}
	}

	message->state = RUNNING_STATE;

	bcm63268_spi_setup_transfer(drv_data);

	bcm63268_spi_fill_tx_fifo(drv_data);

	/* Fill in the Message control register */
	bcm63268_spi_setup_msgtype(drv_data, byte_cnt);

	/* Issue the transfer */
	bcm63268_spi_issue_transfer(drv_data, prepend_byte_cnt);
}

/* pop a msg from queue and kick off real transfer */
static void bcm63268_spi_pump_messages(struct work_struct *work)
{
	struct bcm63268_spi *drv_data;
	unsigned long flags;

	drv_data = container_of(work, struct bcm63268_spi, pump_messages);

	/* Lock queue and check for queue work */
	spin_lock_irqsave(&drv_data->lock, flags);
	if (list_empty(&drv_data->queue) || drv_data->run == QUEUE_STOPPED) {
		/* pumper kicked off but no work to do */
		drv_data->busy = 0;
		spin_unlock_irqrestore(&drv_data->lock, flags);
		return;
	}

	/* Make sure we are not already running a message */
	if (drv_data->cur_msg) {
		spin_unlock_irqrestore(&drv_data->lock, flags);
		return;
	}

	/* Extract head of queue */
	drv_data->cur_msg = list_entry(drv_data->queue.next,
				       struct spi_message, queue);

	list_del_init(&drv_data->cur_msg->queue);

	/* Initial message state */
	drv_data->cur_msg->state = START_STATE;
	drv_data->cur_transfer = list_entry(drv_data->cur_msg->transfers.next,
					    struct spi_transfer, transfer_list);

	dev_dbg(&drv_data->pdev->dev, "got a message to pump\n");

	/* Mark as busy and launch transfers */
	tasklet_schedule(&drv_data->pump_transfers);

	drv_data->busy = 1;
	spin_unlock_irqrestore(&drv_data->lock, flags);
}

/*
 * got a msg to transfer, queue it in drv_data->queue.
 * And kick off message pumper
 */
static int bcm63268_spi_transfer(struct spi_device *spi, struct spi_message *msg)
{
	struct bcm63268_spi *drv_data = spi_master_get_devdata(spi->master);
	unsigned long flags;

	spin_lock_irqsave(&drv_data->lock, flags);

	if (drv_data->run == QUEUE_STOPPED) {
		spin_unlock_irqrestore(&drv_data->lock, flags);
		return -ESHUTDOWN;
	}

	msg->actual_length = 0;
	msg->status = -EINPROGRESS;
	msg->state = START_STATE;

	dev_dbg(&spi->dev, "adding an msg in transfer() \n");
	list_add_tail(&msg->queue, &drv_data->queue);

	if (drv_data->run == QUEUE_RUNNING && !drv_data->busy)
		queue_work(drv_data->workqueue, &drv_data->pump_messages);

	spin_unlock_irqrestore(&drv_data->lock, flags);

	return 0;
}

static int bcm63268_spi_setup(struct spi_device *spi)
{
	struct bcm63268_spi *drv_data;

	drv_data = spi_master_get_devdata(spi->master);

	if (!spi->bits_per_word)
		spi->bits_per_word = 8;

	if (spi->mode & ~(SPI_CPOL | SPI_CPHA)) {
		dev_err(&spi->dev, "%s, unsupported mode bits %x\n",
			__func__, spi->mode & ~(SPI_CPOL | SPI_CPHA));
		return -EINVAL;
	}

	dev_dbg(&spi->dev, "%s, mode %d, %u bits/w, %u nsec/bit\n",
		__func__, spi->mode & (SPI_CPOL | SPI_CPHA), spi->bits_per_word, 0);

	return 0;
}

/*
 * callback for spi framework.
 * clean driver specific data
 */
static void bcm63268_spi_cleanup(struct spi_device *spi)
{
	/* do nothing here */
}

static inline int bcm63268_spi_init_queue(struct bcm63268_spi *drv_data)
{
	INIT_LIST_HEAD(&drv_data->queue);
	spin_lock_init(&drv_data->lock);

	drv_data->run = QUEUE_STOPPED;
	drv_data->busy = 0;

	/* init transfer tasklet */
	tasklet_init(&drv_data->pump_transfers,
		     bcm63268_spi_pump_transfers, (unsigned long)drv_data);

	/* init messages workqueue */
	INIT_WORK(&drv_data->pump_messages, bcm63268_spi_pump_messages);
	drv_data->workqueue = create_singlethread_workqueue(
				dev_name(drv_data->master->dev.parent));
	if (drv_data->workqueue == NULL)
		return -EBUSY;

	return 0;
}

static inline int bcm63268_spi_start_queue(struct bcm63268_spi *drv_data)
{
	unsigned long flags;

	spin_lock_irqsave(&drv_data->lock, flags);

	if (drv_data->run == QUEUE_RUNNING || drv_data->busy) {
		spin_unlock_irqrestore(&drv_data->lock, flags);
		return -EBUSY;
	}

	drv_data->run = QUEUE_RUNNING;
	drv_data->cur_msg = NULL;
	drv_data->cur_transfer = NULL;
	spin_unlock_irqrestore(&drv_data->lock, flags);

	queue_work(drv_data->workqueue, &drv_data->pump_messages);

	return 0;
}

static inline int bcm63268_spi_stop_queue(struct bcm63268_spi *drv_data)
{
	unsigned long flags;
	unsigned limit = 500;
	int status = 0;

	spin_lock_irqsave(&drv_data->lock, flags);

	/*
	 * This is a bit lame, but is optimized for the common execution path.
	 * A wait_queue on the drv_data->busy could be used, but then the common
	 * execution path (pump_messages) would be required to call wake_up or
	 * friends on every SPI message. Do this instead
	 */
	drv_data->run = QUEUE_STOPPED;
	while (!list_empty(&drv_data->queue) && drv_data->busy && limit--) {
		spin_unlock_irqrestore(&drv_data->lock, flags);
		msleep(10);
		spin_lock_irqsave(&drv_data->lock, flags);
	}

	if (!list_empty(&drv_data->queue) || drv_data->busy)
		status = -EBUSY;

	spin_unlock_irqrestore(&drv_data->lock, flags);

	return status;
}

static inline int bcm63268_spi_destroy_queue(struct bcm63268_spi *drv_data)
{
	int status;

	status = bcm63268_spi_stop_queue(drv_data);
	if (status != 0)
		return status;

	destroy_workqueue(drv_data->workqueue);

	return 0;
}

static int __init bcm63268_spi_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct bcm63268_hsspi_pdata *platform_data;
	struct bcm63268_spi *drv_data;
	struct clk *clk;
	struct spi_master *master;
	struct resource *res_mem, *res_irq;
	int ret;
	u16 intr_sts;
	u32 reg;

	clk = NULL;
	platform_data = dev->platform_data;

	/* Allocate master with space for drv_data */
	master = spi_alloc_master(dev, sizeof(struct bcm63268_spi) + 16);
	if (!master) {
		dev_err(&pdev->dev, "can not alloc spi_master\n");
		ret = -ENOMEM;
		goto out;
	}

	drv_data = spi_master_get_devdata(master);
	drv_data->master = master;
	drv_data->pdata = platform_data;
	drv_data->pdev = pdev;

	master->bus_num = pdev->id;
	master->num_chipselect = platform_data->num_chipselect;
	master->cleanup = bcm63268_spi_cleanup;
	master->setup = bcm63268_spi_setup;
	master->transfer = bcm63268_spi_transfer;
#ifdef CONFIG_ARCADYAN_KERNEL_ENHANCE
	master->fifo_size = platform_data->fifo_size;
#endif /* CONFIG_ARCADYAN_KERNEL_ENHANCE */

	/* Find and map our resources */
	res_mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (res_mem == NULL) {
		dev_err(dev, "unable to get IORESOURCE_MEM\n");
		ret = -ENOENT;
		goto out_free;
	}

	if (!request_mem_region(res_mem->start,
			res_mem->end - res_mem->start, DRV_NAME)) {
		ret = -ENXIO;
		goto out_free;
	}

	drv_data->mapbase = res_mem->start;
	drv_data->membase = ioremap(res_mem->start, resource_size(res_mem));
	if (drv_data->membase == NULL) {
		dev_err(dev, "unable to ioremap mapbase\n");
		ret = -ENXIO;
		goto out_free;
	}

	bcm63268_hsspi_regs_init();

	clk = clk_get(&pdev->dev, "spi");
	if (IS_ERR(clk)) {
		dev_err(&pdev->dev, "No clock for device\n");
		ret = -ENODEV;
		goto out_unmap;
	}

	clk_enable(clk);
	drv_data->clk = clk;

	res_irq = platform_get_resource(pdev, IORESOURCE_IRQ, 0);
	if (res_irq == NULL) {
		dev_err(dev, "unable to get IORESOURCE_IRQ\n");
		ret = -ENOENT;
		goto out_unmap;
	}

	drv_data->irq = res_irq->start;

	ret = request_irq(drv_data->irq, bcm63268_spi_interrupt, 0, pdev->name, master);
	if (ret) {
		dev_err(dev, "unable to request irq\n");
		goto out_unmap;
	}

	/* Initial and start queue */
	ret = bcm63268_spi_init_queue(drv_data);
	if (ret != 0) {
		dev_err(dev, "unable to initialize queue\n");
		goto out_queue;
	}

	ret = bcm63268_spi_start_queue(drv_data);
	if (ret != 0) {
		dev_err(dev, "unable to start queue\n");
		goto out_queue;
	}

	/* Initialize hardware */
	reg = (V_HSSPI_FLASH_CTL_SS_NUM(0) & M_HSSPI_FLASH_CTL_SS_NUM)
			| (V_HSSPI_FLASH_CTL_PROFILE_NUM(0) & M_HSSPI_FLASH_CTL_PROFILE_NUM)
			| (V_HSSPI_FLASH_CTL_DUMMY_BYTES(1) & M_HSSPI_FLASH_CTL_DUMMY_BYTES)
			| (V_HSSPI_FLASH_CTL_ADDR_BYTES(1) & M_HSSPI_FLASH_CTL_ADDR_BYTES)
			| (V_HSSPI_FLASH_CTL_OPCODE(0x0b) & M_HSSPI_FLASH_CTL_OPCODE);
	bcm_hsspi_writel(reg, drv_data->mapbase, HSSPI_FLASH_CTL);

	bcm_hsspi_writel(0, drv_data->mapbase, HSSPI_INT_MASK);
	intr_sts = bcm_hsspi_readl(drv_data->mapbase, HSSPI_INT_STATUS);
	bcm_hsspi_writel(intr_sts, drv_data->mapbase, HSSPI_INT_STATUS);


	/* Register with the SPI framework */
	platform_set_drvdata(pdev, drv_data);
	ret = spi_register_master(master);
	if (ret != 0) {
		dev_err(dev, "unable to register spi master\n");
		goto out_queue;
	}

	dev_info(&pdev->dev, " at 0x%08x (irq %d, FIFOs size %d) v%s\n",
				drv_data->mapbase, drv_data->irq, platform_data->fifo_size, DRV_VER);
	return ret;

out_queue:
	free_irq(drv_data->irq, master);
	bcm63268_spi_destroy_queue(drv_data);
out_unmap:
	iounmap(drv_data->membase);
out_free:
	clk_put(clk);
	spi_master_put(master);
out:
	return ret;
}

/* stop hardware and remove the driver */
static int __devexit bcm63268_spi_remove(struct platform_device *pdev)
{
	struct bcm63268_spi *drv_data = platform_get_drvdata(pdev);
	struct spi_master *master = drv_data->master;

	free_irq(drv_data->irq, master);
	bcm63268_spi_destroy_queue(drv_data);
	iounmap(drv_data->membase);
	clk_disable(drv_data->clk);
	clk_put(drv_data->clk);
	spi_master_put(master);
	release_mem_region(drv_data->mapbase, RSET_HSSPI_SIZE);
	platform_set_drvdata(pdev, NULL);

	return 0;
}

#ifdef CONFIG_PM
static int bcm63268_spi_suspend(struct platform_device *pdev, pm_message_t mesg)
{
	struct bcm63268_spi *drv_data = platform_get_drvdata(pdev);

	clk_disable(drv_data->clk);

	return 0;
}

static int bcm63268_spi_resume(struct platform_device *pdev)
{
	struct bcm63268_spi *drv_data = platform_get_drvdata(pdev);

	clk_enable(drv_data->clk);

	return 0;
}
#else
#define bcm63268_spi_suspend	NULL
#define bcm63268_spi_resume	NULL
#endif /* CONFIG_PM */

MODULE_ALIAS("platform:" DRV_NAME);
static struct platform_driver bcm63268_spi_driver = {
	.driver = {
		.name	= DRV_NAME,
		.owner	= THIS_MODULE,
	},
	.probe		= bcm63268_spi_probe,
	.suspend	= bcm63268_spi_suspend,
	.resume		= bcm63268_spi_resume,
	.remove		= __devexit_p(bcm63268_spi_remove),
};


static int __init bcm63268_spi_init(void)
{
	return platform_driver_register(&bcm63268_spi_driver);
}

static void __exit bcm63268_spi_exit(void)
{
	platform_driver_unregister(&bcm63268_spi_driver);
}

module_init(bcm63268_spi_init);
module_exit(bcm63268_spi_exit);

MODULE_AUTHOR(DRV_AUTHOR);
MODULE_DESCRIPTION(DRV_DESC);
MODULE_LICENSE("GPL");
MODULE_VERSION(DRV_VER);

