/*
<:copyright-gpl 
 Copyright 2010 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#if defined(CONFIG_SERIAL_BL234X_CONSOLE) && defined(CONFIG_MAGIC_SYSRQ)
#define SUPPORT_SYSRQ
#endif

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/ioport.h>
#include <linux/init.h>
#include <linux/console.h>
#include <linux/sysrq.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/serial_reg.h>
#include <linux/serial_core.h>
#include <linux/serial.h>
#include <linux/serial_8250.h>
#include <linux/nmi.h>
#include <linux/mutex.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/serial.h>

#include <bl234x_clk.h>
#include <bl234x_irq.h>
#include <bl234x_uart.h>
#include "8250.h"
#include "BL234x_sw_reset.h"
#include "BL234x_ic.h"

#define PFX 		KBUILD_MODNAME
#define DRV_VER		"0.0.1"
#undef DEBUG

#define BL234X_NR_UARTS	CONFIG_SERIAL_BL234X_NR_UARTS

/*
 * Debugging.
 */
#ifdef DEBUG
#define DEBUG_AUTOCONF(fmt...)	printk(fmt)
#else
#define DEBUG_AUTOCONF(fmt...)	do { } while (0)
#endif

#ifdef DEBUG
#define DEBUG_INTR(fmt...)	printk(fmt)
#else
#define DEBUG_INTR(fmt...)	do { } while (0)
#endif

#define BL234X_LSR_SAVE_FLAGS UART_LSR_BRK_ERROR_BITS

struct bl234x_uart_port {
	struct uart_port	port;
	unsigned int		tx_loadsz;	/* transmit fifo load size */
	unsigned char		ier;
	unsigned char		lcr;
	unsigned char		lsr_saved_flags;
};

static struct bl234x_uart_port bl234x_uart_ports[BL234X_NR_UARTS];

static void bl234x_uart_serial_out(struct uart_port *p, int offset, int value)
{
	offset = offset << p->regshift;
	writel(value, p->membase + offset);
}

static unsigned int bl234x_uart_serial_in(struct uart_port *p, int offset)
{
	offset = offset << p->regshift;
	return readl(p->membase + offset);
}


#define serial_in(up, offset)		\
	(bl234x_uart_serial_in(&(up)->port, (offset)))
#define serial_out(up, offset, value)	\
	(bl234x_uart_serial_out(&(up)->port, (offset), (value)))
/*
 * We used to support using pause I/O for certain machines.  We
 * haven't supported this for a while, but just in case it's badly
 * needed for certain old 386 machines, I've left these #define's
 * in....
 */
#define serial_inp(up, offset)		serial_in(up, offset)
#define serial_outp(up, offset, value)	serial_out(up, offset, value)

/* Uart divisor latch read */
static inline int _serial_dl_read(struct bl234x_uart_port *up)
{
	return serial_inp(up, UART_DLL) | serial_inp(up, UART_DLM) << 8;
}

/* Uart divisor latch write */
static inline void _serial_dl_write(struct bl234x_uart_port *up, int value)
{
	serial_outp(up, UART_DLL, value & 0xff);
	serial_outp(up, UART_DLM, value >> 8 & 0xff);
}

#define serial_dl_read(up) _serial_dl_read(up)
#define serial_dl_write(up, value) _serial_dl_write(up, value)

/*
 * FIFO support.
 */
static void bl234x_uart_clear_fifos(struct bl234x_uart_port *p)
{
	serial_outp(p, UART_FCR, UART_FCR_ENABLE_FIFO);
	serial_outp(p, UART_FCR, UART_FCR_ENABLE_FIFO |
		       UART_FCR_CLEAR_RCVR | UART_FCR_CLEAR_XMIT);
	serial_outp(p, UART_FCR, 0);
}


/*
 * serial core request to disable tx ASAP (used for flow control)
 */
static inline void __bl234x_uart_stop_tx(struct bl234x_uart_port *p)
{
	if (p->ier & UART_IER_THRI) {
		p->ier &= ~UART_IER_THRI;
		serial_out(p, UART_IER, p->ier);
	}
}

static void bl234x_uart_stop_tx(struct uart_port *port)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;

	__bl234x_uart_stop_tx(bl234x_up);
}

static void __bl234x_uart_transmit_chars(struct bl234x_uart_port *up);

/*
 * serial core request to (re)enable tx
 */
static void bl234x_uart_start_tx(struct uart_port *port)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;

	if (!(bl234x_up->ier & UART_IER_THRI)) {
		bl234x_up->ier |= UART_IER_THRI;
		serial_out(bl234x_up, UART_IER, bl234x_up->ier);
	}
}

/*
 * serial core request to stop rx, called before port shutdown
 */
static void bl234x_uart_stop_rx(struct uart_port *port)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;

	bl234x_up->ier &= ~UART_IER_RLSI;
	port->read_status_mask &= ~UART_LSR_DR;
	serial_out(bl234x_up, UART_IER, bl234x_up->ier);
}

/*
 * serial core request to enable modem status interrupt reporting
 */
static void bl234x_uart_enable_ms(struct uart_port *port)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;

	bl234x_up->ier |= UART_IER_MSI;
	serial_out(bl234x_up, UART_IER, bl234x_up->ier);
}

static void
__bl234x_uart_receive_chars(struct bl234x_uart_port *up, unsigned int *status)
{
	struct tty_struct *tty = up->port.info->port.tty;
	unsigned char ch, lsr = *status;
	int max_count = 256;
	char flag;

	do {
		if (likely(lsr & UART_LSR_DR))
			ch = serial_inp(up, UART_RX);
		else
			/*
			 * Intel 82571 has a Serial Over Lan device that will
			 * set UART_LSR_BI without setting UART_LSR_DR when
			 * it receives a break. To avoid reading from the
			 * receive buffer without UART_LSR_DR bit set, we
			 * just force the read character to be 0
			 */
			ch = 0;

		flag = TTY_NORMAL;
		up->port.icount.rx++;

		lsr |= up->lsr_saved_flags;
		up->lsr_saved_flags = 0;

		if (unlikely(lsr & UART_LSR_BRK_ERROR_BITS)) {
			/*
			 * For statistics only
			 */
			if (lsr & UART_LSR_BI) {
				lsr &= ~(UART_LSR_FE | UART_LSR_PE);
				up->port.icount.brk++;
				/*
				 * We do the SysRQ and SAK checking
				 * here because otherwise the break
				 * may get masked by ignore_status_mask
				 * or read_status_mask.
				 */
				if (uart_handle_break(&up->port))
					goto ignore_char;
			} else if (lsr & UART_LSR_PE)
				up->port.icount.parity++;
			else if (lsr & UART_LSR_FE)
				up->port.icount.frame++;
			if (lsr & UART_LSR_OE)
				up->port.icount.overrun++;

			/*
			 * Mask off conditions which should be ignored.
			 */
			lsr &= up->port.read_status_mask;

			if (lsr & UART_LSR_BI) {
				DEBUG_INTR("handling break....");
				flag = TTY_BREAK;
			} else if (lsr & UART_LSR_PE)
				flag = TTY_PARITY;
			else if (lsr & UART_LSR_FE)
				flag = TTY_FRAME;
		}
		if (uart_handle_sysrq_char(&up->port, ch))
			goto ignore_char;

		uart_insert_char(&up->port, lsr, UART_LSR_OE, ch, flag);

ignore_char:
		lsr = serial_inp(up, UART_LSR);
	} while ((lsr & (UART_LSR_DR | UART_LSR_BI)) && (max_count-- > 0));
	spin_unlock(&up->port.lock);
	tty_flip_buffer_push(tty);
	spin_lock(&up->port.lock);
	*status = lsr;
}

static void __bl234x_uart_transmit_chars(struct bl234x_uart_port *up)
{
	struct circ_buf *xmit = &up->port.info->xmit;
	int count;

	if (up->port.x_char) {
		serial_outp(up, UART_TX, up->port.x_char);
		up->port.icount.tx++;
		up->port.x_char = 0;
		return;
	}
	if (uart_tx_stopped(&up->port)) {
		bl234x_uart_stop_tx(&up->port);
		return;
	}
	if (uart_circ_empty(xmit)) {
		__bl234x_uart_stop_tx(up);
		return;
	}

	count = up->tx_loadsz;
	do {
		serial_out(up, UART_TX, xmit->buf[xmit->tail]);
		xmit->tail = (xmit->tail + 1) & (UART_XMIT_SIZE - 1);
		up->port.icount.tx++;
		if (uart_circ_empty(xmit))
			break;
	} while (--count > 0);

	if (uart_circ_chars_pending(xmit) < WAKEUP_CHARS)
		uart_write_wakeup(&up->port);

	DEBUG_INTR("THRE...");

	if (uart_circ_empty(xmit))
		__bl234x_uart_stop_tx(up);
}

/*
 * This handles the interrupt from one port.
 */
static void bl234x_uart_handle_port(struct bl234x_uart_port *up)
{
	unsigned int status;
	unsigned long flags;

	spin_lock_irqsave(&up->port.lock, flags);

	status = serial_inp(up, UART_LSR);

	DEBUG_INTR("status = %x...", status);

	if (status & (UART_LSR_DR | UART_LSR_BI))
		__bl234x_uart_receive_chars(up, &status);
	if (status & UART_LSR_THRE)
		__bl234x_uart_transmit_chars(up);

	spin_unlock_irqrestore(&up->port.lock, flags);
}

/*
 * process uart interrupt
 */
static irqreturn_t bl234x_uart_interrupt(int irq, void *dev_id)
{
	struct uart_port *port = (struct uart_port *)dev_id;
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;
	unsigned int iir;

	spin_lock(&port->lock);

	iir = serial_in(bl234x_up, UART_IIR);
	if (!(iir & UART_IIR_NO_INT))
		bl234x_uart_handle_port(bl234x_up);

	fi_bl234x_drv_ic_isr_ack(irq - BL234X_INT_IRQ_BASE);

	spin_unlock(&port->lock);

	return IRQ_HANDLED;
}

/*
 * serial core request to check if uart tx fifo is empty
 */
static unsigned int bl234x_uart_tx_empty(struct uart_port *port)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;
	unsigned long flags;
	unsigned int lsr;

	spin_lock_irqsave(&port->lock, flags);
	lsr = serial_in(bl234x_up, UART_LSR);
	bl234x_up->lsr_saved_flags |= lsr & BL234X_LSR_SAVE_FLAGS;
	spin_unlock_irqrestore(&port->lock, flags);

	return lsr & UART_LSR_TEMT ? TIOCSER_TEMT : 0;
}

static unsigned int bl234x_uart_check_modem_status(struct bl234x_uart_port *up)
{
	unsigned int status = serial_in(up, UART_MSR);

	if (status & UART_MSR_ANY_DELTA && up->ier & UART_IER_MSI &&
	    up->port.info != NULL) {
		if (status & UART_MSR_TERI)
			up->port.icount.rng++;
		if (status & UART_MSR_DDSR)
			up->port.icount.dsr++;
		if (status & UART_MSR_DDCD)
			uart_handle_dcd_change(&up->port, status & UART_MSR_DCD);
		if (status & UART_MSR_DCTS)
			uart_handle_cts_change(&up->port, status & UART_MSR_CTS);

		wake_up_interruptible(&up->port.info->delta_msr_wait);
	}

	return status;
}

/*
 * serial core request to return RI, CTS, DCD and DSR pin state
 */
static unsigned int bl234x_uart_get_mctrl(struct uart_port *port)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;
	unsigned int status;
	unsigned int ret;

	status = bl234x_uart_check_modem_status(bl234x_up);

	ret = 0;
	if (status & UART_MSR_DCD)
		ret |= TIOCM_CAR;
	if (status & UART_MSR_RI)
		ret |= TIOCM_RNG;
	if (status & UART_MSR_DSR)
		ret |= TIOCM_DSR;
	if (status & UART_MSR_CTS)
		ret |= TIOCM_CTS;
	return ret;
}

/*
 * serial core request to set RTS and DTR pin state and loopback mode
 */
static void bl234x_uart_set_mctrl(struct uart_port *port, unsigned int mctrl)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;
	unsigned char mcr = 0;

	if (mctrl & TIOCM_RTS)
		mcr |= UART_MCR_RTS;
	if (mctrl & TIOCM_DTR)
		mcr |= UART_MCR_DTR;
	if (mctrl & TIOCM_OUT1)
		mcr |= UART_MCR_OUT1;
	if (mctrl & TIOCM_OUT2)
		mcr |= UART_MCR_OUT2;
	if (mctrl & TIOCM_LOOP)
		mcr |= UART_MCR_LOOP;

	serial_out(bl234x_up, UART_MCR, mcr);
}

/*
 * serial core request to start/stop emitting break char
 */
static void bl234x_uart_break_ctl(struct uart_port *port, int break_state)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;
	unsigned long flags;

	spin_lock_irqsave(&port->lock, flags);
	if (break_state == -1)
		bl234x_up->lcr |= UART_LCR_SBC;
	else
		bl234x_up->lcr &= ~UART_LCR_SBC;
	serial_out(bl234x_up, UART_LCR, bl234x_up->lcr);
	spin_unlock_irqrestore(&port->lock, flags);
}

#define BOTH_EMPTY (UART_LSR_TEMT | UART_LSR_THRE)

/*
 * return port type in string format
 */
static const char *bl234x_uart_type(struct uart_port *port)
{
	return (port->type == PORT_BL234X) ? "bl234x-uart" : NULL;
}

/*
 * serial core request to initialize uart and start rx operation
 */
static int bl234x_uart_startup(struct uart_port *port)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;
	unsigned long flags;
	int retval;
	
	/*
	 * Clear the FIFO buffers and disable them.
	 * (they will be reenabled in set_termios())
	 */
	bl234x_uart_clear_fifos(bl234x_up);

	/*
	 * Clear the interrupt registers.
	 */
	(void) serial_inp(bl234x_up, UART_LSR);
	(void) serial_inp(bl234x_up, UART_RX);
	(void) serial_inp(bl234x_up, UART_IIR);
	(void) serial_inp(bl234x_up, UART_MSR);

	/*
	 * At this point, there's no way the LSR could still be 0xff;
	 * if it is, then bail out, because there's likely no UART
	 * here.
	 */
	if (serial_inp(bl234x_up, UART_LSR) == 0xff) {
		printk(KERN_INFO "ttyS%d: LSR safety check engaged!\n", port->line);
		return -ENODEV;
	}

	/* register irq and enable rx interrupts */
	retval = request_irq(port->irq, bl234x_uart_interrupt, 0,
			  bl234x_uart_type(port), port);

	if (retval)
		return retval;

	/*
	 * Now, initialize the UART
	 */
	serial_outp(bl234x_up, UART_LCR, UART_LCR_WLEN8);

	spin_lock_irqsave(&port->lock, flags);
	/*
	 * Most PC uarts need OUT2 raised to enable interrupts.
	 */
	port->mctrl |= TIOCM_OUT2;

	bl234x_uart_set_mctrl(port, port->mctrl);

	spin_unlock_irqrestore(port->lock, flags);

	/*
	 * Clear the interrupt registers again for luck, and clear the
	 * saved flags to avoid getting false values from polling
	 * routines or the previous session.
	 */
	serial_inp(bl234x_up, UART_LSR);
	serial_inp(bl234x_up, UART_RX);
	serial_inp(bl234x_up, UART_IIR);
	serial_inp(bl234x_up, UART_MSR);

	return 0;
}

/*
 * serial core request to flush & disable uart
 */
static void bl234x_uart_shutdown(struct uart_port *port)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;
	unsigned long flags;

	/*
	 * Disable interrupts from this port
	 */
	bl234x_up->ier = 0;
	serial_outp(bl234x_up, UART_IER, 0);

	spin_lock_irqsave(&port->lock, flags);
	port->mctrl &= ~TIOCM_OUT2;

	bl234x_uart_set_mctrl(port, port->mctrl);
	spin_unlock_irqrestore(&port->lock, flags);

	/*
	 * Disable break condition and FIFOs
	 */
	serial_out(bl234x_up, UART_LCR, serial_inp(bl234x_up, UART_LCR) & ~UART_LCR_SBC);
	bl234x_uart_clear_fifos(bl234x_up);

	/*
	 * Read data port to reset things, and then free
	 * the IRQ.
	 */
	(void) serial_in(bl234x_up, UART_RX);
	free_irq(port->irq, port);
}

/*
 * serial core request to change current uart setting
 */
static void bl234x_uart_set_termios(struct uart_port *port, 
									struct ktermios *new,
									struct ktermios *old)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;
	unsigned char cval, fcr = 0;
	unsigned long flags;
	unsigned int baud, quot;

	switch (new->c_cflag & CSIZE) {
	case CS5:
		cval = UART_LCR_WLEN5;
		break;
	case CS6:
		cval = UART_LCR_WLEN6;
		break;
	case CS7:
		cval = UART_LCR_WLEN7;
		break;
	default:
	case CS8:
		cval = UART_LCR_WLEN8;
		break;
	}

	if (new->c_cflag & CSTOPB)
		cval |= UART_LCR_STOP;
	if (new->c_cflag & PARENB)
		cval |= UART_LCR_PARITY;
	if (!(new->c_cflag & PARODD))
		cval |= UART_LCR_EPAR;
#ifdef CMSPAR
	if (new->c_cflag & CMSPAR)
		cval |= UART_LCR_SPAR;
#endif

	/*
	 * Ask the core to calculate the divisor for us.
	 */
	baud = uart_get_baud_rate(port, new, old, 0, port->uartclk/16);
	quot = uart_get_divisor(port, baud);

	if (baud < 2400)
		fcr = UART_FCR_ENABLE_FIFO | UART_FCR_TRIGGER_1;
	else
		fcr = UART_FCR_ENABLE_FIFO | UART_FCR_R_TRIG_01;

	/*
	 * Ok, we're now changing the port state.  Do it with
	 * interrupts disabled.
	 */
	spin_lock_irqsave(&port->lock, flags);

	/*
	 * Update the per-port timeout.
	 */
	uart_update_timeout(port, new->c_cflag, baud);

	port->read_status_mask = UART_LSR_OE | UART_LSR_THRE | UART_LSR_DR;
	if (new->c_iflag & INPCK)
		port->read_status_mask |= UART_LSR_FE | UART_LSR_PE;
	if (new->c_iflag & (BRKINT | PARMRK))
		port->read_status_mask |= UART_LSR_BI;

	/*
	 * Characteres to ignore
	 */
	port->ignore_status_mask = 0;
	if (new->c_iflag & IGNPAR)
		port->ignore_status_mask |= UART_LSR_PE | UART_LSR_FE;
	if (new->c_iflag & IGNBRK) {
		port->ignore_status_mask |= UART_LSR_BI;
		/*
		 * If we're ignoring parity and break indicators,
		 * ignore overruns too (for real raw support).
		 */
		if (new->c_iflag & IGNPAR)
			port->ignore_status_mask |= UART_LSR_OE;
	}

	/*
	 * ignore all characters if CREAD is not set
	 */
	if ((new->c_cflag & CREAD) == 0)
		port->ignore_status_mask |= UART_LSR_DR;

	/*
	 * CTS flow control flag and modem status interrupts
	 */
	bl234x_up->ier &= ~UART_IER_MSI;
	if (UART_ENABLE_MS(port, new->c_cflag))
		bl234x_up->ier |= UART_IER_MSI;

	bl234x_up->ier |= UART_IER_RLSI | UART_IER_RDI;

	serial_out(bl234x_up, UART_IER, bl234x_up->ier);

	serial_outp(bl234x_up, UART_LCR, cval | UART_LCR_DLAB);/* set DLAB */

	serial_dl_write(bl234x_up, quot);

	serial_outp(bl234x_up, UART_LCR, cval);		/* reset DLAB */
	bl234x_up->lcr = cval;					/* Save LCR */
	if (fcr & UART_FCR_ENABLE_FIFO) {
		/* emulated UARTs (Lucent Venus 167x) need two steps */
		serial_outp(bl234x_up, UART_FCR, UART_FCR_ENABLE_FIFO);
	}
	serial_outp(bl234x_up, UART_FCR, fcr);		/* set fcr */
	bl234x_uart_set_mctrl(port, port->mctrl);
	spin_unlock_irqrestore(&port->lock, flags);
	/* Don't rewrite B0 */
	if (tty_termios_baud_rate(new))
		tty_termios_encode_baud_rate(new, baud, baud);
}

/*
 * serial core request to release uart iomem
 */
static void bl234x_uart_release_port(struct uart_port *port)
{
	release_mem_region(port->mapbase, RSET_UART_SIZE);
	iounmap(port->membase);
}


/*
 * serial core request to claim uart iomem
 */
static int bl234x_uart_request_port(struct uart_port *port)
{
	if (!request_mem_region(port->mapbase, RSET_UART_SIZE, PFX)) {
		dev_err(port->dev, PFX " Memory region busy\n");
		return -EBUSY;
	}

	port->membase = ioremap_nocache(port->mapbase, RSET_UART_SIZE);
	if (!port->membase) {
		dev_err(port->dev, PFX " Unable to map registers\n");
		release_mem_region(port->mapbase, RSET_UART_SIZE);
		return -EBUSY;
	}
	return 0;
}

static void bl234x_uart_autoconfig(struct bl234x_uart_port *up)
{
	unsigned char save_lcr, save_mcr;
	unsigned long flags;

	if (!up->port.mapbase && !up->port.membase)
		return;

	/*
	 * We really do need global IRQs disabled here - we're going to
	 * be frobbing the chips IRQ enable register to see if it exists.
	 */
	spin_lock_irqsave(&up->port.lock, flags);

	save_mcr = serial_in(up, UART_MCR);
	save_lcr = serial_in(up, UART_LCR);

	serial_outp(up, UART_LCR, 0xBF);
	serial_outp(up, UART_EFR, 0);
	serial_outp(up, UART_LCR, 0);

	serial_outp(up, UART_FCR, UART_FCR_ENABLE_FIFO);

	serial_outp(up, UART_LCR, save_lcr);

	up->tx_loadsz = 16;

	/*
	 * Reset the UART.
	 */
	serial_outp(up, UART_MCR, save_mcr);
	bl234x_uart_clear_fifos(up);
	serial_in(up, UART_RX);
	serial_outp(up, UART_IER, 0);

	spin_unlock_irqrestore(&up->port.lock, flags);
	DEBUG_AUTOCONF("type=%s\n", uart_config[up->port.type].name);
}

/*
 * serial core request to do any port required autoconfiguration
 */
static void bl234x_uart_config_port(struct uart_port *port, int flags)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;
	
	if (flags & UART_CONFIG_TYPE) {
		if (bl234x_uart_request_port(port))
			return;

		bl234x_uart_autoconfig(bl234x_up);
		
		port->type = PORT_BL234X;
	}
}

/*
 * serial core request to check that port information in serinfo are
 * suitable
 */
static int
bl234x_uart_verify_port(struct uart_port *port, struct serial_struct *serinfo)
{
	if (port->type != PORT_BCM681X)
		return -EINVAL;
	if (port->irq != serinfo->irq)
		return -EINVAL;
	if (port->iotype != serinfo->io_type)
		return -EINVAL;
	if (port->mapbase != (unsigned long)serinfo->iomem_base)
		return -EINVAL;
	return 0;
}

static struct uart_ops bl234x_uart_ops = {
	.tx_empty	= bl234x_uart_tx_empty,
	.set_mctrl	= bl234x_uart_set_mctrl,
	.get_mctrl	= bl234x_uart_get_mctrl,
	.stop_tx	= bl234x_uart_stop_tx,
	.start_tx	= bl234x_uart_start_tx,
	.stop_rx	= bl234x_uart_stop_rx,
	.enable_ms	= bl234x_uart_enable_ms,
	.break_ctl	= bl234x_uart_break_ctl,
	.startup	= bl234x_uart_startup,
	.shutdown	= bl234x_uart_shutdown,
	.set_termios	= bl234x_uart_set_termios,
	.type		= bl234x_uart_type,
	.release_port	= bl234x_uart_release_port,
	.request_port	= bl234x_uart_request_port,
	.config_port	= bl234x_uart_config_port,
	.verify_port	= bl234x_uart_verify_port,
};

#ifdef CONFIG_SERIAL_BL234X_CONSOLE

/*
 *	Wait for transmitter & holding register to empty
 */
static inline void wait_for_xmitr(struct bl234x_uart_port *up, int bits)
{
	unsigned int status, tmout = 10000;

	/* Wait up to 10ms for the character(s) to be sent. */
	do {
		status = serial_in(up, UART_LSR);

		up->lsr_saved_flags |= status & BL234X_LSR_SAVE_FLAGS;

		if (--tmout == 0)
			break;
		udelay(1);
	} while ((status & bits) != bits);
}

static void bl234x_console_putchar(struct uart_port *port, int ch)
{
	struct bl234x_uart_port *bl234x_up = (struct bl234x_uart_port *)port;
	
	wait_for_xmitr(bl234x_up, UART_LSR_THRE);
	serial_out(bl234x_up, UART_TX, ch);
}

/*
 *	Print a string to the serial port trying not to disturb
 *	any possible real use of the port...
 *
 *	The console_lock must be held when we get here.
 */
static void
bl234x_console_write(struct console *co, const char *s, unsigned int count)
{
	struct bl234x_uart_port *bl234x_up = &bl234x_uart_ports[co->index];
	struct uart_port *port = &bl234x_up->port;
	unsigned long flags;
	unsigned int ier;
	int locked = 1;

	touch_nmi_watchdog();

	local_irq_save(flags);
	if (port->sysrq) {
		/* bl234x_uart_handle_port() already took the lock */
		locked = 0;
	} else if (oops_in_progress) {
		locked = spin_trylock(port->lock);
	} else
		spin_lock(port->lock);

	/*
	 *	First save the IER then disable the interrupts
	 */
	ier = serial_in(bl234x_up, UART_IER);

	serial_out(bl234x_up, UART_IER, 0);

	uart_console_write(port, s, count, bl234x_console_putchar);

	/*
	 *	Finally, wait for transmitter to become empty
	 *	and restore the IER
	 */
	wait_for_xmitr(bl234x_up, BOTH_EMPTY);
	serial_out(bl234x_up, UART_IER, ier);

	if (locked)
		spin_unlock(port->lock);
	local_irq_restore(flags);
}

static int __init bl234x_console_setup(struct console *co, char *options)
{
	struct uart_port *port;
	int baud = 9600;
	int bits = 8;
	int parity = 'n';
	int flow = 'n';

	if (co->index < 0 || co->index >= BL234X_NR_UARTS)
		return -EINVAL;
	
	port = &bl234x_uart_ports[co->index].port;
	if (!port->membase)
		return -ENODEV;

	if (options)
		uart_parse_options(options, &baud, &parity, &bits, &flow);

	return uart_set_options(port, co, baud, parity, bits, flow);
}

static struct uart_driver bl234x_uart_driver;

static struct console bl234x_console = {
	.name		= "ttyS",
	.write		= bl234x_console_write,
	.device		= uart_console_device,
	.setup		= bl234x_console_setup,
	.flags		= CON_PRINTBUFFER,
	.index		= -1,
	.data		= &bl234x_uart_driver,
};

static void __init bl234x_console_init_ports(void)
{
	static int first = 1;
	int i;

	if (!first)
		return;
	
	first = 0;

	for (i = 0; i < BL234X_NR_UARTS; i++) {
		struct uart_port *port = &bl234x_uart_ports[i].port;

		port->line = i;
		spin_lock_init(&port->lock);

		port->ops = &bl234x_uart_ops;
	}
}

static int __init bl234x_console_init(void)
{
	bl234x_console_init_ports();
	register_console(&bl234x_console);
	return 0;
}
console_initcall(bl234x_console_init);

#define BL234X_CONSOLE	&bl234x_console
#else
#define BL234X_CONSOLE	NULL
#endif /* CONFIG_SERIAL_BL234X_CONSOLE */


static struct uart_driver bl234x_uart_driver = {
	.owner		= THIS_MODULE,
	.driver_name	= "bl234x-uart",
	.dev_name	= "ttyS",
	.major		= TTY_MAJOR,
	.minor		= 64,
	.nr			= 1,
	.cons		= BL234X_CONSOLE,
};

/*
 * Register a set of serial devices attached to a platform device.  The
 * list is terminated with a zero flags entry, which means we expect
 * all entries to have at least UPF_BOOT_AUTOCONF set.
 */
static int __devinit bl234x_uart_probe(struct platform_device *pdev)
{
	struct resource *res_mem, *res_irq;
	struct uart_port *port;
	struct clk *clk;
	int ret;

	if (pdev->id < 0 || pdev->id >= BL234X_NR_UARTS)
		return -EINVAL;

	port = &bl234x_uart_ports[pdev->id].port;

	if (port->membase)
		return -EBUSY;

	res_mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res_mem)
		return -ENODEV;

	res_irq = platform_get_resource(pdev, IORESOURCE_IRQ, 0);
	if (!res_irq)
		return -ENODEV;

	clk = clk_get(&pdev->dev, "periph");
	if (IS_ERR(clk))
		return -ENODEV;
	
	memset(port, 0, sizeof(*port));
	port->iotype = UPIO_MEM32;
	port->mapbase = res_mem->start;
	port->irq = res_irq->start;
	port->ops = &bl234x_uart_ops;
	port->flags = UPF_BOOT_AUTOCONF | UPF_SKIP_TEST | UPF_IOREMAP;
	port->dev = &pdev->dev;
	port->fifosize = 16;
	port->regshift = 2;
	port->uartclk = clk_get_rate(clk);

	ret = uart_add_one_port(&bl234x_uart_driver, port);
	if (ret) {
		return ret;
	}

	return 0;
}

/*
 * Remove serial ports registered against a platform device.
 */
static int __devexit bl234x_uart_remove(struct platform_device *pdev)
{
	struct bl234x_uart_port *bl234x_up;
	struct uart_port *port;

	bl234x_up = platform_get_drvdata(pdev);
	port = &bl234x_up->port;
	uart_remove_one_port(&bl234x_uart_driver, port);
	platform_set_drvdata(pdev, NULL);
	/* mark port as free */
	port->membase = 0;
	return 0;
}

static int bl234x_uart_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct bl234x_uart_port *bl234x_up;
	struct uart_port *port;

	bl234x_up = platform_get_drvdata(pdev);
	port = &bl234x_up->port;
	uart_suspend_port(&bl234x_uart_driver, port);
	
	return 0;
}

static int bl234x_uart_resume(struct platform_device *pdev)
{
	struct bl234x_uart_port *bl234x_up;
	struct uart_port *port;

	bl234x_up = platform_get_drvdata(pdev);
	port = &bl234x_up->port;
	uart_resume_port(&bl234x_uart_driver, port);
	
	return 0;
}

static struct platform_driver bl234x_uart_platform_driver = {
	.probe		= bl234x_uart_probe,
	.remove		= __devexit_p(bl234x_uart_remove),
	.suspend	= bl234x_uart_suspend,
	.resume		= bl234x_uart_resume,
	.driver		= {
		.name	= "bl234x-uart",
		.owner	= THIS_MODULE,
	},
};

static int __init bl234x_uart_init(void)
{
	int ret;

	ret = uart_register_driver(&bl234x_uart_driver);
	if (ret)
		return ret;

	ret = platform_driver_register(&bl234x_uart_platform_driver);
	if (ret)
		uart_unregister_driver(&bl234x_uart_driver);

	return ret;
}

static void __exit bl234x_uart_exit(void)
{
	platform_driver_unregister(&bl234x_uart_platform_driver);
	uart_unregister_driver(&bl234x_uart_driver);
}

module_init(bl234x_uart_init);
module_exit(bl234x_uart_exit);

MODULE_ALIAS("platform:bl234x-uart");
MODULE_AUTHOR("Arcadyan Technology");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Broadlight 234x inetgrated uart driver");
MODULE_VERSION(DRV_VER);

MODULE_ALIAS_CHARDEV_MAJOR(TTY_MAJOR);

