/*
 * Copyright (c) 2011  Bjorn Mork <bjorn@xxxxxxx>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/ethtool.h>
#include <linux/mii.h>
#include <linux/usb.h>
#include <linux/usb/cdc.h>
#include <linux/usb/usbnet.h>
#include "qmi.h"

/* overloading the cdc_state structure */
struct qmi_wwan_state {
	struct qmi_state		*qmi;
	struct usb_cdc_union_desc	*u;
	struct usb_cdc_ether_desc	*ether;
	struct usb_interface		*control;
	struct usb_interface		*data;
};


static void qmi_wwan_cdc_status(struct usbnet *dev, struct urb *urb)
{
	struct usb_cdc_notification	*event;
	struct qmi_wwan_state *info = (void *)&dev->data;

	if (urb->actual_length < sizeof *event) {
		devdbg(dev, "short status urb rcvd: %d bytes\n", urb->actual_length);
		return;
	}

	event = urb->transfer_buffer;
	switch (event->bNotificationType) {
	case USB_CDC_NOTIFY_RESPONSE_AVAILABLE:
		qmi_recv(info->qmi);	/* request response */
		break;
	default:
		/* reuse usbnet() for handling other messages */
		usbnet_cdc_status(dev, urb);
	}
}

static int qmi_wwan_cdc_ecmlike_bind(struct usbnet *dev, struct usb_interface *intf)
{
	int status;
	struct qmi_wwan_state *info = (void *)&dev->data;

	status = usbnet_generic_cdc_bind(dev, intf);
	if (status < 0)
		return status;

	/* initialize QMI data */
	info->qmi = qmi_init(intf);
	if (!info->qmi) {
		usbnet_cdc_unbind(dev, intf);
		return -1;
	}
	return 0;
}

static int qmi_wwan_cdc_bind(struct usbnet *dev, struct usb_interface *intf)
{
	int status;
	struct qmi_wwan_state *info = (void *)&dev->data;

	memset(info, 0, sizeof(*info));
	status = usbnet_get_endpoints(dev, intf);
	if (status < 0)
		return status;

	/* initialize QMI data */
	info->qmi = qmi_init(intf);
	if (!info->qmi)
		return -1;

	info->control = intf;
	return 0;
}

static void qmi_wwan_cdc_unbind(struct usbnet *dev, struct usb_interface *intf)
{
	struct qmi_wwan_state *info = (void *)&dev->data;

	/* release our private structure */
	qmi_exit(info->qmi);
	info->qmi = NULL;

	/* disconnect */
	usbnet_cdc_unbind(dev, intf);
}

static int qmi_wwan_reset(struct usbnet *dev)
{
	struct qmi_wwan_state *info = (void *)&dev->data;

	qmi_reset(info->qmi);
	return 1;
}

static int qmi_wwan_stop(struct usbnet *dev)
{
	struct qmi_wwan_state *info = (void *)&dev->data;

	qmi_disconnect(info->qmi);
	return 1;
}

/* abusing check_connect for triggering QMI state transitions */
static int qmi_wwan_check_connect(struct usbnet *dev)
{
	struct qmi_wwan_state *info = (void *)&dev->data;
	return qmi_conn_state(info->qmi);
}

static const struct driver_info	qmi_wwan_ecm_info = {
	.description =	"QMI speaking CDC ECM like wwan device",
	.flags =	 FLAG_WWAN,
	.bind =		 qmi_wwan_cdc_ecmlike_bind,
	.unbind =	 qmi_wwan_cdc_unbind,
	.status =	 qmi_wwan_cdc_status,
	.check_connect = qmi_wwan_check_connect,
	.stop =          qmi_wwan_stop,
	.reset =         qmi_wwan_reset,
};

static const struct driver_info	qmi_wwan_info = {
	.description =	"QMI speaking wwan device",
	.flags =	 FLAG_WWAN,
	.bind =		 qmi_wwan_cdc_bind,
	.unbind =	 qmi_wwan_cdc_unbind,
	.status =	 qmi_wwan_cdc_status,
	.check_connect = qmi_wwan_check_connect,
	.stop =          qmi_wwan_stop,
	.reset =         qmi_wwan_reset,
};

#define HUAWEI_VENDOR_ID	0x12D1

static const struct usb_device_id products[] = {
{
	/* Qmi_Wwan E392, E398, ++? */
	.match_flags    =   USB_DEVICE_ID_MATCH_VENDOR
		 | USB_DEVICE_ID_MATCH_INT_INFO,
	.idVendor               = HUAWEI_VENDOR_ID,
	.bInterfaceClass	= USB_CLASS_VENDOR_SPEC,
	.bInterfaceSubClass	= 1,
	.bInterfaceProtocol	= 9,
	.driver_info = (unsigned long)&qmi_wwan_ecm_info,
}, {
	/* Qmi_Wwan device id 1413 ++? */
	.match_flags    =   USB_DEVICE_ID_MATCH_VENDOR
		 | USB_DEVICE_ID_MATCH_INT_INFO,
	.idVendor               = HUAWEI_VENDOR_ID,
	.bInterfaceClass	= USB_CLASS_VENDOR_SPEC,
	.bInterfaceSubClass	= 6,
	.bInterfaceProtocol	= 255,
	.driver_info = (unsigned long)&qmi_wwan_ecm_info,
}, {
	/* Qmi_Wwan E392, E398, ++? "Windows mode" */
	.match_flags    =   USB_DEVICE_ID_MATCH_VENDOR
		 | USB_DEVICE_ID_MATCH_INT_INFO,
	.idVendor               = HUAWEI_VENDOR_ID,
	.bInterfaceClass	= USB_CLASS_VENDOR_SPEC,
	.bInterfaceSubClass	= 1,
	.bInterfaceProtocol	= 17,
	.driver_info = (unsigned long)&qmi_wwan_info,
},
{ },		/* END */
};
MODULE_DEVICE_TABLE(usb, products);

static struct usb_driver qmi_wwan_driver = {
	.name =		"qmi_wwan",
	.id_table =	products,
	.probe =	usbnet_probe,
	.disconnect =	usbnet_disconnect,
	.suspend =	usbnet_suspend,
	.resume =	usbnet_resume,
	.reset_resume =	usbnet_resume,
	.supports_autosuspend = 1,
};

static int __init qmi_wwan_init(void)
{
	/* we remap struct (cdc_state) so we should be compatible */
	BUILD_BUG_ON(sizeof(struct cdc_state) != sizeof(struct qmi_wwan_state));

	return usb_register(&qmi_wwan_driver);
}
module_init(qmi_wwan_init);

static void __exit qmi_wwan_exit(void)
{
	usb_deregister(&qmi_wwan_driver);
}
module_exit(qmi_wwan_exit);

MODULE_AUTHOR("Bjorn Mork <bjorn@xxxxxxx>");
MODULE_DESCRIPTION("Qualcomm Messaging Interface (QMI) WWAN driver");
MODULE_LICENSE("GPL");