/*
 * Copyright (c) 2011  Bjorn Mork <bjorn@xxxxxxx>
 * Copyright (c) 2010, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 */

/*
 * Dealing with devices using Qualcomm Messaging Interface (QMI) for
 * configuration.  Full documentation of the protocol can be obtained
 * from http://developer.qualcomm.com/
 *
 * Some of this code may be inspired by code from the Qualcomm Gobi
 * 2000 and Gobi 3000 drivers, available at http://www.codeaurora.org/
 */

#define QMI_BUFLEN 128

enum qmi_device_state {
	QMI_STATE_INIT = 0, /* allocate CIDs, verify pin code */
	QMI_STATE_DOWN,    /* connection is down */
	QMI_STATE_UP,      /* connection is up */
	QMI_STATE_ERR,     /* fatal and final error state, e.g. no CID available - no way out of here! */
};

/* the different QMI subsystems we are using */
enum qmi_subsystems {
	QMI_CTL = 0,
	QMI_WDS,
	QMI_DMS,
#if defined(ARCADYAN)
	QMI_NAS,
#endif
	QMI_SYSMAX,
};

#define QMI_FLAG_RECV		0x00000001
#define QMI_FLAG_PINOK		0x00000002

struct qmi_state {
	struct usb_device *dev;
	struct urb *urb;		/* receive urb */
	unsigned char *rcvbuf;		/* pre-allocated receive buffer */
	struct usb_ctrlrequest setup;	/* the receive setup - 8 bytes */
	unsigned long flags;		/* used for en/dis-abling functionality on the fly */
	u8 state;			/* for connection state machine */
	u8 wds_status;			/* current value for QMI_WDS message 0x0022 or 0 if uninitialized */
	__le16 intfnr;			/* keeping track of the interface we are referring to */
	u8 handle[4];			/* connection handle needed for disconnect */
	u8 cid[QMI_SYSMAX];		/* keeping track of cid per subsystem */
};

/* combined QMUX and SDU */
struct qmux {
	u8 tf;		/* always 1 */
	__le16 len;	/* excluding tf */
	u8 ctrl;	/* b7: sendertype 1 => service, 0 => control point */
	u8 service;	/* 0 => QMI_CTL, 1 => QMI_WDS, .. */
	u8 qmicid;	/* client id or 0xff for broadcast */
	u8 flags;	/* always 0 for req */
	union {
		__le16 w;  /* each control point maintains a transaction id counter - non-zero */
		u8 b;	/* system QMI_CTL uses one byte transaction ids! */
	} tid;
	u8 msg[];	/* one or more messages */
} __packed;

struct qmi_msg {
	__le16 msgid;
	__le16 len;
	u8 tlv[];	/* zero or more tlvs */
} __packed;

struct qmi_tlv {
	u8 type;
	__le16 len;
	u8 bytes[];
} __packed;

struct qmi_tlv_response_data {
	__le16 error;
	__le16 code;
} __packed;


/* reset state to INIT */
extern int qmi_reset(struct qmi_state *qmi);

/* disconnect WWAN connection */
extern int qmi_disconnect(struct qmi_state *qmi);

/* must be called whenever USB_CDC_NOTIFY_RESPONSE_AVAILABLE is received */
extern int qmi_recv(struct qmi_state *qmi);

/* called periodically, whenever a state transition is wanted */
extern int qmi_conn_state(struct qmi_state *qmi);

/* initialize */
extern struct qmi_state *qmi_init(struct usb_interface *intf);

/* clean up */
extern int qmi_exit(struct qmi_state *qmi);