/*
 * Copyright (c) 2011  Bjorn Mork <bjorn@xxxxxxx>
 * Copyright (c) 2010, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 */

/*
 * Dealing with devices using Qualcomm Messaging Interface (QMI) for
 * configuration.  Full documentation of the protocol can be obtained
 * from http://developer.qualcomm.com/
 *
 * Some of this code may be inspired by code from the Qualcomm Gobi
 * 2000 and Gobi 3000 drivers, available at http://www.codeaurora.org/
 */

#include <linux/module.h>
#include <linux/slab.h>
#include <linux/usb.h>
#include <linux/usb/cdc.h>
#include "qmi.h"


/* QMI protocol debug output */
static int qmi_debug;
module_param(qmi_debug, int, 0);
MODULE_PARM_DESC(qmi_debug, "Enable QMI protocol debugging");
#if defined(ARCADYAN)
static  int qmi_tech = 0; /* 0 is AUTO, 1 is GSM, 2 is UMTS, 3 is LTE */
module_param(qmi_tech, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(qmi_tech, "Technology");
static char *qmi_apn = NULL;
module_param(qmi_apn, charp, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(qmi_apn, "APN");
static  int qmi_auto = 1;
module_param(qmi_auto, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(qmi_auto, "Autoconnect");
#endif

/* find and return a pointer to the requested tlv */
struct qmi_tlv *qmi_get_tlv(u8 type, u8 *buf, size_t len)
{
	u8 *p;
	struct qmi_tlv *t;

	for (p = buf; p < buf + len; p += le16_to_cpu(t->len) + sizeof(struct qmi_tlv)) {
		t = (struct qmi_tlv *)p;
		if (t->type == type)
			return (p + le16_to_cpu(t->len) <= buf + len) ? t : NULL;
	}
	return NULL;
}


/* TLV 0x02 is status: return a negative QMI error code or 0 if OK */
int qmi_verify_status_tlv(u8 *buf, size_t len)
{
	struct qmi_tlv *tlv = (void *)qmi_get_tlv(0x02, buf, len);
	struct qmi_tlv_response_data *r = (void *)tlv->bytes;

	if (!tlv || le16_to_cpu(tlv->len) != 2 * sizeof(__le16))
		return -1; /* QMI_ERR_MALFORMED_MSG */

	return r->error ? -r->code : 0;
}

static char *decode_trans_flags(const u8 flags)
{
	switch (flags) {
	case 0: return "request";
	case 2:	return "response";
	case 4:	return "indication";
	case 6:	return "reserved";
	default: return "invalid";
	}
}

static char *qmi_system(u8 system)
{
	switch (system) {
	case QMI_CTL:	return "QMI_CTL";
	case QMI_WDS:	return "QMI_WDS";
	case QMI_DMS:	return "QMI_DMS";
#if defined(ARCADYAN)
	case QMI_NAS:	return "QMI_NAS";
#endif
	default: return "(unknown)";
	}
}


/* debug print TLVs from a QMI message */
static void qmi_tlv_dump(struct qmi_state *qmi, u8 *data, size_t len, u8 system, __le16 msgid)
{
	u8 *p;
	struct qmi_tlv *t;
	char linebuf[100];

	for (p = data; p < data + len; p += le16_to_cpu(t->len) + sizeof(struct qmi_tlv)) {
		t = (struct qmi_tlv *)p;

		/* mark an empty string */
		linebuf[0] = 0;
		if (t->type == 0x02) { /* status response */
			struct qmi_tlv_response_data *r = (void *)t->bytes;
			snprintf(linebuf, sizeof(linebuf), "%s (0x%04x)",
				r->error ? "FAILED" : "SUCCESS", r->code);
		} else {
			switch (system) {
			case QMI_WDS:
				switch (msgid) {
				case 0x0001:
					switch (t->type) {
					case 0x16:
						snprintf(linebuf, sizeof(linebuf), "tx: %d bps, rx: %d bps",
							*(__le32 *)t->bytes, *(__le32 *)(t->bytes + 4));
					}
				}
				break;
			case QMI_DMS:
				switch (msgid) {
				case 0x002b:
					if (t->type == 0x11 || t->type == 0x12)
						snprintf(linebuf, sizeof(linebuf),
							"PIN%d status=%d, %d verify retries left, %d unblock retries left",
							t->type & 0x3, t->bytes[0], t->bytes[1], t->bytes[2]);
				}
			}
		}

		/* do a default byte dump if the linebuf is empty */
		if (linebuf[0] == 0)
			hex_dump_to_buffer(t->bytes, le16_to_cpu(t->len), 16, 1, linebuf, sizeof(linebuf), true);

		dev_info(&qmi->dev->dev, "[0x%02x] (%d) %s\n", t->type, le16_to_cpu(t->len), linebuf);
	}
}

/* verify QMUX message header and return pointer to the (first) message if OK */
static struct qmi_msg *qmi_qmux_verify(u8 *data, size_t len)
{
	struct qmux *h =  (void *)data;
	struct qmi_msg *m;
	ssize_t hsize = sizeof(struct qmux);

	if (len < sizeof(struct qmux) || /* short packet */
		h->tf != 0x01 || le16_to_cpu(h->len) != (len - 1)) /* invalid QMUX packet! */
		return NULL;

	/* tid has a different size for QMI_CTL for some fucking stupid reason */
	if (h->service == QMI_CTL)
		hsize--;

	m = (struct qmi_msg *)(data + hsize);

	/* insanity checking before any further dereferencing... */
	if (hsize + sizeof(struct qmi_msg) + le16_to_cpu(m->len) > len)
		return NULL;

	return m;
}

/* debug print a QMUX packet */
static void qmi_dump_qmux(struct qmi_state *qmi, u8 *data, size_t len)
{
	struct qmux *h =  (void *)data;
	struct qmi_msg *m;

	m = qmi_qmux_verify(data, len);
	if (!m) {
		dev_info(&qmi->dev->dev, "invalid QMUX packet (%zu bytes)\n", len);
		return;
	}

	/* dump parts of the QMUX header and the message ID */
	dev_info(&qmi->dev->dev, "%s(0x%02x) %s: msg 0x%04x (len %d) from \"%s\" with cid=0x%02x%s and TLVs:\n",
		qmi_system(h->service),
		h->service,
		decode_trans_flags(h->service == QMI_CTL ? h->flags << 1 : h->flags),
		le16_to_cpu(m->msgid),
		le16_to_cpu(h->len),
		h->ctrl & 0x80 ? "service" : "control point",
		h->qmicid, h->qmicid == 0xff ? " (broadcast)" : "");

	/* dump the TLVs */
	qmi_tlv_dump(qmi, m->tlv, le16_to_cpu(m->len), h->service, le16_to_cpu(m->msgid));
}

static int qmi_dms_parse(struct qmi_state *qmi, struct qmi_msg *m)
{
	int status;
	struct qmi_tlv *tlv;

	if (!m)
		return -1;

	/* check and save status TLV */
	status = qmi_verify_status_tlv(m->tlv, le16_to_cpu(m->len));

	/* no QMI_DMS messages requires parsing unless status is OK */
	if (status < 0)
		return status;

	switch (le16_to_cpu(m->msgid)) {
	case 0x002b: /* QMI_DMS_UIM_GET_PIN_STATUS */
		if (status == 0) {
			/* PIN1 status */
			tlv = qmi_get_tlv(0x11, m->tlv, le16_to_cpu(m->len));
			if (tlv && le16_to_cpu(tlv->len) == 3) {
				switch (tlv->bytes[0]) {
				case 2: /* PIN enabled, verified */
				case 3: /* PIN disabled */
					qmi->flags |= QMI_FLAG_PINOK;
				}
			}
		}
	}

	return 0;
}

static int qmi_ctl_parse(struct qmi_state *qmi, struct qmi_msg *m)
{
	int status;
	struct qmi_tlv *tlv;

	if (!m)
		return -1;

	/* check and save status TLV */
	status = qmi_verify_status_tlv(m->tlv, le16_to_cpu(m->len));

	if (status < 0)
		return status;

	/* TLVs are message dependent */
	switch (le16_to_cpu(m->msgid)) {
	case 0x0022: /* allocate cid */
		/* TLV 0x01 is system + cid */
		tlv = qmi_get_tlv(0x01, m->tlv, le16_to_cpu(m->len));

		/* can only save CID if there is a slot for the subsystem */
		if (tlv && le16_to_cpu(tlv->len) == 2 && tlv->bytes[0] < sizeof(qmi->cid))
			qmi->cid[tlv->bytes[0]] = tlv->bytes[1];
		else
			return -1;
	}

	return status;
}

static int qmi_wds_parse(struct qmi_state *qmi, struct qmi_msg *m)
{
	int status;
	struct qmi_tlv *tlv;

	if (!m)
		return -1;

	/* check and save status TLV */
	status = qmi_verify_status_tlv(m->tlv, le16_to_cpu(m->len));

	if (le16_to_cpu(m->msgid) == 0x0020 && -status == 0x001A) { /* QMI_ERR_NO_EFFECT */
		memset(qmi->handle, 0xff, sizeof(qmi->handle)); /* global handle */
		return 0;
	}

	if (status < 0)
		return status;

	switch (le16_to_cpu(m->msgid)) {
	case 0x0020: /* QMI_WDS_START_NETWORK_INTERFACE */
		/* TLV 0x01 is a 4 byte packet data handle we need to keep */
		tlv = qmi_get_tlv(0x01, m->tlv, le16_to_cpu(m->len));
		if (tlv && le16_to_cpu(tlv->len) == sizeof(qmi->handle))
			memcpy(qmi->handle, tlv->bytes, sizeof(qmi->handle));

		break;

	case 0x0022: /* QMI_WDS_GET_PKT_SRVC_STATUS */
		/* TLV 0x01 is a 1 byte connection status */
		tlv = qmi_get_tlv(0x01, m->tlv, le16_to_cpu(m->len));
		if (tlv && le16_to_cpu(tlv->len) == 1)
			qmi->wds_status = tlv->bytes[0];

	}

	return status;
}

static void qmi_read_callback(struct urb *urb)
{
	struct qmi_state *qmi = (void *)urb->context;

	if (urb->status == 0) {
		struct qmux *h = (void *)urb->transfer_buffer;
		struct qmi_msg *m = qmi_qmux_verify(urb->transfer_buffer, urb->actual_length);

		switch (h->service) {
		case QMI_CTL:
			qmi_ctl_parse(qmi, m);
			break;
		case QMI_WDS:
			qmi_wds_parse(qmi, m);
			break;
		case QMI_DMS:
			qmi_dms_parse(qmi, m);
		}

		if (qmi_debug)
			qmi_dump_qmux(qmi, urb->transfer_buffer, urb->actual_length);
	}
}

static int qmi_recv_sync(struct qmi_state *qmi)
{
	int status;

	status = usb_control_msg(qmi->dev,
			       usb_rcvctrlpipe(qmi->dev, 0),
			       USB_CDC_GET_ENCAPSULATED_RESPONSE, USB_DIR_IN|USB_TYPE_CLASS|USB_RECIP_INTERFACE,
			       0, qmi->intfnr,
			       qmi->rcvbuf, QMI_BUFLEN, 1000);

	if (status > 0 && qmi_debug)
		qmi_dump_qmux(qmi, qmi->rcvbuf, status);

	return status;
}

static int qmi_send_sync(struct qmi_state *qmi, unsigned char *msg, size_t len)
{
	int status;

	status = usb_control_msg(qmi->dev,
				usb_sndctrlpipe(qmi->dev, 0),
				USB_CDC_SEND_ENCAPSULATED_COMMAND, USB_DIR_OUT|USB_TYPE_CLASS|USB_RECIP_INTERFACE,
				0, qmi->intfnr,
				msg, len, 1000);

	if (qmi_debug) /* NOTE: regardless of status! */
		qmi_dump_qmux(qmi, msg, len);

	return status;
}

/* send message and wait for the reply - returns pointer to reply message */
struct qmi_msg *qmi_send_and_wait_for_ack(struct qmi_state *qmi, unsigned char *data, size_t len, int timeout)
{
	int ret;
	u8 system;
	__le16 msgid;
	struct qmi_msg *msg = qmi_qmux_verify(data, len);
	struct qmux *h = (void *)data;

	/* should never happen? */
	if (!msg)
		return NULL;
	msgid = msg->msgid;
	system = h->service;

	/* flush any pending messages  */
	do {
		ret = qmi_recv_sync(qmi);
	} while (ret > 0);

	/* send our message */
	ret = qmi_send_sync(qmi, data, len);
	if (ret != len)
		return NULL;

	/* wait a while for the (correct) reply */
	do {
		ret = qmi_recv_sync(qmi);
		msg = qmi_qmux_verify(qmi->rcvbuf, ret);
		if (msg) {
			h = (struct qmux *)qmi->rcvbuf;
			if (msg->msgid == msgid && h->service == system) /* found it */
				return msg;

		}
		if (timeout--)
			msleep(100);
	} while (timeout > 0);

	/* no reply */
	return NULL;
}

/* add a TLV to an existing QMUX */
static size_t qmi_add_tlv(u8 *buf, size_t buflen, u8 type, const u8 *tlvdata, size_t datalen)
{
	struct qmux *h =  (void *)buf;
	struct qmi_msg *m = (void *)(buf + sizeof(struct qmux) - (h->service == QMI_CTL));
	struct qmi_tlv *tlv = (void *)(buf + le16_to_cpu(h->len) + 1);
	size_t tlvlen = sizeof(struct qmi_tlv) + datalen;

	if (buflen < le16_to_cpu(h->len) + tlvlen - 1)
		return -1;

	tlv->type = type;
	tlv->len = cpu_to_le16(datalen);
	memcpy(tlv->bytes, tlvdata, datalen);
	h->len = cpu_to_le16(le16_to_cpu(h->len) + tlvlen);
	m->len = cpu_to_le16(le16_to_cpu(m->len) + tlvlen);
	return le16_to_cpu(h->len) + 1;
}

/* get a new transaction id */
static __le16 new_tid(void)
{
	static __le16 tid;
	return ++tid;
}

/* assemble a complete QMUX packet with one QMI message */
static size_t qmi_create_msg(char *buf, size_t buflen, u8 cid, u8 system, __le16 msgid)
{
	struct qmux *h =  (void *)buf;
	struct qmi_msg *m;
	size_t hsize = sizeof(struct qmux);

	/* this is weird, but it seems as though QMI_CTL uses 1 byte
	   transaction ids while the other systems use 2 bytes as specified
	   in the standard */
	if (system == QMI_CTL)
		hsize--;

	/* sanity check */
	if (buflen < hsize + sizeof(struct qmi_msg))
		return -1;

	h->tf = 1;     /* always 1 */
	h->ctrl = 0;   /* 0 - control point */
	h->service = system;
	h->qmicid = cid;
	h->flags = 0;  /* 0 - request */

	/* see comment above */
	if (system == QMI_CTL)
		h->tid.b = new_tid() & 0xff;
	else
		h->tid.w = new_tid();

	m = (struct qmi_msg *)(buf + hsize);
	m->msgid = cpu_to_le16(msgid);

	h->len = cpu_to_le16(hsize + sizeof(struct qmi_msg) - 1);
	m->len = 0;
	return hsize + sizeof(struct qmi_msg);
}

/* QMI_CTL msg 0x0022 is "request cid", TLV 0x01 is system */
static int qmi_ctl_request_cid(struct qmi_state *qmi, u8 system)
{
	size_t len;
	u8 tlvdata[1] = { system };
	char buf[32];

	/* return immediately if a CID is already allocated */
	if (qmi->cid[system])
		return qmi->cid[system];

	/* else request a new one */
	len = qmi_create_msg(buf, sizeof(buf), 0, QMI_CTL, 0x0022);
	len = qmi_add_tlv(buf, sizeof(buf), 0x01, tlvdata, sizeof(tlvdata));

	/* send message */
	return qmi_ctl_parse(qmi, qmi_send_and_wait_for_ack(qmi, buf, len, 30));
}

/* QMI_WDS msg 0x0001 is "" */
static int qmi_wds_report(struct qmi_state *qmi)
{
	size_t len;
	u8 cid = qmi->cid[QMI_WDS];
	u8 tlvdata[1] = { 1 };	/* generic "enable" */
	char buf[32];
#if defined(ARCADYAN)
	char stats[5];
	u32 mask = cpu_to_le32(0x000000c3);
#endif

	/* enable connection status reports */
	len = qmi_create_msg(buf, sizeof(buf), cid, QMI_WDS, 0x0001);
	len = qmi_add_tlv(buf, sizeof(buf), 0x10, tlvdata, sizeof(tlvdata));	/* current channel rate */
	len = qmi_add_tlv(buf, sizeof(buf), 0x15, tlvdata, sizeof(tlvdata));	/* current data bearer tech */
#if defined(ARCADYAN)
	stats[0] = 60;
	memcpy(&stats[1], &mask, 4);
	len = qmi_add_tlv(buf, sizeof(buf), 0x11, stats, 5);
#endif

	/* send message */
	return qmi_wds_parse(qmi, qmi_send_and_wait_for_ack(qmi, buf, len, 30));
}

/* QMI_WDS msg 0x0020 is "" */
static int qmi_wds_start(struct qmi_state *qmi)
{
	size_t len;
	u8 cid = qmi->cid[QMI_WDS];
	char buf[32];

	if (!cid)
		return -1;

	len = qmi_create_msg(buf, sizeof(buf), cid, QMI_WDS, 0x0020);
#if defined(ARCADYAN)
	if (qmi_apn && (qmi_apn[0] != '\0') && (qmi_apn[0] != '\n')) {
		char *cp = qmi_apn;
		while (*cp != '\0') { /* remove new line character, don't know why it is appended... */
			if (*cp == '\n') {
				*cp = '\0';
				break;
			}
			++cp;
		}
		len = qmi_add_tlv(buf, sizeof(buf), 0x14, qmi_apn, strlen(qmi_apn));
	}
#endif

	/* send message */
	return qmi_wds_parse(qmi, qmi_send_and_wait_for_ack(qmi, buf, len, 300));
}

/* QMI_WDS msg 0x0021 is "" */
static int qmi_wds_stop(struct qmi_state *qmi)
{
	size_t len;
	u8 cid = qmi->cid[QMI_WDS];
	char buf[32];

	if (!cid)
		return -1;

	len = qmi_create_msg(buf, sizeof(buf), cid, QMI_WDS, 0x0021);
	len = qmi_add_tlv(buf, sizeof(buf), 0x01, qmi->handle, sizeof(qmi->handle));	 /* packet data handle */

	return qmi_wds_parse(qmi, qmi_send_and_wait_for_ack(qmi, buf, len, 30));
}

/* QMI_WDS msg 0x0022 is "" */
static int qmi_wds_status(struct qmi_state *qmi)
{
	int ret;
	size_t len;
	u8 cid = qmi->cid[QMI_WDS];
	char buf[32];

	if (!cid)
		return -1;

	len = qmi_create_msg(buf, sizeof(buf), cid, QMI_WDS, 0x0022);

	ret = qmi_wds_parse(qmi, qmi_send_and_wait_for_ack(qmi, buf, len, 30));
	if (ret < 0)
		return ret;

	return qmi->wds_status;
}

#if defined(ARCADYAN)
static int qmi_nas_sys_selection(struct qmi_state *qmi)
{
	size_t len;
	u8 cid = qmi->cid[QMI_NAS];
	char buf[32];
	u16 mode = cpu_to_le16(0x001c);
	u8 acq_order[4] = {3, 0x08, 0x05, 0x04}; /* the first element is number if orders */

	if (!cid)
		return -1;

	switch (qmi_tech) {
	case 1:
		mode = cpu_to_le16(0x0004);
		acq_order[0] = 1;
		acq_order[1] = 0x04;
		break;

	case 2:
		mode = cpu_to_le16(0x0008);
		acq_order[0] = 1;
		acq_order[1] = 0x05;
		break;

	case 3:
		mode = cpu_to_le16(0x0010);
		acq_order[0] = 1;
		acq_order[1] = 0x08;
		break;
	}

	len = qmi_create_msg(buf, sizeof(buf), cid, QMI_NAS, 0x0033);
	len = qmi_add_tlv(buf, sizeof(buf), 0x11, (u8 *) &mode, sizeof(mode));
	len = qmi_add_tlv(buf, sizeof(buf), 0x1E, acq_order, acq_order[0] + 1);


	qmi_send_and_wait_for_ack(qmi, buf, len, 30);

#if 1
	len = qmi_create_msg(buf, sizeof(buf), cid, QMI_NAS, 0x0024);

	qmi_send_and_wait_for_ack(qmi, buf, len, 30);
#endif

	len = qmi_create_msg(buf, sizeof(buf), cid, QMI_NAS, 0x0034);

	qmi_send_and_wait_for_ack(qmi, buf, len, 30);

#if 1
	len = qmi_create_msg(buf, sizeof(buf), cid, QMI_NAS, 0x004D);

	qmi_send_and_wait_for_ack(qmi, buf, len, 30);
#endif

	return 0;
}

static int qmi_wds_autoconnect(struct qmi_state *qmi)
{
	size_t len;
	u8 cid = qmi->cid[QMI_WDS];
	char buf[32];
	char autoconnect = 0;

	if (!cid)
		return -1;

	autoconnect = qmi_auto;

	len = qmi_create_msg(buf, sizeof(buf), cid, QMI_WDS, 0x0051);
	len = qmi_add_tlv(buf, sizeof(buf), 0x01, &autoconnect, 1);

	/* send message */
	return qmi_wds_parse(qmi, qmi_send_and_wait_for_ack(qmi, buf, len, 30));
}
#endif

/* QMI_CTL msg 0x0023 is "release cid", TLV 0x01 is system + cid */
static int qmi_ctl_release_cid(struct qmi_state *qmi, u8 system)
{
	char buf[32];
	size_t len = qmi_create_msg(buf, sizeof(buf), 0, QMI_CTL, 0x0023);

	if (system < sizeof(qmi->cid)) {
		u8 tlvdata[2] = { system, qmi->cid[system] };
		len = qmi_add_tlv(buf, sizeof(buf), 0x01, tlvdata, sizeof(tlvdata));

	} else {
		return -1;
	}

	/* no need to parse result - cant reuse CID in any case */
	qmi_send_and_wait_for_ack(qmi, buf, len, 30);
	qmi->cid[system] = 0;

	return 1;
}

static int qmi_dms_verify_pin(struct qmi_state *qmi)
{
	int ret;
	size_t len;
	u8 cid = qmi->cid[QMI_DMS];
	char buf[32];

	if (!cid)
		return -1;

	len = qmi_create_msg(buf, sizeof(buf), cid, QMI_DMS, 0x002b);
	ret = qmi_dms_parse(qmi, qmi_send_and_wait_for_ack(qmi, buf, len, 30));
	return qmi->flags && QMI_FLAG_PINOK;
}


/* ----- exported API ----- */

int qmi_reset(struct qmi_state *qmi)
{
	/* NOTE:  We do NOT clear the allocated CIDs! */
	qmi->state = QMI_STATE_INIT;
	qmi->wds_status = 0;
	qmi->flags = 0;
	return 1;
}

int qmi_disconnect(struct qmi_state *qmi)
{
	qmi->flags &= ~QMI_FLAG_RECV;         /* disable async receiving */
	qmi_wds_stop(qmi);
	qmi_wds_status(qmi);
	qmi->state = QMI_STATE_DOWN;
	return 1;
}

/* receive a QMUX can be called in interrupt context! */
int qmi_recv(struct qmi_state *qmi)
{
	if (qmi->flags & QMI_FLAG_RECV) /* async receiving enabled? */
		return usb_submit_urb(qmi->urb, GFP_ATOMIC);
	else
		return 0;
}

/*
 * return current connection state with side effects: will run through
 *  the states from INIT to CONN if possible
 */
int qmi_conn_state(struct qmi_state *qmi)
{
	int ret;

	dev_dbg(&qmi->dev->dev, ".state=%d, .cid[QMI_WDS]=0x%02x, .cid[QMI_DMS]=0x%02x, .flags=0x%08lx, .handle=0x%08x\n",
		qmi->state,
		qmi->cid[QMI_WDS],
		qmi->cid[QMI_DMS],
		qmi->flags,
		*(__le32 *)qmi->handle);

	switch (qmi->state) {
	/* these states don't trigger any action */
	case QMI_STATE_DOWN:
		return 1;
	case QMI_STATE_UP:
		qmi->flags |= QMI_FLAG_RECV;  /* enable async receiving  */
		return 0;
	case QMI_STATE_ERR:
		return -1;

	case QMI_STATE_INIT:
		qmi->flags &= ~QMI_FLAG_RECV;  /* disable async receiving */

		/* request CIDs */
		if (qmi_ctl_request_cid(qmi, QMI_WDS) < 0 || qmi_ctl_request_cid(qmi, QMI_DMS) < 0) {
			qmi->state = QMI_STATE_ERR; /* FATAL - no way out except for unplugging the device */
			return -1;
		}

#if defined(ARCADYAN)
		if (qmi_ctl_request_cid(qmi, QMI_NAS) >= 0)
			qmi_nas_sys_selection(qmi);
#endif

		/* stay in this state until PIN code is entered */
		if (!qmi_dms_verify_pin(qmi)) {
			dev_notice(&qmi->dev->dev, "please enter PIN code using AT+CPIN=\"xxxx\" on a ttyUSB0 device\n");
			return 1;
		}
		/* check status */
		switch (qmi_wds_status(qmi)) {
		case 1: /* DISCONNECTED - attempt to connect */
#if defined(ARCADYAN)
			qmi_wds_autoconnect(qmi);
			if (!qmi_auto) {
#endif
			ret = qmi_wds_start(qmi);
			if (ret < 0) {
				dev_notice(&qmi->dev->dev, "connection failed: 0x%04x\n", -ret);
				break;
			}
#if defined(ARCADYAN)
			}
#endif

			/* fallthrough */
		case 2: /* CONNECTED - note that */
			qmi_wds_report(qmi); /* enable status reports */
			qmi->state = QMI_STATE_UP;
#if defined(ARCADYAN)
			qmi->flags |= QMI_FLAG_RECV;
#endif

		/* do nothing for the other states */
		}
		break;

	default:
		dev_dbg(&qmi->dev->dev, "%s() unknown state=%d\n", __func__, qmi->state);
	}
	return 1;  /* not connected yet - */
}


/* allocate and initiate a QMI state device */
struct qmi_state *qmi_init(struct usb_interface *intf)
{
	struct qmi_state *qmi = kmalloc(sizeof(struct qmi_state), GFP_KERNEL);

	if (!qmi)
		return NULL;

	memset(qmi, 0, sizeof(*qmi));

	/*
	 * keep track of the device receiving the control messages and the
	 * number of the CDC (like) control interface which is our target.
	 * Note that the interface might be disguised as vendor specific,
	 * and be a combined CDC control/data interface
	 */
	qmi->dev = interface_to_usbdev(intf);
	qmi->intfnr = intf->cur_altsetting->desc.bInterfaceNumber;
	/* FIXME: it would be useful to verify that this interface actually talks CDC */

	/* create async receive URB */
	qmi->urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!qmi->urb) {
		kfree(qmi);
		return NULL;
	}

	/* pre-allocate buffer for receiving */
	qmi->rcvbuf = kmalloc(QMI_BUFLEN, GFP_KERNEL);

	if (!qmi->rcvbuf) {
		usb_free_urb(qmi->urb);
		kfree(qmi);
		return NULL;
	}

	/* usb control setup */
	qmi->setup.bRequestType = USB_DIR_IN|USB_TYPE_CLASS|USB_RECIP_INTERFACE;
	qmi->setup.bRequest = USB_CDC_GET_ENCAPSULATED_RESPONSE;
	qmi->setup.wValue = 0; /* zero */
	qmi->setup.wIndex = cpu_to_le16(qmi->intfnr);
	qmi->setup.wLength = cpu_to_le16(QMI_BUFLEN);

	/* prepare the async receive URB */
	usb_fill_control_urb(qmi->urb, qmi->dev,
			usb_rcvctrlpipe(qmi->dev, 0),
			(char *)&qmi->setup,
			qmi->rcvbuf,
			QMI_BUFLEN,
			qmi_read_callback, qmi);

	return qmi;
}

/* disable and free a QMI state device */
int qmi_exit(struct qmi_state *qmi)
{
	int i;

	/* release all CIDs - may need to sleep */
	for (i = 1; i < sizeof(qmi->cid); i++)
		qmi_ctl_release_cid(qmi, i);

	/* free all buffers */
	kfree(qmi->rcvbuf);

	/* free URBs */
	usb_free_urb(qmi->urb);

	kfree(qmi);

	return 0;
}
