/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/clk.h>
#include <linux/err.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <linux/completion.h>

#include <bcm63268_io.h>
#include <bcm63268_regs.h>
#include <bcm63268_dev_nand.h>

#undef BCM63268_MTD_DEBUG

#define DRV_NAME	"bcm63268-nand"
#define DRV_AUTHOR	"Arcadyan Technology"
#define DRV_DESC	"Broadcom on-chip NAND FLASH Controller Driver"
#define DRV_VER		"1.0"

/*
 * helpers for the NAND controller register sets
 */
#define bcm_nand_readl(b,o)      bcm_readl((b->base) + (o))
#define bcm_nand_writel(v,b,o)   bcm_writel((v), (b->base) + (o))
#define bcm_nand_intr_readl(b,o)      bcm_readl((b->intr) + (o))
#define bcm_nand_intr_writel(v,b,o)   bcm_writel((v), (b->intr) + (o))

#define nfc_is_v4(host)		(host->revision & 0x400)

#define BCM63268_NAND_ECC_LAYOUT_CFG(L,S,P)     \
    (((unsigned long)(L)<<28) | ((unsigned long)(S)<<16) | (P))

#define BCM63268_NAND_BUF_SIZE	(NAND_MAX_PAGESIZE + NAND_MAX_OOBSIZE)

struct bcm63268_nand_host_buf {
	int head;
	int tail;
	uint8_t buf[BCM63268_NAND_BUF_SIZE];
};

struct bcm63268_nand_host {
	struct mtd_info		mtd;
	struct nand_chip	nand;
	struct mtd_partition	*parts;
	struct device		*dev;
	struct clk			*clk;
	int					clk_act;

	void __iomem		*base;
	void __iomem		*intr;
	void __iomem		*cache;

	uint32_t	revision;
	int			cachesize;
	int			stepsize;
	int			intfsts_retries;
	int			cmd_retries;
	int			status;
	int			irq;
	spinlock_t	irq_lock;
	uint32_t	irq_status;
	uint32_t	ecclevel;
	uint16_t	pageaddr;

	struct bcm63268_nand_host_buf buf;

	struct completion	op_completion;

	void			(*preset)(struct bcm63268_nand_host *);
	void			(*send_cmd)(struct bcm63268_nand_host *, uint32_t);
	void			(*send_addr)(struct bcm63268_nand_host *, uint32_t);
	void			(*send_page)(struct mtd_info *, unsigned int);
	void			(*send_read_id)(struct bcm63268_nand_host *);
	uint16_t		(*get_dev_status)(struct bcm63268_nand_host *);
	int				(*check_intr)(struct bcm63268_nand_host *);
	void			(*clear_intr)(struct bcm63268_nand_host *, uint32_t);
	void			(*ctrl_intr)(struct bcm63268_nand_host *, int);
};

/* OOB description for 512 byte pages with 16 byte OOB using Hamming */
static struct nand_ecclayout nand_hw_eccoob_hamming_16 = {
	.eccbytes = 3,
	.eccpos = {
		 6, 7, 8
	},
	.oobfree = {
		{.offset = 0, .length = 5},
		{.offset = 9, .length = 7},
	}
};

/* OOB description for 2048 byte pages with 64 byte OOB */
static struct nand_ecclayout nand_hw_eccoob_hamming_64 = {
	.eccbytes = 12,
	.eccpos = {
		 6, 7,  8,  22, 23, 24, 38, 39, 40, 54, 55, 56
	},
	.oobfree = {
		{.offset = 2, .length = 4},
		{.offset = 9, .length = 13},
		{.offset = 25, .length = 13},
		{.offset = 41, .length = 13},
		{.offset = 57, .length = 7}
	}
};

/* OOB description for 4096 byte pages with 128 byte OOB using Hamming */
static struct nand_ecclayout nand_hw_eccoob_hamming_128 = {
	.eccbytes = 24,
	.eccpos = {
		 6, 7, 8, 22, 23, 24, 38, 39, 40, 54, 55, 56,
		 70, 71, 72, 86, 87, 88, 102, 103, 104, 118, 119, 120
	},
	.oobfree = {
		{.offset = 2, .length = 4},
		{.offset = 9, .length = 13},
		{.offset = 25, .length = 13},
		{.offset = 41, .length = 13},
		{.offset = 57, .length = 13},
		{.offset = 73, .length = 13},
		{.offset = 89, .length = 13},
		{.offset = 105, .length = 13},
		{.offset = 121, .length = 7}
	}
};

/* OOB description for 512 byte pages with 16 byte OOB using BCH-4 */
static struct nand_ecclayout nand_hw_eccoob_bch4_512 = {
	.eccbytes = 7,
	.eccpos = {
		 9, 10, 11, 12, 13, 14, 15
	},
	.oobfree = {
		{.offset = 0, .length = 5},
		{.offset = 6, .length = 3}
	}
};

/* OOB description for 2048 byte pages with 64 byte OOB using BCH-4 */
static struct nand_ecclayout nand_hw_eccoob_bch4_2k = {
	.eccbytes = 28,
	.eccpos = {
		 9, 10, 11, 12, 13, 14, 15, 25, 26, 27, 28, 29, 30, 31, 
		 41, 42, 43, 44, 45, 46, 47, 57, 58, 59, 60, 61, 62, 63
	},
	.oobfree = {
		{.offset = 1, .length = 8},
		{.offset = 16, .length = 9},
		{.offset = 32, .length = 9},
		{.offset = 48, .length = 9}
	}
};

/* OOB description for 4096 byte pages with 128 byte OOB using BCH-4 */
static struct nand_ecclayout nand_hw_eccoob_bch4_4k = {
	.eccbytes = 56,
	.eccpos = {
		 9, 10, 11, 12, 13, 14, 15, 25, 26, 27, 28, 29, 30, 31, 
		 41, 42, 43, 44, 45, 46, 47, 57, 58, 59, 60, 61, 62, 63, 
		 73, 74, 75, 76, 77, 78, 79, 89, 90, 91, 92, 93, 94, 95, 
		 105, 106, 107, 108, 109, 110, 111, 121, 122, 123, 124, 125, 126, 127
	},
	.oobfree = {
		{.offset = 1, .length = 8},
		{.offset = 16, .length = 9},
		{.offset = 32, .length = 9},
		{.offset = 48, .length = 9},
		{.offset = 64, .length = 9},
		{.offset = 80, .length = 9},
		{.offset = 96, .length = 9},
		{.offset = 112, .length = 9}
	}
};

/* OOB description for 8192 byte pages with 256 byte OOB using BCH-4 */
static struct nand_ecclayout nand_hw_eccoob_bch4_8k = {
	.eccbytes = 112,
	.eccpos = {
		 9, 10, 11, 12, 13, 14, 15, 25, 26, 27, 28, 29, 30, 31, 
		 41, 42, 43, 44, 45, 46, 47, 57, 58, 59, 60, 61, 62, 63, 
		 73, 74, 75, 76, 77, 78, 79, 89, 90, 91, 92, 93, 94, 95, 
		 105, 106, 107, 108, 109, 110, 111, 121, 122, 123, 124, 125, 126, 127, 
		 137, 138, 139, 140, 141, 142, 143, 153, 154, 155, 156, 157, 158, 159, 
		 169, 170, 171, 172, 173, 174, 175, 185, 186, 187, 188, 189, 190, 191, 
		 201, 202, 203, 204, 205, 206, 207, 217, 212, 213, 214, 215, 216, 217, 
		 233, 234, 235, 236, 237, 238, 239, 249, 250, 251, 252, 253, 254, 255
	},
	.oobfree = {
		{.offset = 1, .length = 8},
		{.offset = 16, .length = 9},
		{.offset = 32, .length = 9},
		{.offset = 48, .length = 9},
		{.offset = 64, .length = 9},
		{.offset = 80, .length = 9},
		{.offset = 96, .length = 9},
		{.offset = 112, .length = 9}
	}
};


/* OOB description for 2048 byte pages with 64 byte OOB using BCH-8, Step-16 */
static struct nand_ecclayout nand_hw_eccoob_bch8_16_2k = {
	.eccbytes = 52,
	.eccpos = {
		 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 
		 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
		 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
		 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63
	},
	.oobfree = {
		{.offset = 1, .length = 2},
		{.offset = 16, .length = 3},
		{.offset = 32, .length = 3},
		{.offset = 48, .length = 3}
	}
};

/* OOB description for 4096 byte pages with 128 byte OOB using BCH-8, Step-16 */
static struct nand_ecclayout nand_hw_eccoob_bch8_16_4k = {
	.eccbytes = 104,
	.eccpos = {
		 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 
		 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
		 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
		 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 
		 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 
		 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 
		 99, 100, 101, 102, 103, 104, 105, 106, 108, 108, 109, 110, 111, 
		 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127
	},
	.oobfree = {
		{.offset = 1, .length = 2},
		{.offset = 16, .length = 3},
		{.offset = 32, .length = 3},
		{.offset = 48, .length = 3},
		{.offset = 64, .length = 3},
		{.offset = 80, .length = 3},
		{.offset = 96, .length = 3},
		{.offset = 112, .length = 3}
	}
};

/* OOB description for 8192 byte pages with 256 byte OOB using BCH-8, Step-16 */
static struct nand_ecclayout nand_hw_eccoob_bch8_16_8k = {
	.eccbytes = 208,
	.eccpos = {
		 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 
		 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
		 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
		 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 
		 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 
		 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 
		 99, 100, 101, 102, 103, 104, 105, 106, 108, 108, 109, 110, 111, 
		 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 
		 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 
		 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 
		 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 
		 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 
		 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 
		 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 
		 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 
		 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
	},
	.oobfree = {
		{.offset = 1, .length = 2},
		{.offset = 16, .length = 3},
		{.offset = 32, .length = 3},
		{.offset = 48, .length = 3},
		{.offset = 64, .length = 3},
		{.offset = 80, .length = 3},
		{.offset = 96, .length = 3},
		{.offset = 112, .length = 3}, 
		{.offset = 128, .length = 3}, 
		{.offset = 144, .length = 3}, 
		{.offset = 160, .length = 3}, 
		{.offset = 176, .length = 3}, 
		{.offset = 192, .length = 3}, 
		{.offset = 208, .length = 3}, 
		{.offset = 224, .length = 3}, 
		{.offset = 240, .length = 3}, 
	}
};

/* OOB description for 4096 byte pages with 216 byte OOB using BCH-8, Step-27 */
static struct nand_ecclayout nand_hw_eccoob_bch8_27_4k = {
	.eccbytes = 104,
	.eccpos = {
		 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 
		 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53,
		 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 
		 95, 96, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107,
		 122, 123, 124, 125, 126, 127, 128, 129, 130, 131, 132, 133, 134, 
		 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 160, 161,
		 176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 
		 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 215
	},
	.oobfree = {
		{.offset = 1, .length = 13},
		{.offset = 27, .length = 14},
		{.offset = 54, .length = 14},
		{.offset = 81, .length = 14},
		{.offset = 108, .length = 14},
		{.offset = 135, .length = 14},
		{.offset = 162, .length = 14},
		{.offset = 189, .length = 14}
	}
};

/* OOB description for 8192 byte pages with 432 byte OOB using BCH-8, Step-27 */
static struct nand_ecclayout nand_hw_eccoob_bch8_27_8k = {
	.eccbytes = 208,
	.eccpos = {
		 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 
		 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53,
		 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 
		 95, 96, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107,
		 122, 123, 124, 125, 126, 127, 128, 129, 130, 131, 132, 133, 134, 
		 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 160, 161,
		 176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 
		 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 215, 
		 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 
		 257, 258, 259, 260, 261, 262, 263, 264, 265, 266, 267, 268, 269, 
		 284, 285, 286, 287, 288, 289, 290, 291, 292, 293, 294, 295, 296, 
		 311, 312, 313, 314, 315, 316, 317, 318, 319, 320, 321, 322, 323,
		 338, 339, 340, 341, 342, 343, 344, 345, 346, 347, 348, 349, 350, 
		 365, 366, 367, 368, 369, 370, 371, 372, 373, 374, 375, 376, 377, 
		 392, 393, 394, 395, 396, 397, 398, 399, 400, 401, 402, 403, 404, 
		 419, 420, 421, 422, 423, 424, 425, 426, 427, 428, 429, 430, 431
	},
	.oobfree = {
		{.offset = 1, .length = 13},
		{.offset = 27, .length = 14},
		{.offset = 54, .length = 14},
		{.offset = 81, .length = 14},
		{.offset = 108, .length = 14},
		{.offset = 135, .length = 14},
		{.offset = 162, .length = 14},
		{.offset = 189, .length = 14}, 
		{.offset = 216, .length = 14},
		{.offset = 243, .length = 14},
		{.offset = 270, .length = 14},
		{.offset = 297, .length = 14},
		{.offset = 324, .length = 14},
		{.offset = 351, .length = 14},
		{.offset = 378, .length = 14},
		{.offset = 405, .length = 14},
	}
};

static uint8_t bbt_pattern[] = { 'B', 'b', 't', '0' };
static uint8_t mirror_pattern[] = { 't', 'b', 'B', '1' };

static struct nand_bbt_descr bbt_main_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
	    | NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs = 0,
	.len = 4,
	.veroffs = 4,
	.maxblocks = 4,
	.pattern = bbt_pattern,
};

static struct nand_bbt_descr bbt_mirror_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
	    | NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs = 0,
	.len = 4,
	.veroffs = 4,
	.maxblocks = 4,
	.pattern = mirror_pattern,
};

static struct nand_bbt_descr bbt_main_bch8_16_2k_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
	    | NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs = 16,
	.len = 3,
	.veroffs = 32,
	.maxblocks = 4,
	.pattern = bbt_pattern,
};

static struct nand_bbt_descr bbt_mirror_bch8_16_2k_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
	    | NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs = 16,
	.len = 3,
	.veroffs = 32,
	.maxblocks = 4,
	.pattern = mirror_pattern,
};


static irqreturn_t bcm63268_nand_isr(int irq, void *dev_id)
{
	struct bcm63268_nand_host *host = dev_id;
	unsigned long flags;

	spin_lock_irqsave(&host->irq_lock, flags);

	if (!host->check_intr(host)) {
		spin_unlock_irq(&host->irq_lock);
		return IRQ_NONE;
	}

	complete(&host->op_completion);

	spin_unlock_irqrestore(&host->irq_lock, flags);

	return IRQ_HANDLED;
}

static int check_intr_v4(struct bcm63268_nand_host *host)
{
	uint32_t intr_sts;

	host->irq_status = 0;
	intr_sts = bcm_nand_intr_readl(host, R_NAND_INTR_CTRL_INTR);

	if ((intr_sts & M_NAND_INTR_CTRL_INTR_ALL_MASK) == 0)
		return 0;

	bcm_nand_intr_writel(intr_sts, host, R_NAND_INTR_CTRL_INTR);
	host->irq_status |= intr_sts & M_NAND_INTR_CTRL_INTR_ALL_MASK;

	return 1;
}

static void control_intr_v4(struct bcm63268_nand_host *host, int activate)
{
	uint32_t val = 0;

	val = activate ? M_NAND_INTR_CTRL_INTR_ENABLE_MASK : 0;
	val |= M_NAND_INTR_CTRL_INTR_ALL_MASK;
	bcm_nand_intr_writel(val, host, R_NAND_INTR_CTRL_INTR);
}

static void clear_intr_v4(struct bcm63268_nand_host *host, uint32_t intr_mask)
{
	bcm_nand_intr_writel(intr_mask, host, R_NAND_INTR_CTRL_INTR);
}

static void send_cmd_v4(struct bcm63268_nand_host *host, uint32_t cmd)
{
	bcm_nand_writel(cmd, host, R_NAND_CTRL_CMD_START);
}

static void send_addr_v4(struct bcm63268_nand_host *host, uint32_t addr)
{
	bcm_nand_writel(addr, host, R_NAND_CTRL_CMD_ADDR);
	bcm_nand_writel(0, host, R_NAND_CTRL_CMD_EXTADDR);
}

static void send_page_v4(struct mtd_info *mtd, unsigned int ops)
{
	printk("%s NIY!\n", __FUNCTION__);
}

static void send_read_id_v4(struct bcm63268_nand_host *host)
{
	printk("%s NIY!\n", __FUNCTION__);
}

static uint16_t get_dev_status_v4(struct bcm63268_nand_host *host)
{
	printk("%s NIY!\n", __FUNCTION__);

	return 0;
}

static void preset_v4(struct bcm63268_nand_host *host)
{
	struct nand_chip *chip = &host->nand;
	struct mtd_info *mtd = &host->mtd;
	uint32_t ecclayout_cfg;
	uint32_t reg_val;

	reg_val = bcm_nand_readl(host, R_NAND_CTRL_ACCESS_CTRL);
	host->ecclevel = (reg_val & M_NAND_CTRL_ACCESS_CTRL_ECC_LVL) >> S_NAND_CTRL_ACCESS_CTRL_ECC_LVL;
	ecclayout_cfg = BCM63268_NAND_ECC_LAYOUT_CFG(host->ecclevel, mtd->oobsize, mtd->writesize);

	host->stepsize = 16;
	/* do not write the BBT to FLASH currently since 
	 * we don't have any update mechanism from the 
	 * bootloader currently
	 */
#if 0
	chip->bbt_td = &bbt_main_descr;
	chip->bbt_md = &bbt_mirror_descr;
#endif
	switch (ecclayout_cfg) {
		case (BCM63268_NAND_ECC_LAYOUT_CFG(NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING, 16, 512)):
			chip->ecc.layout = &nand_hw_eccoob_hamming_16;
			break;

		case (BCM63268_NAND_ECC_LAYOUT_CFG(NAND_CTRL_ACCESS_CTRL_ECC_LVL_BCH_4, 16, 512)):
			chip->ecc.layout = &nand_hw_eccoob_bch4_512;
			break;

		case (BCM63268_NAND_ECC_LAYOUT_CFG(NAND_CTRL_ACCESS_CTRL_ECC_LVL_BCH_4, 64, 2048)):
			chip->ecc.layout = &nand_hw_eccoob_bch4_2k;
			break;

		case (BCM63268_NAND_ECC_LAYOUT_CFG(NAND_CTRL_ACCESS_CTRL_ECC_LVL_BCH_8, 64, 2048)):
			chip->ecc.layout = &nand_hw_eccoob_bch8_16_2k;
#if 0
			chip->bbt_td = &bbt_main_bch8_16_2k_descr;
			chip->bbt_md = &bbt_mirror_bch8_16_2k_descr;
#endif
			break;

		case (BCM63268_NAND_ECC_LAYOUT_CFG(NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING, 64, 2048)):
			chip->ecc.layout = &nand_hw_eccoob_hamming_64;
			break;

		case (BCM63268_NAND_ECC_LAYOUT_CFG(NAND_CTRL_ACCESS_CTRL_ECC_LVL_BCH_4, 128, 4096)):
			chip->ecc.layout = &nand_hw_eccoob_bch4_4k;
			break;

		case (BCM63268_NAND_ECC_LAYOUT_CFG(NAND_CTRL_ACCESS_CTRL_ECC_LVL_BCH_8, 128, 4096)):
			chip->ecc.layout = &nand_hw_eccoob_bch8_16_4k;
			break;

		case (BCM63268_NAND_ECC_LAYOUT_CFG(NAND_CTRL_ACCESS_CTRL_ECC_LVL_BCH_8, 216, 4096)):
			chip->ecc.layout = &nand_hw_eccoob_bch8_27_4k;
			host->stepsize = 27;
			break;

		case (BCM63268_NAND_ECC_LAYOUT_CFG(NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING, 128, 4096)):
			chip->ecc.layout = &nand_hw_eccoob_hamming_128;
			break;

		case (BCM63268_NAND_ECC_LAYOUT_CFG(NAND_CTRL_ACCESS_CTRL_ECC_LVL_BCH_4, 256, 8192)):
			chip->ecc.layout = &nand_hw_eccoob_bch4_8k;
			break;

		case (BCM63268_NAND_ECC_LAYOUT_CFG(NAND_CTRL_ACCESS_CTRL_ECC_LVL_BCH_8, 256, 8192)):
			chip->ecc.layout = &nand_hw_eccoob_bch8_16_8k;
			break;

		case (BCM63268_NAND_ECC_LAYOUT_CFG(NAND_CTRL_ACCESS_CTRL_ECC_LVL_BCH_8, 432, 8192)):
			chip->ecc.layout = &nand_hw_eccoob_bch8_27_8k;
			host->stepsize = 27;
			break;

		default:
			if (host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_DISABLE) {
				pr_err("%s: no valid spare layout for level=%d, oob size=%d, page size=%d\n", 
											__FUNCTION__, host->ecclevel, mtd->oobsize, mtd->writesize);
			}
			break;
	}

	chip->ecc.size = 512;
	chip->ecc.bytes = chip->ecc.layout->eccbytes;
}

static void bcm63268_nand_reset_buf(struct bcm63268_nand_host *host)
{
	host->buf.head = host->buf.tail = 0;
}

static void bcm63268_nand_write_byte_to_buf(struct bcm63268_nand_host *host, uint8_t byte)
{
	BUG_ON(host->buf.tail >= sizeof(host->buf.buf));
	host->buf.buf[host->buf.tail++] = byte;
}

static void bcm63268_nand_clear_intrsts(struct bcm63268_nand_host *host)
{
	unsigned long flags;
	
	spin_lock_irqsave(&host->irq_lock, flags);
	
	host->irq_status = 0;
	
	spin_unlock_irqrestore(&host->irq_lock, flags);
}

static int bcm63268_nand_intrsts_poll(struct bcm63268_nand_host *host, uint32_t intr_mask)
{
	uint32_t intr_sts = 0;
	int retry = 0;
	unsigned long timeout = msecs_to_jiffies(3000);
	unsigned long flags;

	/* Apply this short delay always to ensure that we do wait tWB in
	 * any case on any machine. */
	ndelay(100);

	if (in_interrupt() || oops_in_progress) {
		for (timeout = 1000; timeout > 0; timeout -= 10) {
			spin_lock_irq(&host->irq_lock);
			intr_sts = host->irq_status;
			if ((intr_sts & intr_mask) == intr_mask) {
				host->irq_status &= ~intr_mask;
				spin_unlock_irq(&host->irq_lock);
				return 0;
			}
			spin_unlock_irq(&host->irq_lock);
			mdelay(10);
		}

		if (!timeout) {
			dev_err(host->dev, "oops timeout occurred, expecting mask = 0x%x\n", intr_mask);
			return -ETIMEDOUT;
		}
	} else {

		if (!wait_for_completion_timeout(&host->op_completion, timeout)) {
			dev_err(host->dev, "timeout occurred, expecting mask = 0x%x\n", intr_mask);
			return -ETIMEDOUT;
		}

		spin_lock_irqsave(&host->irq_lock, flags);

		intr_sts = host->irq_status;

		if ((intr_sts & intr_mask) == intr_mask)
			host->irq_status &= ~intr_mask;
		else
			retry = 1;
		
		spin_unlock_irqrestore(&host->irq_lock, flags);
	}
	
	return retry;
}

static int bcm63268_nand_intfsts_poll(struct bcm63268_nand_host *host, uint32_t intf_mask)
{
	int retry = host->intfsts_retries;
	uint32_t intf_sts;
	
	do {

		intf_sts = bcm_nand_readl(host, R_NAND_CTRL_INTFSTATUS);
		
		retry--;
		if (!retry)
			break;
		
		udelay(1);
	} while ((intf_sts & intf_mask) != intf_mask);

	if (!retry)
		return -1;

	return 0;
}

static int bcm63268_nand_pgmsts(struct bcm63268_nand_host *host)
{
	uint32_t intf_sts;

	intf_sts = bcm_nand_readl(host, R_NAND_CTRL_INTFSTATUS);

	if (intf_sts & NAND_STATUS_FAIL)
		return -1;

	return 0;
}

static void bcm63268_nand_reset_flash(struct bcm63268_nand_host *host)
{
	uint32_t intr_mask = M_NAND_INTR_CTRL_INTR_CTRL_READY | M_NAND_INTR_CTRL_INTR_DEV_RBPIN;
	uint32_t intf_mask = M_NAND_CTRL_INTFSTATUS_CTRLER_READY | M_NAND_CTRL_INTFSTATUS_FLASH_READY;
	int rv, retry = host->cmd_retries;

	do {
		bcm63268_nand_clear_intrsts(host);

		host->send_addr(host, 0);
		host->send_cmd(host, V_NAND_CTRL_CMD_START_FLASH_RESET);

		rv = bcm63268_nand_intrsts_poll(host, intr_mask);

		if (rv == -ETIMEDOUT) {
			dev_err(host->dev, "reset flash failed(wait intr timed out).\n");
			return;
		} else if (rv == 0)
			break;
	} while (retry--);

	if (!retry) {
		dev_err(host->dev, "reset flash failed(no intr).\n");
		return;
	}

	if (bcm63268_nand_intfsts_poll(host, intf_mask) < 0)
		dev_err(host->dev, "reset flash failed.\n");
}

/* This functions is used by upper layer to checks if device is ready */
static int bcm63268_nand_dev_ready(struct mtd_info *mtd)
{
	/*
	 * NFC handles R/B internally. Therefore, this function
	 * always returns status as ready.
	 */
	return 1;
}

static void bcm63268_nand_enable_hwecc(struct mtd_info *mtd, int mode)
{
	/*
	 * If HW ECC is enabled, we turn it on during init. There is
	 * no need to enable again here.
	 */
	printk("%s NIY!\n", __FUNCTION__);
}

static int bcm63268_nand_correct_data(struct mtd_info *mtd, u_char *dat,
				 u_char *read_ecc, u_char *calc_ecc)
{
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;
	dev_err(host->dev,
			"%s called unexpectedly\n", __FUNCTION__);
	BUG();
	return -EIO;
}

static int bcm63268_nand_calculate_ecc(struct mtd_info *mtd, const u_char *dat,
				  u_char *ecc_code)
{
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;
	dev_err(host->dev,
			"%s called unexpectedly\n", __FUNCTION__);
	BUG();
	return -EIO;
}

/* NAND core entry points */
/* this function examines buffers to see if they contain data that
 * indicate that the buffer is part of an erased region of flash.
 */
static bool bcm63268_nand_is_erased(uint8_t *buf, int len)
{
	int i = 0;
	for (i = 0; i < len; i++)
		if (buf[i] != 0xFF)
			return false;
	return true;
}

#ifdef BCM63268_NAND_DEBUG
static void bcm63268_nand_dump_buf(uint32_t addr, uint8_t *buf, uint32_t nLen )
{
    static char szHexChars[] = "0123456789abcdef";
    char szLine[80];
    char *p = szLine;
    unsigned char ch, *q;
    int i = 0, j, size = 0;
    unsigned long ul;
    unsigned short us;

    if( (addr & 0xff000000) == 0xff000000 ||
        (addr & 0xff000000) == 0xb0000000 )
    {
        if (nLen == 2) {
            addr = addr & ~0x01;
			buf = (unsigned char *) ((unsigned long) buf & ~0x01);
        } else if (nLen != 1) {
            /* keeping the old logic as is. */
            if( (addr & 0x03) != 0 )
                nLen += 4;
            addr = addr & ~0x03;
			buf = (unsigned char *) ((unsigned long) buf & ~0x03);
        }
    }
    while( nLen > 0 )
    {
        sprintf( szLine, "%8.8x: ", addr );
        p = szLine + strlen(szLine);

        if( (addr & 0xff000000) == 0xff000000 ||
            (addr & 0xff000000) == 0xb0000000 )
        {
            for(i = 0; i < 6 && nLen > 0; i += sizeof(long), nLen -= sizeof(long))
            {
                if (nLen == 1) {
                    q = buf;
                    size = 1;
                } else if (nLen == 2) {
                    us = *(unsigned short *)buf;
                    q = (unsigned char *) &us;
                    size = 2;
                } else {
                    ul = *(unsigned long *) &buf[i];
                    q = (unsigned char *) &ul;
                    size = sizeof(long);
                }
                for( j = 0; j < size; j++ )
                {
                    *p++ = szHexChars[q[j] >> 4];
                    *p++ = szHexChars[q[j] & 0x0f];
                }
                *p++ = ' ';
            }
        }
        else
        {
            for(i = 0; i < 16 && nLen > 0; i++, nLen-- )
            {
                ch =  buf[i];

                *p++ = szHexChars[ch >> 4];
                *p++ = szHexChars[ch & 0x0f];
                *p++ = ' ';
            }
        }

        for( j = 0; j < 16 - i; j++ )
            *p++ = ' ', *p++ = ' ', *p++ = ' ';

        *p++ = ' ', *p++ = ' ', *p++ = ' ';

        for( j = 0; j < i; j++ )
        {
            ch = buf[j];
            *p++ = (ch >= ' ' && ch <= '~') ? ch : '.';
        }

        *p++ = '\0';
        printk( "%s\r\n", szLine );

        addr += i;
		buf += i;
    }
    printk( "\r\n" );
}
#endif

/* this is the callback that the NAND core calls to write a page. Since
 * writing a page with ECC or without is similar, all the work is done
 * by write_page above.
 * */
static void bcm63268_nand_write_page(struct mtd_info *mtd, struct nand_chip *chip,
				const uint8_t *buf)
{
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;
	uint8_t *tmp_buf = (uint8_t *)buf;
	size_t write_size = mtd->writesize;
	uint32_t intr_mask = M_NAND_INTR_CTRL_INTR_CTRL_READY | M_NAND_INTR_CTRL_INTR_PAGE_PROGRAM;
	uint32_t intf_mask = M_NAND_CTRL_INTFSTATUS_CTRLER_READY | M_NAND_CTRL_INTFSTATUS_FLASH_READY | NAND_STATUS_READY;
	int rv, retry;
	int page_addr, i;
	uint8_t *oob_buf = nand_chip->oob_poi;
	uint32_t reg_val;

	page_addr = host->pageaddr << nand_chip->page_shift;

	while (write_size > 0) {
		retry = host->cmd_retries;
		host->status = 0;
		do {
			bcm63268_nand_clear_intrsts(host);
				
			host->send_addr(host, page_addr);
			memcpy(host->cache, tmp_buf, host->cachesize);
			for (i = 0; i < 4; i++) {
				if (oob_buf && *((uint32_t *)oob_buf) != 0xffffffff) {
					reg_val = oob_buf[0] << 24 | oob_buf[1] << 16 | oob_buf[2] << 8 | oob_buf[3];
				} else {
					reg_val = 0xffffffff;
				}
				if (oob_buf)
					oob_buf += 4;
				bcm_nand_writel(reg_val, host, R_NAND_CTRL_OOB_WRITE_OFFSET(i));
			}
			
			host->send_cmd(host, V_NAND_CTRL_CMD_START_PROGRAM_PAGE);
			rv = bcm63268_nand_intrsts_poll(host, intr_mask);

			if (rv == -ETIMEDOUT) {
				dev_err(host->dev, "write_page failed(wait intr timed out).\n");
				host->status = NAND_STATUS_FAIL;
				return;
			} else if (rv == 0)
				break;
		} while (retry--);

		if (!retry) {
			dev_err(host->dev, "write_page failed(no intr).\n");
			host->status = NAND_STATUS_FAIL;
			return;
		}

		if (bcm63268_nand_intfsts_poll(host, intf_mask) < 0) {
			host->status = NAND_STATUS_FAIL;
			dev_err(host->dev, "write_page failed.\n");
			return;
		}

		page_addr += host->cachesize;
		write_size -= host->cachesize;
		tmp_buf += host->cachesize;
	}
}

/* This is the callback that the NAND core calls to write a page without ECC.
 * raw access is similiar to ECC page writes, so all the work is done in the
 * write_page() function above.
 */
static void bcm63268_nand_write_page_raw(struct mtd_info *mtd, struct nand_chip *chip,
					const uint8_t *buf)
{
	/* for raw page writes, we want to disable ECC and simply write
	   whatever data is in the buffer. */
	printk("%s: NIY!\n", __FUNCTION__);
}

static int bcm63268_nand_write_oob(struct mtd_info *mtd, struct nand_chip *chip,
			    int page_addr)
{
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;
	size_t write_size = mtd->writesize;
	uint32_t intr_mask = M_NAND_INTR_CTRL_INTR_CTRL_READY | M_NAND_INTR_CTRL_INTR_PAGE_PROGRAM;
	uint32_t intf_mask = M_NAND_CTRL_INTFSTATUS_CTRLER_READY | M_NAND_CTRL_INTFSTATUS_FLASH_READY | NAND_STATUS_READY;
	int i, rv, retry, first_page = 1;
	uint8_t *oob_buf = nand_chip->oob_poi;
	uint32_t reg_val;
	uint8_t page_cache[512];
	uint32_t acc_save = 0;

	retry = host->cmd_retries;
	host->status = 0;
	page_addr = page_addr << nand_chip->page_shift;
	do {
		bcm63268_nand_clear_intrsts(host);
			
		host->send_addr(host, page_addr);
		memset(page_cache, 0xff, 512);
		memcpy(host->cache, page_cache, host->cachesize);
		for (i = 0; i < 4; i++) {
			if (oob_buf && *((uint32_t *)oob_buf) != 0xffffffff) {
				reg_val = oob_buf[0] << 24 | oob_buf[1] << 16 | oob_buf[2] << 8 | oob_buf[3];
			} else {
				reg_val = 0xffffffff;
			}
			if (oob_buf)
				oob_buf += 4;
			bcm_nand_writel(reg_val, host, R_NAND_CTRL_OOB_WRITE_OFFSET(i));
		}

		if (host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_DISABLE &&
			host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING) {
			/* disable ECC first, will be restored later */
			acc_save = bcm_nand_readl(host, R_NAND_CTRL_ACCESS_CTRL);
			bcm_nand_writel(acc_save & ~(M_NAND_CTRL_ACCESS_CTRL_ECC_LVL | M_NAND_CTRL_ACCESS_CTRL_ECC_LVL_BLOCK0), 
															host, R_NAND_CTRL_ACCESS_CTRL);
		}
		
		host->send_cmd(host, V_NAND_CTRL_CMD_START_PROGRAM_PAGE);
		rv = bcm63268_nand_intrsts_poll(host, intr_mask);

		if (host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_DISABLE &&
			host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING) {
			/* restore ECC */
			bcm_nand_writel(acc_save, host, R_NAND_CTRL_ACCESS_CTRL);
		}

		if (rv == -ETIMEDOUT) {
			dev_err(host->dev, "write_oob failed(wait intr timed out).\n");
			host->status = NAND_STATUS_FAIL;
			return 0;
		} else if (rv == 0) {
			if ((chip->options & NAND_BBT_SCAN2NDPAGE) != 0 && 
				first_page == 1) {
				page_addr += host->cachesize;
				first_page = 0;
				retry = host->cmd_retries;
				continue;
			}
			break;
		}
	} while (retry--);

	if (!retry) {
		dev_err(host->dev, "write_oob failed(no intr).\n");
		host->status = NAND_STATUS_FAIL;
		return 0;
	}

	if (bcm63268_nand_intfsts_poll(host, intf_mask) < 0) {
		host->status = NAND_STATUS_FAIL;
		dev_err(host->dev, "write_oob failed.\n");
		return 0;
	}

	return 0;
}

static int bcm63268_nand_read_oob(struct mtd_info *mtd, struct nand_chip *chip,
			   int page_addr, int sndcmd)
{
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;
	size_t read_size = mtd->oobsize;
	uint32_t intr_mask = M_NAND_INTR_CTRL_INTR_CTRL_READY;
	uint32_t intf_mask = M_NAND_CTRL_INTFSTATUS_CTRLER_READY | M_NAND_CTRL_INTFSTATUS_OOB_VALID;
	uint8_t *oob_buf = nand_chip->oob_poi;
	int i, rv, retry;
	uint32_t reg_val, acc_save;

	page_addr = page_addr << nand_chip->page_shift;

	if (host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_DISABLE &&
		host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING) {
		/* disable ECC first, will be restored later */
		acc_save = bcm_nand_readl(host, R_NAND_CTRL_ACCESS_CTRL);
		bcm_nand_writel(acc_save & ~(M_NAND_CTRL_ACCESS_CTRL_READ_ECC_BLOCK0_ENABLE | M_NAND_CTRL_ACCESS_CTRL_READ_ECC_ENABLE), 
														host, R_NAND_CTRL_ACCESS_CTRL);
	}

	while (read_size > 0) {
		retry = host->cmd_retries;
		do {
			bcm63268_nand_clear_intrsts(host);

			host->send_addr(host, page_addr);
			if (host->ecclevel == NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING)
				host->send_cmd(host, V_NAND_CTRL_CMD_START_OOB_READ);
			else {
				host->send_cmd(host, V_NAND_CTRL_CMD_START_PAGE_READ);
				intf_mask |= M_NAND_CTRL_INTFSTATUS_CACHE_VALID;
			}

			rv = bcm63268_nand_intrsts_poll(host, intr_mask);

			if (rv == -ETIMEDOUT) {
				dev_err(host->dev, "read_oob failed(wait intr timed out).\n");
				return rv;
			} else if (rv == 0)
				break;
		} while (retry--);

		if (!retry) {
			dev_err(host->dev, "read_oob failed(wait intr timed out).\n");
			return -ENODATA;
		}

		if (bcm63268_nand_intfsts_poll(host, intf_mask) < 0)
			dev_err(host->dev, "read_oob failed.\n");

		for (i = 0; i < 4; i++) {
			reg_val = bcm_nand_readl(host, R_NAND_CTRL_OOB_READ_OFFSET(i));
			*oob_buf++ = (uint8_t)((reg_val >> 24) & 0xff);
			*oob_buf++ = (uint8_t)((reg_val >> 16) & 0xff);
			*oob_buf++ = (uint8_t)((reg_val >> 8) & 0xff);
			*oob_buf++ = (uint8_t)(reg_val & 0xff);
		}

		page_addr += host->cachesize;
		read_size -= host->stepsize;
	}

	return 0; /* notify NAND core to send command to
			   NAND device. */
}

static int bcm63268_nand_read_page(struct mtd_info *mtd, struct nand_chip *chip,
			    uint8_t *buf, int page_addr)
{
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;
	size_t read_size = mtd->writesize;
	uint32_t intr_mask = M_NAND_INTR_CTRL_INTR_CTRL_READY;
	uint32_t intf_mask = M_NAND_CTRL_INTFSTATUS_CTRLER_READY;
	uint8_t *oob_buf = nand_chip->oob_poi;
	uint8_t *tmp_buf = buf;
	int corrected = 0;
	int uncorrected = 0;
	int i, rv, retry;
	uint32_t reg_val;
	uint32_t acc_save = 0;

	if (page_addr != host->pageaddr) {
		dev_err(host->dev, "IN %s: page %d is not"
				" equal to host->pageaddr %d, investigate!!",
				__FUNCTION__, page_addr, host->pageaddr);
		BUG();
	}

	if (host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_DISABLE &&
		host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING) {
		/* disable ECC first, will be restored later */
		acc_save = bcm_nand_readl(host, R_NAND_CTRL_ACCESS_CTRL);
		bcm_nand_writel(acc_save & ~(M_NAND_CTRL_ACCESS_CTRL_READ_ECC_BLOCK0_ENABLE | M_NAND_CTRL_ACCESS_CTRL_READ_ECC_ENABLE), 
														host, R_NAND_CTRL_ACCESS_CTRL);
	}

	page_addr = page_addr << nand_chip->page_shift;

	while (read_size > 0) {
		retry = host->cmd_retries;
		do {
			bcm63268_nand_clear_intrsts(host);
			
			host->send_addr(host, page_addr);
			host->send_cmd(host, V_NAND_CTRL_CMD_START_PAGE_READ);

			rv = bcm63268_nand_intrsts_poll(host, intr_mask);
			if (rv == -ETIMEDOUT) {
				dev_err(host->dev, "read_page failed(wait intr timed out).\n");
				return -ETIMEDOUT;
			} else if (rv == 0)
				break;
		} while (retry--);

		if (!retry) {
			dev_err(host->dev, "read_page failed(no intr).\n");
			return -ENODATA;
		}

		if (host->ecclevel == NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING)
			intf_mask |= M_NAND_CTRL_INTFSTATUS_CACHE_VALID;
		else
			intf_mask |= M_NAND_CTRL_INTFSTATUS_CACHE_VALID | M_NAND_CTRL_INTFSTATUS_OOB_VALID;

		if (bcm63268_nand_intfsts_poll(host, intf_mask) < 0)
			dev_err(host->dev, "read_page failed.\n");

		intf_mask &= ~(M_NAND_CTRL_INTFSTATUS_CACHE_VALID | M_NAND_CTRL_INTFSTATUS_OOB_VALID);

		memcpy(tmp_buf, host->cache, host->cachesize);

		if (host->irq_status & M_NAND_INTR_CTRL_INTR_ECC_UNCORR)
			uncorrected++;	
		else if (host->irq_status & M_NAND_INTR_CTRL_INTR_ECC_CORR)
			corrected++;

		host->irq_status &= ~(M_NAND_INTR_CTRL_INTR_ECC_UNCORR | M_NAND_INTR_CTRL_INTR_ECC_CORR);

		if (host->ecclevel == NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING) {
			retry = host->cmd_retries;
			do {
				host->send_addr(host, page_addr);
				host->send_cmd(host, V_NAND_CTRL_CMD_START_OOB_READ);

				rv = bcm63268_nand_intrsts_poll(host, intr_mask);
				if (rv == -ETIMEDOUT) {
					dev_err(host->dev, "read_page failed(wait intr timed out).\n");
					return rv;
				} else if (rv == 0)
					break;
			} while (retry--);

			if (!retry) {
				dev_err(host->dev, "read_page failed(no intr).\n");
				return -ENODATA;
			}
			
			if (bcm63268_nand_intfsts_poll(host, intf_mask | M_NAND_CTRL_INTFSTATUS_OOB_VALID) < 0)
				dev_err(host->dev, "read_page failed.\n");
		}

		for (i = 0; i < 4; i++) {
			reg_val = bcm_nand_readl(host, R_NAND_CTRL_OOB_READ_OFFSET(i));
			*oob_buf++ = (uint8_t)((reg_val >> 24) & 0xff);
			*oob_buf++ = (uint8_t)((reg_val >> 16) & 0xff);
			*oob_buf++ = (uint8_t)((reg_val >> 8) & 0xff);
			*oob_buf++ = (uint8_t)(reg_val & 0xff);
		}

		page_addr += host->cachesize;
		read_size -= host->cachesize;
		tmp_buf += host->cachesize;
	}

	if (host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_DISABLE &&
		host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING) {
		/* restore ECC */
		bcm_nand_writel(acc_save, host, R_NAND_CTRL_ACCESS_CTRL);
	}

	if (uncorrected) {
		/* if the error is not correctable, need to
		 * look at the page to see if it is an erased
		 * page. if so, then it's not a real ECC error
		 */
		if (!bcm63268_nand_is_erased(buf, mtd->writesize))
			host->mtd.ecc_stats.failed++;
		if (!bcm63268_nand_is_erased(nand_chip->oob_poi, mtd->oobsize))
			host->mtd.ecc_stats.failed++;
	} else if (corrected) {
		host->mtd.ecc_stats.corrected++;
	}

#ifdef BCM63268_NAND_DEBUG
	page_addr = host->pageaddr;
	bcm63268_nand_dump_buf(page_addr << nand_chip->page_shift, buf, mtd->writesize);
#endif

	return 0;
}

static int bcm63268_nand_read_page_raw(struct mtd_info *mtd, struct nand_chip *chip,
				uint8_t *buf, int page_addr)
{
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;
	size_t read_size = mtd->writesize;
	uint32_t intr_mask = M_NAND_INTR_CTRL_INTR_CTRL_READY;
	uint32_t intf_mask = M_NAND_CTRL_INTFSTATUS_CTRLER_READY;
	uint8_t *oob_buf = nand_chip->oob_poi;
	uint8_t *tmp_buf = buf;
	int i, rv, retry;
	uint32_t reg_val;
	uint32_t acc_save = 0;

	if (page_addr != host->pageaddr) {
		dev_err(host->dev, "IN %s: page %d is not"
				" equal to host->pageaddr %d, investigate!!",
				__FUNCTION__, page_addr, host->pageaddr);
		BUG();
	}

	if (host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_DISABLE &&
		host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING) {
		/* disable ECC first, will be restored later */
		acc_save = bcm_nand_readl(host, R_NAND_CTRL_ACCESS_CTRL);
		bcm_nand_writel(acc_save & ~(M_NAND_CTRL_ACCESS_CTRL_READ_ECC_BLOCK0_ENABLE | M_NAND_CTRL_ACCESS_CTRL_READ_ECC_ENABLE), 
														host, R_NAND_CTRL_ACCESS_CTRL);
	}

	page_addr = page_addr << nand_chip->page_shift;

	while (read_size > 0) {
		retry = host->cmd_retries;
		do {
			bcm63268_nand_clear_intrsts(host);
			
			host->send_addr(host, page_addr);
			host->send_cmd(host, V_NAND_CTRL_CMD_START_PAGE_READ);

			rv = bcm63268_nand_intrsts_poll(host, intr_mask);
			if (rv == -ETIMEDOUT) {
				dev_err(host->dev, "read_page_raw failed(wait intr timed out).\n");
				return -ETIMEDOUT;
			} else if (rv == 0)
				break;
		} while (retry--);

		if (!retry) {
			dev_err(host->dev, "read_page_raw failed(no intr).\n");
			return -ENODATA;
		}

		if (host->ecclevel == NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING)
			intf_mask |= M_NAND_CTRL_INTFSTATUS_CACHE_VALID;
		else
			intf_mask |= M_NAND_CTRL_INTFSTATUS_CACHE_VALID | M_NAND_CTRL_INTFSTATUS_OOB_VALID;

		if (bcm63268_nand_intfsts_poll(host, intf_mask) < 0)
			dev_err(host->dev, "read_page_raw failed.\n");

		intf_mask &= ~(M_NAND_CTRL_INTFSTATUS_CACHE_VALID | M_NAND_CTRL_INTFSTATUS_OOB_VALID);

		memcpy(tmp_buf, host->cache, host->cachesize);

		host->irq_status &= ~(M_NAND_INTR_CTRL_INTR_ECC_UNCORR | M_NAND_INTR_CTRL_INTR_ECC_CORR);

		if (host->ecclevel == NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING) {
			retry = host->cmd_retries;
			do {
				host->send_addr(host, page_addr);
				host->send_cmd(host, V_NAND_CTRL_CMD_START_OOB_READ);

				rv = bcm63268_nand_intrsts_poll(host, intr_mask);
				if (rv == -ETIMEDOUT) {
					dev_err(host->dev, "read_page_raw failed(wait intr timed out).\n");
					return rv;
				} else if (rv == 0)
					break;
			} while (retry--);

			if (!retry) {
				dev_err(host->dev, "read_page_raw failed(no intr).\n");
				return -ENODATA;
			}
			
			if (bcm63268_nand_intfsts_poll(host, intf_mask | M_NAND_CTRL_INTFSTATUS_OOB_VALID) < 0)
				dev_err(host->dev, "read_page_raw failed.\n");
		}

		for (i = 0; i < 4; i++) {
			reg_val = bcm_nand_readl(host, R_NAND_CTRL_OOB_READ_OFFSET(i));
			*oob_buf++ = (uint8_t)((reg_val >> 24) & 0xff);
			*oob_buf++ = (uint8_t)((reg_val >> 16) & 0xff);
			*oob_buf++ = (uint8_t)((reg_val >> 8) & 0xff);
			*oob_buf++ = (uint8_t)(reg_val & 0xff);
		}

		page_addr += host->cachesize;
		read_size -= host->cachesize;
		tmp_buf += host->cachesize;
	}

	if (host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_DISABLE &&
		host->ecclevel != NAND_CTRL_ACCESS_CTRL_ECC_LVL_HAMMING) {
		/* restore ECC */
		bcm_nand_writel(acc_save, host, R_NAND_CTRL_ACCESS_CTRL);
	}

#ifdef BCM63268_NAND_DEBUG
	page_addr = host->pageaddr;
	bcm63268_nand_dump_buf(page_addr << nand_chip->page_shift, buf, mtd->writesize);
#endif

	return 0;
}

static u_char bcm63268_nand_read_byte(struct mtd_info *mtd)
{
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;
	uint8_t result = 0xff;

	if (host->buf.head < host->buf.tail)
		result = host->buf.buf[host->buf.head++];

	return result;
}

static int bcm63268_nand_waitfunc(struct mtd_info *mtd, struct nand_chip *chip)
{
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;
	int status = host->status;
	host->status = 0;

	return status;
}

/* This function is used by upper layer for select and
 * deselect of the NAND chip */
static void bcm63268_nand_select_chip(struct mtd_info *mtd, int chip)
{
#if 0
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;

	switch (chip) {
	case -1:
		/* Disable the NFC clock */
		if (host->clk_act) {
			clk_disable(host->clk);
			host->clk_act = 0;
		}
		break;
	case 0:
		/* Enable the NFC clock */
		if (!host->clk_act) {
			clk_enable(host->clk);
			host->clk_act = 1;
		}
		break;

	default:
		break;
	}
#endif
}

static void bcm63268_nand_erase(struct mtd_info *mtd, int page_addr)
{
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;
	uint32_t intr_mask = M_NAND_INTR_CTRL_INTR_CTRL_READY | M_NAND_INTR_CTRL_INTR_BLOCK_ERASE;
	uint32_t intf_mask = M_NAND_CTRL_INTFSTATUS_CTRLER_READY | M_NAND_CTRL_INTFSTATUS_FLASH_READY | NAND_STATUS_READY;
	int rv, retry = host->cmd_retries;

	page_addr = page_addr << nand_chip->page_shift;

	do {
		bcm63268_nand_clear_intrsts(host);

		host->send_addr(host, page_addr);
		host->send_cmd(host, V_NAND_CTRL_CMD_START_BLOCK_ERASE);

		host->status = 0;
		rv = bcm63268_nand_intrsts_poll(host, intr_mask);
		if (rv == -ETIMEDOUT) {
			host->status = NAND_STATUS_FAIL;
			dev_err(host->dev, "erase_block failed(wait intr timed out).\n");
			return;
		} else if (rv == 0)
			break;
	} while (retry--);

	if (!retry) {
		host->status = NAND_STATUS_FAIL;
		dev_err(host->dev, "erase_block failed(no intr).\n");
		return;
	}

	if (bcm63268_nand_intfsts_poll(host, intf_mask) < 0) {
		host->status = NAND_STATUS_FAIL;
		dev_err(host->dev, "erase_block failed(no sts).\n");
		return;
	}

	if (bcm63268_nand_pgmsts(host) < 0)
		host->status = NAND_STATUS_FAIL;
}

static void bcm63268_nand_cmd_readid(struct bcm63268_nand_host *host)
{
	uint32_t dev_id;
	
	bcm63268_nand_reset_buf(host);

	dev_id = bcm_nand_readl(host, R_NAND_CTRL_DEVICEID);
	bcm63268_nand_write_byte_to_buf(host, (dev_id >> 24) & 0xff);
	bcm63268_nand_write_byte_to_buf(host, (dev_id >> 16) & 0xff);
	bcm63268_nand_write_byte_to_buf(host, (dev_id >> 8) & 0xff);
	bcm63268_nand_write_byte_to_buf(host, dev_id & 0xff);
	dev_id = bcm_nand_readl(host, R_NAND_CTRL_DEVICEID_EXT);
	bcm63268_nand_write_byte_to_buf(host, (dev_id >> 24) & 0xff);
}

static void bcm63268_nand_cmd_status(struct bcm63268_nand_host *host)
{
	uint32_t intr_mask = M_NAND_INTR_CTRL_INTR_CTRL_READY;
	uint32_t intf_mask = M_NAND_CTRL_INTFSTATUS_CTRLER_READY | M_NAND_CTRL_INTFSTATUS_FLASH_READY;
	uint32_t intf_sts = 0;
	int rv, retry = host->cmd_retries;

	bcm63268_nand_reset_buf(host);

	do {
		bcm63268_nand_clear_intrsts(host);

		host->send_addr(host, 0);
		host->send_cmd(host, V_NAND_CTRL_CMD_START_STATUS_READ);
		
		rv = bcm63268_nand_intrsts_poll(host, intr_mask);
		if (rv == -ETIMEDOUT) {
			dev_err(host->dev, "read status failed(wait intr timed out).\n");
			bcm63268_nand_write_byte_to_buf(host, NAND_STATUS_FAIL);
			return;
		}
	} while (retry--);

	if (!retry) {
		dev_err(host->dev, "read status failed.(no intr)\n");
		bcm63268_nand_write_byte_to_buf(host, NAND_STATUS_FAIL);
		return;
	}

	if (bcm63268_nand_intfsts_poll(host, intf_mask) < 0) {
		dev_err(host->dev, "read status failed.\n");
		bcm63268_nand_write_byte_to_buf(host, NAND_STATUS_FAIL);
		return;
	}

	intf_sts = bcm_nand_readl(host, R_NAND_CTRL_INTFSTATUS);

	bcm63268_nand_write_byte_to_buf(host, intf_sts & 0xff);
}

static void bcm63268_nand_cmd_readoob(struct mtd_info *mtd, int column, int page_addr)
{
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;
	uint32_t oobsize = mtd->oobsize;
	uint32_t reg_val;
	uint32_t intr_mask = M_NAND_INTR_CTRL_INTR_CTRL_READY;
	uint32_t intf_mask = M_NAND_CTRL_INTFSTATUS_CTRLER_READY | M_NAND_CTRL_INTFSTATUS_OOB_VALID;
	int i, rv, retry;
	
	bcm63268_nand_reset_buf(host);

	page_addr = page_addr << nand_chip->page_shift;

	while (oobsize > 0) {
		retry = host->cmd_retries;
		host->status = 0;
		do {
			bcm63268_nand_clear_intrsts(host);
			
			host->send_addr(host, page_addr);
			host->send_cmd(host, V_NAND_CTRL_CMD_START_OOB_READ);

			rv = bcm63268_nand_intrsts_poll(host, intr_mask);
			if (rv == -ETIMEDOUT) {
				dev_err(host->dev, "cmd_readoob failed(wait intr timed out).\n");
				host->status = NAND_STATUS_FAIL;
				return;
			} else if (rv == 0)
				break;
		} while (retry--);

		if (!retry) {
			dev_err(host->dev, "cmd_readoob failed(no intr).\n");
			host->status = NAND_STATUS_FAIL;
			return;
		}

		if (bcm63268_nand_intfsts_poll(host, intf_mask) < 0) {
			dev_err(host->dev, "cmd_readoob failed.\n");
			host->status = NAND_STATUS_FAIL;
			return;
		}

		for (i = 0; i < 4; i++) {
			reg_val = bcm_nand_readl(host, R_NAND_CTRL_OOB_READ_OFFSET(i));
			bcm63268_nand_write_byte_to_buf(host, (uint8_t)((reg_val >> 24) & 0xff));
			bcm63268_nand_write_byte_to_buf(host, (uint8_t)((reg_val >> 16) & 0xff));
			bcm63268_nand_write_byte_to_buf(host, (uint8_t)((reg_val >> 8) & 0xff));
			bcm63268_nand_write_byte_to_buf(host, (uint8_t)(reg_val & 0xff));
		}

		page_addr += host->cachesize;
		oobsize -= host->stepsize;
	}
}

/* Used by the upper layer to write command to NAND Flash for
 * different operations to be carried out on NAND Flash */
static void bcm63268_nand_command(struct mtd_info *mtd, unsigned command,
				int column, int page_addr)
{
	struct nand_chip *nand_chip = mtd->priv;
	struct bcm63268_nand_host *host = nand_chip->priv;
	

#if 0
	if (command == NAND_CMD_SEQIN ||
		command == NAND_CMD_PAGEPROG)
		printk("bcm63268_nand_command (cmd = 0x%x, col = 0x%x, page = 0x%x)\n",
		      command, column, page_addr);
#else
	DEBUG(MTD_DEBUG_LEVEL3,
	      "bcm63268_nand_command (cmd = 0x%x, col = 0x%x, page = 0x%x)\n",
	      command, column, page_addr);
#endif

	/* Command pre-processing step */
	switch (command) {
	case NAND_CMD_READID:
		if (column == 0x20) {
			/* ONFI */
			bcm63268_nand_reset_buf(host);
			bcm63268_nand_write_byte_to_buf(host, 0x00);
			bcm63268_nand_write_byte_to_buf(host, 0x00);
			bcm63268_nand_write_byte_to_buf(host, 0x00);
			bcm63268_nand_write_byte_to_buf(host, 0x00);
		} else {
			bcm63268_nand_cmd_readid(host);
		}
		break;
		
	case NAND_CMD_RESET:
		bcm63268_nand_reset_flash(host);
		break;
		
	case NAND_CMD_STATUS:
	case NAND_CMD_STATUS_MULTI:
		bcm63268_nand_cmd_status(host);
		break;
		
	case NAND_CMD_SEQIN:
	case NAND_CMD_READ0:
		host->pageaddr = page_addr;
		break;
		
	case NAND_CMD_PAGEPROG:
		break;
		
	case NAND_CMD_READOOB:
		bcm63268_nand_cmd_readoob(mtd, column, page_addr);
		break;
		
	default:
		pr_err(": unsupported command"
				" received 0x%x\n", command);
		break;
	}
}

static int __init bcm63268_nand_probe(struct platform_device *pdev)
{
	struct nand_chip *this;
	struct mtd_info *mtd;
	struct bcm63268_nand_platform_data *pdata = pdev->dev.platform_data;
	struct bcm63268_nand_host *host;
	struct resource *res;
	int err = 0, __maybe_unused nr_parts = 0;

	/* Allocate memory for MTD device structure and private data */
	host = kzalloc(sizeof(struct bcm63268_nand_host), GFP_KERNEL);
	if (!host)
		return -ENOMEM;

	host->dev = &pdev->dev;
	/* structures must be linked */
	this = &host->nand;
	mtd = &host->mtd;
	mtd->priv = this;
	mtd->owner = THIS_MODULE;
	mtd->dev.parent = &pdev->dev;
	mtd->name = DRV_NAME;

	/* 50 us command delay time */
	this->chip_delay = 5;

	/* register the driver with the NAND core subsystem */
	this->priv = host;
	this->dev_ready = bcm63268_nand_dev_ready;
	this->cmdfunc = bcm63268_nand_command;
	this->select_chip = bcm63268_nand_select_chip;
	this->read_byte = bcm63268_nand_read_byte;
	this->waitfunc = bcm63268_nand_waitfunc;

	host->clk = clk_get(&pdev->dev, "nfc");
	if (IS_ERR(host->clk)) {
		err = PTR_ERR(host->clk);
		goto eclk;
	}

	clk_enable(host->clk);
	host->clk_act = 1;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		err = -ENODEV;
		goto eres;
	}

	host->base = (void __iomem *)res->start;
	if (!host->base) {
		err = -ENOMEM;
		goto eres;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	if (!res) {
		err = -ENODEV;
		goto eres;
	}

	host->cache = (void __iomem *)res->start;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 2);
	if (!res) {
		err = -ENODEV;
		goto eres;
	}

	host->intr = (void __iomem *)res->start;

	host->revision = bcm_nand_readl(host, NAND_CTRL_REVISION);

	pr_info("Broadcom NAND FLASH Controller Version %d.%d\n", (host->revision & 0xff00) >> 8, host->revision & 0xff);

	if (nfc_is_v4(host)) {
		host->preset = preset_v4;
		host->send_cmd = send_cmd_v4;
		host->send_addr = send_addr_v4;
		host->send_page = send_page_v4;
		host->send_read_id = send_read_id_v4;
		host->get_dev_status = get_dev_status_v4;
		host->check_intr = check_intr_v4;
		host->clear_intr = clear_intr_v4;
		host->ctrl_intr = control_intr_v4;
		host->cachesize = 512;
		host->intfsts_retries = 20000;
		host->cmd_retries = 5;
	} else
		BUG();

	/* setup interrupt handler */
	/* the completion object will be used to notify
	 * the callee that the interrupt is done */
	init_completion(&host->op_completion);

	/* the spinlock will be used to synchronize the ISR
	 * with any element that might be access shared
	 * data (interrupt status) */
	spin_lock_init(&host->irq_lock);

	host->irq = platform_get_irq(pdev, 0);

	err = request_irq(host->irq, bcm63268_nand_isr, 0, pdev->name, host);
	if (err)
		goto eres;

	/* now that our ISR is registered, we can enable interrupts */
	host->ctrl_intr(host, 1);

	/* scan for NAND devices attached to the controller
	 * this is the first stage in a two step process to register
	 * with the nand subsystem */
	if (nand_scan_ident(mtd, 1, NULL)) {
		err = -ENXIO;
		goto escan;
	}

	/* Call preset with correct oobsize and writesize */
	host->preset(host);

	/* NAND bus width determines access funtions used by upper layer */
	if (pdata->width == 2)
		this->options |= NAND_BUSWIDTH_16;

	if (pdata->flash_bbt == 1) {
		/* update flash based bbt */
		this->options |= NAND_USE_FLASH_BBT;
	} else if (pdata->flash_bbt == 2) {
		this->options |= NAND_SKIP_BBTSCAN;
	}

	this->options |= NAND_NO_SUBPAGE_WRITE;
	this->badblockpos = 0;

	/* These functions are required by the NAND core framework, otherwise,
	 * the NAND core will assert. However, we don't need them, so we'll stub
	 * them out. */
	if (pdata->hw_ecc) {
		this->ecc.calculate = bcm63268_nand_calculate_ecc;
		this->ecc.hwctl = bcm63268_nand_enable_hwecc;
		this->ecc.correct = bcm63268_nand_correct_data;
		this->ecc.mode = NAND_ECC_HW;
	} else {
		this->ecc.mode = NAND_ECC_SOFT;
	}

	/* override the default operations */
	this->ecc.read_page = bcm63268_nand_read_page;
	this->ecc.read_page_raw = bcm63268_nand_read_page_raw;
	this->ecc.write_page = bcm63268_nand_write_page;
	this->ecc.write_page_raw = bcm63268_nand_write_page_raw;
	this->ecc.read_oob = bcm63268_nand_read_oob;
	this->ecc.write_oob = bcm63268_nand_write_oob;
	this->erase_cmd = bcm63268_nand_erase;

	/* second phase scan */
	if (nand_scan_tail(mtd)) {
		err = -ENXIO;
		goto escan;
	}

	/* Register the partitions */
#ifdef CONFIG_MTD_PARTITIONS
	if (pdata->parts)
		add_mtd_partitions(mtd, pdata->parts, pdata->nr_parts);
	else
#endif
	{
		pr_info("Registering %s as whole device\n", mtd->name);
		add_mtd_device(mtd);
	}

	platform_set_drvdata(pdev, host);

	dev_info(&pdev->dev, " at 0x%08x (irq %d, cache 0x%08x, ecclevel %d) v%s\n", 
									(uint32_t)host->base, host->irq, (uint32_t)host->cache, host->ecclevel, DRV_VER);

	return 0;

escan:
	free_irq(host->irq, host);
eres:
	clk_put(host->clk);
eclk:
	kfree(host);

	return err;
}

static int __devexit bcm63268_nand_remove(struct platform_device *pdev)
{
	struct bcm63268_nand_host *host = platform_get_drvdata(pdev);

	clk_put(host->clk);

	platform_set_drvdata(pdev, NULL);

	nand_release(&host->mtd);
	free_irq(host->irq, host);
	kfree(host);

	return 0;
}

/* PM Support */
#ifdef CONFIG_PM
static int bcm63268_nand_suspend(struct platform_device *dev, pm_message_t pm)
{
	struct bcm63268_nand_host *host = platform_get_drvdata(dev);

	clk_disable(host->clk);

	return 0;
}

static int bcm63268_nand_resume(struct platform_device *dev)
{
	struct bcm63268_nand_host *host = platform_get_drvdata(dev);

	clk_enable(host->clk);

	return 0;
}

#else
#define bcm63268_nand_suspend NULL
#define bcm63268_nand_resume NULL
#endif

static struct platform_device_id bcm63268_nand_driver_ids[] = {
	{
		.name		= "bcm63268-nand",
	},
	{ }
};
MODULE_DEVICE_TABLE(platform, bcm63268_nand_driver_ids);

static struct platform_driver bcm63268_nand_driver = {
	.probe = bcm63268_nand_probe,
	.remove = __devexit_p(bcm63268_nand_remove),
	.suspend = bcm63268_nand_suspend,
	.resume = bcm63268_nand_resume,
	.id_table = bcm63268_nand_driver_ids,
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
	},
};

static int __init bcm63268_nand_init(void)
{
	return platform_driver_register(&bcm63268_nand_driver);
}

static void __exit bcm63268_nand_cleanup(void)
{
	/* Unregister the device structure */
	platform_driver_unregister(&bcm63268_nand_driver);
}

module_init(bcm63268_nand_init);
module_exit(bcm63268_nand_cleanup);

MODULE_AUTHOR(DRV_AUTHOR);
MODULE_DESCRIPTION(DRV_DESC);
MODULE_LICENSE("GPL");
MODULE_VERSION(DRV_VER);
