/*
<:copyright-gpl 
 Copyright 2011 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/clk.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/spi/spi.h>
#include <linux/completion.h>
#include <linux/err.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>

#include <typedefs.h>
#include <bcmdefs.h>
#include <bcmdevs.h>
#include <osl.h>
#include <sbchipc.h>
#include <bcmutils.h>
#include <siutils.h>
#include <hndsoc.h>

#include <bcm47xx_io.h>
#include <bcm47xx.h>
#include <bcm47xx_board.h>

#define DRV_NAME	"bcm47xx-sflash"
#define DRV_AUTHOR	"Arcadyan Technology"
#define DRV_DESC	"Broadcom BCM47XX Chipcommon Serial FLASH"
#define DRV_VER		"1.0"

#define	ST_RETRIES	3

#ifdef	IL_BIGENDIAN
#ifdef	BCMHND74K
#define	GET_BYTE(ptr)	(*(uint8 *)((uint32)(ptr) ^ 7))
#else	/* !74K, bcm33xx */
#define	GET_BYTE(ptr)	(*(uint8 *)((uint32)(ptr) ^ 3))
#endif	/* BCMHND74K */
#else	/* !IL_BIGENDIAN */
#define	GET_BYTE(ptr)	(*(ptr))
#endif	/* IL_BIGENDIAN */


struct bcm47xx_sflash {
	si_t *sih;
	chipcregs_t *cc;
	struct bcm47xx_sflash_info *flash_info;
	struct mtd_info mtd;
	struct mtd_erase_region_info region;
	struct mutex mutex;
	unsigned int partitioned:1;
};

/* Private global state */
static struct bcm47xx_sflash sflash;

static inline void
bcm47xx_sflash_cmd(osl_t *osh, chipcregs_t *cc, uint opcode)
{
	W_REG(osh, &cc->flashcontrol, SFLASH_START | opcode);
	while (R_REG(osh, &cc->flashcontrol) & SFLASH_BUSY);
}

static int
bcm47xx_sflash_poll(struct bcm47xx_sflash_info *info, 
								si_t *sih, chipcregs_t *cc, uint offset)
{
	osl_t *osh;

	ASSERT(sih);

	osh = si_osh(sih);

	if (offset >= info->size)
		return -22;

	switch (info->type) {
		case SFLASH_ST:
			bcm47xx_sflash_cmd(osh, cc, SFLASH_ST_RDSR);
			return R_REG(osh, &cc->flashdata) & SFLASH_ST_WIP;
		case SFLASH_AT:
			bcm47xx_sflash_cmd(osh, cc, SFLASH_AT_STATUS);
			return !(R_REG(osh, &cc->flashdata) & SFLASH_AT_READY);
	}

	return 0;
}

static int
bcm47xx_sflash_read(struct bcm47xx_sflash_info *info, si_t *sih, 
								chipcregs_t *cc, uint offset, uint len, uchar *buf)
{
	uint8 *from, *to;
	int cnt, i;
	osl_t *osh;

	ASSERT(sih);

	if (!len)
		return 0;

	if ((offset + len) > info->size)
		return -22;

	if ((len >= 4) && (offset & 3))
		cnt = 4 - (offset & 3);
	else if ((len >= 4) && ((uintptr)buf & 3))
		cnt = 4 - ((uintptr)buf & 3);
	else
		cnt = len;

	osh = si_osh(sih);

	if (sih->ccrev == 12)
		from = (uint8 *)OSL_UNCACHED(SI_FLASH2 + offset);
	else
		from = (uint8 *)OSL_CACHED(SI_FLASH2 + offset);
	to = (uint8 *)buf;

	if (cnt < 4) {
		for (i = 0; i < cnt; i ++) {
			/* Cannot use R_REG because in bigendian that will
			 * xor the address and we don't want that here.
			 */
			*to = *from;
			from ++;
			to ++;
		}
		return cnt;
	}

	while (cnt >= 4) {
		*(uint32 *)to = *(uint32 *)from;
		from += 4;
		to += 4;
		cnt -= 4;
	}

	return (len - cnt);
}

static int
bcm47xx_sflash_write(struct bcm47xx_sflash_info *info, si_t *sih, 
								chipcregs_t *cc, uint offset, uint length, const uchar *buffer)
{
	uint off = offset, len = length;
	const uint8 *buf = buffer;
	uint8 data;
	int ret = 0, ntry = 0;
	bool is4712b0;
	uint32 page, byte, mask;
	osl_t *osh;

	ASSERT(sih);

	osh = si_osh(sih);

	if (!len)
		return 0;

	if ((off + len) > info->size)
		return -22;

	switch (info->type) {
	case SFLASH_ST:
		is4712b0 = (sih->chip == BCM4712_CHIP_ID) && (sih->chiprev == 3);
		/* Enable writes */
retry:	
		bcm47xx_sflash_cmd(osh, cc, SFLASH_ST_WREN);
		off = offset;
		len = length;
		buf = buffer;
		ntry++;
		if (is4712b0) {
			mask = 1 << 14;
			W_REG(osh, &cc->flashaddress, off);
			data = GET_BYTE(buf);
			buf++;
			W_REG(osh, &cc->flashdata, data);
			/* Set chip select */
			OR_REG(osh, &cc->gpioout, mask);
			/* Issue a page program with the first byte */
			bcm47xx_sflash_cmd(osh, cc, SFLASH_ST_PP);
			ret = 1;
			off++;
			len--;
			while (len > 0) {
				if ((off & 255) == 0) {
					/* Page boundary, drop cs and return */
					AND_REG(osh, &cc->gpioout, ~mask);
					OSL_DELAY(1);
					if (!bcm47xx_sflash_poll(info, sih, cc, off)) {
						/* Flash rejected command */
						if (ntry <= ST_RETRIES)
							goto retry;
						else
							return -11;
					}
					return ret;
				} else {
					/* Write single byte */
					data = GET_BYTE(buf);
					buf++;
					bcm47xx_sflash_cmd(osh, cc, data);
				}
				ret++;
				off++;
				len--;
			}
			/* All done, drop cs */
			AND_REG(osh, &cc->gpioout, ~mask);
			OSL_DELAY(1);
			if (!bcm47xx_sflash_poll(info, sih, cc, off)) {
				/* Flash rejected command */
				if (ntry <= ST_RETRIES)
					goto retry;
				else
					return -12;
			}
		} else if (sih->ccrev >= 20) {
			W_REG(osh, &cc->flashaddress, off);
			data = GET_BYTE(buf);
			buf++;
			W_REG(osh, &cc->flashdata, data);
			/* Issue a page program with CSA bit set */
			bcm47xx_sflash_cmd(osh, cc, SFLASH_ST_CSA | SFLASH_ST_PP);
			ret = 1;
			off++;
			len--;
			while (len > 0) {
				if ((off & 255) == 0) {
					/* Page boundary, poll droping cs and return */
					W_REG(NULL, &cc->flashcontrol, 0);
					OSL_DELAY(1);
					if (bcm47xx_sflash_poll(info, sih, cc, off) == 0) {
						/* Flash rejected command */
						pr_debug("%s: pp rejected, ntry: %d,"
						         " off: %d/%d, len: %d/%d, ret:"
						         "%d\n", DRV_NAME, ntry, off, offset, len,
						         length, ret);
						if (ntry <= ST_RETRIES)
							goto retry;
						else
							return -11;
					}
					return ret;
				} else {
					/* Write single byte */
					data = GET_BYTE(buf);
					buf++;
					bcm47xx_sflash_cmd(osh, cc, SFLASH_ST_CSA | data);
				}
				ret++;
				off++;
				len--;
			}
			/* All done, drop cs & poll */
			W_REG(NULL, &cc->flashcontrol, 0);
			OSL_DELAY(1);
			if (bcm47xx_sflash_poll(info, sih, cc, off) == 0) {
				/* Flash rejected command */
				pr_debug("%s: pp rejected, ntry: %d, off: %d/%d,"
				         " len: %d/%d, ret: %d\n",
				         DRV_NAME, ntry, off, offset, len, length, ret);
				if (ntry <= ST_RETRIES)
					goto retry;
				else
					return -12;
			}
		} else {
			ret = 1;
			W_REG(osh, &cc->flashaddress, off);
			data = GET_BYTE(buf);
			buf++;
			W_REG(osh, &cc->flashdata, data);
			/* Page program */
			bcm47xx_sflash_cmd(osh, cc, SFLASH_ST_PP);
		}
		break;
	case SFLASH_AT:
		mask = info->block_size - 1;
		page = (off & ~mask) << 1;
		byte = off & mask;
		/* Read main memory page into buffer 1 */
		if (byte || (len < info->block_size)) {
			W_REG(osh, &cc->flashaddress, page);
			bcm47xx_sflash_cmd(osh, cc, SFLASH_AT_BUF1_LOAD);
			/* 250 us for AT45DB321B */
			SPINWAIT(bcm47xx_sflash_poll(info, sih, cc, off), 1000);
			ASSERT(!bcm47xx_sflash_poll(info, sih, cc, off));
		}
		/* Write into buffer 1 */
		for (ret = 0; (ret < (int)len) && (byte < info->block_size); ret++) {
			W_REG(osh, &cc->flashaddress, byte++);
			W_REG(osh, &cc->flashdata, *buf++);
			bcm47xx_sflash_cmd(osh, cc, SFLASH_AT_BUF1_WRITE);
		}
		/* Write buffer 1 into main memory page */
		W_REG(osh, &cc->flashaddress, page);
		bcm47xx_sflash_cmd(osh, cc, SFLASH_AT_BUF1_PROGRAM);
		break;
	}

	return ret;
}

static int
bcm47xx_sflash_erase(struct bcm47xx_sflash_info *info, si_t *sih, chipcregs_t *cc, uint offset)
{
	osl_t *osh;

	ASSERT(sih);

	osh = si_osh(sih);

	if (offset >= info->size)
		return -22;

	switch (info->type) {
	case SFLASH_ST:
		bcm47xx_sflash_cmd(osh, cc, SFLASH_ST_WREN);
		W_REG(osh, &cc->flashaddress, offset);
		bcm47xx_sflash_cmd(osh, cc, SFLASH_ST_SE);
		return info->block_size;
	case SFLASH_AT:
		W_REG(osh, &cc->flashaddress, offset << 1);
		bcm47xx_sflash_cmd(osh, cc, SFLASH_AT_PAGE_ERASE);
		return info->block_size;
	}

	return 0;
}

static int
bcm47xx_sflash_mtd_poll(struct bcm47xx_sflash *flash, unsigned int offset, int timeout)
{
	int now = jiffies;
	int ret = 0;

	for (;;) {
		if (!bcm47xx_sflash_poll(flash->flash_info, flash->sih, flash->cc, offset))
			break;

		if (time_after((unsigned long)jiffies, (unsigned long)now + timeout)) {
			if (!bcm47xx_sflash_poll(flash->flash_info, flash->sih, flash->cc, offset))
				break;

			pr_err("%s: timeout\n", DRV_NAME);
			ret = -ETIMEDOUT;
			break;
		}
		udelay(1);
	}

	return ret;
}

static int
bcm47xx_sflash_mtd_read(struct mtd_info *mtd, loff_t from, 
									size_t len, size_t *retlen, u_char *buf)
{
	struct bcm47xx_sflash *flash = (struct bcm47xx_sflash *)mtd->priv;
	int bytes, ret = 0;

	/* Check address range */
	if (!len)
		return 0;
	
	if ((from + len) > mtd->size)
		return -EINVAL;
	
	mutex_lock(&flash->mutex);

	*retlen = 0;
	while (len) {
		if ((bytes = bcm47xx_sflash_read(flash->flash_info, 
				flash->sih, flash->cc, (uint) from, len, buf)) < 0) {
			ret = bytes;
			break;
		}
		from += (loff_t) bytes;
		len -= bytes;
		buf += bytes;
		*retlen += bytes;
	}
	
	mutex_unlock(&flash->mutex);
	return ret;
}

static int
bcm47xx_sflash_mtd_write(struct mtd_info *mtd, loff_t to, 
										size_t len, size_t *retlen, const u_char *buf)
{
	struct bcm47xx_sflash *flash = (struct bcm47xx_sflash *) mtd->priv;
	int bytes, ret = 0;

	/* Check address range */
	if (!len)
		return 0;
	if ((to + len) > mtd->size)
		return -EINVAL;

	mutex_lock(&flash->mutex);
	*retlen = 0;
	while (len) {
		if ((bytes = bcm47xx_sflash_write(flash->flash_info, 
						flash->sih, flash->cc, (uint) to, len, buf)) < 0) {
			ret = bytes;
			break;
		}
		if ((ret = bcm47xx_sflash_mtd_poll(flash, (unsigned int) to, HZ)))
			break;
		to += (loff_t) bytes;
		len -= bytes;
		buf += bytes;
		*retlen += bytes;
	}

	mutex_unlock(&flash->mutex);
	return ret;
}

static int
bcm47xx_sflash_mtd_erase(struct mtd_info *mtd, struct erase_info *erase)
{
	struct bcm47xx_sflash *flash = (struct bcm47xx_sflash *) mtd->priv;
	int i, j, ret = 0;
	unsigned int addr, len;

	/* Check address range */
	if (!erase->len)
		return 0;
	if ((erase->addr + erase->len) > mtd->size)
		return -EINVAL;

	mutex_lock(&flash->mutex);
	addr = erase->addr;
	len = erase->len;

	/* Ensure that requested region is aligned */
	for (i = 0; i < mtd->numeraseregions; i++) {
		for (j = 0; j < mtd->eraseregions[i].numblocks; j++) {
			if (addr == mtd->eraseregions[i].offset +
			    mtd->eraseregions[i].erasesize * j &&
			    len >= mtd->eraseregions[i].erasesize) {
				if ((ret = bcm47xx_sflash_erase(flash->flash_info, flash->sih, flash->cc, addr)) < 0)
					break;
				if ((ret = bcm47xx_sflash_mtd_poll(flash, addr, 10 * HZ)))
					break;
				addr += mtd->eraseregions[i].erasesize;
				len -= mtd->eraseregions[i].erasesize;
			}
		}
		if (ret)
			break;
	}

	/* Set erase status */
	if (ret)
		erase->state = MTD_ERASE_FAILED;
	else
		erase->state = MTD_ERASE_DONE;

	mutex_unlock(&flash->mutex);

	/* Call erase callback */
	if (erase->callback)
		erase->callback(erase);

	return ret;
}

static int __init
bcm47xx_sflash_mtd_init(void)
{
	int ret = 0;
	struct bcm47xx_sflash_info *info;
	struct pci_dev *dev = NULL;

	list_for_each_entry(dev, &((pci_find_bus(0, 0))->devices), bus_list) {
		if ((dev != NULL) && (dev->device == CC_CORE_ID))
			break;
	}

	if (!dev)
		return -ENODEV;

	memset(&sflash, 0, sizeof(struct bcm47xx_sflash));

	/* attach to the backplane */
	if (!(sflash.sih = si_kattach(SI_OSH))) {
		pr_err("%s: attach to backplane failed\n", DRV_NAME);
		return -ENODEV;
	}

	/* Map registers and flash base */
	if (!(sflash.cc = ioremap_nocache(pci_resource_start(dev, 0), pci_resource_len(dev, 0)))) {
		pr_err("%s: ioremap register failed\n", DRV_NAME);
		ret = -ENODEV;
		goto out_si_detach;
	}

	/* Initialize serial flash access */
	if (!(info = bcm47xx_board_get_sflash_info())) {
		pr_err("%s: flash info unvailable\n", DRV_NAME);
		ret = -ENODEV;
		goto out_iounmap;
	}

	/* Setup region info */
	sflash.region.offset = 0;
	sflash.region.erasesize = info->block_size;
	sflash.region.numblocks = info->n_blocks;
	if (sflash.region.erasesize > sflash.mtd.erasesize)
		sflash.mtd.erasesize = sflash.region.erasesize;
	info->size = info->block_size * info->n_blocks;
	sflash.mtd.size = info->size;
	sflash.mtd.numeraseregions = 1;

	/* Register with MTD */
	sflash.mtd.name = "sflash";
	sflash.mtd.type = MTD_NORFLASH;
	sflash.mtd.flags = MTD_CAP_NORFLASH;
	sflash.mtd.eraseregions = &sflash.region;
	sflash.mtd.erase = bcm47xx_sflash_mtd_erase;
	sflash.mtd.read = bcm47xx_sflash_mtd_read;
	sflash.mtd.write = bcm47xx_sflash_mtd_write;
	sflash.mtd.writesize = 1;
	sflash.mtd.priv = &sflash;
	sflash.mtd.owner = THIS_MODULE;
	sflash.flash_info = info;
	
	mutex_init(&sflash.mutex);

	if (mtd_has_partitions()) {
		struct mtd_partition	*parts = NULL;
		int	nr_parts = 0;
		int i;

		if (mtd_has_cmdlinepart()) {
			static const char *part_probes[]
					= { "cmdlinepart", NULL, };

			nr_parts = parse_mtd_partitions(&sflash.mtd,
					part_probes, &parts, 0);
		}

		if (nr_parts <= 0 && info->parts) {
			parts = info->parts;
			nr_parts = info->nr_parts;
		}

		if (nr_parts > 0) {
			for (i = 0; i < nr_parts; i++) {
				DEBUG(MTD_DEBUG_LEVEL2, "partitions[%d] = "
					"{.name = %s, .offset = 0x%llx, "
						".size = 0x%llx (%lldKiB) }\n",
					i, parts[i].name,
					(long long)parts[i].offset,
					(long long)parts[i].size,
					(long long)(parts[i].size >> 10));
			}
			sflash.partitioned = 1;
			return add_mtd_partitions(&sflash.mtd, parts, nr_parts);
		}
	} else if (info->nr_parts)
		pr_warning("%s: ignoring %d default partitions on %s\n",
				DRV_NAME, info->nr_parts, info->name);
	
	return 0;

out_si_detach:
	si_detach(sflash.sih);

out_iounmap:
	iounmap((void *) sflash.cc);
	return ret;
}

static void __exit
bcm47xx_sflash_mtd_exit(void)
{
	int status;
	
	if (mtd_has_partitions() && sflash.partitioned)
		status = del_mtd_partitions(&sflash.mtd);
	else
		status = del_mtd_device(&sflash.mtd);

	iounmap((void *)sflash.cc);

	si_detach(sflash.sih);
}

module_init(bcm47xx_sflash_mtd_init);
module_exit(bcm47xx_sflash_mtd_exit);

MODULE_AUTHOR(DRV_AUTHOR);
MODULE_DESCRIPTION(DRV_DESC);
MODULE_LICENSE("GPL");
MODULE_VERSION(DRV_VER);

