/*
<:copyright-gpl 
 Copyright 2010 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/i2c.h>
#include <asm/uaccess.h>
#include <linux/proc_fs.h>
#include "m02098.h"
#include "m02098_lut.h"


#define DRV_NAME	"m02098"
#define DRV_AUTHOR	"Arcadyan Technology"
#define DRV_DESC	"Broadcom on-chip I2C controller Proc for M02098"
#define DRV_VER		"1.0"
#define M02098_DEVICE_MAJOR 239
#define M02098_DEVICE_NAME  "m02098"

#define M02098_I2C_CHIP_ADDRESS				0x4e
#define M02098_I2C_CHIP_PROC_ARG_NUM_MAX	0xff
#define M02098_I2C_CHIP_PROC_ARG_SIZE_MAX	32
/*#define M02098_I2C_CHIP_PROC_KBUF_SIZE_MAX	128*/
#define M02098_I2C_CHIP_PROC_KBUF_SIZE_MAX	192
static unsigned short normal_i2c[] = { M02098_I2C_CHIP_ADDRESS, I2C_CLIENT_END };
char m02098loaded = 0;
static int M02098Major = M02098_DEVICE_MAJOR;
static int rx_slope = 100 , tx_slope = 100, temp_slope = 10;
static int rx_calibdB = 0 , tx_calibdB = 0;

I2C_CLIENT_INSMOD_1(m02098);

extern const m02098_lut m02098_lut_table[];

static int m02098_i2c_chip_attach_adapter(struct i2c_adapter *adapter);
static int m02098_i2c_chip_detect(struct i2c_adapter *adapter, int address, int kind);
static int m02098_i2c_chip_detach_client(struct i2c_client *client);

/* Each client has this additional data */
struct m02098_i2c_chip_data {
	struct i2c_client client;
	struct semaphore update_lock;
};

static struct m02098_i2c_chip_data *chip_data;

/* This is the driver that will be inserted */
static struct i2c_driver m02098_i2c_chip_driver = {
	.driver = {
		.name	= DRV_NAME,
		.owner	= THIS_MODULE,
	},
	.attach_adapter	= m02098_i2c_chip_attach_adapter,
	.detach_client	= m02098_i2c_chip_detach_client,
};

static void m02098_i2c_chip_read(unsigned long arg)
{
    struct i2c_msg msg[2];
	m02098_readregs receiveMsg;
    char buf[9];	
    int i;
	u8 offset, length;

	memset(&receiveMsg, 0, sizeof(receiveMsg));
    copy_from_user(&receiveMsg, (m02098_readregs *)arg,
                           sizeof(m02098_readregs));
	offset = (u8)receiveMsg.offset;
	length = (u8)receiveMsg.length;
	msg[0].addr = M02098_I2C_CHIP_ADDRESS;
	msg[0].flags = 0;
	msg[1].addr = M02098_I2C_CHIP_ADDRESS;
	msg[1].flags = 0;

    msg[0].len = 1;
    buf[0] = offset;
    msg[0].buf = buf;

    msg[1].flags |= I2C_M_RD;
    msg[1].len = length;
    msg[1].buf = buf;
    if(i2c_transfer(chip_data->client.adapter, msg, 2) == 2)
        {
        for (i=0; i <  receiveMsg.length; i++) {
			receiveMsg.buf[i] = buf[i] & 0xFF;
            copy_to_user( (char *) arg , (char *) &receiveMsg, sizeof(m02098_readregs));			
            }
        } else {
            pr_info(DRV_NAME "read failed\n");
        	}
}

static void m02098_i2c_chip_generic_access(u8 offset, u8 length, 
															int val, u8 write)
{
#if 1
    struct i2c_msg msg[2];
    char buf[8];	
    int i;

	msg[0].addr = M02098_I2C_CHIP_ADDRESS;
	msg[0].flags = 0;
	msg[1].addr = M02098_I2C_CHIP_ADDRESS;
	msg[1].flags = 0;

	if (write) {
		msg[0].len = length + 1;
		buf[0] = offset;
		/* On the I2C bus, LS byte should go first */
		val = swab32(val);
		memcpy(&buf[1], (char*)&val, length);
        msg[0].buf = buf;
		if(i2c_transfer(chip_data->client.adapter, msg, 1) != 1) {
			pr_info(DRV_NAME "write failed\n");
		}
	}
	else {
        msg[0].len = 1;
        buf[0] = offset;
        msg[0].buf = buf;

        msg[1].flags |= I2C_M_RD;
        msg[1].len = length;
        msg[1].buf = buf;

        /* On I2C bus, we receive LS byte first. So swap bytes as necessary */
        if(i2c_transfer(chip_data->client.adapter, msg, 2) == 2)
        {
            for (i=0; i < length; i++) {
                printk("0x%02x = 0x%02x \n", offset + i, buf[i] & 0xFF);
            }
            printk("\n");
        } else {
            pr_info(DRV_NAME "read failed\n");
        }
	}
#else
    struct i2c_msg msg[2];
    char buf[8];
    int i, j, len, loop;
	msg[0].addr = M02098_I2C_CHIP_ADDRESS;
	msg[0].flags = 0;
	msg[1].addr = M02098_I2C_CHIP_ADDRESS;
	msg[1].flags = 0;
/*	pr_info("m02098_i2c_chip_generic_access %x %d %x %d\n", offset, length, val, write);*/

	if (write) {
		msg[0].len = length + 1;
		buf[0] = offset;
		/* On the I2C bus, LS byte should go first */
		val = swab32(val);
		memcpy(&buf[1], (char*)&val, length);
        msg[0].buf = buf;
		if(i2c_transfer(chip_data->client.adapter, msg, 1) != 1) {
			pr_info(DRV_NAME "write failed\n");
			}
		}
	else {
		if (length < 1)
			{
			loop = 0;
			}
		else{
			loop = length/8 + (((length%8) > 0) ? 1: 0);
			}
		
		for (j = 0; j< loop; j++)
			{
       		msg[0].len = 1;
			buf[0] = offset+j*8;
        	msg[0].buf = buf;

        	msg[1].flags |= I2C_M_RD;
			len = ((length - 8*j) > 8 ? 8:(length - 8*j));
			msg[1].len =  ((length - 8*j) > 8 ? 8:(length - 8*j));
        	msg[1].buf = buf;

        	/* On I2C bus, we receive LS byte first. So swap bytes as necessary */
        	if(i2c_transfer(chip_data->client.adapter, msg, 2) == 2)
       		 {
       		 for (i=0; i < (len > 8 ? 8:len); i++)
			 	{
              	printk("offset 0x%02x = 0x%02x \n", offset+j*8 + i, buf[i] & 0xFF);
            	}
            printk("\n");
        	} else {
            pr_info(DRV_NAME "read failed\n");
        	}
		}
	}
#endif
}

static ssize_t m02098_i2c_chip_proc_fop_read(struct file *f, 
													char *buf, size_t count, loff_t *pos)
{
	return 0;
}


typedef struct 
{
  u8 regnum;
  u8 value;
} regform;

#define ishex(v)	(((v)>='0'&&(v)<='9') || ((v)>='A'&&(v)<='F'))
#define isnumber(v)	((v)>='0'&&(v)<='9')


char proc_argv[M02098_I2C_CHIP_PROC_ARG_NUM_MAX][M02098_I2C_CHIP_PROC_ARG_SIZE_MAX];
u8 kbuf[M02098_I2C_CHIP_PROC_KBUF_SIZE_MAX];
regform m02098reg[96];
	
static ssize_t m02098_i2c_chip_proc_fop_write(struct file *f, 
													char *ubuf, size_t count, loff_t *pos)
{
	int proc_argc = 0, i, j, val, index = -1, cmdlen = 0;
	char cmd;
/*	u8 kbuf[M02098_I2C_CHIP_PROC_KBUF_SIZE_MAX];*/
	u8 off, len, counter;



	if (count > (M02098_I2C_CHIP_PROC_KBUF_SIZE_MAX - 1))
		return -EINVAL;
	
	if (copy_from_user(kbuf, ubuf, count) != 0)
		return -EFAULT;

	kbuf[count] = 0;
	
#if 0
	proc_argc = sscanf(kbuf, "%c %s %s %s", &cmd, 
						proc_argv[0], proc_argv[1], proc_argv[2], proc_argv[3]);

#else/* the useful M02098 control registers is from 0x00 to 0x60 (total 96 registers)*/
	proc_argc = sscanf(&kbuf[0], "%c", &cmd); 
	for (j = (sizeof(cmd)+1), i = 0 ; j <= count -(sizeof(cmd)+1)  && i <= 96; i ++)
		{
		sscanf(&kbuf[j], "%s", proc_argv[i]);
		cmdlen = (strlen(proc_argv[i]) +1);
		j+= cmdlen;
		proc_argc ++;

		}
#endif

	if (proc_argc < 1)
		return -EINVAL;

	for (i = 0; i < M02098_I2C_CHIP_PROC_ARG_NUM_MAX; i++)
		proc_argv[i][M02098_I2C_CHIP_PROC_ARG_SIZE_MAX - 1] = 0;

	switch (cmd) {
		case 'a':
			if (proc_argc < 3)
				return -EINVAL;

			off = (u8)simple_strtoul(proc_argv[0], NULL, 0);
			len = (u8)simple_strtoul(proc_argv[1], NULL, 0);
			if (proc_argc == 4) {
				val = (int)simple_strtoul(proc_argv[2], NULL, 0);
				m02098_i2c_chip_generic_access(off, len, val, 1);
			}
			else {
				m02098_i2c_chip_generic_access(off, len, 0, 0);
			}
			break;
			
		case 's':
			if (proc_argc < 3)
				return -EINVAL;
			counter = 0;

			do{
			m02098reg[counter].regnum = (u8)simple_strtoul(proc_argv[++index], NULL, 0);				
			m02098reg[counter].value = (u8)simple_strtoul(proc_argv[++index], NULL, 0);
			pr_info("regnum = %d, value = %d \n",m02098reg[counter].regnum, m02098reg[counter].value);			
			counter ++;			
				}while (counter < 96 && proc_argc > index +2);
			len = 1;
			for (i = 0 ; i < counter; i++)
				{
				off = m02098reg[i].regnum;
				val = m02098reg[i].value;
				m02098_i2c_chip_generic_access(off, len, val, 1);
				}
			
			break;
			
		default:
			pr_info("usage :\n");
			pr_info("a <offset> <length> [<write_data>] :\n");			
			pr_info("s [<reg-offset> <value>] [<reg-offset> <value>].. :\n");
			

	}

	return count;
}


/* 0x00 ~ 0x2F*/
char initreg[48] = {0x20,0x01,0x02,0x09,0x00,0xB4,0x00,0x77,0x01,0xC0,0x4D,0x02,0x02,0x32,0x00,0x00,
					0x24,0x14,0xEA,0x0E,0x00,0x09,0x00,0x00,0x28,0x6F,0xFF,0xFF,0x2F,0xFF,0xFF,0xFF,
					0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x00,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x00};


#define FIFO_LEN 7
int m02098_reg_Init(void)
{
	struct i2c_msg msg[2];
    char buf[9];	
    int i, j, loop, length = 48;
	int val;

	loop = length/FIFO_LEN + (((length%FIFO_LEN) > 0) ? 1: 0);

	for (j = 0; j< loop; j++)
		{
		msg[0].addr = M02098_I2C_CHIP_ADDRESS;
		msg[0].flags = 0;
		msg[1].addr = M02098_I2C_CHIP_ADDRESS;
		msg[1].flags = 0;
		if (length - FIFO_LEN*j > FIFO_LEN)	
			msg[0].len = FIFO_LEN + 1;
		else
			msg[0].len = length - (FIFO_LEN*j) + 1;
			
		for(i = 0; i < FIFO_LEN; i ++)
			{
			val = initreg[j*FIFO_LEN+i];
			/*val = swab32(val);*/
			buf[i+1] = (unsigned char)val;		
			}
		 msg[0].buf = buf;
  		 buf[0] = j * FIFO_LEN;
		 if(i2c_transfer(chip_data->client.adapter, msg, 1) != 1) {
			pr_info(DRV_NAME "write failed\n");
			}
		}		 
	m02098loaded = 1;
return 0;
}


int m02098_reg_Init_microCtrlOnly(void)
{
	int temp, temperature;
	struct i2c_msg msg[2];
    char buf[9];	
    int i, j, loop, length = 28;/* 0x00 ~ 0x1B */
	int val;

	loop = length/FIFO_LEN + (((length%FIFO_LEN) > 0) ? 1: 0);

	for (j = 0; j< loop; j++)
		{
		msg[0].addr = M02098_I2C_CHIP_ADDRESS;
		msg[0].flags = 0;
		msg[1].addr = M02098_I2C_CHIP_ADDRESS;
		msg[1].flags = 0;
		if (length - FIFO_LEN*j > FIFO_LEN)	
			msg[0].len = FIFO_LEN + 1;
		else
			msg[0].len = length - (FIFO_LEN*j) + 1;
			
		for(i = 0; i < FIFO_LEN; i ++)
			{
			val = initreg[j*FIFO_LEN+i];
			/*val = swab32(val);*/
			buf[i+1] = (unsigned char)val;		
			}
		 msg[0].buf = buf;
  		 buf[0] = j * FIFO_LEN;
		 if(i2c_transfer(chip_data->client.adapter, msg, 1) != 1) {
			pr_info(DRV_NAME "write failed\n");
			}
		}		 
	/* matched look up table */

	msg[0].addr = M02098_I2C_CHIP_ADDRESS;
	msg[0].flags = 0;
	msg[1].addr = M02098_I2C_CHIP_ADDRESS;
	msg[1].flags = 0;

	msg[0].len = 1;
	buf[0] = 0x57;
	msg[0].buf = buf;

	msg[1].flags |= I2C_M_RD;
	msg[1].len = 2;
	msg[1].buf = buf;

	if(i2c_transfer(chip_data->client.adapter, msg, 2) != 2)
			pr_info(DRV_NAME "read temperature failed\n");
	else{
		/* search corresponding LUT */
		temp = (((buf[1] &0xFF)<<2) &0x3FC)+(buf[0] &0x003);
		/* -40 ~ 125 degree /10 bits resolution */
		temperature = -40 + (temp * 165 / 1023);
/*		pr_info("temperature = %d degree (count: 0x%x)\n",temperature, temp);*/
		temperature = temperature * 1000;
		for (i = START_TEMP, j = 0; i <END_TEMP; i+= INTERVAL, j++)
			{
			if(temperature > i)
				break;
			}

		/* load i0_msb, ib0_lsb and im0_lsb to 0x0D, 0x0E, 0x0F respectively */
		msg[0].addr = M02098_I2C_CHIP_ADDRESS;
		msg[0].flags = 0;
		msg[1].addr = M02098_I2C_CHIP_ADDRESS;
		msg[1].flags = 0;

		msg[0].len = 4;
		buf[0] = 0x0D;

		buf[1] = m02098_lut_table[j].i0_msb & 0xff; /* loaded to 0x0D*/
		buf[2] = m02098_lut_table[j].ib0_lsb & 0xff; /* loaded to 0x0E */
		buf[3] = m02098_lut_table[j].im0_lsb & 0xff;	/* loaded to 0x0F */
		msg[0].buf = buf;
		if(i2c_transfer(chip_data->client.adapter, msg, 1) != 1) {
			pr_info(DRV_NAME "write failed\n");
			}

		}
	

return 0;
}


static int m02098Ioctl(struct inode *ip, struct file *fp, unsigned int cmd, unsigned long arg)
{
    int ret = 0;


switch((m02098_ioctlCmd_t)(cmd)) 
    {
        case M02098_IOC_READREG:
        {
          m02098_readregs m02098param;

 	      copy_from_user(&m02098param, (m02098_readregs *)arg,
                           sizeof(m02098_readregs));

		  m02098_i2c_chip_read(arg);	

            break;
        }             
        case M02098_IOC_WRITEREG:
        {
          m02098_writeregs m02098param;

 	      copy_from_user(&m02098param, (m02098_writeregs *)arg,
                           sizeof(m02098_writeregs));
		  m02098_i2c_chip_generic_access(m02098param.offset, 1, m02098param.value, 1);

            break;
        }             
		case M02098_IOC_INITREGS:
			m02098_reg_Init();
			break;

		case M02098_IOC_INITREGS_MICROCTRL:
			m02098_reg_Init_microCtrlOnly();
			break;

        case M02098_LOADED:
			copy_to_user( (char *) arg , (char *) &m02098loaded, sizeof(char));			
			ret = -1;
			break;
	   case M02098_CALIB_READ:
	   		{
             m02098_calibration calib;
			 calib.rx = rx_slope;
			 calib.tx = tx_slope;
			 calib.temp = temp_slope;			 
	   		copy_to_user( (char *) arg , (char *) &calib, sizeof(calib));			
			
		   	break;
	   		}
	   case M02098_CALIB_WRITE:
			{
    	    m02098_calibration calib;
 	        copy_from_user(&calib, (m02098_calibration *)arg,
                           sizeof(m02098_calibration));
			rx_slope = calib.rx;
			tx_slope = calib.tx;
			temp_slope = calib.temp;			
			break;
		}
	   case M02098_CALIB_READ_DBM:
	   		{
             m02098_calibration calib;
			 calib.rx = rx_calibdB;
			 calib.tx = tx_calibdB;
			 calib.temp = temp_slope;			 
	   		copy_to_user( (char *) arg , (char *) &calib, sizeof(calib));			
			
		   	break;
	   		}
	   case M02098_CALIB_WRITE_DBM:
			{
    	    m02098_calibration calib;
 	        copy_from_user(&calib, (m02098_calibration *)arg,
                           sizeof(m02098_calibration));
			rx_calibdB = calib.rx;
			tx_calibdB = calib.tx;
			temp_slope = calib.temp;			
			break;
		}
        default:
			pr_info("invalid command, %d\n",cmd);
			
            ret = -1;
    }

    return ret;
}


static ssize_t m02098_i2c_chip_proc_read(char *page, char **start, 
								off_t off, int count, int *eof, void *data) 
{
    struct i2c_msg msg[2];
    struct m02098_i2c_chip_data *chip_data = (struct m02098_i2c_chip_data *)data; 
	struct i2c_client *client = &chip_data->client;

    msg[0].addr = client->addr;
	msg[0].flags = client->flags;
	msg[0].len = 1;
	page[0] = (u8)off;
	msg[0].buf = page;

	msg[1].addr = client->addr;
	msg[1].flags = client->flags;
	msg[1].flags |= I2C_M_RD;
	msg[1].len = count;
	msg[1].buf = page;

	if (i2c_transfer(client->adapter, msg, 2) != 2)
		return -1;
	
    *start = page;
    *eof = 1;

    return count;
}

static ssize_t m02098_i2c_chip_proc_write(struct file *file, 
							const char __user *buffer, unsigned long count, void *data)
{
    struct m02098_i2c_chip_data *chip_data = (struct m02098_i2c_chip_data *)data; 
	struct i2c_client *client = &chip_data->client;
	u8 kbuf[32];

	kbuf[0] = (u8)file->f_pos;
	copy_from_user((void *)&kbuf[1], buffer, count);

    return i2c_master_send(client, kbuf, count + 1) - 1;
}


static struct file_operations m02098_i2c_chip_proc_fops = {
    read: m02098_i2c_chip_proc_fop_read,
    write: m02098_i2c_chip_proc_fop_write,
};


/* File operations struct for character device */
static const struct file_operations bosa_fops = {
	.owner		= THIS_MODULE,
	.ioctl		= m02098Ioctl,
	.release	= NULL
};


int m02098_Init(void)
{
    int ret;
	
	if (M02098Major == -1) {
		if ((M02098Major = register_chrdev (0, M02098_DEVICE_NAME, &bosa_fops)) < 0)
			{
			return -1;
			}
		}
	else
		{
		ret = register_chrdev(M02098Major, M02098_DEVICE_NAME, &bosa_fops);
	    if(ret < 0)
    		{
	        return ret;
    		}
	    if(M02098Major == 0)
    		{
	        M02098Major = ret;
    		}				
		}	
    return 0;
}

/* This function is called by i2c_detect */
static int m02098_i2c_chip_detect(struct i2c_adapter *adapter, int address, int kind)
{
	struct i2c_client *new_client;
	struct proc_dir_entry *m02098_proc;
	int err = 0;

	if (!i2c_check_functionality(adapter, I2C_FUNC_I2C))
		goto out;

	chip_data = kmalloc(sizeof(struct m02098_i2c_chip_data), GFP_KERNEL);
	if (!chip_data) {
		err = -ENOMEM;
		goto out;
	}
			
	memset(chip_data, 0x00, sizeof(struct m02098_i2c_chip_data));

	new_client = &chip_data->client;
	i2c_set_clientdata(new_client, chip_data);
	new_client->addr = address;
	new_client->adapter = adapter;
	new_client->driver = &m02098_i2c_chip_driver;
	new_client->flags = 0;

	/* Fill in the remaining client fields */
	strncpy(new_client->name, DRV_NAME, I2C_NAME_SIZE);
	init_MUTEX(&chip_data->update_lock);

	/* Tell the I2C layer a new client has arrived */
	err = i2c_attach_client(new_client);
	if (err)
		goto out_kfree;

	m02098_proc = create_proc_entry("m02098", S_IWUSR | S_IRUGO, NULL);
	if (!m02098_proc) {
		err = -ENODEV;
		goto out_detach;
	}
	m02098_proc->proc_fops = &m02098_i2c_chip_proc_fops;
	m02098_proc->read_proc = m02098_i2c_chip_proc_read;
	m02098_proc->write_proc = m02098_i2c_chip_proc_write;
	m02098_proc->data = (void *)chip_data;
	m02098_reg_Init();

	return 0;

out_detach:
	i2c_detach_client(new_client);
	
out_kfree:
	kfree(new_client);
	kfree(chip_data);
out:
	return err;
}

static int m02098_i2c_chip_attach_adapter(struct i2c_adapter *adapter)
{
    return i2c_probe(adapter, &addr_data, m02098_i2c_chip_detect);
}

static int m02098_i2c_chip_detach_client(struct i2c_client *client)
{
	struct chip_data *data = i2c_get_clientdata(client);
	int err;

	err = i2c_detach_client(client);
	if (err) {
		dev_err(&client->dev, "Client deregistration failed, client not detached.\n");
		return err;
	}

	kfree(client);
	kfree(data);
	return 0;
}


void M02098_release(void)
{

	if (M02098Major >= 0) {
		unregister_chrdev(M02098Major, "M02098_DEVICE_NAME");
		M02098Major = -1;
    }

}

static int __init m02098_i2c_chip_init(void)
{
	m02098_Init();

	return i2c_add_driver(&m02098_i2c_chip_driver);
}

static void __exit m02098_i2c_chip_exit(void)
{
	M02098_release();
	i2c_del_driver(&m02098_i2c_chip_driver);
}

module_init(m02098_i2c_chip_init);
module_exit(m02098_i2c_chip_exit);

MODULE_AUTHOR(DRV_AUTHOR);
MODULE_DESCRIPTION(DRV_DESC);
MODULE_LICENSE("GPL");
MODULE_VERSION(DRV_VER);

