/*
<:copyright-gpl 
 Copyright 2011 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/paccess.h>

#include <bl234x_cpu.h>
#include <bl234x_regs.h>
#include <bl234x_io.h>
#include <bl234x_irq.h>
#include <bl234x_pci.h>

#include "BL234x_ic.h"
#include "BL234x_pci.h"

#undef BL234X_PCI_DEBUG

#define VERSION     "1.1.0"
#define VER_STR     "v" VERSION " " __DATE__ " " __TIME__

#define BL234X_PCI_PROC_NAME	"bl234x_pci"

spinlock_t bl234x_pci_lock;

#ifdef BL234X_PCI_DEBUG
unsigned int debugflags = 0x7fffffff;
unsigned int debugmem = 0x7fffffff;
unsigned int debugmemv = 0x7fffffff;
#else
unsigned int debugflags = 0x0;
unsigned int debugmem = 0x0;
unsigned int debugmemv = 0x0;
#endif /* BL234X_PCI_DEBUG */

/* dma word -buffer - will used for reaad/write opperations */
u32 dma_word_buffer = 0;
u32 *dma_word_buffer_ptr = NULL;
u32 dma_word_buffer_addr = 0;

bl234x_pci_dev_stat_t bl234x_pci_stats[BL234X_PCI_DEV_NUM_MAX];

static void bl234x_pci_proc_create(void);

static struct proc_dir_entry *bl234x_pci_proc = NULL;

static char helpstring [] = {
    "\n\tBL234X PCI debug commands - use 'echo > blpci [command]'\n\t \
               - fullstat                  \t\tprint full error statistics \n\t \
               - debugc <on|off> <all|0x...\t\tdebug configuration space\n\t \
               - debugm <on|off|onv> <all|0x...\tdebug memory space\n\t \
               - getc <0x...> <0x...> <l>     <n>\tread  config : offset len   size\n\t \
               - setc <0x...> <0x...> <0x...> <n>\twrite config : offset value size\n\t \
               - <dml dmw dmb> <0x...> <n>\t\tread memory\n\t \
               - <mml mmw mmb> <0x...> <0x...>\t\twrite memory\n\t \
               - <fill> <0x...> <0x...> <n>\t\tfill bytes memory\n\t \
               - <readb readw readl>    <0x...> <n>\tread with/without DMA\n\t \
               - <writeb writew writel> <0x...> <0x..> write with/without DMA\n\n"
};

extern struct pci_ops bl234x_pci_ops;

static irqreturn_t bl234x_pci_interrupt(int irq, void *dev)
{
	unsigned long istatus,flags;
	PCI_CPU_ISTATUS_DTE mask;
	int rv = IRQ_HANDLED;

	spin_lock_irqsave(&bl234x_pci_lock, flags);
	
	istatus = fi_bl234x_pci_drv_get_int();

	mask = *(PCI_CPU_ISTATUS_DTE *)&istatus;

	if (mask.pci_ahb_window_post_err) {
		pi_bl234x_pci_drv_ack_int(CE_PCI_AHB_POST_ERROR);
		pr_err("%s: PCI->AHB window post error\n", __FUNCTION__);
	}

	if (mask.pci_ahb_window_fetch_err) {
		pi_bl234x_pci_drv_ack_int(CE_PCI_AHB_FETCH_ERROR);
		pr_err("%s: PCI->AHB window fetch error\n", __FUNCTION__);
	}

	if (mask.ahb_pci_window_post_err) {
		pi_bl234x_pci_drv_ack_int(CE_AHB_PCI_POST_ERROR);
		pr_err("%s: AHB->PCI window post error\n", __FUNCTION__);
	}

	if (mask.ahb_pci_window_fetch_err) {
		pi_bl234x_pci_drv_ack_int(CE_AHB_PCI_FETCH_ERROR);
		pr_err("%s: AHB->PCI window fetch error\n", __FUNCTION__);
	}
	
	fi_bl234x_drv_ic_isr_ack((stt_int32)irq - BL234X_INT_IRQ_BASE);

	spin_unlock_irqrestore(&bl234x_pci_lock, flags);
	return rv;
}

static struct resource bl234x_pci_mem_resource = {
	.name	= "BL234X PCI MEM",
	.start = IOMEM_NP_RESOURCE_START,
	.end   = IOMEM_NP_RESOURCE_END,
	.flags = IORESOURCE_MEM
};

static struct resource bl234x_pci_io_resource = {
	.name	= "BL234X PCI IO",
	.start = IOPORT_RESOURCE_START,
	.end   = IOPORT_RESOURCE_END,
	.flags = IORESOURCE_IO
};

struct pci_controller bl234x_pci_controller = {
	.pci_ops	    = &bl234x_pci_ops,
	.mem_resource	= &bl234x_pci_mem_resource,
	.io_resource	= &bl234x_pci_io_resource
};

static int __init bl234x_pci_setup(void)
{
	int ret;
	PCI_VERSIONS_DTE version;
	
	dma_word_buffer_addr = ((u32)&dma_word_buffer) | 0xA0000000; 
	dma_word_buffer_ptr  = (u32 *)dma_word_buffer_addr; /* set not cacheable */

    /* clear statistics */
    memset(bl234x_pci_stats, 0, sizeof(bl234x_pci_dev_stat_t));
	
	/* set I/O resource limits. - unlimited for now to accomodate HT */
	ioport_resource.end = 0xffffffffUL;
	iomem_resource.end  = 0xffffffffUL;
	
    /* create semaphore */
	spin_lock_init(&bl234x_pci_lock);

	ret = request_irq(bl234x_get_irq_number(IRQ_PCI), bl234x_pci_interrupt, IRQF_SAMPLE_RANDOM, "bl234x-pci", NULL);
	if (ret) {
		printk(KERN_ERR "bl234x-pci: failed to register irq\n");
		return ret;
	}
	
    /* register the controller to Linux data base */
	register_pci_controller(&bl234x_pci_controller);

	fi_bl234x_drv_ic_isr_ack(bl234x_get_irq_number(IRQ_PCI) - BL234X_INT_IRQ_BASE);
	fi_bl234x_drv_ic_enable(bl234x_get_irq_number(IRQ_PCI) - BL234X_INT_IRQ_BASE);
	
    /* create a char device for debugging purpose */
    bl234x_pci_proc_create();

	/* print controller ID */
	pi_bl234x_pci_drv_get_version(&version);
	pr_info("bl234x-pci: vendor = 0x%x device = 0x%x"
			" ver 0x%x type 0x%x rev 0x%x cls 0x%x\n", 
			(u32)version.vendor_id, (u32)version.device_id, (u32)version.version,
			(u32)version.type, (u32)version.revision, (u32)version.class_id);

    return 0;
}

arch_initcall(bl234x_pci_setup);

static int bl234x_pci_proc_read(char *buf , char **start, off_t offset,
                         int count, int *eof, void *data)
{
	printk("\nBL234X PCI driver version %s\n\n", VER_STR);
	
	printk("\ndebug configuration : config=0x%08x memory=0x%08x 0x%08x\n", 
													debugflags, debugmem, debugmemv);
	printk(helpstring);
	*eof = 1;
	return 0;
}

static char * bl234x_pci_str_trim(char * buffer)
{
	if(!buffer)
		return NULL;

	while(*buffer && (*buffer == ' ' || *buffer == '\t' ))
		buffer++;

	return(buffer);
}

static void bl_pci_show_stat(void)
{
    int i;
	struct pci_dev *dev = NULL;

	for_each_pci_dev(dev) {
		printk("  bus=%02x, devfn=0x%02x(dev=%d func=%d), vendor=0x%04x, device=0x%04x, irq_pin=%d, irq_line=%d\n",
				dev->bus->number,
				dev->devfn,
				PCI_SLOT(dev->devfn),
				PCI_FUNC(dev->devfn),
				dev->vendor,
				dev->device,
				dev->pin,
				dev->irq);
		
		printk("\tBase addresses (start : size)\n");

		for (i = 0; i < 6; i++) {
			resource_size_t start, end;
			pci_resource_to_user(dev, i, &dev->resource[i], &start, &end);
			printk( "  %08x : %08x",
					(unsigned int)(start |
					(dev->resource[i].flags & PCI_REGION_FLAG_MASK)),
					dev->resource[i].start < dev->resource[i].end ?
					(unsigned int)(end - start) + 1 : 0);
			if ( i == 3) {
				printk("\n");
			}
		}
		printk( "\n\n");
	}
}

static int bl234x_pci_proc_write(struct file *file, const char *user_buffer,
                         unsigned long count, void *data) 
{
	char name[100];
	char *nameptr = bl234x_pci_str_trim(name);
	unsigned int device;
	unsigned int n;
	unsigned int datar = 0;

	if (copy_from_user(name, user_buffer, count)) {
		printk("'copy_from_user' failed\n");
		return -EFAULT;
	}
	
	if (!strncmp(nameptr, "help", 4)) {
		printk(helpstring);
		return count;
	}

	if (!strncmp(nameptr, "fullstat", 8)) {
		bl_pci_show_stat();
		return count;
	}

	/* debugm */
	if (!strncmp(nameptr, "debugm", 6)) {
		nameptr = &name[7];
		nameptr = bl234x_pci_str_trim(nameptr);

		/* debugm onv */
		if ( !strncmp(nameptr, "onv", 3)) {
			nameptr += 4;
			nameptr = bl234x_pci_str_trim(nameptr);
			if (!strncmp(nameptr, "all", 3)) {
				debugmem = 0x7FFFFFFF;
				debugmemv = debugmem;
				return count;
			}
			
			if (sscanf(nameptr, "%x", &device) == 1) {
				debugmem |= 1 << PCI_SLOT(device) ;
				debugmemv = debugmem;
				return count;
			}
			return -EFAULT;
		}

		/* debugm off */
		if (!strncmp(nameptr, "off", 3)) {
			nameptr += 4;
			nameptr = bl234x_pci_str_trim(nameptr);
			if (!strncmp(nameptr, "all", 3)) {
				debugmem = 0;
				return count;
			}
			if (sscanf(nameptr, "%x", &device) == 1) {
				debugmem &= ~(1 << PCI_SLOT(device)) ;
				return count;
			}
			return -EFAULT;
		}

		/* debugm on */
		if (!strncmp(nameptr, "on", 2)) {
			nameptr += 3;
			nameptr = bl234x_pci_str_trim(nameptr);
			if ( !strncmp(nameptr, "all", 3)) {
				debugmem = 0x7FFFFFFF;
				return count;
			}
			
			if (sscanf(nameptr, "%x", &device) == 1) {
				debugmem |= 1 << PCI_SLOT(device) ;
				return count;
			}
			return -EFAULT;
		}
	
		return -EFAULT;
	}

	/* debugc */
	if (!strncmp(nameptr, "debugc", 6)) {
		nameptr = &name[7];
		nameptr = bl234x_pci_str_trim(nameptr);
		/* debugc on */
		if (!strncmp(nameptr, "on", 2)) {
			nameptr += 3;
			nameptr = bl234x_pci_str_trim(nameptr);
			if ( !strncmp(nameptr,"all",strlen("all"))) {
				debugflags = 0x7FFFFFFF;
				return count;
			}
			if (sscanf(nameptr, "%x", &device) == 1) {
				debugflags |= 1 << PCI_SLOT(device) ;
				return count;
			}
			return -EFAULT;
		}

		/* debugc off */
		if (!strncmp(nameptr,"off",strlen("off"))) {
			nameptr += 4;
			nameptr = bl234x_pci_str_trim(nameptr);
			if (!strncmp(nameptr, "all", 3)) {
				debugflags = 0;
				return count;
			}
			if (sscanf(nameptr, "%x", &device) == 1) {
				debugflags &= ~(1 << PCI_SLOT(device)) ;
				return count;
			}
			return -EFAULT;
		}
		return -EFAULT;
	}

	/* dml */
	if (!strncmp(nameptr, "dml", strlen("dml"))) {
		nameptr = &name[4];
		nameptr = bl234x_pci_str_trim(nameptr);
		printk("nameptr %s\n", nameptr);
		sscanf(nameptr, "%x %d", &device, &n);
		{
			int i;
			for (i = 0; i < n; i++)
			{
				datar = readl((void __iomem *)device);
				printk("0x%08x => 0x%08x\n", device, datar);
				/* implement PCI Bridge registers work around */
				if ((device >= 0xaff79400) && (device <= 0xaff7941c)) {
					*(volatile unsigned int *)device = datar;
				}
				device += 4;
			}
			return count;
		}
	}

	/* dmw */
	if (!strncmp(nameptr, "dmw", 3)) {
		nameptr = &name[4];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %d", &device, &n);
		{
			int i;
			for (i = 0; i < n; i++) {
				datar = readw((void __iomem *)device);
				printk("0x%08x => 0x%04x\n", device, datar);
				device += 2;
			}
			return count;
		}
	}

	/* dmb */
	if (!strncmp(nameptr, "dmb", 3)) {
		nameptr = &name[4];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %d", &device, &n);
		{
			int i;
			for (i = 0; i < n; i++) {
				datar = readb((void __iomem *)device);
				printk("0x%08x => 0x%02x\n", device, datar);
				device += 1;
			}
			return count;
		}
	}

	/* fill */
	if ( !strncmp(nameptr, "fill", 4)) {
		nameptr = &name[5];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %x %d", &device, &datar, &n);
		memset((unsigned char *)device, (unsigned char)datar, n);
		printk("0x%08x <= 0x%08x for %d bytes\n", device, datar, n);
		return count;
	}

	/* mml */
	if ( !strncmp(nameptr, "mml", 3)) {
		nameptr = &name[4];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %x", &device, &datar);
		writel(datar, (void __iomem *)device);
		printk("0x%08x <= 0x%08x\n", device, datar);
		return count;
	}

	/* mmw */
	if (!strncmp(nameptr, "mmw", 3)) {
		nameptr = &name[4];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %x", &device, &datar);
		writew(datar, (void __iomem *)device);
		printk("0x%08x <= 0x%04x\n", device, datar);
		return count;
	}

	/* mmb */
	if (!strncmp(nameptr, "mmb", 3)) {
		nameptr = &name[4];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %x", &device, &datar);
		writeb(datar, (void __iomem *)device);
		printk("0x%08x <= 0x%02x\n", device, datar);
		return count;
	}

	/* readl */
	if (!strncmp(nameptr, "readl", 5)) {
		nameptr = &name[6];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %d", &device, &n);
		{
			int i;
			for (i = 0; i < n; i++) {
				datar = ioread32((void __iomem *)device);
				printk("0x%08x => 0x%08x\n", device, datar);
				device += 4;
			}
			return count;
		}
	}

	/* readw */
	if (!strncmp(nameptr, "readw", 5)) {
		nameptr = &name[6];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %d", &device, &n);
		{
			int i;
			for (i = 0; i < n; i++) {
				datar = ioread16((void __iomem *)device);
				printk("0x%08x => 0x%04x\n", device, datar);
				device += 2;
			}
			return count;
		}
	}

	/* readb */
	if (!strncmp(nameptr, "readb", 5)) {
		nameptr = &name[6];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %d", &device, &n);
		{
			int i;
			for (i = 0; i < n; i++) {
				datar = ioread8((void __iomem *)device);
				printk("0x%08x => 0x%02x\n", device, datar);
				device += 1;
			}
			return count;
		}
	}

	/* writel */
	if (!strncmp(nameptr, "writel", 6)) {
		nameptr = &name[7];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %x", &device, &datar);
		iowrite32(datar,(void __iomem *)device);
		printk("0x%08x => 0x%08x\n", device, datar);
		return count;
	}

	/* writew */
	if (!strncmp(nameptr, "writew", 6)) {
		nameptr = &name[7];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %x", &device, &datar);
		iowrite16(datar,(void __iomem *)device);
		printk("0x%08x <= 0x%04x\n", device, datar);
		return count;
	}

	/* writeb */
	if (!strncmp(nameptr, "writeb", 6)) {
		nameptr = &name[7];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %x", &device, &datar);
		iowrite8(datar,(void __iomem *)device);
		printk("0x%08x <= 0x%02x\n", device, datar);
		return count;
	}

	/* setc */
	if (!strncmp(nameptr, "setc", 4)) {
		struct pci_dev *dev = NULL;
		int l;
		int found = 0;

		nameptr = &name[5];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %x %x %d", &device, &n, &datar, &l);
		if ((l != 1) && (l != 2) && (l != 4)) {
			printk("ERROR size must be 1, or 2 or 4\n");
			return count;
		}

		for_each_pci_dev(dev) {
			if (dev->bus->number == 0 && 
				dev->devfn == device) {
				found = 1;
				break;
			}
		}
		
		if (found == 0) {
			printk("Device 0x%08x not found !!!\n\r", device);
			return count;
		}
		
		switch (l) {
			case 1:
				pci_write_config_byte(dev,n, datar);
				break;
				
			case 2:
				pci_write_config_word(dev,n, datar);
				break;
				
			case 4:
				pci_write_config_dword(dev,n, datar);            
				break;
				
			default:
				break;
		}
		return count;
	}

	/* getc */
	if (!strncmp(nameptr, "getc", 4)) {
		int i, l;
		u32 offset;
		struct pci_dev *dev = NULL;
		int found = 0;

		nameptr = &name[5];
		nameptr = bl234x_pci_str_trim(nameptr);
		sscanf(nameptr,"%x %x %d %d", &device, & offset, &n, &l);
		if ((l != 1) && (l != 2) && (l != 4)) {
			printk("Error: size must 1 or 2 or 4\n");
			return count;
		}

		for_each_pci_dev(dev) {
			if (dev->bus->number == 0 && 
				dev->devfn == device) {
				found = 1;
				break;
			}
		}
		
		if (found == 0) {
			printk("Device 0x%08x not found !!!\n\r", device);
			return count;
		}
		
		for (i = 0; i < n; i++) {
			unsigned char d[4];
			unsigned int adr;

			switch (l) {
				case 1:
					{
					u8 data;
					adr = offset + i;
					pci_read_config_byte(dev,adr, &data);
					printk("0x%08x => 0x%02x\n\r", adr, data);
					}
					break;
					
				case 2:
					{
						u16 data ;
						adr = offset + i*2;
						pci_read_config_word(dev,adr, &data);
						memcpy (d, &data, 2);
						printk("0x%08x => 0x%04x (%02x %02x)\n\r", adr, data, d[0], d[1]);
					}
					break;
					
				case 4:
					{
						u32 data;
						adr = offset + i*4;
						pci_read_config_dword(dev,adr, &data);
						memcpy (d, &data, 4);
						printk("0x%08x => 0x%08x (%02x %02x %02x %02x)\n\r", 
														adr, data, d[0], d[1], d[2], d[3]);
					}
					break;
					
				default:
					break;
			}
		}
		return count;
	}
	
	return -EFAULT;
}

static void bl234x_pci_proc_create(void)
{
	bl234x_pci_proc = create_proc_entry(BL234X_PCI_PROC_NAME, 0666, 0);

	if (!bl234x_pci_proc) {
		pr_err("%s: unable to create proc entry(%s)\n", __FUNCTION__, BL234X_PCI_PROC_NAME);
	}
	
	bl234x_pci_proc->read_proc = bl234x_pci_proc_read;
	bl234x_pci_proc->write_proc = bl234x_pci_proc_write;
}

