/*
<:copyright-gpl 
 Copyright 2011 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/paccess.h>

#include <bl234x_cpu.h>
#include <bl234x_regs.h>
#include <bl234x_io.h>
#include <bl234x_irq.h>
#include <bl234x_pci.h>

#include <BL234x_ic.h>
#include <BL234x_pci.h>

#undef BL234X_PCI_DEBUG

/* Type 0 */
#define PCI_CFG_TYPE0_REG_SHF           0
#define PCI_CFG_TYPE0_FUNC_SHF          8

/* Type 1 */
#define PCI_CFG_TYPE1_REG_SHF           0
#define PCI_CFG_TYPE1_FUNC_SHF          8
#define PCI_CFG_TYPE1_DEV_SHF           11
#define PCI_CFG_TYPE1_BUS_SHF           16

extern spinlock_t bl234x_pci_lock;
extern u32 debugflags;

extern u32 dma_word_buffer;
extern u32 *dma_word_buffer_ptr;
extern u32 dma_word_buffer_addr;

extern bl234x_pci_dev_stat_t bl234x_pci_stats[];

#ifdef BL234X_PCI_DEBUG
#define BL234X_PCI_DEBUG_PRINT(mask, fmt, args...) \
	do { \
		if (debugflags & mask) \
			printk("%s: " fmt, __func__, ##args); \
	} while (0);
#else
#define BL234X_PCI_DEBUG_PRINT(mask, fmt, args...)
#endif /* BL234X_PCI_DEBUG */

static char *bl234x_pci_dma_err_string[] = {
	"DMA busy    ",    
	"DMA timeout ", 
	"DMA AHB bus ", 
	"DMA aborted ", 
	"DMA parity  ",
	"Last DMA error",
	"NULL address "
};

/* Returns the actual PCI bus address from slot address and required offset */
static inline unsigned int 
bl234x_pci_adress_compose(struct pci_bus *bus, unsigned int devfn, int where)
{
	unsigned int  pci_addr;

	/* we support only one bus (bus #0) */
	if(bus->number > BL234X_PCI_BUS_ROOT)
		return -1;
	
    /* Max supported number od devises - 20 */
    if (PCI_SLOT(devfn) >= BL234X_PCI_DEV_NUM_MAX)
		return -1;
	
	pci_addr = 1 << (PCI_SLOT(devfn) + PCI_CFG_TYPE1_DEV_SHF);
	
	/* Device number */
	pci_addr |= PCI_FUNC(devfn) << PCI_CFG_TYPE1_FUNC_SHF;

	/* Register number (same for Type 0/1)  - clear the 2 lsb bits */
	pci_addr |= (where & ~0x3) << PCI_CFG_TYPE0_REG_SHF;

	return pci_addr;
}

/*
    Read PCI card configuration space; BL PCI controller reads only words(32 bits aligned)
    If the size is byte or short, we read the whole word and do the needed
    manipulation to return the required size
*/
static int 
bl234x_pci_read_config(struct pci_bus* bus, unsigned int devfn, int where, int size, u32 *val)
{
	int rv = PCIBIOS_SUCCESSFUL;
	unsigned long flags;
    u32 pciaddress;
	u32 pcislot = PCI_SLOT(devfn);
    u32 mask = 1 << PCI_SLOT(devfn);
	int tmp_where;
    BL_PCI_RETURN_CODE error = BL_PCI_OK;
	
	if(bus->number > BL234X_PCI_BUS_ROOT)
		return PCIBIOS_DEVICE_NOT_FOUND;
	
	spin_lock_irqsave(&bl234x_pci_lock, flags);
	
	/* calculate the pci bus address for the required slot and register */
	pciaddress = bl234x_pci_adress_compose(bus, devfn, where) ;
	if (pciaddress == 0xFFFFFFFF) {
		rv = PCIBIOS_DEVICE_NOT_FOUND;
		goto read_config_out;
	}

	/* aligned to 4 */
	pciaddress &= ~0x3;

	pi_bl234x_pci_drv_set_direct_address(dma_word_buffer_addr);

	/* update internal statistics */
	bl234x_pci_stats[pcislot].devfn = devfn;
	bl234x_pci_stats[pcislot].address = pciaddress;

	error = fi_bl234x_pci_drv_read_word_spatial(dma_word_buffer_addr, 
											pciaddress, DMA_TRANSACTION_TYPE_CONFIGURATION);
	if (error != BL_PCI_OK) {
		/* error - increment the counter and print */
		bl234x_pci_stats[pcislot].readerrors[error] ++;
		BL234X_PCI_DEBUG_PRINT(mask, "slot %d read for 0x%x failed(%s)\n", 
												pcislot, pciaddress, bl234x_pci_dma_err_string[error]);
		rv = PCIBIOS_DEVICE_NOT_FOUND;
		goto read_config_out;
	}

	tmp_where = where & 3;
	switch (size) {
		case 1:
			*val = (*dma_word_buffer_ptr >> (tmp_where * 8)) & 0xff;
			break;

		case 2:
			if (tmp_where == 1 || tmp_where == 3)
				*val = (*dma_word_buffer_ptr >> (tmp_where * 8)) & 0xff;
			else
				*val = (*dma_word_buffer_ptr >> (tmp_where * 8)) & 0xffff;
			break;

		case 4:
			if (tmp_where == 1 || tmp_where == 3)
				*val = (*dma_word_buffer_ptr >> (tmp_where * 8)) & 0xff;
			else if (tmp_where == 0)
				*val = *dma_word_buffer_ptr;
			break;

		default:
			break;
	}

	BL234X_PCI_DEBUG_PRINT(mask, "bus=%d devfn=0x%x (dev=%d fn=%d) where=0x%x " 
									"size=%d pciaddr=0x%08x val=0x%08x(0x%08x)\n", 
									bus->number, devfn, pcislot, PCI_FUNC(devfn), where, 
									size, pciaddress, (u32)*val, (u32)*dma_word_buffer_ptr);

read_config_out:		
	spin_unlock_irqrestore(&bl234x_pci_lock, flags);
	return rv;
}

/*
    Write PCI card configuration space; BL PCI controller writes only words(32 bits aligned)
    If the size is byte or short, we read the whole word and do the needed
    manipulation to return the required size
*/
static int 
bl234x_pci_write_config(struct pci_bus *bus, unsigned int devfn, int where, int size, u32 val)
{
	unsigned long flags;
	u32 tempread, tmp_where;
	u32 pciaddress;
	u32 pcislot = PCI_SLOT(devfn);
	u32 mask = 1 << PCI_SLOT(devfn);
	BL_PCI_RETURN_CODE error = BL_PCI_OK;
	int rv;

	if(bus->number > BL234X_PCI_BUS_ROOT)
		return PCIBIOS_DEVICE_NOT_FOUND;

	spin_lock_irqsave(&bl234x_pci_lock, flags);

	/* calculate the pci bus address for the required slot and register */
	pciaddress = bl234x_pci_adress_compose(bus, devfn, where);
	if (pciaddress == 0xFFFFFFFF) {
		rv = PCIBIOS_DEVICE_NOT_FOUND;
		goto write_config_out;
	}

	/* aligned to 4 */
	pciaddress &= ~0x3;

	pi_bl234x_pci_drv_set_direct_address(dma_word_buffer_addr);
	error = fi_bl234x_pci_drv_read_word_spatial(dma_word_buffer_addr, 
													pciaddress, DMA_TRANSACTION_TYPE_CONFIGURATION);
	if (error != BL_PCI_OK) {
		/* error - increment the counter and print */
		bl234x_pci_stats[pcislot].readerrors[error]++;
		BL234X_PCI_DEBUG_PRINT(mask, "slot %d read for 0x%x failed(%s)\n", 
												pcislot, pciaddress, bl234x_pci_dma_err_string[error]);
		rv = PCIBIOS_DEVICE_NOT_FOUND;
		goto write_config_out;
	}
	
	tempread = *dma_word_buffer_ptr;
	if ((where >= 0x10) && (where <= 0x24) && (size == 4)) {
		val &= 0x0fffffff;
	}

	tmp_where = where & 0x3;
	switch (size) {
		case 1:
			val &= 0xff;
			tempread &= ~(0xff << (tmp_where * 8));
			val <<= (tmp_where * 8);
			tempread |= val;
			break;

		case 2:
			val &= 0xffff;
			if (!(tmp_where % 2)) {
				tempread &= ~(0xffff << (tmp_where * 8));
				val <<= (tmp_where * 8);
			}
			tempread |= val;
			break;

		case 3:
			val &= 0xffffff;
			tempread &= 0xff000000;
			tempread |= val;
			break;

		default:
			tempread = val;
			break;
	}
	
	*dma_word_buffer_ptr = tempread;

	/* update internal statistics */
	bl234x_pci_stats[pcislot].devfn = devfn;
	bl234x_pci_stats[pcislot].address = pciaddress;

	/* write back the word */
	error = fi_bl234x_pci_drv_write_word_spatial(dma_word_buffer_addr, 
												pciaddress, DMA_TRANSACTION_TYPE_CONFIGURATION);

	if (error != BL_PCI_OK) {
		/* error - increment the counter and print */
		bl234x_pci_stats[PCI_SLOT(devfn)].writeerrors[error]++;
		BL234X_PCI_DEBUG_PRINT(mask, "slot %d write for 0x%x failed(%s)\n", 
												pcislot, pciaddress, bl234x_pci_dma_err_string[error]);
		rv = PCIBIOS_DEVICE_NOT_FOUND;
		goto write_config_out;
	}

	BL234X_PCI_DEBUG_PRINT(mask, "bus=%d devfn=0x%x (dev=%d fn=%d) where=0x%x " 
									"size=%d pciaddr=0x%08x val=0x%08x(0x%08x)\n", 
									bus->number, devfn, pcislot, PCI_FUNC(devfn), where, 
									size, pciaddress, val, tempread);
	
write_config_out:		
	spin_unlock_irqrestore(&bl234x_pci_lock, flags);
	return rv;
}

struct pci_ops bl234x_pci_ops = {
    .read   = bl234x_pci_read_config,
    .write  = bl234x_pci_write_config,
};

