/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/clk.h>
#include <linux/platform_device.h>
#include <bcm63268_cpu.h>
#include <bcm63268_regs.h>
#include <bcm63268_io.h>
#include <bcm63268_pcie.h>
#include <bcm63268_board.h>

#ifdef DEBUG
#define BCM63268_PCI_OPS_DBG(fmt, args...)	pr_printk(fmt, ## args)
#else
#define BCM63268_PCI_OPS_DBG(fmt, args...)
#endif

static u32 bcm63268_woc_cfg_space[1][64] = {
	{BCM63268_PCI_WOC_ID, 0x00100006, 0x02800000, 0x00000010,
	0x10004000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0xdeadbeef,
	0x00000000, 0x00000040, 0x00000000, 0x0000010f,
	0xce035801, 0x00004008, 0x0080d005, 0x00000000,
	0x00000000, 0x00000000, 0x00784809, 0x00000010,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x18001000, 0x00000000, 0xffffffff, 0x00000003,
	0x00000000, 0x00000100, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00010000, 0x18101000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00010010, 0x00288fa0, 0x00190100, 0x00176c11,
	0x30110040, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000},
};

static bool bcm63268_pci_usb11h_mem_size_rd = false;
static u32 bcm63268_pci_usb11h_mem_base = 0;
static u32 bcm63268_pci_usb11h_cfg_space_cmd_reg = 0;

static bool bcm63268_pci_usb20h_mem_size_rd = false;
static u32 bcm63268_pci_usb20h_mem_base = 0;
static u32 bcm63268_pci_usb20h_cfg_space_cmd_reg = 0;

static int bcm63268_pci_woc_write(unsigned int devfn, int where, u32 *val, int size)
{
    u32 data;
        
	if(where >= 256)
		return PCIBIOS_BAD_REGISTER_NUMBER;
	
	data = bcm63268_woc_cfg_space[PCI_SLOT(devfn) - BCM63268_PCI_WOC_SLOT][where / 4];

	switch(size) {
		case 1:
			data = (data & ~(0xff << ((where & 3) << 3))) | (*val << ((where & 3) << 3));
			break;
			
		case 2:
			data = (data & ~(0xffff << ((where & 3) << 3))) | (*val << ((where & 3) << 3));
			break;
			
		case 4:
			data = *val;
			break;
		default:
			break;
	}
	
	bcm63268_woc_cfg_space[PCI_SLOT(devfn) - BCM63268_PCI_WOC_SLOT][where / 4] = data;

    return PCIBIOS_SUCCESSFUL;
}

static int bcm63268_pci_woc_read(unsigned int devfn, int where, u32 *val, int size)
{
    u32 data;

	switch (where) {
		case PCI_INTERRUPT_LINE:
			data = bcm63268_get_irq_number(IRQ_WLAN);
			break;

		case PCI_SUBSYSTEM_VENDOR_ID:
			if (size == 4)
				data = (bcm63268_board_get_woc_subsysid() << 16) | PCI_VENDOR_ID_BROADCOM;
			else
				data = PCI_VENDOR_ID_BROADCOM;
			break;
			
		case PCI_SUBSYSTEM_ID:
			data = bcm63268_board_get_woc_subsysid();
			break;

		default:
			if(where >= 256)
	    		data = 0xffffffff;
	    	else	
	    		data = bcm63268_woc_cfg_space[PCI_SLOT(devfn) - BCM63268_PCI_WOC_SLOT][where / 4];
			break;
	}

	switch(size) {
		case 1:
			*val = (data >> ((where & 3) << 3)) & 0xff;
			break;
		case 2:
			*val = (data >> ((where & 3) << 3)) & 0xffff;
			break;
		case 4:
			*val = data;
			/* Special case for reading PCI device range */
			if ((where >= PCI_BASE_ADDRESS_0) && (where <= PCI_BASE_ADDRESS_5)) {
				if (data == 0xffffffff) {
					if (where == PCI_BASE_ADDRESS_0)
						*val = 0xFFFF0000;//PCI_SIZE_64K;
					else
						*val = 0;
				}
			}
			break;
		default:
			break;
	}
	
    return PCIBIOS_SUCCESSFUL;
}

static int bcm63268_pci_usb11h_write(unsigned int devfn, int where, u32 * value, int size)
{
    switch (size) {
        case 1:
            BCM63268_PCI_OPS_DBG("W => Slot: %d Where: %2X Len: %d Data: %02X\n",
                PCI_SLOT(devfn), where, size, *value);
            break;
			
        case 2:
            BCM63268_PCI_OPS_DBG("W => Slot: %d Where: %2X Len: %d Data: %04X\n",
                PCI_SLOT(devfn), where, size, *value);
			switch (where) {
				case PCI_COMMAND:
					bcm63268_pci_usb11h_cfg_space_cmd_reg = *value;
					break;
				default:
					break;
			}
            break;
			
        case 4:
            BCM63268_PCI_OPS_DBG("W => Slot: %d Where: %2X Len: %d Data: %08lX\n",
                PCI_SLOT(devfn), where, size, *value);
			switch (where) {
				case PCI_BASE_ADDRESS_0:
					if (*value == 0xffffffff) {
						bcm63268_pci_usb11h_mem_size_rd = true;
					} else {
						bcm63268_pci_usb11h_mem_base = *value;
					}
					break;
				default:
					break;
			}
			break;
		default:
			break;
	}

	return PCIBIOS_SUCCESSFUL;
}

static int bcm63268_pci_usb11h_read(unsigned int devfn, int where, u32 * value, int size)
{
    u32 rv = 0xFFFFFFFF;

	switch (where) {
		case PCI_VENDOR_ID:
		case PCI_DEVICE_ID:
			rv = PCI_VENDOR_ID_BROADCOM | 0x63000000;
			break;

		case PCI_COMMAND:
		case PCI_STATUS:
			rv = (0x0006 << 16) | bcm63268_pci_usb11h_cfg_space_cmd_reg;
			break;

		case PCI_CLASS_REVISION:
		case PCI_CLASS_DEVICE:
			rv = (PCI_CLASS_SERIAL_USB << 16) | (0x10 << 8) | 0x01;
			break;

		case PCI_BASE_ADDRESS_0:
			if (bcm63268_pci_usb11h_mem_size_rd) {
				rv = RSET_USB_OHCI_SIZE;
			} else {
				if (bcm63268_pci_usb11h_mem_base != 0)
					rv = bcm63268_pci_usb11h_mem_base;
				else
					rv = bcm63268_regset_address(RSET_USB_OHCI);
			}
			bcm63268_pci_usb11h_mem_size_rd = false;
		break;

	case PCI_CACHE_LINE_SIZE:
		rv = L1_CACHE_BYTES/4;
		break;

	case PCI_LATENCY_TIMER:
		rv = 0;
		break;

	case PCI_HEADER_TYPE:
		rv = PCI_HEADER_TYPE_NORMAL;
		break;

	case PCI_SUBSYSTEM_VENDOR_ID:
		rv = PCI_VENDOR_ID_BROADCOM;
		break;

	case PCI_SUBSYSTEM_ID:
		rv = 0x6300;
		break;

	case PCI_INTERRUPT_LINE:
		rv = bcm63268_get_irq_number(IRQ_USB_OHCI);
		break;

	default:
		break;
	}

	switch (size) {
		case 1:
			*value = (rv >> ((where & 3) << 3)) & 0xff;
			BCM63268_PCI_OPS_DBG("R <= Slot: %d Where: %2X Len: %d Data: %02X\n",
													PCI_SLOT(devfn), where, size, *value);
			break;
			
		case 2:
			*value = (rv >> ((where & 3) << 3)) & 0xffff;
			BCM63268_PCI_OPS_DBG("R <= Slot: %d Where: %2X Len: %d Data: %04X\n",
													PCI_SLOT(devfn), where, size, *value);
			break;
			
		case 4:
			*value = rv;
			BCM63268_PCI_OPS_DBG("R <= Slot: %d Where: %2X Len: %d Data: %08lX\n",
													PCI_SLOT(devfn), where, size, *value);
			break;
			
		default:
			break;
	}
	
	return PCIBIOS_SUCCESSFUL;
}

static int bcm63268_pci_usb20h_write(unsigned int devfn, int where, u32 *value, int size)
{
    switch (size) {
		case 1:
			BCM63268_PCI_OPS_DBG("W => Slot: %d Where: %2X Len: %d Data: %02X\n",
														PCI_SLOT(devfn), where, size, *value);
			break;
		case 2:
			BCM63268_PCI_OPS_DBG("W => Slot: %d Where: %2X Len: %d Data: %04X\n",
														PCI_SLOT(devfn), where, size, *value);
			switch (where) {
				case PCI_COMMAND:
					bcm63268_pci_usb20h_cfg_space_cmd_reg = *value;
					break;
				default:
					break;
			}
			break;
		
        case 4:
			BCM63268_PCI_OPS_DBG("W => Slot: %d Where: %2X Len: %d Data: %08lX\n",
												PCI_SLOT(devfn), where, size, *value);
			switch (where) {
				case PCI_BASE_ADDRESS_0:
					if (*value == 0xffffffff) {
						bcm63268_pci_usb20h_mem_size_rd = true;
					} else {
						bcm63268_pci_usb20h_mem_base = *value;
					}
					break;
				default:
					break;
			}
			break;
			
		default:
			break;
    }

	return PCIBIOS_SUCCESSFUL;
}

static int bcm63268_pci_usb20h_read(unsigned int devfn, int where, u32 *value, int size)
{
    u32 rv = 0xFFFFFFFF;
	
	switch (where) {
		case PCI_VENDOR_ID:
		case PCI_DEVICE_ID:
			rv = PCI_VENDOR_ID_BROADCOM | 0x63000000;
			break;
		
		case PCI_COMMAND:
		case PCI_STATUS:
			rv = (0x0006 << 16) | bcm63268_pci_usb20h_cfg_space_cmd_reg;
			break;
			
		case PCI_CLASS_REVISION:
		case PCI_CLASS_DEVICE:
			rv = (PCI_CLASS_SERIAL_USB << 16) | (0x20 << 8) | 0x01;
			break;
			
		case PCI_BASE_ADDRESS_0:
			if (bcm63268_pci_usb20h_mem_size_rd) {
				rv = RSET_USB_EHCI_SIZE;
			} else {
				if (bcm63268_pci_usb20h_mem_base != 0)
					rv = bcm63268_pci_usb20h_mem_base;
				else
					rv = bcm63268_regset_address(RSET_USB_EHCI);
			}
			bcm63268_pci_usb20h_mem_size_rd = false;
			break;
		case PCI_CACHE_LINE_SIZE:
			rv = L1_CACHE_BYTES/4;
			break;
			
		case PCI_LATENCY_TIMER:
			rv = 0;
			break;
		
		case PCI_HEADER_TYPE:
			rv = PCI_HEADER_TYPE_NORMAL;
			break;
			
		case PCI_SUBSYSTEM_VENDOR_ID:
			rv = PCI_VENDOR_ID_BROADCOM;
			break;
			
		case PCI_SUBSYSTEM_ID:
			rv = 0x6300;
			break;
			
		case PCI_INTERRUPT_LINE:
			rv = bcm63268_get_irq_number(IRQ_USB_EHCI);
			break;
			
		default:
			break;
	}

    switch (size) {
		case 1:
			*value = (rv >> ((where & 3) << 3)) & 0xff;
			BCM63268_PCI_OPS_DBG("R <= Slot: %d Where: %2X Len: %d Data: %02X\n",
														PCI_SLOT(devfn), where, size, *value);
			break;
			
		case 2:
			*value = (rv >> ((where & 3) << 3)) & 0xffff;
			BCM63268_PCI_OPS_DBG("R <= Slot: %d Where: %2X Len: %d Data: %04X\n",
														PCI_SLOT(devfn), where, size, *value);
			break;
			
		case 4:
			*value = rv;
			BCM63268_PCI_OPS_DBG("R <= Slot: %d Where: %2X Len: %d Data: %08lX\n",
														PCI_SLOT(devfn), where, size, *value);
			break;
			
		default:
			break;
    }

    return PCIBIOS_SUCCESSFUL;
}

int bcm63268_pcibios_read(struct pci_bus *bus, unsigned int devfn,
    int where, int size, u32 *val)
{
	if (PCI_SLOT(devfn) == BCM63268_PCI_WOC_SLOT)
		return bcm63268_pci_woc_read(devfn, where, val, size);

#if defined(CONFIG_USB) || defined(CONFIG_USB_MODULE)
	if (PCI_SLOT(devfn) == BCM63268_PCI_USB11H_SLOT)
		return bcm63268_pci_usb11h_read(devfn, where, val, size);
	if (PCI_SLOT(devfn) == BCM63268_PCI_USB20H_SLOT)
		return bcm63268_pci_usb20h_read(devfn, where, val, size);
#endif /* CONFIG_USB || CONFIG_USB_MODULE */

    return PCIBIOS_SUCCESSFUL;
}

int bcm63268_pcibios_write(struct pci_bus *bus, unsigned int devfn,
    int where, int size, u32 val)
{
	if (PCI_SLOT(devfn) == BCM63268_PCI_WOC_SLOT)
		return bcm63268_pci_woc_write(devfn, where, &val, size);

#if defined(CONFIG_USB) || defined(CONFIG_USB_MODULE)
	if (PCI_SLOT(devfn) == BCM63268_PCI_USB11H_SLOT)
		return bcm63268_pci_usb11h_write(devfn, where, &val, size);
	if (PCI_SLOT(devfn) == BCM63268_PCI_USB20H_SLOT)
		return bcm63268_pci_usb20h_write(devfn, where, &val, size);
#endif /* CONFIG_USB || CONFIG_USB_MODULE */

    return PCIBIOS_SUCCESSFUL;
}

struct pci_ops bcm63268_pci_ops = {
    .read   = bcm63268_pcibios_read,
    .write  = bcm63268_pcibios_write
};

static inline u32 __bcm63268_pcie_build_config_addr(int bus, int dev, int fn, int reg)
{
	if (bus == BCM63268_PCIE_BUS_ROOT) {
		return (reg & ~3);
	}

	return ((BCM63268_PCIEH_DEV_OFFSET | reg) & ~3);
}

static int bcm63268_pcie_can_access(int bus, int dev, int fn)
{
	u32 reg32;

	if (bus == BCM63268_PCIE_BUS_ROOT) {
		/* complex topology with a PCIe switch connected */
		return 1;
	}
	
	/* direct-connect */
	reg32 = bcm_pcie_readl(PCIE_BLOCK_1000_DL_STATUS_REG);
	return (reg32 & PCIE_BLOCK_1000_DL_STATUS_PHYLINKUP) ? 1 : 0;
}

static int bcm63268_pcie_read_config(struct pci_bus *bus, unsigned int devfn, 
														int reg, int size, u32 *val)
{
	u32 cfg_addr = 0;
    u32 data = 0;
	int bus_number = bus->number;

	if ((size == 2) && (reg & 1)) {
		return PCIBIOS_BAD_REGISTER_NUMBER;
	} else if ((size == 4) && (reg & 3)) {
		return PCIBIOS_BAD_REGISTER_NUMBER;
	}

	/*	 
	 * PCIe only has a single device connected to BCM63268 bus 1 or bus 2.
	 * It is always device ID 0. Don't bother doing reads for other
	 * device IDs on the first segment.
	 */	
	if (PCI_SLOT(devfn) != 0)
		return PCIBIOS_FUNC_NOT_SUPPORTED;

	if (!bcm63268_pcie_can_access(bus_number, PCI_SLOT(devfn), PCI_FUNC(devfn))) {
		return PCIBIOS_DEVICE_NOT_FOUND;
	}

	cfg_addr = __bcm63268_pcie_build_config_addr(bus_number, PCI_SLOT(devfn), PCI_FUNC(devfn), reg);

	data = bcm_pcie_readl(cfg_addr);

	switch (size) {
		case 1:
			*val = (data >> ((reg & 3) << 3)) & 0xff;
			break;

		case 2:
			*val = (data >> ((reg & 3) << 3)) & 0xffff;
			break;

		case 4:
			*val = data;
			break;

		default:
			return PCIBIOS_FUNC_NOT_SUPPORTED;
	}

    return PCIBIOS_SUCCESSFUL;
}

static int bcm63268_pcie_write_config(struct pci_bus *bus, unsigned int devfn, 
															int reg, int size, u32 val)
{
	u32 cfg_addr = 0;
    u32 data = 0;
	int bus_number = bus->number;

	if ((size == 2) && (reg & 1)) {
		return PCIBIOS_BAD_REGISTER_NUMBER;
	} else if ((size == 4) && (reg & 3)) {
		return PCIBIOS_BAD_REGISTER_NUMBER;
	}

	if (PCI_SLOT(devfn) != 0)
		return PCIBIOS_FUNC_NOT_SUPPORTED;

    if (!bcm63268_pcie_can_access(bus_number, PCI_SLOT(devfn), PCI_FUNC(devfn)))
        return PCIBIOS_DEVICE_NOT_FOUND;

	cfg_addr = __bcm63268_pcie_build_config_addr(bus_number, PCI_SLOT(devfn), PCI_FUNC(devfn), reg);
    data = bcm_pcie_readl(cfg_addr);

	switch (size) {
		case 1:
			data = (data & ~(0xff << ((reg & 3) << 3))) | (val << ((reg & 3) << 3));
			break;

		case 2:
			data = (data & ~(0xffff << ((reg & 3) << 3))) | (val << ((reg & 3) << 3));
			break;

		case 4:
			data = val;
			break;

		default:
			return PCIBIOS_FUNC_NOT_SUPPORTED;
	}

	bcm_pcie_writel(data, cfg_addr);

    return PCIBIOS_SUCCESSFUL;
}

struct pci_ops bcm63268_pcie_ops = {
    .read   = bcm63268_pcie_read_config,
    .write  = bcm63268_pcie_write_config,
};

