/*
<:copyright-gpl 
 Copyright 2010 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/clockchips.h>
#include <linux/interrupt.h>
#include <linux/percpu.h>
#include <linux/clk.h>

#include <asm/smtc_ipi.h>
#include <asm/time.h>
#include <asm/cevt-r4k.h>
#include <bcm681x_cpu.h>
#include <bcm681x_io.h>
#include <bcm681x_regs.h>
#include <bcm681x_irq.h>
#include <bcm681x_clk.h>

DEFINE_PER_CPU(struct clock_event_device, mips_bcm_rl_clockevent_device);
int bcm_rl_timer_irq_installed;
static int bcm_rl_timer_count;

typedef int (* bcm_rl_timer_hook_func_t)(u32);
bcm_rl_timer_hook_func_t bcm_rl_timer_hook_g = (bcm_rl_timer_hook_func_t)NULL;


static int mips_bcm_rl_next_event(unsigned long delta,
                           struct clock_event_device *evt)
{
	u32 val;

	val = TIMER_CTL_ENABLE_MASK | bcm_rl_timer_count;
	bcm_timer_writel(val, TIMER_CTL1_REG);
	
	val = bcm_timer_readl(TIMER_IRQSTAT_REG);
	val |= TIMER_IRQSTAT_TIMER1_CAUSE;
	bcm_timer_writel(val, TIMER_IRQSTAT_REG);

    return 0;
}

void mips_bcm_rl_set_clock_mode(enum clock_event_mode mode,
				struct clock_event_device *evt)
{
	/* do nothing */
}

void mips_bcm_rl_event_handler(struct clock_event_device *dev)
{
	/* do nothing */
}

irqreturn_t mips_bcm_rl_timer_interrupt(int irq, void *dev_id)
{
	struct clock_event_device *cd;
	int cpu = smp_processor_id();
	u32 val;

    val = bcm_timer_readl(TIMER_IRQSTAT_REG);
	val |= TIMER_IRQSTAT_TIMER1_CAUSE;
	bcm_timer_writel(val, TIMER_IRQSTAT_REG);

    cd = &per_cpu(mips_bcm_rl_clockevent_device, cpu);
    cd->event_handler(cd);

	val = TIMER_CTL_ENABLE_MASK | bcm_rl_timer_count;
	bcm_timer_writel(val, TIMER_CTL1_REG);

	val = bcm_timer_readl(TIMER_IRQSTAT_REG);
	val |= TIMER_IRQSTAT_TIMER1_IR_EN;
	bcm_timer_writel(val, TIMER_IRQSTAT_REG);

    /* call the timer handler function if attached */
    if (bcm_rl_timer_hook_g != (bcm_rl_timer_hook_func_t)NULL)
        bcm_rl_timer_hook_g(read_c0_count());

	return IRQ_HANDLED;
}

struct irqaction bcm_rl_timer_irqaction = {
	.handler = mips_bcm_rl_timer_interrupt,
	.flags = IRQF_DISABLED,
	.name = "cevt-bcm_rl",
};


/* start the periodic timer - keep accurate 125 usec tick count */
int bcm_rl_timer_start( void )
{
	u32 val;
	
    val = TIMER_CTL_ENABLE_MASK | bcm_rl_timer_count;
	bcm_timer_writel(val, TIMER_CTL1_REG);

	val = bcm_timer_readl(TIMER_IRQSTAT_REG);
	val |= TIMER_IRQSTAT_TIMER1_IR_EN;
	bcm_timer_writel(val, TIMER_IRQSTAT_REG);
	
    return 0;
}

/* stop the periodic timer */
int bcm_rl_timer_stop( void )
{
	u32 val;
	
    val = 0;
	bcm_timer_writel(val, TIMER_CTL1_REG);

	val = bcm_timer_readl(TIMER_IRQSTAT_REG);
	val &= ~TIMER_IRQSTAT_TIMER1_IR_EN;
	bcm_timer_writel(val, TIMER_IRQSTAT_REG);

    return 0;
}


/*
 * use timer1 to create a rate-limiter which refreshes every 125 usec
 */
int __cpuinit r4k_clockevent_bcm_rl_timer_init(void)
{
	struct clk *clk;
	unsigned int cpu = smp_processor_id();
	struct clock_event_device *cd;
	int irq;

	cd = &per_cpu(mips_bcm_rl_clockevent_device, cpu);

	cd->name		= "cevt-bcm_rl";
	cd->features	= CLOCK_EVT_FEAT_PERIODIC;

	clk = clk_get(NULL, "periph");
	if (IS_ERR(clk)) {
		pr_err("No peripheral clock for cevt-bcm_rl\n");
		return -1;
	}

	if (BCMCPU_IS_6812()) {
		irq = BCM_6812_TIMER_IRQ;
	}
	else if (BCMCPU_IS_6816()) {
		irq = BCM_6816_TIMER_IRQ;
	}
	else if (BCMCPU_IS_6818()) {
		irq = BCM_6818_TIMER_IRQ;
	}
	
	/* Calculate the min / max delta */
	cd->mult	= div_sc((unsigned long)clk->rate, NSEC_PER_SEC, 32);
	cd->shift		= 32;
	cd->max_delta_ns	= clockevent_delta2ns(0x3fffffff, cd);
	cd->min_delta_ns	= clockevent_delta2ns(0x300, cd);

	cd->rating		= 300;
	cd->irq			= irq;
	cd->cpumask		= cpumask_of(cpu);
	cd->set_next_event	= mips_bcm_rl_next_event;
	cd->set_mode		= mips_bcm_rl_set_clock_mode;
	cd->event_handler	= mips_bcm_rl_event_handler;

	bcm_rl_timer_count = (125 * (clk->rate / 1000000)) - 1;

	clockevents_register_device(cd);

	if (bcm_rl_timer_irq_installed)
		return 0;

    setup_irq(irq, &bcm_rl_timer_irqaction);
	bcm_rl_timer_irq_installed = 1;

	return 0;
}

EXPORT_SYMBOL(bcm_rl_timer_hook_g);
EXPORT_SYMBOL(bcm_rl_timer_start);
EXPORT_SYMBOL(bcm_rl_timer_stop);

