/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/
/***********************************************************
 *
 * Warning... This code is not yes SMP-compatible. So power management
 * cannot be enabled at the same time as SMP for the moment.
 *
 * This file implements clock events for the Broadcom DSL and GPON CPE
 * when the power management feature is enabled. When the processor
 * is found to be mostly idle, the main CPU clock is slowed down to
 * save power. By slowing down the clock, the C0 counter unfortunately
 * also slows down. This file replaces the (typical) 1 msec clock tick
 * interrupt processing with a reliable timer source which is unaffected
 * by the change in MIPS clock changes.
 *
 * The timer available to replace the C0 timer works differently.
 * The design needs to be adjusted accordingly. The C0 counter is a free
 * running counter which wraps at 0xFFFFFFFF and which runs at different
 * frequencies depending on the MIPS frequency. The C0 compare register
 * requires to be programmed to stay ahead of the C0 counter, to generate
 * an interrupt in the future.
 *
 * The peripheral timers (there are 3 of them) wrap at 0x3fffffff and
 * run at 50 MHz. When the timer reaches a programmed value, it can generate
 * and interrupt and then either stops counting or restarts at 0.
 * This difference in behavior between the C0 counter and the peripheral timers
 * required to use 2 timers for power management. One to generate the periodic
 * interrupts required by the clock events (Timer 0), and one to keep an accurate
 * reference when the clock is slowed down for saving power (Timer 2). Timer 1
 * is planned to be used by the second processor to support SMP.
 *
 ************************************************************/


#include <linux/clockchips.h>
#include <linux/interrupt.h>
#include <linux/percpu.h>
#include <linux/smp.h>
#include <linux/clk.h>

#include <asm/smtc_ipi.h>
#include <asm/time.h>
#include <asm/cevt-r4k.h>
#include <bcm632x_cpu.h>
#include <bcm632x_io.h>
#include <bcm632x_regs.h>
#include <bcm632x_irq.h>
#include <bcm632x_clk.h>
#include <bcm632x_timer.h>
#include <bcm632x_pwrmngt.h>

extern void BcmPwrMngtCheckWaitCount(void);
extern unsigned int TimerC0Snapshot0;
#if defined(CONFIG_SMP)
extern unsigned int TimerC0Snapshot1;
extern unsigned int C0divider, C0multiplier, C0ratio;
extern struct plat_smp_ops *mp_ops;
#endif /* CONFIG_SMP */

DEFINE_PER_CPU(struct clock_event_device, mips_bcm_pwr_clockevent_device);
static int bcm_pwr_timer_count;
static int bcm_pwr_high_precision_timer_count;
int bcm_pwr_timer_irq_installed;

/* 
 * TP0 is responsible for reprogramming the timer with the required delta
 */
static int mips_bcm_pwr_next_event_tp0(unsigned long delta,
                           struct clock_event_device *evt)
{
#ifdef CONFIG_BCM_TIMER
	bcm632x_timer_set(TIMER_T0_ID, 0, 1, delta - 1);
	bcm632x_timer_enable(TIMER_T0_ID, 1);
    return 0;
#else
	u32 reg;

	reg = 0;
	bcm_timer_writel(reg, TIMER_CTLx_REG(TIMER_T0_ID));
	bcm_timer_writel(reg, TIMER_CNTx_REG(TIMER_T0_ID));
	reg |= TIMER_CTL_ENABLE_MASK | TIMER_CTL_MONOTONIC_MASK | (delta - 1);
	bcm_timer_writel(reg, TIMER_CTLx_REG(TIMER_T0_ID));

	reg = bcm_timer_readl(TIMER_IRQSTAT_REG);
	reg |= TIMER_IRQSTAT_TIMER0_IR_EN;
	bcm_timer_writel(reg, TIMER_IRQSTAT_REG);

	return 0;
#endif /* CONFIG_BCM_TIMER */
}

#if defined(CONFIG_SMP)
/*
 * TP1 has nothing to do to generate the next timer interrupt
 * It will receive an interrupt from TP0 through SW IPI
 */
static int mips_bcm_pwr_next_event_tp1(unsigned long delta,
                           struct clock_event_device *evt)
{
#ifdef CONFIG_BCM_TIMER
	bcm632x_timer_set(TIMER_T1_ID, 0, 1, delta - 1);
	bcm632x_timer_enable(TIMER_T1_ID, 1);
    return 0;
#else
	u32 reg;

	reg = 0;
	bcm_timer_writel(reg, TIMER_CTLx_REG(TIMER_T1_ID));
	bcm_timer_writel(reg, TIMER_CNTx_REG(TIMER_T1_ID));
	reg |= TIMER_CTL_ENABLE_MASK | TIMER_CTL_MONOTONIC_MASK | (delta - 1);
	bcm_timer_writel(reg, TIMER_CTLx_REG(TIMER_T1_ID));

	reg = bcm_timer_readl(TIMER_IRQSTAT_REG);
	reg |= TIMER_IRQSTAT_TIMER1_IR_EN;
	bcm_timer_writel(reg, TIMER_IRQSTAT_REG);

	return 0;
#endif /* CONFIG_BCM_TIMER */
}
#endif /* CONFIG_SMP */

void mips_bcm_pwr_set_clock_mode(enum clock_event_mode mode,
				struct clock_event_device *evt)
{
	/* do nothing */
}

void mips_bcm_pwr_event_handler(struct clock_event_device *dev)
{
	/* do nothing */
}

#ifdef CONFIG_BCM_TIMER
static void mips_bcm_pwr_timer_interrupt(u32 stat, void *data)
{
	struct clock_event_device *cd;
	int cpu = smp_processor_id();

	if (stat & TIMER_IRQSTAT_TIMER_CAUSE(TIMER_T0_ID)) {
		bcm_timer_writel(0, TIMER_CTLx_REG(TIMER_T0_ID));
		cd = &per_cpu(mips_bcm_pwr_clockevent_device, cpu);
    	cd->event_handler(cd);

		BcmPwrMngtCheckWaitCount();
	}
	
#if defined(CONFIG_SMP)
	if (stat & TIMER_IRQSTAT_TIMER_CAUSE(TIMER_T1_ID)) {
		bcm_timer_writel(0, TIMER_CTLx_REG(TIMER_T1_ID));

		mp_ops->send_ipi_single(1, SMP_BCM_PWRSAVE_TIMER);
	}
#endif /* CONFIG_SMP */
}
#else
irqreturn_t mips_bcm_pwr_timer_interrupt(int irq, void *dev_id)
{
	struct clock_event_device *cd;
	u32 reg, stat;
	int rv = IRQ_NONE;

	stat = bcm_timer_readl(TIMER_IRQSTAT_REG);
	/* ack the timer interrupt */
	bcm_timer_writel(stat, TIMER_IRQSTAT_REG);

	if (stat & TIMER_IRQSTAT_TIMER_CAUSE(TIMER_T0_ID)) {
		/* turn-off timer */
		reg = 0;
		bcm_timer_writel(reg, TIMER_CTLx_REG(TIMER_T0_ID));

		cd = &per_cpu(mips_bcm_pwr_clockevent_device, 0);
		cd->event_handler(cd);

		BcmPwrMngtCheckWaitCount();

		rv = IRQ_HANDLED;
	}

#if defined(CONFIG_SMP)
	if (stat & TIMER_IRQSTAT_TIMER_CAUSE(TIMER_T1_ID)) {
		/* turn-off timer */
		reg = 0;
		bcm_timer_writel(reg, TIMER_CTLx_REG(TIMER_T1_ID));

		mp_ops->send_ipi_single(1, SMP_BCM_PWRSAVE_TIMER);

		rv = IRQ_HANDLED;
	}

#endif /* CONFIG_SMP */

	return rv;
}
#endif /* CONFIG_BCM_TIMER */

#ifndef CONFIG_BCM_TIMER
struct irqaction bcm632x_pwrmngt_timer_irqaction = {
	.handler = mips_bcm_pwr_timer_interrupt,
	.flags = IRQF_DISABLED | IRQF_SHARED,
	.name = "bcm632x-pwrmngt-timer",
};
#endif /* !CONFIG_BCM_TIMER */

/*
 * use timer0 to create a power-management timer thread
 */
int __cpuinit r4k_clockevent_init(void)
{
	struct clk *clk;
	unsigned int cpu = smp_processor_id();
	struct clock_event_device *cd;
	int irq;

	cd = &per_cpu(mips_bcm_pwr_clockevent_device, cpu);

	cd->name		= "cevt-bcm_pwrmngt";
	cd->features	= CLOCK_EVT_FEAT_ONESHOT;

	clk = clk_get(NULL, "periph");
	if (IS_ERR(clk)) {
		pr_err("No peripheral clock for cevt-bcm_rl\n");
		return -1;
	}

	if (BCMCPU_IS_63281()) {
		irq = bcm632x_get_irq_number(IRQ_TIMER);
	}
	
	/* Calculate the min / max delta */
	cd->mult	= div_sc((unsigned long)clk->rate, NSEC_PER_SEC, 32);
	cd->shift		= 32;
	cd->max_delta_ns	= clockevent_delta2ns(0x3fffffff, cd);
	cd->min_delta_ns	= clockevent_delta2ns(0x300, cd);

	cd->rating		= 300;
	cd->irq			= irq;
	cd->cpumask		= cpumask_of(cpu);
	if (cpu == 0)
		cd->set_next_event	= mips_bcm_pwr_next_event_tp0;
#if defined(CONFIG_SMP)
	else
		cd->set_next_event	= mips_bcm_pwr_next_event_tp1;
#endif /* CONFIG_SMP */
	cd->set_mode		= mips_bcm_pwr_set_clock_mode;
	cd->event_handler	= mips_bcm_pwr_event_handler;

	/* 1 msec */
	bcm_pwr_timer_count = 50000 - 1;

	/* high precision */
	bcm_pwr_high_precision_timer_count = 0x3fffffff;

	clockevents_register_device(cd);

	if (cpu == 0) {
#ifdef CONFIG_BCM_TIMER
		if (bcm_pwr_timer_irq_installed)
			return 0;
		
		bcm632x_timer_register(0, (bcm_timer_hook_func_t)mips_bcm_pwr_timer_interrupt, NULL);
		bcm_pwr_timer_irq_installed = 1;
		
		/* start timer0 and keep accurate 1 msec tick count */
		bcm632x_timer_set(TIMER_T0_ID, 0, 1, bcm_pwr_timer_count);
		bcm632x_timer_enable(TIMER_T0_ID, 1);

#ifdef CONFIG_BCM_HOSTMIPS_PWRSAVE_TIMERS
		/* 
		 * Take a snapshot of the C0 timer when timer2 was started.
		 * This will be needed later when having to make adjustments
		 */
		TimerC0Snapshot0 = read_c0_count();
#endif /* CONFIG_BCM_HOSTMIPS_PWRSAVE_TIMERS */

		/* start timer2 to keep an accurate free running high presicion tick count */
		bcm632x_timer_set(TIMER_T2_ID, 1, 1, bcm_pwr_high_precision_timer_count);
		bcm632x_timer_enable(TIMER_T2_ID, 0);
#else
		u32 reg;

		irq_set_affinity(irq, cpumask_of(0));
		setup_irq(irq, &bcm632x_pwrmngt_timer_irqaction);

		/* start timer0 and keep accurate 1 msec tick count */
		reg = bcm_timer_readl(TIMER_CTLx_REG(TIMER_T0_ID));
		reg |= TIMER_CTL_ENABLE_MASK | TIMER_CTL_MONOTONIC_MASK | bcm_pwr_timer_count;
		bcm_timer_writel(reg, TIMER_CTLx_REG(TIMER_T0_ID));

		reg = bcm_timer_readl(TIMER_IRQSTAT_REG);
		reg |= TIMER_IRQSTAT_TIMER0_IR_EN;
		bcm_timer_writel(reg, TIMER_IRQSTAT_REG);

#ifdef CONFIG_BCM_HOSTMIPS_PWRSAVE_TIMERS
		/* 
		 * Take a snapshot of the C0 timer when timer2 was started.
		 * This will be needed later when having to make adjustments
		 */
		TimerC0Snapshot0 = read_c0_count();
#endif /* CONFIG_BCM_HOSTMIPS_PWRSAVE_TIMERS */

		/* start timer2 to keep an accurate free running high presicion tick count */
		reg = bcm_timer_readl(TIMER_CTLx_REG(TIMER_T2_ID));
		reg |= TIMER_CTL_ENABLE_MASK | bcm_pwr_high_precision_timer_count;
		bcm_timer_writel(reg, TIMER_CTLx_REG(TIMER_T2_ID));
#endif /* CONFIG_BCM_TIMER */
	}
#if defined(CONFIG_SMP)
	else {
		u32 newTimerCnt, mult, rem, result;

#ifdef CONFIG_BCM_TIMER
		/* start timer0 and keep accurate 1 msec tick count */
		bcm632x_timer_set(TIMER_T1_ID, 0, 1, bcm_pwr_timer_count);
		bcm632x_timer_enable(TIMER_T1_ID, 1);
#else
		u32 reg;
		/* start timer0 and keep accurate 1 msec tick count */
		reg = bcm_timer_readl(TIMER_CTLx_REG(TIMER_T1_ID));
		reg |= TIMER_CTL_ENABLE_MASK | TIMER_CTL_MONOTONIC_MASK | bcm_pwr_timer_count;
		bcm_timer_writel(reg, TIMER_CTLx_REG(TIMER_T1_ID));

		reg = bcm_timer_readl(TIMER_IRQSTAT_REG);
		reg |= TIMER_IRQSTAT_TIMER1_IR_EN;
		bcm_timer_writel(reg, TIMER_IRQSTAT_REG);
#endif

		/* 
		 * Take a snapshot of the C0 timer when Timer2 was started
		 * This will be needed later when having to make adjustments
		 */
		TimerC0Snapshot1 = read_c0_count();
		reg_val = bcm_timer_readl(TIMER_CNT2_REG);
		newTimerCnt = reg_val & 0x3fffffff;
		mult = newTimerCnt / C0divider;
		rem  = newTimerCnt % C0divider;
		result  = (mult * C0multiplier) + ((rem * C0ratio) >> 10);
		TimerC0Snapshot1 -= result;
	}
#endif /* CONFIG_SMP */

	return 0;
}

