/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/kernel.h>
#include <linux/err.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <bcm63268_cpu.h>
#include <bcm63268_io.h>
#include <bcm63268_timer.h>
#include <bcm63268_regs.h>

static DEFINE_SPINLOCK(timer_reg_lock);
static DEFINE_SPINLOCK(timer_data_lock);
static struct clk *periph_clk = NULL;

static struct timer_data {
	void	(*cb)(u32 stat, void *);
	void	*data;
} timer_data[BCM63268_TIMER_COUNT];

static irqreturn_t bcm63268_timer_interrupt(int irq, void *dev_id)
{
	u32 stat;
	int i;

	spin_lock(&timer_reg_lock);
	stat = bcm_timer_readl(TIMER_IRQSTAT_REG);
	bcm_timer_writel(stat, TIMER_IRQSTAT_REG);
	spin_unlock(&timer_reg_lock);

	for (i = 0; i < BCM63268_TIMER_COUNT; i++) {
		if (!(stat & TIMER_IRQSTAT_TIMER_CAUSE(i)))
			continue;

		spin_lock(&timer_data_lock);
		if (!timer_data[i].cb) {
			spin_unlock(&timer_data_lock);
			continue;
		}

		timer_data[i].cb(stat, timer_data[i].data);
		spin_unlock(&timer_data_lock);
	}

	return IRQ_HANDLED;
}

int bcm63268_timer_enable(int id, int intr)
{
	u32 reg;
	unsigned long flags;

	if (id >= BCM63268_TIMER_COUNT)
		return -EINVAL;

	spin_lock_irqsave(&timer_reg_lock, flags);

	reg = bcm_timer_readl(TIMER_CTLx_REG(id));
	if (!(reg & TIMER_CTL_ENABLE_MASK)) {
		reg |= TIMER_CTL_ENABLE_MASK;
		bcm_timer_writel(reg, TIMER_CTLx_REG(id));
	}

	if (intr) {
		reg = bcm_timer_readl(TIMER_IRQSTAT_REG);
		reg |= TIMER_IRQSTAT_TIMER_IR_EN(id);
		bcm_timer_writel(reg, TIMER_IRQSTAT_REG);
	}

	spin_unlock_irqrestore(&timer_reg_lock, flags);
	return 0;
}

EXPORT_SYMBOL(bcm63268_timer_enable);

int bcm63268_timer_disable(int id)
{
	u32 reg;
	unsigned long flags;

	if (id >= BCM63268_TIMER_COUNT)
		return -EINVAL;

	spin_lock_irqsave(&timer_reg_lock, flags);

	reg = bcm_timer_readl(TIMER_CTLx_REG(id));
	if (reg & TIMER_CTL_ENABLE_MASK) {
		reg &= ~TIMER_CTL_ENABLE_MASK;
		bcm_timer_writel(reg, TIMER_CTLx_REG(id));
	}

	reg = bcm_timer_readl(TIMER_IRQSTAT_REG);
	reg &= ~(TIMER_IRQSTAT_TIMER_IR_EN(id));
	bcm_timer_writel(reg, TIMER_IRQSTAT_REG);

	spin_unlock_irqrestore(&timer_reg_lock, flags);
	return 0;
}

EXPORT_SYMBOL(bcm63268_timer_disable);

int bcm63268_timer_register(int id, bcm_timer_hook_func_t callback, void *data)
{
	unsigned long flags;
	int ret;

	if (id >= BCM63268_TIMER_COUNT || !callback)
		return -EINVAL;

	ret = 0;
	spin_lock_irqsave(&timer_data_lock, flags);
	if (timer_data[id].cb) {
		ret = -EBUSY;
		goto out;
	}

	timer_data[id].cb = callback;
	timer_data[id].data = data;

out:
	spin_unlock_irqrestore(&timer_data_lock, flags);
	return ret;
}

EXPORT_SYMBOL(bcm63268_timer_register);

void bcm63268_timer_unregister(int id)
{
	unsigned long flags;

	if (id >= BCM63268_TIMER_COUNT)
		return;

	spin_lock_irqsave(&timer_data_lock, flags);
	timer_data[id].cb = NULL;
	spin_unlock_irqrestore(&timer_data_lock, flags);
}

EXPORT_SYMBOL(bcm63268_timer_unregister);

unsigned int bcm63268_timer_countdown(unsigned int countdown_us)
{
	if (!periph_clk)
		periph_clk = clk_get(NULL, "periph");

	if (IS_ERR(periph_clk)) {
		pr_err("No peripheral clock for timer\n");
		return 0;
	}

	return (clk_get_rate(periph_clk) / (1000 * 1000)) * countdown_us;
}

EXPORT_SYMBOL(bcm63268_timer_countdown);

int bcm63268_timer_set(int id, int monotonic, int direct_timerval, unsigned int countdown_us)
{
	u32 reg, countdown;
	unsigned long flags;

	if (id >= BCM63268_TIMER_COUNT)
		return -EINVAL;

	if (!direct_timerval) {
		countdown = bcm63268_timer_countdown(countdown_us);
		if (countdown & ~TIMER_CTL_COUNTDOWN_MASK)
			return -EINVAL;
	} else {
		countdown = countdown_us;
	}

	spin_lock_irqsave(&timer_reg_lock, flags);
	reg = 0;
	bcm_timer_writel(reg, TIMER_CTLx_REG(id));
	bcm_timer_writel(reg, TIMER_CNTx_REG(id));

	if (monotonic)
		reg &= ~TIMER_CTL_MONOTONIC_MASK;
	else
		reg |= TIMER_CTL_MONOTONIC_MASK;

	reg &= ~TIMER_CTL_COUNTDOWN_MASK;
	reg |= countdown;
	bcm_timer_writel(reg, TIMER_CTLx_REG(id));

	spin_unlock_irqrestore(&timer_reg_lock, flags);
	return 0;
}

EXPORT_SYMBOL(bcm63268_timer_set);

int bcm63268_timer_is_mytimer(u32 stat, int timer_id)
{
	return stat & TIMER_IRQSTAT_TIMER_CAUSE(timer_id) ? 1 : 0;
}
EXPORT_SYMBOL(bcm63268_timer_is_mytimer);

void bcm63268_timer_clear(int timer_id)
{
	bcm_timer_writel(0, TIMER_CTLx_REG(timer_id));
}
EXPORT_SYMBOL(bcm63268_timer_clear);

#ifdef CONFIG_BCM_HOSTMIPS_PWRSAVE
struct irqaction bcm632x_timer_irqaction = {
	.handler = bcm63268_timer_interrupt,
	.flags = IRQF_DISABLED | IRQF_SHARED,
	.name = "bcm63268-timer",
};
#endif /* CONFIG_BCM_HOSTMIPS_PWRSAVE */

int __init bcm63268_timer_init(void)
{
	int ret, irq;
	u32 reg;

	reg = 0;
	bcm_timer_writel(0, TIMER_CTL0_REG);
	bcm_timer_writel(0, TIMER_CTL1_REG);
	bcm_timer_writel(0, TIMER_CTL2_REG);

	reg = bcm_timer_readl(TIMER_IRQSTAT_REG);
	reg &= ~TIMER_IRQSTAT_TIMER0_IR_EN;
	reg &= ~TIMER_IRQSTAT_TIMER1_IR_EN;
	reg &= ~TIMER_IRQSTAT_TIMER2_IR_EN;
	bcm_timer_writel(reg, TIMER_IRQSTAT_REG);

	if (!periph_clk)
		periph_clk = clk_get(NULL, "periph");
	
	if (IS_ERR(periph_clk)) {
		pr_err("No peripheral clock for timer\n");
		return -ENODEV;
	}

	irq = bcm63268_get_irq_number(IRQ_TIMER);
#ifdef CONFIG_BCM_HOSTMIPS_PWRSAVE
	irq_set_affinity(irq, cpumask_of(0));
	ret = setup_irq(irq, &bcm632x_timer_irqaction);
#else
	ret = request_irq(irq, bcm63268_timer_interrupt, 0, "bcm63268_timer", NULL);
#endif /* CONFIG_BCM_HOSTMIPS_PWRSAVE */
	if (ret) {
		printk(KERN_ERR "bcm63268_timer: failed to register irq\n");
		return ret;
	}

	return 0;
}

#ifndef CONFIG_BCM_HOSTMIPS_PWRSAVE
arch_initcall(bcm63268_timer_init);
#endif /* !CONFIG_BCM_HOSTMIPS_PWRSAVE */

