/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/cpumask.h>
#include <linux/interrupt.h>
#include <linux/compiler.h>

#include <asm/atomic.h>
#include <asm/cacheflush.h>
#include <asm/cpu.h>
#include <asm/processor.h>
#include <asm/system.h>
#include <asm/hardirq.h>
#include <asm/mmu_context.h>
#include <asm/smp.h>
#include <asm/time.h>
#include <asm/mipsregs.h>
#include <asm/mipsmtregs.h>
#include <asm/mips_mt.h>

static int cpu_ipi_irq;
DEFINE_PER_CPU(unsigned int, bcm63268_ipi_pending);
DEFINE_PER_CPU(unsigned int, bcm63268_ipi_flags);

extern spinlock_t bcm63268_irqlock;

/* boot parameters */
struct bcm63268_boot_config_s {
    unsigned int func_addr;
    unsigned int gp;
    unsigned int sp;
};

static struct bcm63268_boot_config_s bcm63268_boot_config;

void bcm63268_install_smp_boot_ex_handler(void);
static void bcm63268_core_send_ipi_single(int cpu, unsigned int action);
static void bcm63268_core_send_ipi_mask(cpumask_t mask, unsigned int action);

void bcm63268_install_smp_boot_ex_handler(void)
{

	asm (
        ".set push\n"
        ".set noreorder\n"
        "lui    $8, 0xa000 \n"
        "ori    $8, $8, 0x0200  # alternative mips exception vector\n"
        "la     $9, 2f\n"
        "la     $10, 3f\n"
    "1:\n"
        "lw     $11, 0($9)\n"
        "sw     $11, 0($8)\n"
        "addiu  $9, $9, 4\n"
        "bne    $9, $10, 1b\n"
        "addiu  $8, $8, 4\n"
        "b      3f\n"
        "nop\n"
    "2:    # begin exception handler code\n"
        "mfc0   $26, $13, 0\n"
        "li     $27, 0x800100   # change back to normal exception vector & ack interrupt\n"
        "xor    $26, $27\n"
        "mtc0   $26, $13, 0\n"
        "la     $27, %0         # pointer to boot_config structure\n"
        "lw     $24, 0($27)     # func_addr - will be loaded into EPC before eret\n"
        "lw     $28, 4($27)     # gp\n"
        "lw     $29, 8($27)     # sp\n"
        "mtc0   $24, $14        # load return address into EPC\n"
        "eret\n"
    "3:\n"
        ".set pop\n"
        :
        : "X" (&bcm63268_boot_config)
	);
}

#ifdef CONFIG_BCM_HOSTMIPS_PWRSAVE_TIMERS
extern void mips_bcm_pwr_timer_interrupt_tp1(void);
#endif

static irqreturn_t bcm63268_ipi_interrupt(int irq, void *dev_id)
{
	unsigned int ipiflags;

	spin_lock(&bcm63268_irqlock);
	ipiflags = __get_cpu_var(bcm63268_ipi_flags);
	__get_cpu_var(bcm63268_ipi_flags) = 0;
	spin_unlock(&bcm63268_irqlock);

#ifdef CONFIG_BCM_HOSTMIPS_PWRSAVE_TIMERS
	/* Process TIMER related interrupt first */
	if (ipiflags & (1 << 2)) {
		mips_bcm_pwr_timer_interrupt_tp1();
	}
#endif

	if (ipiflags & SMP_CALL_FUNCTION) {
		smp_call_function_interrupt();
	}

	return IRQ_HANDLED;
}


static struct irqaction bcm63268_irq_ipi = {
	.handler	= bcm63268_ipi_interrupt,
	.flags		= IRQF_DISABLED|IRQF_PERCPU,
	.name		= "ipi"
};


static void __init bcm63268_smp_setup(void)
{
	int i;

	cpus_clear(cpu_possible_map);

	for (i = 0; i < 2; i++) {
		cpu_set(i, cpu_possible_map);
		__cpu_number_map[i]	= i;
		__cpu_logical_map[i] = i;
	}
}


static void __init bcm63268_prepare_cpus(unsigned int max_cpus)
{
	unsigned int c0tmp;
	int cpu;

	c0tmp = __read_32bit_c0_register($22, 0);
	c0tmp |= CP0_BCM_CFG_NBK;    /* non blocking */
	__write_32bit_c0_register($22, 0, c0tmp);

	c0tmp = __read_32bit_c0_register($22, 2);
	c0tmp &= ~(CP0_CMT_PRIO_TP0 | CP0_CMT_PRIO_TP1); /* equal (random) D-cache priority */
	__write_32bit_c0_register($22, 2, c0tmp);

#if 0
	printk("bcm config0 %08x\n", __read_32bit_c0_register($22, 0));
	printk("cmt control %08x\n", __read_32bit_c0_register($22, 2));
	printk("cmt local %08x\n", __read_32bit_c0_register($22, 3));
	printk("bcm config1 %08x\n", __read_32bit_c0_register($22, 5));
#endif

	for_each_cpu_mask(cpu, cpu_possible_map) {
		per_cpu(bcm63268_ipi_pending, cpu) = 0;
		per_cpu(bcm63268_ipi_flags, cpu) = 0;
	}

	c0tmp = __read_32bit_c0_register($22, 1);
	c0tmp |= CP0_CMT_SIR_0;
	__write_32bit_c0_register($22, 1, c0tmp);

	cpu_ipi_irq = 0;
	setup_irq(cpu_ipi_irq, &bcm63268_irq_ipi);
	set_irq_handler(cpu_ipi_irq, handle_percpu_irq);
}


// Pass PC, SP, and GP to a secondary core and start it up by sending an inter-core interrupt
static void __cpuinit bcm63268_boot_secondary(int cpu, struct task_struct *idle)
{
	unsigned int cause;

	bcm63268_boot_config.func_addr = (unsigned long)smp_bootstrap;
	bcm63268_boot_config.sp = (unsigned int)__KSTK_TOS(idle);
	bcm63268_boot_config.gp = (unsigned int)task_thread_info(idle);

	bcm63268_install_smp_boot_ex_handler();

	cause = read_c0_cause();
	cause |= CAUSEF_IP0;
	write_c0_cause(cause);
}

static void __cpuinit bcm63268_init_secondary(void)
{
#if 0
	printk("bcm config0 %08x\n", __read_32bit_c0_register($22, 0));
	printk("cmt control %08x\n", __read_32bit_c0_register($22, 2));
	printk("cmt local %08x\n", __read_32bit_c0_register($22, 3));
	printk("bcm config1 %08x\n", __read_32bit_c0_register($22, 5));
#endif

	clear_c0_status(ST0_BEV);
#ifdef CONFIG_BCM_HOSTMIPS_PWRSAVE_TIMERS
	// CP0 timer interrupt (IRQ5) is not used for TP1 when pwrsave is enabled
	change_c0_status(ST0_IM, IE_SW0 | IE_IRQ0 | IE_IRQ1);
#else
	change_c0_status(ST0_IM, IE_SW0 | IE_IRQ0 | IE_IRQ1 | IE_IRQ5);
#endif
}


static void __cpuinit bcm63268_smp_finish(void)
{
	/* to generate the first CPU timer interrupt */
	write_c0_compare(read_c0_count() + mips_hpt_frequency / HZ);
}


static void bcm63268_cpus_done(void)
{
}


// send inter-core interrupts
static void bcm63268_core_send_ipi_single(int cpu, unsigned int action)
{
	unsigned long flags;
	unsigned int cause;

#if 0
	printk("== from_cpu %d    to_cpu %d    action %u\n", smp_processor_id(), cpu, action);
#endif

	spin_lock_irqsave(&bcm63268_irqlock, flags);

	switch (action) {
	case SMP_RESCHEDULE_YOURSELF:
		per_cpu(bcm63268_ipi_pending, cpu) = 1;
		per_cpu(bcm63268_ipi_flags, cpu) |= 1<<0;
		break;
	case SMP_CALL_FUNCTION:
		per_cpu(bcm63268_ipi_pending, cpu) = 1;
		per_cpu(bcm63268_ipi_flags, cpu) |= 1<<1;
		break;
#if defined(CONFIG_BCM_HOSTMIPS_PWRSAVE_TIMERS)
	case SMP_BCM_PWRSAVE_TIMER:
		per_cpu(bcm63268_ipi_pending, cpu) = 1;
		per_cpu(bcm63268_ipi_flags, cpu) |= 1<<2;
		break;
#endif
	default:
		goto errexit;
	}

	mb();

	cause = read_c0_cause();
	cause |= CAUSEF_IP0;
	write_c0_cause(cause);

errexit:
	spin_unlock_irqrestore(&bcm63268_irqlock, flags);
}


static void bcm63268_core_send_ipi_mask(cpumask_t mask, unsigned int action)
{
	unsigned int cpu;

	for_each_cpu_mask(cpu, mask) {
		bcm63268_core_send_ipi_single(cpu, action);
	}
}


struct plat_smp_ops bcm63268_smp_ops = {
	.send_ipi_single	= bcm63268_core_send_ipi_single,
	.send_ipi_mask		= bcm63268_core_send_ipi_mask,
	.init_secondary		= bcm63268_init_secondary,
	.smp_finish		= bcm63268_smp_finish,
	.cpus_done		= bcm63268_cpus_done,
	.boot_secondary		= bcm63268_boot_secondary,
	.smp_setup		= bcm63268_smp_setup,
	.prepare_cpus		= bcm63268_prepare_cpus
};
