/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/


#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/bootmem.h>
#include <linux/ioport.h>
#include <linux/pm.h>
#include <asm/bootinfo.h>
#include <asm/time.h>
#include <asm/reboot.h>
#include <asm/cacheflush.h>
#include <bcm63268_board.h>
#include <bcm63268_cpu.h>
#include <bcm63268_regs.h>
#include <bcm63268_io.h>
#include <bcm63268_woc.h>
#include <bcm63268_pcie.h>
#include <bcm63268_usb_hc.h>
#include <bcm63268_timer.h>
#include <bcm63268_xdsl.h>

#ifdef CONFIG_SMP
extern void stop_other_cpu(void);
#endif /* CONFIG_SMP */

#define RAC_FLH         (1 << 8)
#define RAC_DPF         (1 << 6)
#define RAC_NCH         (1 << 5)
#define RAC_C_INV       (1 << 4)
#define RAC_PF_D        (1 << 3)
#define RAC_PF_I        (1 << 2)
#define RAC_D           (1 << 1)
#define RAC_I           (1 << 0)

void bcm63268_machine_halt(void)
{
	printk(KERN_INFO "System halted\n");
	while (1)
		;
}

void bcm63268_machine_reboot(void)
{
	u32 reg;

#ifdef CONFIG_SMP
	stop_other_cpu();
#endif /* CONFIG_SMP */

	local_irq_disable();

#ifdef CONFIG_BCM_ETH_PWRSAVE
	/* 
	 * Power Management on Ethernet Ports may have brought down EPHY PLL
     * and soft reset below will lock-up 63268 if the PLL is not up
     * therefore bring it up here to give it time to stabilize
     */
    reg = bcm_gpio_readl(GPIO_ROBOSW_EPHY_CTRL);
	reg &= ~GPIO_ROWOSW_EPHY_PWR_DOWN_DLL;
	bcm_gpio_writel(reg, GPIO_ROBOSW_EPHY_CTRL);
#endif /* CONFIG_BCM_ETH_PWRSAVE */

	udelay(100);

	/* mask and clear all external irq */
	reg = bcm_perf_readl(PERF_EXTIRQ_CFG_REG);
	reg &= ~(EXTIRQ_CFG_MASK(0));
	reg &= ~(EXTIRQ_CFG_MASK(1));
	reg &= ~(EXTIRQ_CFG_MASK(2));
	reg &= ~(EXTIRQ_CFG_MASK(3));
	reg |= EXTIRQ_CFG_CLEAR(0);
	reg |= EXTIRQ_CFG_CLEAR(1);
	reg |= EXTIRQ_CFG_CLEAR(2);
	reg |= EXTIRQ_CFG_CLEAR(3);
	bcm_perf_writel(reg, PERF_EXTIRQ_CFG_REG);

	printk(KERN_INFO "triggering pll soft-reset...\n");
	reg = bcm_perf_readl(PERF_PLLCTRL_REG);
	reg |= PLLCTRL_RESET;
	bcm_perf_writel(reg, PERF_PLLCTRL_REG);
	while (1)
		;
}

static void __bcm63268_machine_reboot(char *p)
{
	bcm63268_machine_reboot();
}

/*
 * return system type in /proc/cpuinfo
 */
const char *get_system_type(void)
{
	return bcm63268_board_get_sys_type();
}

void __init plat_time_init(void)
{
	mips_hpt_frequency = bcm63268_get_cpu_freq() / 2;
}

void __init plat_mem_setup(void)
{

	extern int panic_timeout;
	uint32_t reg32;
	volatile unsigned long *cr;
	u32 mips_addr = 0xff400000;
	u32 ringosc_startcount;
	u32 ringosc_endcount;
	u32 ringosc_countdiff;

#if defined(CONFIG_BCM_XTMCFG) || defined(CONFIG_BCM_XTMCFG_MODULE)
	/*
		we have to reserve memory for ADSL, the MACRO ADSL_SDRAM_IMAGE_SIZE is
		according to ADSL header files, we need to use a general way, TBD
	*/
	add_memory_region(0, bcm63268_get_memory_size() - CONFIG_BCM_XDSL_DSP_FW_SIZE, BOOT_MEM_RAM);
#else	
	add_memory_region(0, bcm63268_get_memory_size(), BOOT_MEM_RAM);
#endif

	cr = (void *)mips_addr;
	*cr = *cr | RAC_D | RAC_PF_D;

	cr = (void *)(mips_addr + 0x8);
	*cr = *cr | RAC_D | RAC_PF_D;

	_machine_halt = bcm63268_machine_halt;
	_machine_restart = __bcm63268_machine_reboot;
	pm_power_off = bcm63268_machine_halt;
	panic_timeout = 1;

#if defined(CONFIG_BCM_ADSL) || defined(CONFIG_BCM_ADSL_MODULE)
	bcm63268_xdsl_init();
#endif /* CONFIG_BCM_ADSL */

#if defined(CONFIG_USB) || defined(CONFIG_USB_MODULE)
	bcm63268_usb_hc_init();
#endif /* CONFIG_USB */

	bcm63268_woc_init();

#if defined(CONFIG_PCI)
	bcm63268_pcie_init();
#endif /* CONFIG_PCI */

#if defined(CONFIG_BCM_HOSTMIPS_PWRSAVE)
	/* Enable power savings from DDR pads on this chip when DDR goes in Self-Refresh mode */
	reg32 = 0x00000172;
	bcm_ddr_writel(reg32, DDR_PHYCTRL_IDLE_PAD_CTRL_REG);
	reg32 = 0x000fffff;
	bcm_ddr_writel(reg32, DDR_PHYBYTELANE_0_CTRL_IDLE_PAD_CTRL_REG);
	bcm_ddr_writel(reg32, DDR_PHYBYTELANE_1_CTRL_IDLE_PAD_CTRL_REG);
#endif

	/*
     * Determine if internal VREG is used.
     * If not, disable it to improve WLAN performance at 5GHz
     * The ring oscillators are affected when varying the 1V2 voltage
     * So take a measure of the ring osc count, then raise the internal regulator voltage and remeasure
     * If the ring osc count changed as expected than internal regulators are used
     */
    printk("Internal 1P2 VREG will be shutdown if unused...");

	/* Take the first ring osc measurement */
	reg32 = (GPIO_RING_OSC_CTRL1_ENABLE_MASK | GPIO_RING_OSC_CTRL1_COUNT_RESET | GPIO_RING_OSC_CTRL1_IRQ);
	bcm_gpio_writel(reg32, GPIO_RING_OSC_CTRL1);

	reg32 = (GPIO_RING_OSC_CTRL1_ENABLE_MASK | (2 << GPIO_RING_OSC_CTRL1_SELECT_SHIFT));
	bcm_gpio_writel(reg32, GPIO_RING_OSC_CTRL1);

	reg32 = GPIO_RING_OSC_CTRL0_512_CYCLES;
	bcm_gpio_writel(reg32, GPIO_RING_OSC_CTRL0);

	do {
		reg32 = bcm_gpio_readl(GPIO_RING_OSC_CTRL1);
	} while (!(reg32 & GPIO_RING_OSC_CTRL1_IRQ));
	
	ringosc_startcount = bcm_gpio_readl(GPIO_RING_OSC_CTRL1);
	ringosc_startcount &= GPIO_RING_OSC_CTRL1_COUNT_MASK;

	/* Increase internal 1V2 slightly and see if the ring osc is speeding up */
	reg32 = bcm_misc_readl(MISC_VREG_CTRL1_REG);
	reg32 += 8;
	bcm_misc_writel(reg32, MISC_VREG_CTRL1_REG);

	reg32 = bcm_misc_readl(MISC_VREG_CTRL0_REG);
	reg32 |= MISC_VREG_CTRL0_RESET;
	bcm_misc_writel(reg32, MISC_VREG_CTRL0_REG);

	/* Take the second ring osc measurement */
	reg32 = (GPIO_RING_OSC_CTRL1_ENABLE_MASK | GPIO_RING_OSC_CTRL1_COUNT_RESET | GPIO_RING_OSC_CTRL1_IRQ);
	bcm_gpio_writel(reg32, GPIO_RING_OSC_CTRL1);

	reg32 = (GPIO_RING_OSC_CTRL1_ENABLE_MASK | (2 << GPIO_RING_OSC_CTRL1_SELECT_SHIFT));
	bcm_gpio_writel(reg32, GPIO_RING_OSC_CTRL1);

	reg32 = GPIO_RING_OSC_CTRL0_512_CYCLES;
	bcm_gpio_writel(reg32, GPIO_RING_OSC_CTRL0);

	do {
		reg32 = bcm_gpio_readl(GPIO_RING_OSC_CTRL1);
	} while (!(reg32 & GPIO_RING_OSC_CTRL1_IRQ));
	
	ringosc_endcount = bcm_gpio_readl(GPIO_RING_OSC_CTRL1);
	ringosc_endcount &= GPIO_RING_OSC_CTRL1_COUNT_MASK;

	reg32 = bcm_misc_readl(MISC_VREG_CTRL1_REG);
	reg32 -= 8;
	bcm_misc_writel(reg32, MISC_VREG_CTRL1_REG);

	/*
     * A negative difference or a small positive difference indicates that an external regulator is used
     * This code was calibrated by repeating the measurements thousands of times and looking for a safe value
     * Safe means avoiding at all costs being wrong by shutting down the internal regulator when it is in use
     * It is better to be wrong by leaving the internal regulator running when an external regulator is used
     */
    ringosc_countdiff = ringosc_startcount - ringosc_endcount;
	if (ringosc_countdiff < 300) {
		printk("Unused, turn it off (%08x-%08x=%d<300)\n", ringosc_startcount, ringosc_endcount, ringosc_countdiff);
		/* Turn off internal 1P2 regulator */
		reg32 = bcm_misc_readl(MISC_VREG_CTRL0_REG);
		reg32 |= (MISC_VREG_CTRL0_RESET | MISC_VREG_CTRL0_POWER_DOWN1);
		bcm_misc_writel(reg32, MISC_VREG_CTRL0_REG);
	} else {
		printk("Used, leave it on (%08x-%08x=%d>=300)\n", ringosc_startcount, ringosc_endcount, ringosc_countdiff);
	}

	bcm63268_board_setup();
}

