/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/kernel.h>
#include <linux/err.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <asm/time.h>
#include <bcm63268_cpu.h>
#include <bcm63268_io.h>
#include <bcm63268_regs.h>
#include <bcm63268_irq.h>
#include <bcm63268_clk.h>
#include <bcm63268_timer.h>
#include <bcm63268_pwrmngt.h>

#define MASK_ASCR_BITS 0x7
#define MASK_ASCR_SHFT 28
#define MASK_ASCR (MASK_ASCR_BITS << MASK_ASCR_SHFT)

#ifdef ARCADYAN
unsigned int power_management_enabled = 0;
EXPORT_SYMBOL(power_management_enabled);
#endif /* ARCADYAN */
unsigned int originalMipsAscr = 0; // To keep track whether MIPS was in Async mode to start with at boot time
unsigned int originalMipsAscrChecked = 0;
unsigned int keepme;
#ifdef ARCADYAN
#if defined(CONFIG_BCM_DDR_SELF_REFRESH_PWRSAVE)
unsigned int self_refresh_enabled = 0; // Wait for the module to control if it is enabled or not
#endif
unsigned int clock_divide_enabled = 0; // Wait for the module to control if it is enabled or not
#else
#if defined(CONFIG_BCM_PWRMNGT_MODULE)
#if defined(CONFIG_BCM_DDR_SELF_REFRESH_PWRSAVE)
unsigned int self_refresh_enabled = 0; // Wait for the module to control if it is enabled or not
#endif
#if defined(CONFIG_BCM_HOSTMIPS_PWRSAVE)
unsigned int clock_divide_enabled = 0; // Wait for the module to control if it is enabled or not
#endif
#else
#if defined(CONFIG_BCM_DDR_SELF_REFRESH_PWRSAVE)
unsigned int self_refresh_enabled = 1;
#endif
#if defined(CONFIG_BCM_HOSTMIPS_PWRSAVE)
unsigned int clock_divide_enabled = 1;
#endif
#endif
#endif /* ARCADYAN */

unsigned int clock_divide_low_power0 = 0;
unsigned int clock_divide_active0 = 0;
unsigned int wait_count0 = 0;

#if defined(CONFIG_SMP)
unsigned int clock_divide_low_power1 = 0;
unsigned int clock_divide_active1 = 0;
unsigned int wait_count1 = 0;
#endif /* CONFIG_SMP */

volatile int isVoiceIdle = 1;
#if defined(ARCADYAN) /* for EndPoint driver */
EXPORT_SYMBOL(isVoiceIdle);
#endif /* ARCADYAN */

#ifdef ARCADYAN
static DEFINE_SPINLOCK(pwrmgnt_clk_irqlock);
#else
spinlock_t pwrmgnt_clk_irqlock = SPIN_LOCK_UNLOCKED;
#endif /* ARCADYAN */
 
#if defined(CONFIG_BCM_HOSTMIPS_PWRSAVE)
/* To put CPU in ASYNC mode and change CPU clock speed */
void __BcmPwrMngtSetASCR(unsigned int freq_div)
{
	register unsigned int temp;
	
	if (freq_div == RATIO_ONE_ASYNC) {
		// Gradually bring the processor speed back to 1:1
		// If it is done in one step, CP0 timer interrupts are missed.

		// E/ SYNC instruction   // Step E SYNC instruction  
		asm("sync" : : );
		
		// Step F1 change to 1/4
		asm("mfc0 %0,$22,5" : "=d"(temp) :);
		temp = ( temp & ~MASK_ASCR) | (RATIO_ONE_QUARTER << MASK_ASCR_SHFT);
		asm("mtc0 %0,$22,5" : : "d" (temp));

		// Step F2 change to 1/2
		temp = ( temp & ~MASK_ASCR) | (RATIO_ONE_HALF << MASK_ASCR_SHFT);
		asm("mtc0 %0,$22,5" : : "d" (temp));

		// Step F3 change to 1/1, high performance memory access
		temp = ( temp & ~MASK_ASCR);
		asm("mtc0 %0,$22,5" : : "d" (temp));

	} else {
		// E/ SYNC instruction   // Step E SYNC instruction 
		asm("sync" : : );

		// F/ change to 1/2, or 1/4, or 1/8 by setting cp0 sel 5 bits[30:28] (sel 4 bits[24:22] for single core mips)  to 011, 101, or 111 respectively
		// Step F change to 1/2, or 1/4, or 1/8 by setting cp0 bits[30:28]
		asm("mfc0 %0,$22,5" : "=d"(temp) :);
		temp = ( temp & ~MASK_ASCR) | (freq_div << MASK_ASCR_SHFT);
		asm("mtc0 %0,$22,5" : : "d" (temp));
	}

   return;
} /* BcmPwrMngtSetASCR */

void BcmPwrMngtSetASCR(unsigned int freq_div)
{
   unsigned long flags;

   if (!freq_div) {
      // Can't use this function to set to SYNC mode
      return;
   }

   spin_lock_irqsave(&pwrmgnt_clk_irqlock, flags);
   __BcmPwrMngtSetASCR(freq_div);
   spin_unlock_irqrestore(&pwrmgnt_clk_irqlock, flags);
   return;
} /* BcmPwrMngtSetASCR */
EXPORT_SYMBOL(BcmPwrMngtSetASCR);

/* To put CPU in SYNC mode and change CPU clock speed to 1:1 ratio */
/* No SYNC mode in newer MIPS core, use the __BcmPwrMngtSetASCR with ratio 1:1 instead */
void __BcmPwrMngtSetSCR(void)
{
   register unsigned int cp0_ascr_asc;

   // It is important to go back to divide by 1 async mode first, don't jump directly from divided clock back to SYNC mode.
   // A/ set cp0 reg 22 sel 5 bits[30:28]  (sel 4 bits[24:22] for single core mips)  to 001
   asm("mfc0 %0,$22,5" : "=d"(cp0_ascr_asc) :);
   if (!originalMipsAscrChecked) {
      originalMipsAscr = cp0_ascr_asc & MASK_ASCR;
      originalMipsAscrChecked = 1;
   }
   if (originalMipsAscr)
      return;
   cp0_ascr_asc = ( cp0_ascr_asc & ~MASK_ASCR) | (RATIO_ONE_ASYNC << MASK_ASCR_SHFT);
   asm("mtc0 %0,$22,5" : : "d" (cp0_ascr_asc));

   // B/ 16 nops // Was 32 nops (wait a while to make sure clk is back to full speed)
   asm("nop" : : ); asm("nop" : : );
   asm("nop" : : ); asm("nop" : : ); 
   asm("nop" : : ); asm("nop" : : );
   asm("nop" : : ); asm("nop" : : );
   asm("nop" : : ); asm("nop" : : );
   asm("nop" : : ); asm("nop" : : );
   asm("nop" : : ); asm("nop" : : );
   asm("nop" : : ); asm("nop" : : );

   // C/ SYNC instruction
   asm("sync" : : );

   // D/ set cp0 reg 22 sel 5 bits[30:28]  (sel 4 bits[24:22] for single core mips)  to 000
   asm("mfc0 %0,$22,5" : "=d"(cp0_ascr_asc) :);
   cp0_ascr_asc = ( cp0_ascr_asc & ~MASK_ASCR);
   asm("mtc0 %0,$22,5" : : "d" (cp0_ascr_asc));

   // E/ SYNC instruction 
   asm("sync" : : );

   return;
} /* BcmPwrMngtSetSCR */

void BcmPwrMngtSetSCR(void)
{
   unsigned long flags;

   spin_lock_irqsave(&pwrmgnt_clk_irqlock, flags);
   __BcmPwrMngtSetSCR();
   spin_unlock_irqrestore(&pwrmgnt_clk_irqlock, flags);

   return;
} /* BcmPwrMngtSetSCR */
EXPORT_SYMBOL(BcmPwrMngtSetSCR);

void BcmPwrMngtSetAutoClkDivide(unsigned int enable)
{
   printk("Host MIPS Clock divider pwrsaving is %s\n", enable ? "enabled" : "disabled");
   clock_divide_enabled = enable;
}
EXPORT_SYMBOL(BcmPwrMngtSetAutoClkDivide);

int BcmPwrMngtGetAutoClkDivide(void)
{
   return (clock_divide_enabled);
}
EXPORT_SYMBOL(BcmPwrMngtGetAutoClkDivide);
#endif

#if defined(CONFIG_BCM_DDR_SELF_REFRESH_PWRSAVE)
void BcmPwrMngtSetDRAMSelfRefresh(unsigned int enable)
{
   printk("DDR Self Refresh pwrsaving is %s\n", enable ? "enabled" : "disabled");
   self_refresh_enabled = enable;

#if defined(CONFIG_BCM_DDR_SELF_REFRESH_PWRSAVE) && defined(CONFIG_USB)
   if (0xd0 == bcm63268_get_cpu_rev()) {
   		u32 reg32;

		reg32 = bcm_usbh_readl(USB_CTRL_SIMCTL_REG);
		if (enable) {
			// Configure USB port to not access DDR if unused, to save power
			reg32 |= USB_CTRL_SIMCTL_MEM_REQ_DIS;
		} else {
			reg32 &= ~USB_CTRL_SIMCTL_MEM_REQ_DIS;
		}
		bcm_usbh_writel(reg32, USB_CTRL_SIMCTL_REG);
   }
#endif
}
EXPORT_SYMBOL(BcmPwrMngtSetDRAMSelfRefresh);

int BcmPwrMngtGetDRAMSelfRefresh(void)
{
   return (self_refresh_enabled);
}
EXPORT_SYMBOL(BcmPwrMngtGetDRAMSelfRefresh);

#if defined(CONFIG_BCM_ADSL_MODULE) || defined(CONFIG_BCM_ADSL)
PWRMNGT_DDR_SR_CTRL *pDdrSrCtrl = NULL;
void BcmPwrMngtRegisterLmemAddr(PWRMNGT_DDR_SR_CTRL *pDdrSr)
{
    pDdrSrCtrl = pDdrSr;

	// Initialize tp0 to busy status and tp1 to idle
	// for cases where SMP is not compiled in.
	if(NULL != pDdrSrCtrl) {
		pDdrSrCtrl->tp0Busy = 1;
		pDdrSrCtrl->tp1Busy = 0;
	}
}
EXPORT_SYMBOL(BcmPwrMngtRegisterLmemAddr);
#else
PWRMNGT_DDR_SR_CTRL ddrSrCtl = {{.word=0}};
PWRMNGT_DDR_SR_CTRL *pDdrSrCtrl = &ddrSrCtl;
#endif
#endif

#ifdef ARCADYAN
void BcmPwrMngtSetGlobal(unsigned int enable)
{
	printk("System Power Management is now %s\n", enable ? "enabled" : "disabled");
	power_management_enabled = enable;
}
EXPORT_SYMBOL(BcmPwrMngtSetGlobal);

int BcmPwrMngtGetGlobal(void)
{
	return (power_management_enabled);
}
EXPORT_SYMBOL(BcmPwrMngtGetGlobal);
#endif /* ARCADYAN */


// Determine if cpu is busy by checking the number of times we entered the wait
// state in the last milisecond. If we entered the wait state only once or
// twice, then the processor is very likely not busy and we can afford to slow
// it down while on wait state. Otherwise, we don't slow down the processor
// while on wait state in order to avoid affecting the time it takes to
// process interrupts
void BcmPwrMngtCheckWaitCount (void)
{
    int cpu = smp_processor_id();

    if (cpu == 0) {
        if (isVoiceIdle) {
           if (wait_count0 > 0 && wait_count0 < 3) {
              clock_divide_low_power0 = 1;
           }
           else {
              clock_divide_low_power0 = 0;
           }
        }
        else {
           clock_divide_low_power0 = 0;
        }
        wait_count0 = 0;
    }
#if defined(CONFIG_SMP)
    else {
		if (wait_count1 > 0 && wait_count1 < 3) {
			clock_divide_low_power1 = 1;
		}
		else {
			clock_divide_low_power1 = 0;
		}
		wait_count1 = 0;
    }
#endif /* CONFIG_SMP */
}

// When entering wait state, consider reducing the MIPS clock speed.
// Clock speed is reduced if it has been determined that the cpu was
// mostly idle in the previous milisecond. Clock speed is reduced only
// once per 1 milisecond interval.
void BcmPwrMngtReduceCpuSpeed (void)
{
    int cpu = smp_processor_id();
    unsigned long flags;

    spin_lock_irqsave(&pwrmgnt_clk_irqlock, flags);

	if (!power_management_enabled) {
		spin_unlock_irqrestore(&pwrmgnt_clk_irqlock, flags);
		return;
	}

    if (cpu == 0) {
        // Slow down the clock when entering wait instruction
        // only if the cpu is not busy
        if (clock_divide_low_power0) {
            if (wait_count0 < 2) {
                clock_divide_active0 = 1;
#if defined(CONFIG_BCM_DDR_SELF_REFRESH_PWRSAVE)
                if (pDdrSrCtrl && self_refresh_enabled) {
                    // Communicate TP status to PHY MIPS
                    pDdrSrCtrl->tp0Busy = 0;
            	}
#endif
            }
        }
        wait_count0++;
    }
#if defined(CONFIG_SMP)
    else {
        // Slow down the clock when entering wait instruction
        // only if the cpu is not busy
        if (clock_divide_low_power1) {
            if (wait_count1 < 2) {
                clock_divide_active1 = 1;
#if defined(CONFIG_BCM_DDR_SELF_REFRESH_PWRSAVE)
                if (pDdrSrCtrl && self_refresh_enabled) {
                    // Communicate TP status to PHY MIPS
                    pDdrSrCtrl->tp1Busy = 0;
            	}
#endif
            }
        }
        wait_count1++;
    }
#endif

#if defined(CONFIG_SMP)
    if (clock_divide_active0 && clock_divide_active1) {
#else
    if (clock_divide_active0) {
#endif
#if defined(CONFIG_BCM_HOSTMIPS_PWRSAVE)
        if (clock_divide_enabled) {
            __BcmPwrMngtSetASCR(RATIO_ONE_EIGHTH);
		}
#endif

#if defined(CONFIG_BCM_DDR_SELF_REFRESH_PWRSAVE)
        // Place DDR in self-refresh mode if enabled and other processors are OK with it
		if (pDdrSrCtrl && !pDdrSrCtrl->word && self_refresh_enabled) {
			bcm_ddr_writel((bcm_ddr_readl(DDR_DRAM_CFG_REG) | DDR_DRAM_CFG_SLEEP), DDR_DRAM_CFG_REG);
		}
#endif
    }

    spin_unlock_irqrestore(&pwrmgnt_clk_irqlock, flags);
}

// Full MIPS clock speed is resumed on the first interrupt following
// the wait instruction. If the clock speed was reduced, the MIPS
// C0 counter was also slowed down and its value needs to be readjusted.
// The adjustments are done based on a reliable timer from the peripheral
// block, timer2. The adjustments are such that C0 will never drift
// but will see minor jitter.
void BcmPwrMngtResumeFullSpeed (void)
{
    int cpu = smp_processor_id();
    unsigned long flags;

	spin_lock_irqsave(&pwrmgnt_clk_irqlock, flags);

	if (!power_management_enabled) {
		spin_unlock_irqrestore(&pwrmgnt_clk_irqlock, flags);
		return;
	}

#if defined(CONFIG_BCM_DDR_SELF_REFRESH_PWRSAVE)
	if (pDdrSrCtrl) {
        // Communicate TP status to PHY MIPS
        // Here I don't check if Self-Refresh is enabled because when it is,
        // I want PHY MIPS to think the Host MIPS is always busy so it won't assert SR
        if (cpu == 0) {
            pDdrSrCtrl->tp0Busy = 1;
        } else {
            pDdrSrCtrl->tp1Busy = 1;
        }
    }
#endif


#if defined(CONFIG_BCM_HOSTMIPS_PWRSAVE)

#if defined(CONFIG_SMP)
    if (clock_divide_enabled && clock_divide_active0 && clock_divide_active1) {
#else
    if (clock_divide_enabled && clock_divide_active0) {
#endif
        // In newer MIPS core, there is no SYNC mode, simply use 1:1 async
        __BcmPwrMngtSetASCR(RATIO_ONE_ASYNC);
    }
#endif

    // On chips not requiring the PERIPH Timers workaround,
    // only need to clear the active flags, no need to adjust timers
    if (cpu == 0) {
       clock_divide_active0 = 0;
    }
#if defined(CONFIG_SMP)
    else {
       clock_divide_active1 = 0;
    }
#endif
    spin_unlock_irqrestore(&pwrmgnt_clk_irqlock, flags);
}

