/*
<:copyright-gpl 
 Copyright 2012 Arcadyan Technology 
 All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <asm/irq_cpu.h>
#include <asm/mipsregs.h>
#include <bcm63268_cpu.h>
#include <bcm63268_regs.h>
#include <bcm63268_io.h>
#include <bcm63268_irq.h>


#if defined(CONFIG_SMP)
    #define AFFINITY_OF(d) (*(d)->affinity)
#else
    #define AFFINITY_OF(d) ((void)(d), CPU_MASK_CPU0)
#endif

typedef struct bcm63268_irq_ctrl_s {
    u64 ext_irq_mask;
    u64 irq_mask;
    u64 ext_irq_status;
    u64 irq_status;
} bcm63268_irq_ctrl_t;

static volatile bcm63268_irq_ctrl_t *bcm63268_irq_ctrl[NR_CPUS];
spinlock_t bcm63268_irqlock;

#if defined(CONFIG_SMP)
extern DEFINE_PER_CPU(unsigned int, bcm63268_ipi_pending);
#endif

/*
 * dispatch internal devices IRQ (uart, enet, watchdog, ...). do not
 * prioritize any interrupt relatively to another. the static counter
 * will resume the loop where it ended the last time we left this
 * function.
 */
static void bcm63268_irq_dispatch_internal(void)
{
	int cpu = smp_processor_id();
	u64 pending, ext_pending;
	u32 extirq_cfg;
	static int i, j;
	int to_call;

	spin_lock(&bcm63268_irqlock);
	pending = bcm63268_irq_ctrl[cpu]->irq_status & bcm63268_irq_ctrl[cpu]->irq_mask;
	ext_pending = bcm63268_irq_ctrl[cpu]->ext_irq_status & bcm63268_irq_ctrl[cpu]->ext_irq_mask;
	spin_unlock(&bcm63268_irqlock);

	if (pending) {
		while (1) {
			to_call = i;

			i = (i + 1) & 0x3f;
			if (pending & ((u64)1 << to_call)) {
				spin_lock(&bcm63268_irqlock);
				if (((to_call + IRQ_INTERNAL_BASE) >= IRQ_EXT_0) && 
					((to_call + IRQ_INTERNAL_BASE) <= IRQ_EXT_3)) {
					extirq_cfg = bcm_perf_readl(PERF_EXTIRQ_CFG_REG);
					extirq_cfg |= EXTIRQ_CFG_CLEAR(to_call - IRQ_EXT_0);
					bcm_perf_writel(extirq_cfg, PERF_EXTIRQ_CFG_REG);
				}
				spin_unlock(&bcm63268_irqlock);
				do_IRQ(to_call + IRQ_INTERNAL_BASE);
				break;
			}
		}
	}

	if (ext_pending) {
		while (1) {
			to_call = j;

			j = (j + 1) & 0x3f;
			if (ext_pending & ((u64)1 << to_call)) {
				do_IRQ(to_call + (IRQ_INTERNAL_BASE + 64));
				break;
			}
		}
	}
}

#ifdef CONFIG_BCM_HOSTMIPS_PWRSAVE
extern void BcmPwrMngtResumeFullSpeed (void);
#endif

asmlinkage void plat_irq_dispatch(void)
{
	u32 cause;

#ifdef CONFIG_BCM_HOSTMIPS_PWRSAVE
    BcmPwrMngtResumeFullSpeed();
#endif

	while ((cause = (read_c0_cause() & read_c0_status() & ST0_IM))) {
		if (cause & CAUSEF_IP7)
			do_IRQ(7);
#if defined(CONFIG_SMP)
		else if ((cause & CAUSEF_IP2) || (cause & CAUSEF_IP3))
#else
		else if (cause & CAUSEF_IP2)
#endif
			bcm63268_irq_dispatch_internal();
		else if (cause & CAUSEF_IP0)
			do_IRQ(0);
		else if (cause & CAUSEF_IP1)
			do_IRQ(1);
	}
}

void enable_brcm_irq(unsigned int irq)
{
    int cpu;
    struct irq_desc *desc = irq_desc + irq;
    unsigned long flags;
	u32 extcfg;
	int temp = irq - IRQ_INTERNAL_BASE;

    spin_lock_irqsave(&bcm63268_irqlock, flags);

	if (irq >= IRQ_INTERNAL_BASE &&
		irq < (IRQ_INTERNAL_BASE + 64)) {
		for_each_cpu_mask(cpu, AFFINITY_OF(desc)) {
			bcm63268_irq_ctrl[cpu]->irq_mask |= (((u64)1) << temp);
		}
	} else if (irq >= (IRQ_INTERNAL_BASE + 64) &&
		irq < (IRQ_INTERNAL_BASE + 128)) {
		for_each_cpu_mask(cpu, AFFINITY_OF(desc)) {
			bcm63268_irq_ctrl[cpu]->ext_irq_mask |= (((u64)1) << temp);
		}
	} else if ((irq == IRQ_MIPS_SOFT0) || (irq == IRQ_MIPS_SOFT1)) {
        set_c0_status(0x1 << (STATUSB_IP0 + irq - IRQ_MIPS_SOFT0));
    }

    if ((irq >= IRQ_EXT_0) && (irq <= IRQ_EXT_3)) {
		extcfg = bcm_perf_readl(PERF_EXTIRQ_CFG_REG);
		extcfg &= ~(EXTIRQ_CFG_BOTHEDGE(irq - IRQ_EXT_0));
		bcm_perf_writel(extcfg, PERF_EXTIRQ_CFG_REG);
		extcfg = bcm_perf_readl(PERF_EXTIRQ_CFG_REG);
		extcfg |= EXTIRQ_CFG_LEVELSENSE(irq - IRQ_EXT_0);
		bcm_perf_writel(extcfg, PERF_EXTIRQ_CFG_REG);
		extcfg = bcm_perf_readl(PERF_EXTIRQ_CFG_REG);
		extcfg &= ~(EXTIRQ_CFG_SENSE(irq - IRQ_EXT_0));
		bcm_perf_writel(extcfg, PERF_EXTIRQ_CFG_REG);
		extcfg = bcm_perf_readl(PERF_EXTIRQ_CFG_REG);
		extcfg |= EXTIRQ_CFG_CLEAR(irq - IRQ_EXT_0);
		bcm_perf_writel(extcfg, PERF_EXTIRQ_CFG_REG);
		extcfg = bcm_perf_readl(PERF_EXTIRQ_CFG_REG);
		extcfg |= EXTIRQ_CFG_MASK(irq - IRQ_EXT_0);
		bcm_perf_writel(extcfg, PERF_EXTIRQ_CFG_REG);
    }

    spin_unlock_irqrestore(&bcm63268_irqlock, flags);
}

void __disable_ack_brcm_irq(unsigned int irq)
{
    int cpu;
    struct irq_desc *desc = irq_desc + irq;
	int temp = irq - IRQ_INTERNAL_BASE;

    if (irq >= IRQ_INTERNAL_BASE &&
		irq < (IRQ_INTERNAL_BASE + 64)) {
        for_each_cpu_mask(cpu, AFFINITY_OF(desc)) {
			bcm63268_irq_ctrl[cpu]->irq_mask &= ~(((u64)1) << temp);
        }
    } else if (irq >= (IRQ_INTERNAL_BASE + 64) &&
		irq < (IRQ_INTERNAL_BASE + 128)) {
		for_each_cpu_mask(cpu, AFFINITY_OF(desc)) {
			bcm63268_irq_ctrl[cpu]->ext_irq_mask &= ~(((u64)1) << temp);
		}
	} 
}


void disable_brcm_irq(unsigned int irq)
{
    unsigned long flags;

    spin_lock_irqsave(&bcm63268_irqlock, flags);
    __disable_ack_brcm_irq(irq);
    if ((irq == IRQ_MIPS_SOFT0) || (irq == IRQ_MIPS_SOFT1)) {
        clear_c0_status(0x1 << (STATUSB_IP0 + irq - IRQ_MIPS_SOFT0));
    }
    spin_unlock_irqrestore(&bcm63268_irqlock, flags);
}


void ack_brcm_irq(unsigned int irq)
{
    unsigned long flags;

    spin_lock_irqsave(&bcm63268_irqlock, flags);
    __disable_ack_brcm_irq(irq);

#if defined(CONFIG_SMP)
    if (irq == IRQ_MIPS_SOFT0) {
        int this_cpu = smp_processor_id();
        int other_cpu = !this_cpu;
        per_cpu(bcm63268_ipi_pending, this_cpu) = 0;
        mb();
        clear_c0_cause(1<<CAUSEB_IP0);
        if (per_cpu(bcm63268_ipi_pending, other_cpu)) {
            set_c0_cause(1<<CAUSEB_IP0);
        }
    }
#else
    if (irq == IRQ_MIPS_SOFT0) {
        clear_c0_cause(1<<CAUSEB_IP0);
    }
#endif

    if (irq == IRQ_MIPS_SOFT1) {
        clear_c0_cause(1<<CAUSEB_IP1);
    }

    spin_unlock_irqrestore(&bcm63268_irqlock, flags);
}


void mask_ack_brcm_irq(unsigned int irq)
{
    unsigned long flags;

    spin_lock_irqsave(&bcm63268_irqlock, flags);
    __disable_ack_brcm_irq(irq);

#if defined(CONFIG_SMP)
    if (irq == IRQ_MIPS_SOFT0) {
        int this_cpu = smp_processor_id();
        int other_cpu = !this_cpu;
        per_cpu(bcm63268_ipi_pending, this_cpu) = 0;
        mb();
        clear_c0_cause(1<<CAUSEB_IP0);
        if (per_cpu(bcm63268_ipi_pending, other_cpu)) {
            set_c0_cause(1<<CAUSEB_IP0);
        }
        clear_c0_status(1<<STATUSB_IP0);
    }
#else
    if (irq == IRQ_MIPS_SOFT0) {
        clear_c0_status(1<<STATUSB_IP0);
        clear_c0_cause(1<<CAUSEB_IP0);
    }
#endif

    if (irq == IRQ_MIPS_SOFT1) {
        clear_c0_status(1<<STATUSB_IP1);
        clear_c0_cause(1<<CAUSEB_IP1);
    }

    spin_unlock_irqrestore(&bcm63268_irqlock, flags);
}


void unmask_brcm_irq_noop(unsigned int irq)
{
}

void set_brcm_affinity(unsigned int irq, const struct cpumask *mask)
{
    int cpu;
    struct irq_desc *desc = irq_desc + irq;
    unsigned long flags;
	int temp = irq - IRQ_INTERNAL_BASE;

    spin_lock_irqsave(&bcm63268_irqlock, flags);

    if (irq >= IRQ_INTERNAL_BASE &&
		irq < (IRQ_INTERNAL_BASE + 64)) {
        for_each_online_cpu(cpu) {
            if (cpu_isset(cpu, *mask) && !(desc->status & IRQ_DISABLED)) {
                bcm63268_irq_ctrl[cpu]->irq_mask |= (((u64)1) << temp);
            }
            else {
                bcm63268_irq_ctrl[cpu]->irq_mask &= ~(((u64)1) << temp);
            }
        }
    } else if (irq >= (IRQ_INTERNAL_BASE + 64) &&
		irq < (IRQ_INTERNAL_BASE + 128)) {
        for_each_online_cpu(cpu) {
            if (cpu_isset(cpu, *mask) && !(desc->status & IRQ_DISABLED)) {
                bcm63268_irq_ctrl[cpu]->ext_irq_mask |= (((u64)1) << temp);
            }
            else {
                bcm63268_irq_ctrl[cpu]->ext_irq_mask &= ~(((u64)1) << temp);
            }
        }
    }

    spin_unlock_irqrestore(&bcm63268_irqlock, flags);
}

#define ALLINTS_NOTIMER IE_IRQ0

static struct irq_chip bcm63268_irq_chip = {
    .name = "bcm63268_ipic",
    .enable = enable_brcm_irq,
    .disable = disable_brcm_irq,
    .ack = ack_brcm_irq,
    .mask = disable_brcm_irq,
    .mask_ack = mask_ack_brcm_irq,
    .unmask = enable_brcm_irq,
    .set_affinity = set_brcm_affinity
};

static struct irq_chip bcm63268_irq_chip_no_unmask = {
    .name = "bcm63268_ipic_n",
    .enable = enable_brcm_irq,
    .disable = disable_brcm_irq,
    .ack = ack_brcm_irq,
    .mask = disable_brcm_irq,
    .mask_ack = mask_ack_brcm_irq,
    .unmask = unmask_brcm_irq_noop,
    .set_affinity = set_brcm_affinity
};

#if 0
static struct irqaction cpu_ip2_cascade_action = {
	.handler	= no_action,
	.name		= "cascade_ip2",
};

#ifdef CONFIG_SMP
static struct irqaction cpu_ip3_cascade_action = {
	.handler	= no_action,
	.name		= "cascade_ip3",
};
#endif /* CONFIG_SMP */
#endif

void __init arch_init_irq(void)
{
    int i;

#ifdef CONFIG_SMP
	/* Set the default affinity to the boot cpu. */
	cpumask_clear(irq_default_affinity);
	cpumask_set_cpu(smp_processor_id(), irq_default_affinity);
#endif /* CONFIG_SMP */

	spin_lock_init(&bcm63268_irqlock);

	for (i = 0; i < NR_CPUS; i++) {
        bcm63268_irq_ctrl[i] = (bcm63268_irq_ctrl_t *)(0xb0000000 + (0x20 * (i + 1)));
    }

	mips_cpu_irq_init();
	set_irq_chip_and_handler(IRQ_MIPS_SOFT0, &bcm63268_irq_chip,
						 handle_level_irq);
	set_irq_chip_and_handler(IRQ_MIPS_SOFT1, &bcm63268_irq_chip,
						 handle_level_irq);
	for (i = IRQ_INTERNAL_BASE; i < NR_IRQS; ++i)
		set_irq_chip_and_handler(i, &bcm63268_irq_chip,
					 handle_level_irq);

#if 0
	setup_irq(IRQ_MIPS_BASE + 2, &cpu_ip2_cascade_action);
#ifdef CONFIG_SMP
	setup_irq(IRQ_MIPS_BASE + 3, &cpu_ip3_cascade_action);
#endif /* ARCADYAN */
#endif

	clear_c0_status(ST0_BEV);

#ifdef CONFIG_SMP
	change_c0_status(ST0_IM, IE_IRQ0|IE_IRQ1);
#else
	change_c0_status(ST0_IM, IE_IRQ0);
#endif /* CONFIG_SMP */

#ifdef CONFIG_REMOTE_DEBUG
    rs_kgdb_hook(0);
#endif
}


// This is a wrapper to standand Linux request_irq
// Differences are:
//    - The irq won't be renabled after ISR is done and needs to be explicity re-enabled, which is good for NAPI drivers.
//      The change is implemented by filling in an no-op unmask function in brcm_irq_chip_no_unmask and set it as the irq_chip
//    - IRQ flags and interrupt names are automatically set
// Either request_irq or BcmHalMapInterrupt can be used. Just make sure re-enabling IRQ is handled correctly.

unsigned int BcmHalMapInterrupt(FN_HANDLER pfunc, unsigned int param, unsigned int irq)
{
    char *devname;
    unsigned long irqflags;
	unsigned int rv;

    devname = kmalloc(16, GFP_KERNEL);
    if (!devname) {
        return -1;
    }
	
    sprintf(devname, "brcm_%d", irq);

	/* If this is for the timer interrupt, do not invoke the following code
	   because doing so kills the timer interrupts that may already be running */
	if (irq != bcm63268_get_irq_number(IRQ_TIMER))
    	set_irq_chip_and_handler(irq, &bcm63268_irq_chip_no_unmask, handle_level_irq);

    irqflags = IRQF_DISABLED | IRQF_SAMPLE_RANDOM;

    if (irq == bcm63268_get_irq_number(IRQ_TIMER)) {
        irqflags |= IRQF_SHARED;
    }

    rv = request_irq(irq, (void*)pfunc, irqflags, devname, (void *) param);

	if (rv != 0) {
		printk(KERN_WARNING "request_irq failed for irq=%d (%s) rv=%d\n", irq, devname, rv);
		kfree(devname);
	}

	return rv;
}

// This is a wrapper to standand Linux request_irq for the VOIP driver
// Differences are:
// The irq is not automatically enabled when the ISR is registered.
// The irq is automatically re-enabled when the ISR is done.
// Interrupts are re-enabled when the ISR is invoked.
unsigned int BcmHalMapInterruptVoip(FN_HANDLER pfunc, unsigned int param, unsigned int irq)
{
    char *devname;
    unsigned long irqflags;
    unsigned long flags;
    struct irq_desc *desc = irq_desc + irq;
	
    devname = kmalloc(16, GFP_KERNEL);
    if (!devname) {
        return -1;
    }
    sprintf( devname, "brcm_%d", irq );

    set_irq_chip_and_handler(irq, &bcm63268_irq_chip, handle_level_irq);

    spin_lock_irqsave(&desc->lock, flags);   
    desc->status |= IRQ_NOAUTOEN;
    desc->status &= ~IRQ_DISABLED;
    spin_unlock_irqrestore(&desc->lock, flags);

    irqflags = IRQF_SAMPLE_RANDOM;

	if (irq == IRQ_INTERNAL_BASE + 13) {
        irqflags |= IRQF_SHARED;
    }

    return request_irq(irq, (void*)pfunc, irqflags, devname, (void *) param);
}

//***************************************************************************
//  void  BcmHalGenerateSoftInterrupt
//
//   Triggers a software interrupt.
//
//***************************************************************************
void BcmHalGenerateSoftInterrupt( unsigned int irq )
{
    unsigned long flags;

    local_irq_save(flags);

    set_c0_cause(0x1 << (CAUSEB_IP0 + irq - IRQ_MIPS_SOFT0));

    local_irq_restore(flags);
}


EXPORT_SYMBOL(enable_brcm_irq);
EXPORT_SYMBOL(disable_brcm_irq);
EXPORT_SYMBOL(BcmHalMapInterrupt);
EXPORT_SYMBOL(BcmHalMapInterruptVoip);
EXPORT_SYMBOL(BcmHalGenerateSoftInterrupt);

